# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import difflib
import io
import sys

from TypeFuture import TacLazyType
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
from CliToken.Flow import flowMatcherForConfig, flowMatcherForClear
from CliToken.Clear import clearKwNode
import Tracing
from FlowTrackerConst import (
      hoHeuristicsConstants,
      sampleRate,
      sampleLimit,
      spaceConst,
)
from FlowTrackerCliUtil import (
      encapTypeKwStr,
      encapIpv4,
      encapIpv6,
      encapMpls,
      encapBase,
      filterRpf,
      encapMplsSupportedForType,
      encapVxlanSupportedForType,
      encapGreSupportedForType,
      ftrConfigReqPathPrefix,
      ftrConfigType,
      ftrMirrorOnDropConfigType,
      ftrConfigReqType,
      ftrTypeByName,
      ftrTypes,
      ftrTypeKwStr,
      ftrTypeMirrorOnDrop,
      getFtrTypeFromArgs,
      getFlowTrackingCliConfigPath,
      showFlowTracking,
      showFlowTrackingTrailer,
)
from CliPlugin.FlowTrackingCliLib import (
      countersClearKw,
      cpuQueueClearKw,
      cpuQueueConfigKw,
      firewallConfigKw,
      distributedConfigKw,
      defaultSampleLimitOnPlatform,
      exporterKw,
      exporterNameMatcher,
      FlowTrackingMode,
      EncapFilterMode,
      SampledFlowTrackerCliModelet,
      guardFtrEncapsulation,
      guardFtrEncapFilter,
      guardFtrTrapSuppression,
      guardHwOffload,
      guardHwOffloadIpv4,
      guardHwOffloadIpv6,
      guardRecordUpdate,
      guardRewriteDscp,
      guardSampleLimit,
      guardSampleRate,
      guardSwExport,
      guardPacketBufferSamplingProb,
      guardPacketBufferShaper,
      hardwareConfigKw,
      hardwareClearKw,
      firewallClearKw,
      distributedClearKw,
      inbandClearKw,
      inbandConfigKw,
      matcherRecord,
      mirrorOnDropClearKw,
      mirrorOnDropConfigKw,
      packetBufferKw,
      packetBufferDropKw,
      packetBufferSampleProbKw25,
      packetBufferSampleProbKw50,
      packetBufferSampleProbKw75,
      packetBufferSampleProbKw100,
      packetBufferQueueUsageMatcher,
      getMplsHwCaps,
      getBridgingHwCaps,
      getFtrCaps,
      sampledClearKw,
      sampledConfigKw,
      sampleRateRangeFn,
      trackingClearKw,
      trackingConfigKw,
      trackerKw,
      trackerNameMatcher,
      telemetryClearKw,
      telemetryConfigKw,
      modDefaultTrapSuppression,
)
from CliPlugin.FlowTrackerCli import (
      AbortCmd,
      TrackerContext,
      showActiveTracker,
      showPendingTracker,
)
from ShowCommand import ShowCliCommandClass
import Tac
from Toggles.FlowTrackerToggleLib import toggleSftBgpExportCliEnabled

traceHandle = Tracing.Handle( 'FlowTrackingCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

trackingConfig = {}
trackingConfigReq = {}
bridgingHwCapabilities = None
PacketBufferRate = TacLazyType( 'FlowTracking::PacketBufferRate' )
PacketBufferBurst = TacLazyType( 'FlowTracking::PacketBufferBurst' )
SampleProbabilityType = TacLazyType( 'FlowTracking::SampleProbability' )

class FlowTrackingContext:
   def __init__( self, ftrType, config ):
      self.config = config
      self.ftrType = ftrType
      self.enabled = self.config.enabled
      self.bgpExport = self.config.bgpExport
      self.sampleRate = self.config.sampleRate
      self.sampleLimit = self.config.sampleLimit
      self.sampleSize = self.config.sampleSize
      self.swExport = self.config.swExport
      self.encap = list( self.config.encap )
      self.filter = list( self.config.filter )
      self.hwOffloadIpv4 = self.config.hwOffloadIpv4
      self.hwOffloadIpv6 = self.config.hwOffloadIpv6
      self.minSamplesBeforeHwOffload = self.config.minSamplesBeforeHwOffload
      self.rewriteDscp = self.config.rewriteDscp
      self.trapSuppression = self.config.trapSuppression
      self.trCtx = {}
      self.packetBufferSampleProb = self.config.packetBufferSampleProb
      self.packetBufferSampleProb25 = self.config.packetBufferSampleProb25
      self.packetBufferSampleProb50 = self.config.packetBufferSampleProb50
      self.packetBufferSampleProb75 = self.config.packetBufferSampleProb75
      self.packetBufferSampleProb100 = self.config.packetBufferSampleProb100
      self.packetBufferShaperRate = self.config.packetBufferShaperRate
      self.packetBufferShaperBurst = self.config.packetBufferShaperBurst
      for trName, trConfig in self.config.flowTrackerConfig.items():
         self.trCtx[ trName ] = TrackerContext(
               self.config, self.ftrType, trName, trConfig )

   def commit( self ):
      self.config.sampleRate = self.sampleRate
      self.config.sampleLimit = self.sampleLimit
      self.config.sampleSize = self.sampleSize
      self.config.enabled = self.enabled
      self.config.swExport = self.swExport
      self.config.hwOffloadIpv4 = self.hwOffloadIpv4
      self.config.hwOffloadIpv6 = self.hwOffloadIpv6
      self.config.minSamplesBeforeHwOffload = self.minSamplesBeforeHwOffload

      staleEncap = set( self.config.encap )
      t1( 'staleEncap', staleEncap )
      for encap in self.encap:
         self.config.encap.add( encap )
         staleEncap.discard( encap )
      t1( 'staleEncap', staleEncap )
      for encap in staleEncap:
         self.config.encap.remove( encap )

      staleFilter = set( self.config.filter )
      t1( 'previous configured filters', staleFilter )
      for item in self.filter:
         self.config.filter.add( item )
         staleFilter.discard( item )
      t1( 'staleFilter', staleFilter )
      for item in staleFilter:
         self.config.filter.remove( item )

      self.config.rewriteDscp = self.rewriteDscp
      self.config.trapSuppression = self.trapSuppression
      self.config.packetBufferSampleProb = self.packetBufferSampleProb
      self.config.packetBufferSampleProb25 = self.packetBufferSampleProb25
      self.config.packetBufferSampleProb50 = self.packetBufferSampleProb50
      self.config.packetBufferSampleProb75 = self.packetBufferSampleProb75
      self.config.packetBufferSampleProb100 = self.packetBufferSampleProb100
      self.config.packetBufferShaperRate = self.packetBufferShaperRate
      self.config.packetBufferShaperBurst = self.packetBufferShaperBurst

   def newTrackerContext( self, trName ):
      # Config could be changed after entering flow-tracking mode.
      # Create context with existing Sysdb config.
      trConfig = self.config.flowTrackerConfig.get( trName )
      self.trCtx[ trName ] = TrackerContext(
                              self.config, self.ftrType, trName, trConfig )
      return self.trCtx[ trName ]

   def trackerContext( self, trName ):
      return self.trCtx.get( trName )

   def trackingConfig( self ):
      return self.config

   def encapsulationIs( self, mode, args ):
      setEncapStrs = args[ 'ENCAPS' ]
      t1( 'Flow tracking encapsulationIs ', setEncapStrs )
      encaps = list( self.encap )
      for encap, encapKwStr in encapTypeKwStr.items():
         if encapKwStr in setEncapStrs and encap not in encaps:
            encaps.append( encap )

      if ( encapIpv4 not in encaps or encapIpv6 not in encaps ):
         mode.addError( "Both ipv4 and ipv6 encapsulations must be set" )
      else:
         self.encap = encaps

   def encapsulationDel( self, mode, args ):
      delEncapStrs = args.get( 'ENCAPS', [] )
      t1( 'Flow tracking encapsulationDel ', delEncapStrs )
      if not delEncapStrs:
         self.encap = []
      else:
         encaps = list( self.encap )
         for encap, encapKwStr in encapTypeKwStr.items():
            if encapKwStr in delEncapStrs and encap in encaps:
               encaps.remove( encap )

         if encaps and ( encapIpv4 not in encaps or encapIpv6 not in encaps ):
            mode.addError( "Both ipv4 and ipv6 encapsulations must be set" )
         else:
            self.encap = encaps

   def encapsulationMirrorOnDropIs( self, mode, args ):
      setEncapStrs = args[ 'ENCAPS' ]
      t1( 'Flow tracking mirror-on-drop encapsulationIs ', setEncapStrs )
      encapKwIpv4 = encapTypeKwStr[ encapIpv4 ]
      encapKwIpv6 = encapTypeKwStr[ encapIpv6 ]
      encapKwMpls = encapTypeKwStr[ encapMpls ]
      defaultEncaps = [ encapKwIpv4, encapKwIpv6 ]
      mplsEncaps = [ encapKwMpls ]
      allEncaps = [ encapKwIpv4, encapKwIpv6, encapKwMpls ]

      if set( setEncapStrs ) == set( defaultEncaps ):
         self.encap = [ encapIpv4, encapIpv6 ]
      elif setEncapStrs == mplsEncaps:
         self.encap = [ encapMpls ]
      elif set( setEncapStrs ) == set( allEncaps ):
         self.encap = [ encapIpv4, encapIpv6, encapMpls ]
      else:
         mode.addError( "Both IPv4 and IPv6 encapsulations must be set" )

   def encapsulationMirrorOnDropDefault( self, mode, args ):
      t1( 'Flow tracking encapsulationMirrorOnDropDefault' )
      self.encap = [ encapIpv4, encapIpv6 ]

   def gotoEncapFilterMode( self, mode ):
      childMode = mode.childMode( EncapFilterMode, context=mode.context() )
      mode.session_.gotoChildMode( childMode )

   def noEncapFilterMode( self, mode ):
      self.filter.clear()

   def packetBufferSamplingProbIs( self, mode, args, probability ):
      if usage := args.get( 'QUEUE_USAGE' ):
         t1( 'packetBufferSamplingProbIs : ', usage, ', ', probability, '%' )
         if usage == packetBufferSampleProbKw25:
            self.packetBufferSampleProb25 = probability
         elif usage == packetBufferSampleProbKw50:
            self.packetBufferSampleProb50 = probability
         elif usage == packetBufferSampleProbKw75:
            self.packetBufferSampleProb75 = probability
         elif usage == packetBufferSampleProbKw100:
            self.packetBufferSampleProb100 = probability
      else:
         t1( 'packetBufferSamplingProbIs : ', probability, '%' )
         self.packetBufferSampleProb = probability

   def packetBufferSamplingProb( self, mode, args ):
      t1( 'packetBufferSamplingProb' )
      self.packetBufferSamplingProbIs( mode, args, args.get( 'PROBABILITY' ) )

   def packetBufferSamplingProbDefault( self, mode, args ):
      t1( 'packetBufferSamplingProbDefault' )
      probability = SampleProbabilityType.probabilityDefault
      if 'QUEUE_USAGE' in args:
         probability = None
      self.packetBufferSamplingProbIs( mode, args, probability )

#-------------------------------------------------------------------------------
# "[no|default] flow tracking ( sampled | hardware | (telemetry inband) |
#                               mirror-on-drop | cpu-queue |
#                               ( firewall distributed ) )"
# in config mode
#-------------------------------------------------------------------------------

def gotoFlowTrackingMode( mode, ftrType ):
   t1( 'gotoFlowTrackingMode ', ftrType )
   context = FlowTrackingContext( ftrType, trackingConfig[ ftrType ] )
   childMode = mode.childMode( FlowTrackingMode, context=context )
   mode.session_.gotoChildMode( childMode )

def noFlowTrackingMode( mode, ftrType ):
   t1( 'noFlowTrackingMode', ftrType )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, FlowTrackingMode ) and lastMode.context() ):
      # If no flow tracking is issued in flow-tracking mode then delete
      # context to avoid committing it again
      lastMode.context_ = None

   config = trackingConfig[ ftrType ]
   config.enabled = False
   config.bgpExport = True
   config.sampleRate = sampleRate.rateDefault
   # sampleLimit is an optional parameter. We need to
   # set it back to default.
   config.sampleLimit = None
   # sampleSize is an optional parameter. Reset it back to default
   config.sampleSize = None
   config.encap.clear()
   config.filter.clear()
   if ftrType == ftrTypeMirrorOnDrop:
      config.setDefaultEncapFlags()
   config.flowTrackerConfig.clear()
   config.swExport = False
   config.hwOffloadIpv4 = False
   config.hwOffloadIpv6 = False
   config.minSamplesBeforeHwOffload = hoHeuristicsConstants.minSamplesDefault
   config.rewriteDscp = False
   config.trapSuppression = None
   config.packetBufferSampleProb = SampleProbabilityType.probabilityDefault
   config.packetBufferSampleProb25 = None
   config.packetBufferSampleProb50 = None
   config.packetBufferSampleProb75 = None
   config.packetBufferSampleProb100 = None
   config.packetBufferShaperRate = PacketBufferRate.rateDefault
   config.packetBufferShaperBurst = PacketBufferBurst.burstDefault

class FlowTrackingConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''flow tracking
               ( sampled
               | hardware
               | ( telemetry inband )
               | mirror-on-drop
               | cpu-queue
               | ( firewall distributed ) )'''
   noOrDefaultSyntax = syntax

   data = {
         'flow' : flowMatcherForConfig,
         'tracking' : trackingConfigKw,
         'sampled' : sampledConfigKw,
         'hardware' : hardwareConfigKw,
         'telemetry' : telemetryConfigKw,
         'inband' : inbandConfigKw,
         'mirror-on-drop' : mirrorOnDropConfigKw,
         'cpu-queue' : cpuQueueConfigKw,
         'firewall' : firewallConfigKw,
         'distributed' : distributedConfigKw,
   }

   @staticmethod
   def handler( mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      gotoFlowTrackingMode( mode, ftrType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      noFlowTrackingMode( mode, ftrType )

BasicCli.GlobalConfigMode.addCommandClass( FlowTrackingConfigCmd )

#-------------------------------------------------------------------------------
# "clear flow tracking (sampled | hardware | (telemetry inband) |
#  mirror-on-drop | cpu-queue) counters [tracker <name> [ exporter <name> ] ]"
#-------------------------------------------------------------------------------

def clearFlowTrackingCounters( mode, args ):
   ftrType = getFtrTypeFromArgs( args )
   config = trackingConfig[ ftrType ]
   configReq = trackingConfigReq[ ftrType ]

   tracker = args.get( 'TRACKER_NAME', '' )
   exporter = args.get( 'EXPORTER_NAME', '' )
   if tracker:
      trConfig = config.flowTrackerConfig.get( tracker )
      if not trConfig:
         mode.addError( 'Tracker %s not found' % tracker )
         return
      if exporter:
         expConfig = trConfig.expConfig.get( exporter )
         if not expConfig:
            mode.addError( 'Exporter %s not found' % exporter )
            return

   configReq.clearCounters = Tac.Value( "FlowTracking::ClearCountersReq",
                                         tracker=tracker,
                                         exporter=exporter,
                                         reqTime=Tac.now() )

class FlowTrackingClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = '''clear flow tracking
               ( sampled
               | hardware
               | ( telemetry inband )
               | mirror-on-drop
               | ( firewall distributed ) )
               counters [ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'''
   data = {
      'clear' : clearKwNode,
      'flow' : flowMatcherForClear,
      'tracking' : trackingClearKw,
      'sampled' : sampledClearKw,
      'hardware' : hardwareClearKw,
      'firewall' : firewallClearKw,
      'distributed' : distributedClearKw,
      'telemetry' : telemetryClearKw,
      'inband' : inbandClearKw,
      'mirror-on-drop' : mirrorOnDropClearKw,
      'counters' : countersClearKw,
      'tracker' : trackerKw,
      'TRACKER_NAME' : trackerNameMatcher,
      'exporter' : exporterKw,
      'EXPORTER_NAME' : exporterNameMatcher,
   }
   handler = clearFlowTrackingCounters

BasicCli.EnableMode.addCommandClass( FlowTrackingClearCountersCmd )

class CpuQueueClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = '''clear flow tracking cpu-queue counters'''
   data = {
      'clear' : clearKwNode,
      'flow' : flowMatcherForClear,
      'tracking' : trackingClearKw,
      'cpu-queue' : cpuQueueClearKw,
      'counters' : countersClearKw,
   }
   handler = clearFlowTrackingCounters

BasicCli.EnableMode.addCommandClass( CpuQueueClearCountersCmd )

#-----------------------------------------------------------------
# "[no|default] sample <rate>" command in "config-flow-tracking" mode
#-----------------------------------------------------------------

class FtrSampleRateCmd( CliCommand.CliCommandClass ):
   syntax = 'sample RATE'
   noOrDefaultSyntax = 'sample ...'

   data =  {
      'sample' : CliCommand.guardedKeyword( 'sample',
                        helpdesc='Set sample characteristics for flow tracking',
                        guard=guardSampleRate ),
      'RATE' : CliMatcher.DynamicIntegerMatcher( sampleRateRangeFn,
                        helpdesc='Sampling rate' ),
   }
   @staticmethod
   def handler( mode, args ):
      mode.context().sampleRate = args.get( 'RATE', sampleRate.rateDefault )

   noOrDefaultHandler = handler

FlowTrackingMode.addCommandClass( FtrSampleRateCmd )

# ------------------------------------------------------------------------------
# "[no|default] sample limit ( N pps | disabled )" command in
# "config-flow-tracking" mode
# ------------------------------------------------------------------------------
def sampleLimitRangeFn( mode, context ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   ftrCaps = getFtrCaps( ftrType )
   return ( sampleLimit.minRate + 1,
            ftrCaps.maxSampleLimit if ftrCaps else sampleLimit.maxRate )

class FtrSampleLimit( CliCommand.CliCommandClass ):
   syntax = 'sample limit ( ( NUM pps ) | disabled )'
   noOrDefaultSyntax = 'sample limit ...'
   data = {
      'sample' : CliCommand.guardedKeyword( 'sample',
                        helpdesc='Set sample characteristics for flow tracking',
                        guard=guardSampleLimit ),
      'limit' : 'Limit the number of packets sampled',
      'NUM' : CliMatcher.DynamicIntegerMatcher(
                        rangeFn=sampleLimitRangeFn,
                        helpdesc='Number of packets to sample' ),
      'pps' : 'Packets per second',
      'disabled' : 'Disable sample limit',
   }
   @staticmethod
   def handler( mode, args ):
      defaultRate = defaultSampleLimitOnPlatform( mode )
      sampleRatePps = args.get( 'NUM', None )
      # 'sample limit disabled' is the knob used to turn off enforcing sample limit.
      if 'disabled' in args:
         sampleRatePps = 0
      # If the rate is same as default rate, we need to set sampleRate to optional.
      if sampleRatePps == defaultRate:
         sampleRatePps = None
      mode.context().sampleLimit = sampleRatePps

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Always set it back to nullopt. This would indicate to the downstream SMs
      # that they need to use default sampleRate capabilities of the platform.
      mode.context().sampleLimit = None

FlowTrackingMode.addCommandClass( FtrSampleLimit )

#-----------------------------------------------------------------
# "[no|default] encapsulation { ipv4 | ipv6 | mpls | vxlan | gre }" command in
# "config-flow-tracking" mode
#-----------------------------------------------------------------

def getFtrEncapOptions( mode, context ):
   options = dict( encapBase )
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   mplsSupported = encapMplsSupportedForType( ftrTypeByName[ mode.ftrTypeStr ],
                                              getMplsHwCaps(),
                                              getFtrCaps( ftrType ) )
   vxlanSupported = encapVxlanSupportedForType( ftrType, getBridgingHwCaps() )
   guardsEnabled = mode.session.guardsEnabled()
   greEnabled = encapGreSupportedForType( ftrType )
   t0( "getFtrEncapOptions, mplsSupported:", mplsSupported, "vxlanSupported",
       vxlanSupported, "guardsEnabled", guardsEnabled )
   if mplsSupported or not guardsEnabled:
      options[ 'mpls' ] = 'MPLS flows'
   if vxlanSupported or not guardsEnabled:
      options[ 'vxlan' ] = 'VXLAN flows'
   if greEnabled:
      options[ 'gre' ] = 'GRE flows'
   return options

class FtrEncapsulation( CliCommand.CliCommandClass ):
   syntax = 'encapsulation ( ENCAPS | filter )'
   noOrDefaultSyntax = syntax

   data = {
      'encapsulation' : CliCommand.guardedKeyword( 'encapsulation',
                        helpdesc='Packet encapsulation',
                        guard=guardFtrEncapsulation ),
      'ENCAPS' : CliCommand.SetEnumMatcher( getFtrEncapOptions ),
      'filter' : CliCommand.guardedKeyword( 'filter',
                 helpdesc='Configure encapsulation filter',
                 guard=guardFtrEncapFilter ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'filter' in args:
         mode.context().gotoEncapFilterMode( mode )
      elif mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeMirrorOnDrop ]:
         mode.context().encapsulationMirrorOnDropIs( mode, args )
      else:
         mode.context().encapsulationIs( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'filter' in args:
         mode.context().noEncapFilterMode( mode )
      elif mode.ftrTypeStr == ftrTypeKwStr[ ftrTypeMirrorOnDrop ]:
         mode.context().encapsulationMirrorOnDropDefault( mode, args )
      else:
         mode.context().encapsulationDel( mode, args )

FlowTrackingMode.addCommandClass( FtrEncapsulation )

class EncapFilters( CliCommand.CliCommandClass ):
   syntax = 'ipv4 ipv6 verify unicast source'
   noOrDefaultSyntax = syntax

   data = {
      'ipv4' : 'IPv4 Flows',
      'ipv6' : 'IPv6 Flows',
      'verify' : 'Source verification method',
      'unicast' : 'Unicast RPF configuration',
      'source' : 'RPF source verification',
   }

   @staticmethod
   def handler( mode, args ):
      if filterRpf not in mode.context().filter:
         mode.context().filter.append( filterRpf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if filterRpf in mode.context().filter:
         mode.context().filter.remove( filterRpf )

EncapFilterMode.addCommandClass( EncapFilters )

# -------------------------------------------------------------------------------
# "[no|default] information element bgp disabled" command in
# "config-flow-tracking-sampled" mode
# -------------------------------------------------------------------------------

def sftBgpExportDisabled( mode, args ):
   config = trackingConfig[ ftrTypeByName[ mode.ftrTypeStr ] ]
   config.bgpExport = False

def noOrDefaultSftBgpExportDisabled( mode, args ):
   config = trackingConfig[ ftrTypeByName[ mode.ftrTypeStr ] ]
   config.bgpExport = True

class SftBgpExportDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'information element bgp disabled'
   noOrDefaultSyntax = syntax
   data = {
      'information' : 'Configure Information Elements in SFT',
      'element' : 'Configure Information Elements in SFT',
      'bgp' : 'Configure BGP Information Element',
      'disabled' : 'Turn off Information Element export',
   }
   handler = sftBgpExportDisabled
   noOrDefaultHandler = noOrDefaultSftBgpExportDisabled

if toggleSftBgpExportCliEnabled():
   SampledFlowTrackerCliModelet.addCommandClass( SftBgpExportDisabledCmd )
   FlowTrackingMode.addModelet( SampledFlowTrackerCliModelet )

# -------------------------------------------------------------------------------
# "[no|default] trap suppression " command
# -------------------------------------------------------------------------------

class FtrTrapSuppression( CliCommand.CliCommandClass ):
   syntax = 'trap suppression [ disabled ]'
   noOrDefaultSyntax = 'trap suppression ...'

   data = {
      'trap' : CliCommand.guardedKeyword( 'trap',
                        helpdesc='Trap configuration',
                        guard=guardFtrTrapSuppression ),
      'suppression' : 'Configure trap suppression',
      'disabled' : 'Disable trap suppression'
    }

   @staticmethod
   def handler( mode, args ):
      if modDefaultTrapSuppression() is None:
         if 'disabled' in args:
            mode.context().trapSuppression = False
         else:
            mode.context().trapSuppression = True
      else:
         if 'disabled' in args:
            if modDefaultTrapSuppression() is True:
               mode.context().trapSuppression = False
            else:
               mode.context().trapSuppression = None
         elif modDefaultTrapSuppression() is False:
            mode.context().trapSuppression = True
         else:
            mode.context().trapSuppression = None

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().trapSuppression = None

FlowTrackingMode.addCommandClass( FtrTrapSuppression )

#-----------------------------------------------------------------
# Commands:
# "packet-buffer drop [ queue usage QUEUE_USAGE ] sample probability PROBABILITY"
# "[no|default] packet-buffer drop [ queue usage QUEUE_USAGE ] sample probability"
# in "config-flow-tracking" mode.
#-----------------------------------------------------------------
class FtrPacketBufferSamplingProbability( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = \
         'packet-buffer drop [ queue usage QUEUE_USAGE ] sample probability ...'
   syntax = noOrDefaultSyntax.replace( '...', ' PROBABILITY' )
   data = {
      'packet-buffer' : packetBufferKw,
      'drop' : packetBufferDropKw,
      'queue' : CliCommand.guardedKeyword( 'queue',
         helpdesc='Configure sampling behavior based on queue usage',
         guard=guardPacketBufferSamplingProb ),
      'usage' : 'Configure sampling behavior based on queue usage',
      'QUEUE_USAGE' : packetBufferQueueUsageMatcher,
      'sample' : CliCommand.guardedKeyword( 'sample',
         helpdesc='Configure sampling behavior',
         guard=guardPacketBufferSamplingProb ),
      'probability' : 'Configure sampling probability',
      'PROBABILITY' : CliMatcher.IntegerMatcher(
         SampleProbabilityType.minProbability, SampleProbabilityType.maxProbability,
         helpdesc='Percent probability of sampling a dropped packet' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().packetBufferSamplingProb( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().packetBufferSamplingProbDefault( mode, args )

FlowTrackingMode.addCommandClass( FtrPacketBufferSamplingProbability )

#-----------------------------------------------------------------
# "[no|default] packet-buffer drop rate RATE pps"
#-----------------------------------------------------------------
def packetBufferShaperRateRangeFn( mode, context ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   ftrCaps = getFtrCaps( ftrType )
   if not ftrCaps:
      return ( 0, 0 )

   return ( 0, ftrCaps.packetBufferShaperMaxRate )

class FtrPacketBufferShaperRate( CliCommand.CliCommandClass ):
   syntax = 'packet-buffer drop rate RATE pps'
   noOrDefaultSyntax = 'packet-buffer drop rate [ RATE pps ]'

   data = {
      'packet-buffer' : packetBufferKw,
      'drop' : packetBufferDropKw,
      'rate' : CliCommand.guardedKeyword( 'rate',
         helpdesc='Set rate',
         guard=guardPacketBufferShaper ),
      'RATE' : CliMatcher.DynamicIntegerMatcher(
         rangeFn=packetBufferShaperRateRangeFn,
         helpdesc='Shaper rate in packets per second' ),
      'pps' : 'Packets per second',
   }

   @staticmethod
   def handler( mode, args ):
      rate = args[ 'RATE' ]
      mode.context().packetBufferShaperRate = rate

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().packetBufferShaperRate = PacketBufferRate.rateDefault

FlowTrackingMode.addCommandClass( FtrPacketBufferShaperRate )

#-----------------------------------------------------------------
# "[no|default] packet-buffer drop burst BURST packets
#-----------------------------------------------------------------
def packetBufferShaperBurstRangeFn( mode, context ):
   ftrType = ftrTypeByName[ mode.ftrTypeStr ]
   ftrCaps = getFtrCaps( ftrType )
   if not ftrCaps:
      return ( 0, 0 )

   return ( 0, ftrCaps.packetBufferShaperMaxBurst )

class FtrPacketBufferShaperBurst( CliCommand.CliCommandClass ):
   syntax = 'packet-buffer drop burst BURST packets'
   noOrDefaultSyntax = 'packet-buffer drop burst [ BURST packets ]'

   data = {
      'packet-buffer' : packetBufferKw,
      'drop' : packetBufferDropKw,
      'burst' : CliCommand.guardedKeyword( 'burst',
         helpdesc='Set burst',
         guard=guardPacketBufferShaper ),
      'BURST' : CliMatcher.DynamicIntegerMatcher(
         rangeFn=packetBufferShaperBurstRangeFn,
         helpdesc='Shaper burst in packets' ),
      'packets' : 'Packets',
   }

   @staticmethod
   def handler( mode, args ):
      burst = args[ 'BURST' ]
      mode.context().packetBufferShaperBurst = burst

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().packetBufferShaperBurst = PacketBufferBurst.burstDefault

FlowTrackingMode.addCommandClass( FtrPacketBufferShaperBurst )

#-----------------------------------------------------------------
# "[no|defalt] shutdown" command in "config-flow-tracking" mode
#-----------------------------------------------------------------

class FtrShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax

   data = {
      'shutdown' : 'Enable or disable the flow tracking feature',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().enabled = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      mode.context().enabled = True

FlowTrackingMode.addCommandClass( FtrShutdown )

#-----------------------------------------------------------------
# "[no|default] record format ipfix standard timestamps counters" command in
# "config-flow-tracking" mode
#-----------------------------------------------------------------

formatNode = CliCommand.guardedKeyword( 'format',
   helpdesc='Configure flow record format',
   guard=guardSwExport )
recordNode = CliCommand.Node( matcherRecord,
   guard=guardRecordUpdate )

class RecordFormatIpfixCmd( CliCommand.CliCommandClass ):
   syntax = 'record format ipfix standard timestamps counters'
   noOrDefaultSyntax = syntax
   data = {
      'record' : recordNode,
      'format' : formatNode,
      'ipfix' : 'Configure IPFIX record format',
      'standard' : 'Configure IPFIX record format standard',
      'timestamps' : 'Configure IPFIX standard record format for timestamps',
      'counters' : 'Configure IPFIX standard record format for counters',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().swExport = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().swExport = False

FlowTrackingMode.addCommandClass( RecordFormatIpfixCmd )

#-----------------------------------------------------------------
# "[no|default] record rewrite dscp" command in
# "config-flow-tracking" mode
#-----------------------------------------------------------------

rewriteNode = CliCommand.guardedKeyword( 'rewrite',
      helpdesc='Configure record rewrite',
      guard=guardRewriteDscp )

class RecordRewriteDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'record rewrite dscp'
   noOrDefaultSyntax = 'record rewrite dscp'
   data = {
      'record' : recordNode,
      'rewrite' : rewriteNode,
      'dscp' : 'Configure rewriting DSCP value',
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().rewriteDscp = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().rewriteDscp = False

FlowTrackingMode.addCommandClass( RecordRewriteDscpCmd )

#-----------------------------------------------------------------
# "[no|default] hardware offload AFs" command in
# "config-flow-tracking" mode
#-----------------------------------------------------------------

matchHardware = CliMatcher.KeywordMatcher( 'hardware',
      helpdesc='Configure hardware parameters' )
matchOffload = CliMatcher.KeywordMatcher( 'offload',
      helpdesc='Configure hardware offload' )
hwOffloadIpv4Node = CliCommand.guardedKeyword( 'ipv4',
      helpdesc='Configure hardware offload for IPv4 traffic',
      guard=guardHwOffloadIpv4 )
hwOffloadIpv6Node = CliCommand.guardedKeyword( 'ipv6',
      helpdesc='Configure hardware offload for IPv6 traffic',
      guard=guardHwOffloadIpv6 )

class HwOffloadIpv4AndIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = '''hardware offload
               ( ( ipv4 [ ipv6 ] )
               | ( ipv6 [ ipv4 ] ) )'''
   noOrDefaultSyntax = '''hardware offload
                          [ ( ipv4 [ ipv6 ] )
                          | ( ipv6 [ ipv4 ] ) ]'''

   data = {
      'hardware' : CliCommand.Node( matchHardware, guard=guardHwOffload ),
      'offload' : matchOffload,
      'ipv4' : hwOffloadIpv4Node,
      'ipv6' : hwOffloadIpv6Node,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().hwOffloadIpv4 = 'ipv4' in args
      mode.context().hwOffloadIpv6 = 'ipv6' in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      if ipv4 or not ipv6:
         mode.context().hwOffloadIpv4 = False

      if ipv6 or not ipv4:
         mode.context().hwOffloadIpv6 = False

FlowTrackingMode.addCommandClass( HwOffloadIpv4AndIpv6Cmd )

# -----------------------------------------------------------------
# "[no|default] hardware offload threshold minimum SAMPLES samples"
# command in "config-flow-tracking" mode
# -----------------------------------------------------------------

class HwOffloadThresholdMinSamplesCmd( CliCommand.CliCommandClass ):
   syntax = '''hardware offload threshold minimum SAMPLES samples'''
   noOrDefaultSyntax = '''hardware offload threshold minimum ...'''

   data = {
      'hardware' : CliCommand.Node( matchHardware, guard=guardHwOffload ),
      'offload' : matchOffload,
      'threshold' : CliMatcher.KeywordMatcher(
            'threshold',
            helpdesc='Configure hardware offload threshold values for heuristics' ),
      'minimum' : CliMatcher.KeywordMatcher(
            'minimum',
            helpdesc='Configure minimum threshold values for hardware offload' ),
      'SAMPLES' : CliMatcher.IntegerMatcher(
            hoHeuristicsConstants.minSamplesLb,
            hoHeuristicsConstants.minSamplesUb,
            helpdesc='Minimum number of samples' ),
      'samples' : CliMatcher.KeywordMatcher(
            'samples',
            helpdesc='Number of samples per active interval for each packet' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().minSamplesBeforeHwOffload = \
         args.get( 'SAMPLES', hoHeuristicsConstants.minSamplesDefault )

   noOrDefaultHandler = handler

FlowTrackingMode.addCommandClass( HwOffloadThresholdMinSamplesCmd )

# Simply abort configuring the current mode.
FlowTrackingMode.addCommandClass( AbortCmd )

#-----------------------------------------------------------------
# show active/pending/diff
#-----------------------------------------------------------------

def _showPending( mode, output=None ):
   ftrCtx = mode.context()
   if ftrCtx is None:
      return
   if output is None:
      output = sys.stdout
   ftrType = ftrCtx.ftrType
   ftrCaps = getFtrCaps( ftrType )
   t3( 'ftrCtx.enabled', ftrCtx.enabled )
   ftrLines = showFlowTracking( ftrCtx, ftrCtx.ftrType,
                                ftrTypeCaps=ftrCaps, cliSave=False,
                                mplsHwCaps=getMplsHwCaps(),
                                bridgingHwCaps=getBridgingHwCaps() )
   for line in ftrLines:
      output.write( line + '\n' )
   t3( 'ftrCtx.trCtx: ', ftrCtx.trCtx )
   # Create trackerCtx that are:
   # 1. Not visited by this CLI session.
   # 2. Added directly in Sysdb or by some other CLI
   config = mode.context().trackingConfig()
   for trName, trConfig in config.flowTrackerConfig.items():
      if trName not in ftrCtx.trCtx:
         t0( 'Creating new tracker context', trName )
         ftrCtx.trCtx[ trName ] = TrackerContext(
                                    config,
                                    ftrCtx.ftrType,
                                    trName,
                                    trConfig )

   if ftrCtx.trCtx and not ftrLines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrCtx.ftrType ] )

   for trName in sorted( ftrCtx.trCtx ):
      trCtx = ftrCtx.trCtx[ trName ]
      showPendingTracker( trCtx, output, space=spaceConst )

   lines = showFlowTrackingTrailer( ftrCtx, ftrCtx.ftrType, cliSave=False )

   # Only Trailer is non-empty, so display flow tracking before trailer lines
   if not ftrCtx.trCtx and not ftrLines and lines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrCtx.ftrType ] )
   for line in lines:
      output.write( line + '\n' )

def _showActive( mode, output=None ):
   config = mode.context().trackingConfig()
   t3( 'config: ', config )
   t3( 'config.enabled', config.enabled )
   t3( 'ftrCtx.enabled', mode.context().enabled )
   ftrType = mode.context().ftrType
   ftrCaps = getFtrCaps( ftrType )
   if config is None:
      return
   if output is None:
      output = sys.stdout
   lines = showFlowTracking( config, ftrType,
                             ftrTypeCaps=ftrCaps, cliSave=False,
                             mplsHwCaps=getMplsHwCaps(),
                             bridgingHwCaps=getBridgingHwCaps() )
   for line in lines:
      output.write( line + '\n' )
   t3( 'trConfigs: ', list( config.flowTrackerConfig ) )

   printHeader = not lines

   for trName, trConfig in sorted( config.flowTrackerConfig.items() ):
      if printHeader:
         output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrType ] )
         printHeader = False
      showActiveTracker( ftrType, trName, trConfig, output,
                         space=spaceConst )

   lines = showFlowTrackingTrailer( config, ftrType, cliSave=False )
   # Only Trailer is non-empty, so display flow tracking before trailer lines
   if printHeader and lines:
      output.write( 'flow tracking %s\n' % ftrTypeKwStr[ ftrType ] )
   for line in lines:
      output.write( line + '\n' )

def _showDiff( mode ):
   # generate diff between active and pending
   activeOutput = io.StringIO()
   _showActive( mode, output=activeOutput )
   pendingOutput = io.StringIO()
   _showPending( mode, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print( '\n'.join( diff ) )

class ShowFtr( ShowCliCommandClass ):
   syntax = 'show ( pending | active | diff )'

   data = {
      'pending': 'Display the new flow tracking configuration to be applied',
      'active': 'Display the flow tracking configuration in the running-config',
      'diff' : ( 'Display the diff between active flow tracking configuration '
                'and to be applied' ),
    }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         _showDiff( mode )
      elif 'active' in args:
         _showActive( mode )
      else:
         _showPending( mode )

FlowTrackingMode.addShowCommandClass( ShowFtr )

#--------------------------
def Plugin( em ):
   for ftrType in ftrTypes:
      ftrEntityType = ( ftrMirrorOnDropConfigType if ftrType == ftrTypeMirrorOnDrop
                        else ftrConfigType )
      trackingConfig[ ftrType ] = \
         ConfigMount.mount( em,
                            getFlowTrackingCliConfigPath( ftrType ),
                            ftrEntityType, 'w' )
      trackingConfigReq[ ftrType ] = ConfigMount.mount( em,
                                        ftrConfigReqPathPrefix + ftrType,
                                        ftrConfigReqType, 'w' )
