# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import typing

import Intf.IntfRange as IntfRange
import Tac
import Arnet
import TableOutput
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from IntfModels import Interface
from Toggles.FlowTrackerToggleLib import toggleSftBgpExportCliEnabled
from FlowTrackerCliUtil import (
   ftrTypeCpuQueue,
   ftrTypeMirrorOnDrop,
   ftrTypeSampled,
   ftrTypeShowStr,
   collectorInactiveReasonStr,
   exporterInactiveReasonStr,
   trackerInactiveReasonStr,
   getExpReasonFlagsFromExpReasonModel,
   exporterInactiveReasonEnum,
   encapTypeEnum,
   encapTypeShowStr,
   filterTypeEnum,
   filterTypeShowStr,
   reservedGroupToSeqnoMap,
   sampledHoInactiveReasonStr,
   packetBufferStatusEnum,
   packetBufferStatusShowStr,
)
from CliPlugin.FlowTrackingCliLib import (
   packetBufferSampleProbKw25,
   packetBufferSampleProbKw50,
   packetBufferSampleProbKw75,
   packetBufferSampleProbKw100,
)
from collections import namedtuple

CollectorPortAndReason = namedtuple( 'Collector', [ 'port', 'inactiveReason' ] )

collectorInactiveReason = Tac.Type( "FlowTracking::CollectorInactiveReason" )
trackerInactiveReason= Tac.Type( "FlowTracking::TrackerInactiveReason" )
hoState = Tac.Type( "FlowTracking::HoState" )
mirrorSelectorAlgo = Tac.Type( "FlowTracking::MirrorSelectorAlgo" )
sampleMode = Tac.Type( "FlowTracking::FlowTrackerSampleMode" )

def printIndented( text, indentation ):
   print( f"{' ' * indentation}{text}" )

def _renderDampingTimeout(
      tab: str,
      dampingTimeout: typing.Optional[ int ] ) -> None:
   if dampingTimeout:
      print( tab + f'Damping timeout: {dampingTimeout} seconds' )

def _renderActiveInterval( tab:str, ftrType, activeInterval ):
   if ftrType != ftrTypeCpuQueue:
      activeIntervalStr = Tac.Value( 'FlowTracking::ActiveInterval',
                                  activeInterval ).stringValue()
      print( tab + "Active interval: " + activeIntervalStr )

def _renderInactiveTimeout( tab:str, ftrType, inactiveTimeout ):
   if ftrType != ftrTypeCpuQueue:
      inactiveTimeoutStr = Tac.Value( 'FlowTracking::InactiveTimeout',
                                  inactiveTimeout ).stringValue()
      print( tab + "Inactive timeout: " + inactiveTimeoutStr )

# Collector
class CollectorModel( Model ):
   port = Int( "Collector port" )
   active = Bool( help="Collector is ready for transmission" )
   inactiveReason = Enum( values=collectorInactiveReason.attributes, 
                          help="Collector inactive reason",
                          optional=True )

# Export Format
class ExpFormatModel( Model ):
   name = Str( help="Export format name" )
   mtu = Int( help="MTU of the export format", optional=True )
   version = Int( help="Version of the export format", optional=True )

class ExpReason( Model ):
   expReason = Enum( values=exporterInactiveReasonEnum.attributes,
                     help="Exporter inactive reason" )

class ExpReasonModel( Model ):
   expReasons = List( valueType=ExpReason,
                      help="List of reasons for inactive exporter" )


# Exporter
class ExpModel( Model ):
   __revision__ = 2

   vrfName = Str( help="VRF name", optional=True )
   localIntf = \
      Interface( help="Local interface to be used for IPFIX source IP4/6 addr" )
   srcIpAddr = Ip4Address(
      help="Local IPv4 address to be used for IPFIX source addr" )
   srcIp6Addr = Ip6Address(
      help="Local IPv6 address to be used for IPFIX source addr" )
   exportFormat = Submodel( valueType=ExpFormatModel,
                            help="Export format information" )
   dscpValue = Int( help="DSCP value in IP header", optional=True )
   templateInterval = Int( help="Template interval in ms", optional=True )
   active = Bool( help="Exporter is ready for transmission" )
   collectors = Dict( keyType=str, valueType=CollectorModel,
                help="A mapping between collector address and collector port" )
   inactiveReasons = Dict( keyType=str, valueType=ExpReasonModel,
                           help="A mapping between type of reason and "
                                "list of reasons for inactive exporter",
                           optional=True )
   useSflowCollectorConfig = Bool( help="Use the configured sFlow collectors",
                                   optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'vrfName' ] = dictRepr.get( 'vrfName', '' )
         dictRepr[ 'dscpValue' ] = dictRepr.get( 'dscpValue', 0 )
         dictRepr[ 'templateInterval' ] = dictRepr.get( 'templateInterval', 0 )

      return dictRepr

class EncapTypeModel( Model ):
   encapType = Enum( values=encapTypeEnum.attributes,
                     help="Encapsulation" )

class FilterTypeModel( Model ):
   filterType = Enum( values=filterTypeEnum.attributes,
                      help="Encapsulation filters" )

class TrapSuppressionModel( Model ):
   trapSuppression = Bool( help="Trap suppression configured value",
                           optional=True )
   active = Bool( help="Trap Suppression operational value", optional=True )
   reason = Str( help="Reason for inactive trap suppression", optional=True )

probHelpStr = 'Configured percent probability of sampling dropped packets'
class PacketBufferModel( Model ):
   status = Enum( values=packetBufferStatusEnum.attributes,
                  help="Packet buffer hardware status" )
   prob25 = Int( help=probHelpStr, optional=True )
   prob50 = Int( help=probHelpStr, optional=True )
   prob75 = Int( help=probHelpStr, optional=True )
   prob100 = Int( help=probHelpStr, optional=True )
   shaperRate = Int( help="Packet buffer drop shaper rate in kbps", optional=True )
   shaperBurst = Int( help="Packet buffer drop shaper burst in packets",
                      optional=True )

# For now it has only one attribute, but in future it will also have
# active/inactive state
class GroupExpModel( Model ):
   expName = Str( help="Exporter name" )

# For now it has only one attribute, but in future it will have more
class AccessListModel( Model ):
   aclName = Str( help="Access list name" )

class MirrorModel( Model ):
   sessionName = Str( help="Mirror session name" )
   initialCopy = Int( help="Initial packets to be mirrored",
                      optional=True )
   mirrorInterval = Int( help="Sampling interval for packets to be mirrored",
                         optional=True )
   mirrorSelectorAlgorithm = Enum( values=mirrorSelectorAlgo.attributes,
                                   help="Mirror sampling algorithm",
                                   optional=True )

# For now it has only one attribute, but in future it will have more
class HwFgModel( Model ):
   hwFgName = Str( help="Hardware flow group name" )
   active = Bool( help="Hardware flow group is active", optional=True )

# Flow group
class FgModel( Model ):
   seqno = Int( help="Flow group sequence number",
                optional=True )
   encapTypes = List( valueType=EncapTypeModel,
                      help="List of encapsulations",
                      optional=True )
   ipAccessLists = Dict( keyType=int, valueType=AccessListModel,
                         help="IPv4 ACLs, keyed by their sequence number",
                         optional=True )
   ip6AccessLists = Dict( keyType=int, valueType=AccessListModel,
                          help="IPv6 ACLs, keyed by their sequence number",
                          optional=True )
   expNames = Dict( keyType=str, valueType=GroupExpModel,
                    help="Exporters, keyed by exporter name",
                    optional=True )
   mirroring = Submodel( valueType=MirrorModel,
                         help="Mirroring details",
                         optional=True )
   hwGroups = Dict( keyType=int, valueType=HwFgModel,
                    help="Flow groups programmed in hardware,"
                         " keyed by their sequence number",
                    optional=True )

# Flow interface
class IntfModel( Model ):
   interface = Interface( help="Name of the interface" )
   reason = Str( help="Reason for interface inactive", optional=True )
   sampleMode = Enum( values=( "all", "filtered" ),
                      help="Sampling mode for the interface", optional=True )

class CongestionIntervalModel ( Model ):
   before = Int( help="Number of seconds of data before event to report" )
   after = Int( help="Number of seconds of data after event to report" )

def _renderExportTriggers(
      tab: str,
      exportTriggers: typing.Optional[ typing.List[ str ] ] ) -> None:
   if exportTriggers is not None:
      triggers = ", ".join( exportTriggers ) if exportTriggers != [] else "disabled"
      print( f'{tab}Export triggers: {triggers}' )

def _renderCongestionInterval(
      tab: str,
      congestionInterval: typing.Optional[ CongestionIntervalModel ] ) -> None:
   if congestionInterval:
      print(
         f'{tab}Capture interval: '
         f'before {congestionInterval.before} seconds, '
         f'after {congestionInterval.after} seconds' )

# Flow tracker
class FtModel( Model ):
   __revision__ = 2

   inactiveTimeout = Int( help="Inactive timeout in ms" )
   activeInterval = Int( help="Active interval in ms" )
   activeIntfs = List( valueType=IntfModel, help="Active Ingress Interfaces" )
   inactiveIntfs = List( valueType=IntfModel, help="Inactive Ingress Interfaces" )
   activeEgressIntfs = List( valueType=IntfModel, help="Active Egress Interfaces" )
   inactiveEgressIntfs = List( valueType=IntfModel,
                               help="Inactive Egress Interfaces" )
   # FgModel is per user defined group
   # Hardware groups corresponding to user defined groups are within FgModel
   groups = Dict( keyType=str, valueType=FgModel, help="Flow groups" )
   exporters = Dict( keyType=str, valueType=ExpModel,
                     help="A mapping between exporter name and exporter" )
   active = Bool( help="Tracker is active", default=False )
   inactiveReason = Enum( values=trackerInactiveReason.attributes,
                          help="Tracker inactive reason",
                          optional=True )
   dampingTimeout = Int( help="Seconds to wait between reports", optional=True )
   congestionInterval = Submodel( valueType=CongestionIntervalModel,
         help="Number of seconds of data to report",
         optional=True )
   # exportTriggers is a list to allow for future growth.
   exportTriggers = List( valueType=str,
         help="Causes of exporter being used",
         optional=True )
   monitoredCpuQueues = List( valueType=str,
         help="CPU Queues that are being monitored",
         optional=True )

FLOWTRACKING_TYPES = Tac.Type( "FlowTracking::FlowTrackingType" )

class HoInactiveReasonModel( Model ):
   inactiveReason = Enum( values=hoState.attributes,
                          help="Hardware offload inactive reason" )

class FtrHoStatusModel( Model ):
   status = Enum( values=( "active", "inactive", "unconfigured" ),
                  help="Flow tracking hardware offload status", optional=True )
   inactiveReasons = List( valueType=HoInactiveReasonModel,
                           help="List of hardware offload inactive reasons",
                           optional=True )

   def renderHoStatus( self, ingress: bool ):
      printIndented( f"{'Ingress' if ingress else 'Egress'} hardware offload:", 4 )
      printIndented( f"Status: {self.status}", 6 )
      if self.inactiveReasons:
         printIndented( "Inactive reason(s):", 6 )
         for hoReason in self.inactiveReasons:
            reasonStr = sampledHoInactiveReasonStr( hoReason.inactiveReason )
            printIndented( reasonStr, 8 )

class FtrTrafficTypeModel( Model ):
   sampleRate = Int( help="Flow tracking sample every x packets", optional=True )
   hwOffloadStatus = Submodel( valueType=FtrHoStatusModel,
                               help="Flow tracking ingress hardware offload status",
                               optional=True )
   egressHwOffloadStatus = Submodel(
         valueType=FtrHoStatusModel,
         help="Flow tracking egress hardware offload status",
         optional=True )

# Flow Tracking
class FtrModel( Model ):
   __revision__ = 2

   trackers = Dict( keyType=str, valueType=FtModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Flow tracking is running" )
   ftrType = Enum( values=FLOWTRACKING_TYPES.attributes, help="Flow tracking type" )
   ftrTrafficGroups = Dict( keyType=str, valueType=FtrTrafficTypeModel,
                            help="Flow tracking traffic types", optional=True )
   standardFormatForCounters = Bool( help="Flow tracking standard record format for"
                                     " counters", optional=True )
   standardFormatForTimeStamps = Bool( help="Flow tracking standard record format "
                                       "for timestamps", optional=True )
   enabledBy = Str( help="The command that enables flow tracking",
                    optional=True )
   if toggleSftBgpExportCliEnabled():
      bgpElementsExported = Bool( help="BGP Export in SFT", optional=True )
   encapTypes = List( valueType=EncapTypeModel,
                      help="List of encapsulations",
                      optional=True )
   filterTypes = List( valueType=FilterTypeModel,
                       help="List of encapsulation filters",
                       optional=True )
   sampleSize = Int( help="Number of bytes used when sampling",
                                  optional=True )
   sampleLimit = Int( help="Maximum sample rate in packets per second "
                                       "per interface",
                                  optional=True )
   trapSuppressionDetails = Submodel( valueType=TrapSuppressionModel,
                                      help="Trap suppression status", optional=True )
   packetBuffer = Submodel( valueType=PacketBufferModel,
                            help="Packet buffer status", optional=True )

   def renderEncapTypes( self, tab, encapTypes ):
      encaps = []
      for encap in encapTypes:
         if encap.encapType == encapTypeEnum.encapNone:
            continue
         encaps.append( encapTypeShowStr[ encap.encapType ] )
      encapStr = ", ".join( sorted( encaps ) ) if encaps else "none"
      print( tab + "Encapsulation: " + encapStr )

   def renderFilterTypes( self, tab, filterTypes ):
      filters = []
      for encapFilter in filterTypes:
         encapFilterStr = filterTypeShowStr.get( encapFilter.filterType )
         if encapFilterStr:
            filters.append( encapFilterStr )
      if filters:
         filterStr = ", ".join( sorted( filters ) )
         print( tab + "Encapsulation filter: " + filterStr )

   @staticmethod
   def renderCollector( ip, collector ):
      tab = " " * 8
      activeStr = ""
      if not collector.active:
         activeStr = collectorInactiveReasonStr( collector.inactiveReason )
      print( tab + ip + " port " + str( collector.port ) + activeStr )

   @staticmethod
   def renderExporter( expName, exporter: ExpModel ):
      tab = " " * 4
      if not exporter.active:
         reasonUnknown = True
         for reason in exporter.inactiveReasons.values():
            expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
            if expInactiveReason.value != 0:
               reasonUnknown = False
               break

         padding = tab + 'Exporter: ' + expName
         for reasonType, reason in sorted( exporter.inactiveReasons.items() ):
            expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
            print( padding + exporterInactiveReasonStr( expInactiveReason,
                                                       reasonType ) )
            padding = ' ' * len( padding )

            # We should print the following reasons only once
            if expInactiveReason.maxExportersLimit or \
               expInactiveReason.exporterNotProcessed:
               return
            if reasonUnknown:
               break
      else:
         print( tab + "Exporter: " + expName )

      tab = " " * 6
      if exporter.vrfName is not None:
         print( tab + "VRF: " + exporter.vrfName )
      if exporter.localIntf:
         ips = []
         if exporter.srcIpAddr.stringValue != exporter.srcIpAddr.ipAddrZero:
            ips.append( exporter.srcIpAddr.stringValue )
         if not exporter.srcIp6Addr.isZero:
            ips.append( exporter.srcIp6Addr.stringValue )
         print( tab + "Local interface: " + exporter.localIntf.stringValue +
                ( ( " (" + ", ".join( ips ) + ")" ) if ips else "" ) )
      formatInfo = exporter.exportFormat
      formatDetails = []
      if formatInfo.version:
         formatDetails.append( "version " + str( formatInfo.version ) )
      if formatInfo.mtu:
         formatDetails.append( "MTU " + str( formatInfo.mtu ) )
      formatOutput = "Export format: " + exporter.exportFormat.name
      if formatDetails:
         formatOutput += " " + ", ".join( formatDetails )
      print( tab + formatOutput )
      if exporter.dscpValue is not None:
         print( tab + "DSCP: " + str( exporter.dscpValue ) )
      if exporter.templateInterval is not None:
         templateIntervalStr = Tac.Value( 'FlowTracking::TemplateInterval',
                                          exporter.templateInterval ).stringValue()
         print( tab + "Template interval: " + templateIntervalStr )

      # First print all active collectors,
      # then inactive collectors which are within max collector limit,
      # then inactive collectors due to max collector limit.
      inactiveCollector = {}
      beyondLimitCollector = {}
      if exporter.useSflowCollectorConfig:
         print( tab + "Collectors: Sflow" )
      elif exporter.collectors:
         print( tab + "Collectors:" )
         for ip, collector in sorted( exporter.collectors.items() ): 
            if not collector.active:
               if collector.inactiveReason == \
                  collectorInactiveReason.maxCollectorsLimit:
                  beyondLimitCollector[ ip ] = collector
               else:
                  inactiveCollector[ ip ] = collector
               continue
            FtrModel.renderCollector( ip, collector )

         for ip in sorted( inactiveCollector ):
            FtrModel.renderCollector( ip, inactiveCollector[ ip ] )

         for ip in sorted( beyondLimitCollector ):
            FtrModel.renderCollector( ip, beyondLimitCollector[ ip ] )

   def renderGroup( self, groupName, group ):
      tab = " " * 4
      print( tab + "Group: " + groupName )

      tab = " " * 6
      # Encapsulation and access list are only supported for non-reserved groups
      if groupName not in reservedGroupToSeqnoMap:
         self.renderEncapTypes( tab, group.encapTypes )

         ip4Acls = []
         for aclSeqno in sorted( group.ipAccessLists ):
            ip4Acls.append( group.ipAccessLists[ aclSeqno ].aclName )
         # ip4Acls are sorted by the seqno, do NOT sort here by name
         ip4AclStr = ", ".join( ip4Acls ) if ip4Acls else "none"
         print( tab + "IPv4 access list: " + ip4AclStr )

         ip6Acls = []
         for aclSeqno in sorted( group.ip6AccessLists ):
            ip6Acls.append( group.ip6AccessLists[ aclSeqno ].aclName )
         # ip6Acls are sorted by the seqno, do NOT sort here by name
         ip6AclStr = ", ".join( ip6Acls ) if ip6Acls else "none"
         print( tab + "IPv6 access list: " + ip6AclStr )

      expStr = ", ".join( sorted( group.expNames ) ) if group.expNames else "none"
      print( tab + "Exporters: " + expStr )

      if group.mirroring:
         groupMirroring = group.mirroring
         if groupMirroring.sessionName:
            print( tab + "Mirroring:" )
            tabMirroring = " " * 8
            sessionStr = groupMirroring.sessionName
            initialCopyStr = str( groupMirroring.initialCopy ) if \
                             groupMirroring.initialCopy else "none"
            if groupMirroring.mirrorSelectorAlgorithm != \
               mirrorSelectorAlgo.sampleModeNone:
               if groupMirroring.mirrorSelectorAlgorithm == \
                  mirrorSelectorAlgo.fixed:
                  selectorAlgoStr = " (fixed)"
               elif groupMirroring.mirrorSelectorAlgorithm == \
                    mirrorSelectorAlgo.random:
                  selectorAlgoStr = " (random)"
               mirrorIntervalStr = str( groupMirroring.mirrorInterval ) + \
                                   selectorAlgoStr
            else:
               mirrorIntervalStr = "none"

            print( tabMirroring + "Monitor session: " + sessionStr )
            print( tabMirroring + "Number of initial packets: " + initialCopyStr )
            print( tabMirroring + "Sample interval: " + mirrorIntervalStr )
         else:
            print( tab + "Mirroring: none" )

      hwGroupsStr = "Hardware groups:" if group.hwGroups else \
                    "Hardware groups: none"
      print( tab + hwGroupsStr )
      tabHwGroups = " " * 8
      for hwFgSeqno in sorted( group.hwGroups ):
         hwGroupActiveStr = "" if group.hwGroups[ hwFgSeqno ].active else \
                            " (inactive)"
         print( tabHwGroups + group.hwGroups[ hwFgSeqno ].hwFgName
            + hwGroupActiveStr )

   @staticmethod
   def _renderMonitoredCpuQueues( tab:str, monitoredCpuQueues ):
      if monitoredCpuQueues:
         print( tab + "Monitored Cpu Queues:" )
         tab = tab + "  "
         strQueues = ", ".join( monitoredCpuQueues )
         print( tab + strQueues )

   def renderTracker( self, trackerName, tracker ):
      tab = " " * 2
      if not tracker.active:
         print( tab + "Tracker: " + trackerName +
                trackerInactiveReasonStr( tracker.inactiveReason ) )
         return
      print( tab + "Tracker: " + trackerName )
      tab = " " * 4
      _renderActiveInterval( tab, self.ftrType, tracker.activeInterval )
      _renderInactiveTimeout( tab, self.ftrType, tracker.inactiveTimeout )
      _renderExportTriggers( tab, tracker.exportTriggers )
      _renderCongestionInterval( tab, tracker.congestionInterval )
      _renderDampingTimeout( tab, tracker.dampingTimeout )

      userConfiguredflowGroups = False
      if tracker.groups:
         for group in tracker.groups.values():
            if group.seqno is not None:
               userConfiguredflowGroups = True
               break
         if not userConfiguredflowGroups:
            print( tab + "Groups: " + ", ".join( sorted( tracker.groups ) ) )
      # First print all active exporters,
      # then inactive exporters which are within max exporter limit,
      # then inactive exporters due to max exporter limit.
      inactiveExporter = {}
      beyondLimitExporter = {}
      for expName, exporter in sorted( tracker.exporters.items() ):
         if not exporter.active:
            maxExportersLimit = False
            for reason in exporter.inactiveReasons.values():
               expInactiveReason = getExpReasonFlagsFromExpReasonModel( reason )
               if expInactiveReason.maxExportersLimit:
                  maxExportersLimit = True
                  break
            if maxExportersLimit:
               beyondLimitExporter[ expName ] = exporter
            else:
               inactiveExporter[ expName ] = exporter
            continue
         FtrModel.renderExporter( expName, exporter )

      for expName in sorted( inactiveExporter ):
         FtrModel.renderExporter( expName, inactiveExporter[ expName ] )

      for expName in sorted( beyondLimitExporter ):
         FtrModel.renderExporter( expName, beyondLimitExporter[ expName ] )

      if userConfiguredflowGroups:
         seqnoToGroupMap = {}
         for groupName, group in tracker.groups.items():
            seqnoToGroupMap[ group.seqno ] = groupName
         for groupSeqno in sorted( seqnoToGroupMap ):
            groupName = seqnoToGroupMap[ groupSeqno ]
            self.renderGroup( groupName, tracker.groups[ groupName ] )
      if tracker.activeIntfs:
         activeIntfs = [ intf for intf in tracker.activeIntfs
                         if intf.sampleMode != "filtered" ]
         activeFilteredIntfs = [ intf for intf in tracker.activeIntfs
                                 if intf.sampleMode == "filtered" ]
         if activeIntfs:
            self.printActiveIntfs( "Active Ingress Interfaces:", activeIntfs )

         if activeFilteredIntfs:
            self.printActiveIntfs( "Active Ingress Filtered Sampling Interfaces:",
                                activeFilteredIntfs )
      if tracker.inactiveIntfs:
         self.printInactiveIntfs( "Inactive Ingress Interfaces:",
                                  tracker.inactiveIntfs )
      if tracker.activeEgressIntfs:
         self.printActiveIntfs( "Active Egress Interfaces:",
                                tracker.activeEgressIntfs )
      if tracker.inactiveEgressIntfs:
         self.printInactiveIntfs( "Inactive Egress Interfaces:",
                                  tracker.inactiveEgressIntfs )
      FtrModel._renderMonitoredCpuQueues( "  ", tracker.monitoredCpuQueues )

   def renderPacketBuffer( self, tab, packetBuffer ):
      print( tab + 'Packet buffer drops: ' +
              packetBufferStatusShowStr[ packetBuffer.status ] )
      if ( packetBuffer.prob25 or packetBuffer.prob50 or
           packetBuffer.prob75 or packetBuffer.prob100 ):
         table = TableOutput.createTable(
               [ 'Buffer Usage', 'Sampling Probability' ],
               indent=tab.count( ' ' ) )
         f = TableOutput.Format( justify="right" )
         f.noPadLeftIs( True )
         f.padLimitIs( True )
         table.formatColumns( f, f )
         table.newRow( packetBufferSampleProbKw25, f'{packetBuffer.prob25}%' )
         table.newRow( packetBufferSampleProbKw50, f'{packetBuffer.prob50}%' )
         table.newRow( packetBufferSampleProbKw75, f'{packetBuffer.prob75}%' )
         table.newRow( packetBufferSampleProbKw100, f'{packetBuffer.prob100}%' )
         print( table.output() )
      elif packetBuffer.shaperRate or packetBuffer.shaperBurst:
         if packetBuffer.shaperRate:
            print( tab + f'Packet buffer drops rate: {packetBuffer.shaperRate}' )
         if packetBuffer.shaperBurst:
            print( tab + f'Packet buffer drops burst: {packetBuffer.shaperBurst}' )

   def printActiveIntfs( self, desc, intfs ):
      tab = " " * 4
      print( tab + desc )
      tab = " " * 6
      activeIntfs = [ i.interface.shortName for i in intfs ]
      activeIntfs = IntfRange.intfListToCanonical( Arnet.sortIntf( activeIntfs ) )
      print( tab + ", ".join( activeIntfs ) )

   def printInactiveIntfs( self, desc, intfs ):
      tab = " " * 4
      print( tab + desc )
      tab = " " * 6
      inactiveIntfs = [ i.interface.shortName
                        for i in intfs if not i.reason ]
      inactiveIntfs = IntfRange.intfListToCanonical(
         Arnet.sortIntf( inactiveIntfs ) )
      inactiveIntfs2 = [ i.interface.shortName + "(" + i.reason + ")"
                         for i in intfs if i.reason ]
      print( tab + ", ".join( Arnet.sortIntf( inactiveIntfs + inactiveIntfs2 ) ) )

   def _renderSampleSize( self, tab ):
      if self.ftrType != ftrTypeCpuQueue:
         return
      if self.sampleSize and self.sampleSize > 0:
         print( tab + f"Sample size: {self.sampleSize} bytes" )

   def _renderSampleLimit( self, tab ):
      if self.sampleLimit is not None and self.sampleLimit > 0:
         if self.ftrType in [ ftrTypeSampled, ftrTypeMirrorOnDrop ]:
            print( tab + f"Sample limit: {self.sampleLimit}" )
         elif self.ftrType == ftrTypeCpuQueue:
            print( tab + f"Sample limit: {self.sampleLimit} packets per second" )

   def render( self ):
      print( "Flow Tracking Status" )
      tab = " " * 2
      print( tab + "Type: " + ftrTypeShowStr[ self.ftrType ] )

      if self.running is not None:
         suffix = ""
         if self.running:
            suffix = ( ", enabled by the '%s' command" % self.enabledBy )
         elif self.ftrType == ftrTypeSampled:
            suffix = ', check "show agent SftAgent runnability" '\
               'for more information.'
         print( tab + "Running: " + ( "yes" if self.running else "no" ) + suffix )

      if not self.running:
         return

      def renderYesNo( attr=None, text=None ):
         if attr is not None:
            print( tab + text + ": " + ( "yes" if attr else "no" ) )

      renderYesNo( attr=self.standardFormatForCounters,
            text="Standard record format for counters" )
      renderYesNo( attr=self.standardFormatForTimeStamps,
            text="Standard record format for timestamps" )

      self._renderSampleSize( tab )
      self._renderSampleLimit( tab )

      if self.ftrType == ftrTypeMirrorOnDrop:
         if not self.trapSuppressionDetails.active:
            print( tab + "Trap suppression: inactive (" +
                  f"{self.trapSuppressionDetails.reason})" )
         else:
            print( tab + "Trap suppression: active" )

      if toggleSftBgpExportCliEnabled():
         if self.ftrType == ftrTypeSampled:
            if self.bgpElementsExported:
               print( tab + "BGP information elements: exported" )
            else:
               print( tab + "BGP information elements: not exported" )

      if self.encapTypes:
         self.renderEncapTypes( tab, self.encapTypes )

      if self.filterTypes:
         self.renderFilterTypes( tab, self.filterTypes )

      if self.packetBuffer:
         self.renderPacketBuffer( tab, self.packetBuffer )

      if self.ftrTrafficGroups:
         for addrFamily, trafficInfo in sorted( self.ftrTrafficGroups.items() ):
            tab = " " * 2
            print( tab + addrFamily + ":" )
            tab = " " * 4
            print( tab + "Sample rate: " + str( trafficInfo.sampleRate ) )
            if trafficInfo.hwOffloadStatus:
               trafficInfo.hwOffloadStatus.renderHoStatus( ingress=True )
            if trafficInfo.egressHwOffloadStatus:
               trafficInfo.hwOffloadStatus.renderHoStatus( ingress=False )

      # First print all active trackers,
      # then inactive trackers which are within max tracker limit (TO DO),
      # then inactive trackers due to max tracker limit.
      inactiveTracker = {}
      beyondLimitTracker = {}
      for trackerName, tracker in sorted( self.trackers.items() ):
         if not tracker.active:
            if tracker.inactiveReason == trackerInactiveReason.maxTrackersLimit:
               beyondLimitTracker[ trackerName ] = tracker
            else:
               inactiveTracker[ trackerName ] = tracker
            continue
         self.renderTracker( trackerName, tracker )

      # This is a no-op for now, existing behavior since beginning.
      for trackerName in sorted( inactiveTracker ):
         self.renderTracker( trackerName, inactiveTracker[ trackerName ] )

      for trackerName in sorted( beyondLimitTracker ):
         self.renderTracker( trackerName, beyondLimitTracker[ trackerName ] )
