# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BasicCli import addShowCommandClass
import CliCommand
from CliMatcher import (
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
)
from CliPlugin import TechSupportCli
from CliPlugin.FlowWatcherCliLib import (
   guardMonitorSecurity,
   guardMonitorSecurityForShowTech,
   nucleusNameMatcher,
)
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliToken.Ip import afMatcherForShow
from CliToken.Monitor import monitorMatcherForShow
import ShowCommand
import Tracing

traceHandle = Tracing.Handle( 'FlowWatcherShowCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

#
# SHOW COMMANDS
#

awakeKw = CliCommand.singleKeyword( 'awake',
                                    helpdesc='Show monitor security awake' )
securityKw = CliCommand.guardedKeyword( 'security',
                                        helpdesc='Show monitor security',
                                        guard=guardMonitorSecurity )
dpiKw = CliCommand.singleKeyword( 'dpi', helpdesc='Show deep packet inspection' )
memoryKw = CliCommand.singleKeyword( 'memory', helpdesc='Show memory use' )

ipGenAddrMatcher = CliCommand.singleNode( IpGenAddrMatcher( 'IP address of the flow',
                                                            helpdesc4='IPv4 address',
                                                           helpdesc6='IPv6 address' )
                                         )

tokenPort = CliCommand.singleNode( IntegerMatcher( 0, 65535, helpdesc='IP port' ) )

# ----------------------------------------------------------------------
# show monitor security awake [nucleus <nucleus>]
# ----------------------------------------------------------------------
class ShowMsa( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor security awake [ nucleus [ NUCLEUS_NAME ] ]'

   data = {
      'monitor': monitorMatcherForShow,
      'security': securityKw,
      'awake': awakeKw,
      'nucleus': 'Show nucleus',
      'NUCLEUS_NAME': nucleusNameMatcher,
   }

   handler = 'FlowWatcherShowCli.showMsaHandler'
   cliModel = 'FlowWatcherModel.Msa'

addShowCommandClass( ShowMsa )

# ----------------------------------------------------------------------
# show monitor security awake counters [flows|ipfix|nucleus [<nucleus>]]
# ----------------------------------------------------------------------
class ShowMsaCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor security awake counters ' + \
            '[ flows | ipfix | ( nucleus [ NUCLEUS_NAME ] ) ]'

   data = {
      'monitor': monitorMatcherForShow,
      'security': securityKw,
      'awake': awakeKw,
      'counters': 'Show monitor security awake counters',
      'flows': 'Show flows counters',
      'ipfix': 'Show ipfix counters',
      'nucleus': 'Show nucleus counters',
      'NUCLEUS_NAME': nucleusNameMatcher,
   }

   handler = 'FlowWatcherShowCli.showMsaCountersHandler'
   cliModel = 'FlowWatcherModel.MsaCounters'

addShowCommandClass( ShowMsaCounters )

# ----------------------------------------------------------------------
# show monitor security awake flow-table [detail]
# [ group <group-name> ][ protocol <protocol> ]
# [ ip <ip> ] [ peer-ip <ip> ] [ port <port> ] [ peer-port <port> ]
# ----------------------------------------------------------------------
class ShowMsaFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor security awake flow-table
               [ { ( detail )
                 | ( group GROUP_NAME )
                 | ( protocol PROTOCOL )
                 | ( ip IP )
                 | ( peer-ip PEER_IP )
                 | ( port PORT )
                 | ( peer-port PEER_PORT ) }  ]'''

   data = {
      'monitor': monitorMatcherForShow,
      'security': CliCommand.guardedKeyword( 'security',
                                             helpdesc='Show monitor security',
                                             guard=guardMonitorSecurity ),
      'awake': awakeKw,
      'flow-table': CliCommand.singleKeyword( 'flow-table',
                                              helpdesc='Show current flows status' ),
      'detail': CliCommand.singleKeyword( 'detail',
                                          helpdesc='More comprehensive output' ),
      'group': CliCommand.singleKeyword( 'group', helpdesc='Flow group' ),
      'GROUP_NAME': CliCommand.singleNode( afMatcherForShow ),
      'protocol': KeywordMatcher( 'protocol', "Flow IP protocol" ),
      'PROTOCOL': CliCommand.singleNode(
         EnumMatcher( { 'tcp': 'TCP', 'udp': 'UDP' } ) ),
      'ip': CliCommand.singleKeyword( 'ip',
                        helpdesc='Source or destination IP address of the flow' ),
      'IP': ipGenAddrMatcher,
      'peer-ip': CliCommand.singleKeyword( 'peer-ip',
                           helpdesc='Source or destination IP address of the flow' ),
      'PEER_IP': ipGenAddrMatcher,
      'port': CliCommand.singleKeyword( 'port',
                              helpdesc='Source or destination port of the flow' ),
      'PORT': tokenPort,
      'peer-port': CliCommand.singleKeyword( 'peer-port',
                                 helpdesc='Source or destination port of the flow' ),
      'PEER_PORT': tokenPort,
   }

   handler = 'FlowWatcherShowCli.showFlowTableHandler'
   cliModel = 'FlowWatcherModel.FlowTable'

addShowCommandClass( ShowMsaFlowTable )

# ----------------------------------------------------------------
# show monitor security awake dpi memory
# ----------------------------------------------------------------
class ShowMsaDpiMemory( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor security awake dpi memory'

   data = {
      'monitor': monitorMatcherForShow,
      'security': securityKw,
      'awake': awakeKw,
      'dpi': dpiKw,
      'memory': memoryKw
   }

   handler = 'FlowWatcherShowCli.showMsaDpiMemoryHandler'
   cliModel = 'FlowWatcherModel.MsaDpiMemory'

addShowCommandClass( ShowMsaDpiMemory )

# ----------------------------------------------------------------
# show monitor security awake dpi memory allocations [ nz ] [ nt ]
# [ sort ( current | minimum | maximum ) ]
# ----------------------------------------------------------------
class ShowMsaDpiMemoryAllocations( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor security awake dpi memory allocations [ nz ] [ nt ]'
              '[ sort ( current | minimum | maximum ) ]' )

   data = {
      'monitor': monitorMatcherForShow,
      'security': securityKw,
      'awake': awakeKw,
      'dpi': dpiKw,
      'memory': memoryKw,
      'allocations': 'Show memory allocations',
      'nz': 'Only non-zero allocations',
      'nt': 'Exclude time ago in text output',
      'sort': 'Sort order for text output',
      'current': 'Sort by current allocations',
      'minimum': 'Sort by minimum allocations',
      'maximum': 'Sort by maximum allocations',
   }

   handler = 'FlowWatcherShowCli.showMsaDpiMemoryAllocationsHandler'
   cliModel = 'FlowWatcherModel.MsaDpiMemoryAllocations'

addShowCommandClass( ShowMsaDpiMemoryAllocations )

# ----------------------------------------------------------------
# show monitor security awake dpi memory pools
# ----------------------------------------------------------------
class ShowMsaDpiMemoryPools( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor security awake dpi memory pools'

   data = {
      'monitor': monitorMatcherForShow,
      'security': securityKw,
      'awake': awakeKw,
      'dpi': dpiKw,
      'memory': memoryKw,
      'pools': 'Show pre-allocated memory pools'
   }

   handler = 'FlowWatcherShowCli.showMsaDpiMemoryPoolsHandler'
   cliModel = 'FlowWatcherModel.MsaDpiMemoryPools'

addShowCommandClass( ShowMsaDpiMemoryPools )

# -------------------------------------------------------------------------------
# Commands to be run for 'show tech-support'
# -------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2022-02-11 15:45:00',
   cmds=[
      'show monitor security awake',
      'show monitor security awake counters',
      'show monitor security awake dpi memory',
      'show monitor security awake dpi memory pools',
   ],
   cmdsGuard=guardMonitorSecurityForShowTech )
