#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import DeferredModel, Submodel, Str, Int, List, Dict, Bool, Float, Enum
from ArnetModel import IpGenericPrefix, IpGenericAddress
from IntfModels import Interface
from TunnelTypeLib import tunnelTypeEnumValues

class BitMask( DeferredModel ):
   value = Int( help="Value" )
   mask = Int( help="Mask" )

class NumericalRange( DeferredModel ):
   minVal = Int( help="Minimum value" )
   maxVal = Int( help="Maximum value" )

class Matches( DeferredModel ):
   destPrefix = IpGenericPrefix( help="Destination prefix", optional=True )
   srcPrefix = IpGenericPrefix( help="Source prefix", optional=True )
   nextProtocol = List( valueType=NumericalRange, help="Next protocol",
                        optional=True )
   port = List( valueType=NumericalRange, help="Port", optional=True )
   destPort = List( valueType=NumericalRange, help="Destination port",
                    optional=True )
   srcPort = List( valueType=NumericalRange, help="Source port", optional=True )
   icmpType = List( valueType=NumericalRange, help="ICMP type", optional=True )
   icmpCode = List( valueType=NumericalRange, help="ICMP code", optional=True )
   packetLength = List( valueType=NumericalRange, help="Packet length",
                        optional=True )
   dscp = List( valueType=NumericalRange, help="DSCP", optional=True )
   tcpFlags = List( valueType=BitMask, help="TCP flags", optional=True )
   fragmentFlags = List( valueType=BitMask, help="Fragment flags",
                         optional=True )
   flowLabels = List( valueType=NumericalRange, help="Flow label",
                      optional=True )

class ViaNexthop( DeferredModel ):
   nexthopIp = IpGenericAddress( help="IP address of nexthop for redirect" )
   tunnelType = Enum( values=tunnelTypeEnumValues, help="Tunnel type",
                      optional=True )
   tunnelIndex = Int( help="Tunnel index within tunnel type", optional=True )
   mplsLabel = Int( help="MPLS label", optional=True )

class TrafficRedirect( DeferredModel ):
   vrf = Str( help="VRF name" )
   nexthop = IpGenericAddress( help="IP address of nexthop for redirect",
                               optional=True )
   mirror = Bool( help="Mirror the packet instead of redirecting it" )
   viaNexthop = List( valueType=ViaNexthop, help="Redirect via the nexthop" )

class TrafficAction( DeferredModel ):
   sample = Bool( help="Traffic sampling and logging is enabled" )
   nonTerminal = Bool( help="Traffic-filtering engine may evaluate "
                       "or stops filtering based on this bit" )

class Actions( DeferredModel ):
   trafficDrop = Bool( help="Drop action", optional=True )
   trafficRate = Float( help="Police traffic based on bits per second (bps)",
                        optional=True )
   trafficRedirect = Submodel( valueType=TrafficRedirect,
                               help="Traffic redirect action", optional=True )
   trafficMarking = Int( help="Mark DSCP action", optional=True )
   trafficRatePkt = Float( help="Police traffic based on packets "
                           "per second (pps)", optional=True )
   trafficAction = Submodel( valueType=TrafficAction, help="Traffic action to sample"
                            " or stop traffic filtering", optional=True )

class CounterInfo( DeferredModel ):
   packetCount = Int( help="Number of packets", optional=True )
   byteCount = Int( help="Number of bytes", optional=True )

class PoliceCount( DeferredModel ):
   belowCount = Submodel( valueType=CounterInfo,
                          help="Number of packets below police rate", optional=True )
   aboveCount = Submodel( valueType=CounterInfo,
                         help="Number of packets above police rate", optional=True )

class Status( DeferredModel ):
   installed = Bool( help="Rule installed in hardware" )
   reason = Str( help="Reason for not programmed in hardware", optional=True )
   packetCount = Int( help="Number of packets hitting the rule", optional=True )
   byteCount = Int( help="Number of bytes hitting the rule", optional=True )
   policeCount = Submodel( valueType=PoliceCount,
                           help="Descriptive count for police", optional=True )

class FeasibleFlowspecRule( DeferredModel ):
   ruleString = Str( help="Flow-spec rule string representation" )
   ruleId = Int( help="Flow-spec rule identifier" )
   matches = Submodel( valueType=Matches, help="Matches for the rule" )
   actions = Submodel( valueType=Actions, help="Actions for the rule" )
   interfaceSets = List( valueType=int,
                         help="List of interface-sets for the rule",
                         optional=True )
   unsupportedActions = Submodel( valueType=Actions,
                                  help="Unsupported actions", optional=True )
   status = Submodel( valueType=Status, help="Status for the rule" )

class InfeasibleFlowspecRule( DeferredModel ):
   ruleString = Str( help="Flow-spec rule string representation" )
   ruleId = Str( help="Flow-spec rule identier" )
   infeasibleMatch = Str( help="Infeasible match" )

class FlowspecInterfaceSet( DeferredModel ):
   activeIntfs = List( valueType=Interface,
                       help="List of interface-set active interfaces" )

class VrfFlowspecRule( DeferredModel ):
   configuredIntfs = List( valueType=Interface,
                           help="List of interfaces configured with flow-spec" )
   activeIntfs = List( valueType=Interface,
                       help="List of interfaces applied with flow-spec" )
   feasibleRules = List( valueType=FeasibleFlowspecRule,
                         help="Feasible flow-spec rules indexed by rule string" )
   infeasibleRules = List( valueType=InfeasibleFlowspecRule,
                           help="Infeasible flow-spec rules indexed by rule string" )
   interfaceSets = Dict( valueType=FlowspecInterfaceSet,
                         help="Flow-spec interface-sets indexed by set identifier",
                         optional=True )

class FlowspecModel( DeferredModel ):
   vrfs = Dict( valueType=VrfFlowspecRule,
                help="Flow-spec rules indexed by VRF name" )

class VrfFlowspecRuleSummary( DeferredModel ):
   totalCount = Int( help="Total number of flow-spec rules" )
   installedCount = Int( help="Number of flow-spec rules installed in hardware" )
   unInstalledCount = Int(
      help="Number of flow-spec rules not installed in hardware" )

class FlowspecSummaryModel( DeferredModel ):
   vrfs = Dict( valueType=VrfFlowspecRuleSummary,
                help="Flow-spec rule summary indexed by VRF name" )
   allVrfTotalCount = Int( help="Total number of flow-spec rules across all VRFs" )
   allVrfInstalledCount = Int(
      help="Number of flow-spec rules installed in hardware across all VRFs" )
