#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

''' CLI commands for configuring Flowspec. '''

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliGlobal
import ConfigMount
import LazyMount
from Arnet import IpGenAddr
from CliPlugin import (
   IntfCli,
   VrfCli,
   IpGenAddrMatcher,
)
from CliPlugin.TrafficPolicyCliContext import TrafficPolicyBaseContext
from CliPlugin.TrafficPolicyCliLib import ( PolicyConfigCmdBase,
                                            matchIpAccessGroup,
                                            matchIpv6AccessGroup,
                                            MatchRuleContext,
                                            ActionType,
                                            RateUnit,
                                            rateUnitTokens,
                                            UniqueId )
from CliPlugin.TrafficPolicyCli import matchRuleName as matchRuleNameMatcher
from CliPlugin.TrafficPolicyCli import ( ActionsConfigCmdBase,
                                         DscpConfigCmd,
                                         MatchAllFragmentConfigCmd,
                                         IpLengthConfigCmd,
                                         MatchRuleConfigCmdBase,
                                         ProtocolIcmpV4ConfigCmd,
                                         ProtocolIcmpV4TypeCodeConfigCmd,
                                         ProtocolIpv4ListConfigCmd,
                                         ProtocolIcmpV6ConfigCmd,
                                         ProtocolIcmpV6TypeCodeConfigCmd,
                                         ProtocolIpv6ListConfigCmd,
                                         ProtocolTcpFlagsOnlyCmd,
                                         ActionCmdBase,
                                         DropActionWithoutNotifCmd,
                                         SetDscpActionCmd, )
from CliPlugin.ClassificationCliLib import ( PrefixCmdMatcher,
                                             PrefixCmdBaseV2,
                                             CommitAbortModelet, )
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher
from RouteMapLib import getExtCommValue, getNumberFromCommValue

import os
import Tac
import Tracing
from CliMode.Flowspec import (
   FlowspecConfigMode,
   POLICYMAP_FEATURE,
   FlowspecPoliciesConfigMode,
   FlowspecPolicyConfigMode,
   FlowspecPolicyMatchIpv4RuleConfigMode,
   FlowspecPolicyMatchIpv6RuleConfigMode,
   FlowspecPolicyActionsConfigMode,
)
from FlowspecCliLib import flowspecCapable
from TypeFuture import TacLazyType
from MultiRangeRule import MultiRangeMatcher
from Toggles import (
   RoutingLibToggleLib,
   PolicyMapToggleLib,
)

t0 = Tracing.trace0
UnresolvedNexthopAction = TacLazyType( "PolicyMap::PbrUnresolvedNexthopAction" )
FlowspecConfig = TacLazyType( "Flowspec::Config" )
FlowspecInterfaceSetCfg = TacLazyType( "Flowspec::FlowspecIntfInterfaceSetConfig" )
routingHwStatus = None

gv = CliGlobal.CliGlobal(
   cliConfig=None,
   flowspecPbrConfig=None,
   flowspecPolicyConfig=None,
   flowspecPolicyStatus=None,
   flowspecPolicyStatusReqDir=None
)

ipv4MatcherForConfig = CliMatcher.KeywordMatcher(
      'ipv4', helpdesc="Match on IPv4 traffic" )
ipv6MatcherForConfig = CliMatcher.KeywordMatcher(
      'ipv6', helpdesc="Match on IPv6 traffic" )
flowSpecMatcherForInterfaceSet = CliMatcher.KeywordMatcher(
      'interface-set', helpdesc='Configure interface-set membership' )
sourceMatcherForConfig = CliMatcher.KeywordMatcher(
   'source', helpdesc="Specify a different source vrf for flowspec rules" )
vrfMatcherForConfig = CliMatcher.KeywordMatcher(
   'vrf', helpdesc="Specify a different source vrf for flowspec rules" )
overrideMatcherForConfig = CliMatcher.KeywordMatcher(
   'override', helpdesc="Specify a different source vrf for flowspec rules" )

# If Interface-set is enabled change the help string from the original
# enable-centric text to a more general configuration string
if RoutingLibToggleLib.toggleBgpFlowspecInterfaceSetEnabled():
   flowSpecMatcherForConfigIf = CliMatcher.KeywordMatcher(
      'flow-spec', helpdesc='Configure flow-spec parameters' )
else:
   flowSpecMatcherForConfigIf = CliMatcher.KeywordMatcher(
      'flow-spec', helpdesc='Enable flow-spec' )

#-------------------------------------------------------------------------------
# Adds flowspec specific CLI commands to the "config-if" mode for routed ports.
#-------------------------------------------------------------------------------
class FlowspecIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return flowspecCapable( mode.intf.name )

#-------------------------------------------------------------------------------
# Associate the FlowspecIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( FlowspecIntfConfigModelet )

modelet = FlowspecIntfConfigModelet

#-------------------------------------------------------------------------------
# Register a dependent class for all interfaces to cleanup on Intf deletion
#-------------------------------------------------------------------------------
class FlowspecIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "Intf deletion. Cleanup", self.intf_.name )
      del gv.cliConfig.enabledIntf[ self.intf_.name ]
      del gv.cliConfig.intfInterfaceSetConfig[ self.intf_.name ]

def configureFlowspec( intfName, ipv4=None, ipv6=None, sourceVrf=None ):
   t0( "intf", intfName, "ipv4", ipv4, "ipv6", ipv6 )
   currCfg = gv.cliConfig.enabledIntf.get( intfName )
   newIpVersion = Tac.Value( 'Flowspec::IpVersion' )
   # Initialize the newly configured ip version to what's currently
   # available and override the current config with the args of
   # ipv4/ipv6 provided they are not None.
   if currCfg:
      newIpVersion.ipv4 = currCfg.ipVersion.ipv4
      newIpVersion.ipv6 = currCfg.ipVersion.ipv6
   if ipv4 is not None:
      newIpVersion.ipv4 = ipv4
   if ipv6 is not None:
      newIpVersion.ipv6 = ipv6
   nextCfg = Tac.Value( 'Flowspec::InterfaceEnableConfig', intfName )
   nextCfg.ipVersion = newIpVersion
   if nextCfg.ipVersion and sourceVrf is not None:
      nextCfg.sourceVrfOverride = sourceVrf
   if not nextCfg.isDefaults():
      t0( "Enabling", nextCfg.ipVersion, "on", intfName )
      gv.cliConfig.enabledIntf.addMember( nextCfg )
   else:
      t0( "Interface config is at defaults. Delete", intfName )
      del gv.cliConfig.enabledIntf[ intfName ]

#-------------------------------------------------------------------------------
# config-if# [ no | default ] flow-spec ipv4 ipv6
# config-if# [ no | default ] flow-spec ipv4
# config-if# [ no | default ] flow-spec ipv6
#-------------------------------------------------------------------------------

# When ipv4 (or ipv6) is enabled, the user only wants ipv4 (or ipv6) to be turned on
# and not ipv6 (or ipv4).
# When ipv4 (or ipv6) is disabled, the user only wants to turn off ipv4 (or ipv6)
# and use the current operational state of ipv6 (or ipv4).
def handleFlowspecIpv4Ipv6( mode, args, enable=True ):
   # if enable=False, this function has been called as the noOrDefaultHandler:
   # when not in the args, the value will be None
   disable = False if enable else None
   ipv4 = enable if 'ipv4' in args else disable
   ipv6 = enable if 'ipv6' in args else disable
   sourceVrf = args.get( 'VRFNAME', None ) if enable else None
   configureFlowspec( mode.intf.name, ipv4=ipv4, ipv6=ipv6, sourceVrf=sourceVrf )

def disableFlowspecIpv4Ipv6( mode, args ):
   handleFlowspecIpv4Ipv6( mode, args, enable=False )

flowSpecSyntax = 'flow-spec ( ipv4 ipv6 )'
# Selective toggle of ipv4 is not supported yet.
if os.environ.get( 'FLOWSPEC_ENABLE_IPV4' ):
   flowSpecSyntax += ' | ipv4'
# Selective toggle of ipv6 is not supported yet.
if os.environ.get( 'FLOWSPEC_ENABLE_IPV6' ):
   flowSpecSyntax += ' | ipv6'
flowSpecSyntax += ' [ source vrf VRFNAME override ]'

# When FLOWSPEC_ENABLE_IPV{4,6} are in env, the full syntax would be
# flow-spec ( ipv4 ipv6 ) | ipv4 | ipv6

# -------------------------------------------------------------------------------
# Flowspec Interface enable configuration command
# config-if
#    [ no|default ] flow-spec { ipv4 [ ipv6 ] | ipv6 }
#          [ source vrf VRFNAME override ]
# -------------------------------------------------------------------------------
class FlowSpecIpv4Ipv6Cmd( CliCommand.CliCommandClass ):
   syntax = flowSpecSyntax
   noOrDefaultSyntax = syntax

   data = {
      'flow-spec' : flowSpecMatcherForConfigIf,
      'ipv4' : ipv4MatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'source' : sourceMatcherForConfig,
      'vrf' : vrfMatcherForConfig,
      'VRFNAME' : CliMatcher.DynamicNameMatcher(
         VrfCli.getAllVrfNames,
         helpdesc='VRF name' ),
      'override' : overrideMatcherForConfig,
   }
   handler = handleFlowspecIpv4Ipv6
   noOrDefaultHandler = disableFlowspecIpv4Ipv6

modelet.addCommandClass( FlowSpecIpv4Ipv6Cmd )

def setFlowspecIntfInterfaceSetMembership( intfName, setIds ):
   '''Set interface interface-set configuration'''
   t0( "intf", intfName, "interface-set", setIds )
   intfSets = gv.cliConfig.intfInterfaceSetConfig.get( intfName, None )
   # Prune removed sets
   if not setIds:
      if intfSets:
         del gv.cliConfig.intfInterfaceSetConfig[ intfName ]
      return
   elif intfSets:
      for setId in intfSets.interfaceSet:
         if setId not in setIds:
            del intfSets.interfaceSet[ setId ]
   else:
      intfSets = gv.cliConfig.intfInterfaceSetConfig.newMember( intfName )
   # Add set membership
   for setId in setIds:
      intfSets.interfaceSet[ setId ] = True

def clearFlowspecIntfInterfaceSetMembership( intfName ):
   '''Clear interface interface-set configuration'''
   t0( "intf", intfName )
   del gv.cliConfig.intfInterfaceSetConfig[ intfName ]

# -------------------------------------------------------------------------------
# Flowspec Interface interface-set configuration command to add if platform supports
# an interface in many interface-sets.
# config-if
#    [ no|default ] flow-spec interface-set IDs
# -------------------------------------------------------------------------------
class FlowSpecOneToManyInterfaceSetCmd( CliCommand.CliCommandClass ):
   syntax = 'flow-spec interface-set IDs'
   noOrDefaultSyntax = 'flow-spec interface-set ...'

   data = {
      'flow-spec' : flowSpecMatcherForConfigIf,
      'interface-set' : flowSpecMatcherForInterfaceSet,
      'IDs' : MultiRangeMatcher(
         rangeFn=lambda : ( FlowspecConfig.minInterfaceSetId,
                            FlowspecConfig.maxInterfaceSetId ),
         noSingletons=False,
         helpdesc="Interface-set IDs" )
   }

   @staticmethod
   def handler( mode, args ):
      # arg IDs is a MultiRangeMatcher capable of matching input ranges
      # (e.g. 1-2,3,5) that is expanded to get the list of all values in the
      # range (e.g. [ 1, 2, 3, 5] )
      rangeMatcher = args.get( 'IDs' )
      setFlowspecIntfInterfaceSetMembership( mode.intf.name, rangeMatcher.values() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearFlowspecIntfInterfaceSetMembership( mode.intf.name )

# -------------------------------------------------------------------------------
# Flowspec Interface interface-set configuration command to add if an interface is
# only allowed in one interface-set.
# config-if
#    [ no|default ] flow-spec interface-set ID
# -------------------------------------------------------------------------------
class FlowSpecOneToOneInterfaceSetCmd( CliCommand.CliCommandClass ):
   syntax = 'flow-spec interface-set ID'
   noOrDefaultSyntax = 'flow-spec interface-set ...'

   data = {
      'flow-spec' : flowSpecMatcherForConfigIf,
      'interface-set' : flowSpecMatcherForInterfaceSet,
      'ID' : CliMatcher.IntegerMatcher( FlowspecConfig.minInterfaceSetId,
                                        FlowspecConfig.maxInterfaceSetId )
   }

   @staticmethod
   def handler( mode, args ):
      setFlowspecIntfInterfaceSetMembership( mode.intf.name, [ args.get( 'ID' ) ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearFlowspecIntfInterfaceSetMembership( mode.intf.name )

if RoutingLibToggleLib.toggleBgpFlowspecInterfaceSetEnabled():
   if os.environ.get( 'FLOWSPEC_INTF_ENABLE_ONE_TO_MANY_INTFSET' ):
      modelet.addCommandClass( FlowSpecOneToManyInterfaceSetCmd )
   else:
      modelet.addCommandClass( FlowSpecOneToOneInterfaceSetCmd )

def removeAllFlowspecPolicies( mode ):
   if gv.flowspecPolicyConfig:
      for policyName in gv.flowspecPolicyConfig.pmapType.pmap:
         # pylint: disable-next=protected-access
         FlowspecPolicyConfigCmd._removePolicy( mode, policyName )

# -------------------------------------------------------------------------------
# Flowspec controls configuration command
#    [ no|default ] flow-spec
# -------------------------------------------------------------------------------
class FlowspecConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'flow-spec'
   noOrDefaultSyntax = syntax
   data = {
      'flow-spec' : 'Configure flow-spec controls'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( FlowspecConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noAction = UnresolvedNexthopAction.na
      gv.cliConfig.interfaceSetCountLimit = 0
      gv.flowspecPbrConfig.pbrUnresolvedNexthopAction = noAction
      gv.flowspecPbrConfig.counterPollInterval = 0.0
      gv.cliConfig.isConfigDriven = False
      removeAllFlowspecPolicies( mode )

BasicCli.GlobalConfigMode.addCommandClass( FlowspecConfigCmd )

def guardL2Flowspec( mode, token ):
   if routingHwStatus.l2FlowspecSupported:
      return None
   return CliParser.guardNotThisPlatform

# -------------------------------------------------------------------------------
#    [ no|default ] hardware persistent
# -------------------------------------------------------------------------------
class HardwarePersistentFlowspecCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware persistent'
   noOrDefaultSyntax = syntax
   data = {
      'hardware' : CliCommand.guardedKeyword( 'hardware',
                                              'Hardware update configuration',
                                              guard=guardL2Flowspec ),
      'persistent' : 'persistence of rules in the hardware across interface '
                     'state changes'
   }

   @staticmethod
   def handler( mode, args ):
      gv.cliConfig.isConfigDriven = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.cliConfig.isConfigDriven = False

if PolicyMapToggleLib.toggleFlowspecConfigDrivenEnabled():
   FlowspecConfigMode.addCommandClass( HardwarePersistentFlowspecCmd )

# -------------------------------------------------------------------------------
#    [ no|default ] next-hop unresolved drop [ immediate ]
# -------------------------------------------------------------------------------
class NexhopUnresolvedDropCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop unresolved drop [ immediate ]'
   noOrDefaultSyntax = 'next-hop unresolved [ drop [ immediate ] ]'
   data = {
      'next-hop' : 'Configure next-hop behavior for policy',
      'unresolved' : ( 'Configure the action to take when next-hops in the policy '
                       'are unresolved' ),
      'drop' : 'Drop packets',
      'immediate' : 'Drop packets immediately',
   }

   @staticmethod
   def handler( mode, args ):
      if 'immediate' in args:
         dropImmediate = UnresolvedNexthopAction.dropImmediate
         gv.flowspecPbrConfig.pbrUnresolvedNexthopAction = dropImmediate
      else:
         dropDelayed = UnresolvedNexthopAction.dropDelayed
         gv.flowspecPbrConfig.pbrUnresolvedNexthopAction = dropDelayed

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hasDrop = 'drop' in args
      hasDropImmediate = 'immediate' in args
      currAction = gv.flowspecPbrConfig.pbrUnresolvedNexthopAction

      noAction = UnresolvedNexthopAction.na
      dropImmediate = UnresolvedNexthopAction.dropImmediate
      dropDelayed = UnresolvedNexthopAction.dropDelayed
      # Remove dropAction only when current unresolved next-hop action matches
      # the configured drop type
      if hasDropImmediate:
         if currAction != dropImmediate:
            return
      elif hasDrop:
         if currAction != dropDelayed:
            return
      gv.flowspecPbrConfig.pbrUnresolvedNexthopAction = noAction

FlowspecConfigMode.addCommandClass( NexhopUnresolvedDropCmd )

# -------------------------------------------------------------------------------
#    [ no|default ] counters poll interval SECONDS seconds
# -------------------------------------------------------------------------------
class CountersPollIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'counters poll interval SECONDS seconds'
   noOrDefaultSyntax = 'counters poll interval ...'
   data = {
      'counters' : 'Counters update configuration',
      'poll' : 'Counters polling configuration',
      'interval' : 'Counters update interval',
      'SECONDS' : CliMatcher.IntegerMatcher( 30, 300,
         helpdesc='Seconds between consecutive polls' ),
      'seconds' : 'Time unit in seconds',
   }

   @staticmethod
   def handler( mode, args ):
      gv.flowspecPbrConfig.counterPollInterval = args[ 'SECONDS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.flowspecPbrConfig.counterPollInterval = 0.0

if PolicyMapToggleLib.toggleFlowspecSharkCountersEnabled():
   FlowspecConfigMode.addCommandClass( CountersPollIntervalCmd )

interfaceSetCountLimitMatcher = CliMatcher.EnumMatcher( {
      '3' : 'Limit to 3 interface-sets',
      '7' : 'Limit to 7 interface-sets',
      '15' : 'Limit to 15 interface-sets',
      '31' : 'Limit to 31 interface-sets',
      '63' : 'Limit to 63 interface-sets',
      '127' : 'Limit to 127 interface-sets',
      '255' : 'Limit to 255 interface-sets',
       } )
# -------------------------------------------------------------------------------
#    [ no|default ] interface-set count limit LIMIT
# -------------------------------------------------------------------------------

class InterfaceSetCountLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'interface-set count limit LIMIT'
   noOrDefaultSyntax = ' interface-set count limit ...'
   data = {
      'interface-set' : 'Configure interface-set count limit',
      'count' : 'count',
      'limit' : 'limit',
      'LIMIT' : interfaceSetCountLimitMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      gv.cliConfig.interfaceSetCountLimit = int( args[ 'LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.cliConfig.interfaceSetCountLimit = 0

if RoutingLibToggleLib.toggleBgpFlowspecInterfaceSetEnabled():
   FlowspecConfigMode.addCommandClass( InterfaceSetCountLimitCmd )

class FlowspecPolicyContext( TrafficPolicyBaseContext ):
   pmapType = 'mapTrafficPolicy'

   def childMode( self ):
      return FlowspecPolicyConfigMode

   def mapTypeStr( self ):
      return "flow-spec-policy"

   def reservedClassMapNames( self ):
      # No 'ipvX-all-default' rules needed in flowspec
      return []

   def copyAction( self, src ):
      actionType = src.actionType
      actions = self.config().actions
      if actionType == ActionType.redirectVrfRouteTarget:
         return actions.redirectVrfRtAction.newMember( src.className, UniqueId(),
                                                       src.vrfRt )
      return super().copyAction( src )

   def removeAction( self, action ):
      actionType = action.actionType
      actions = self.config().actions
      if actionType == ActionType.redirectVrfRouteTarget:
         del actions.redirectVrfRtAction[ action.id ]
      else:
         super().removeAction( action )

class FlowspecPolicyMatchRuleContext( MatchRuleContext ):
   def childMode( self, matchRuleName, matchOption ):
      if matchOption == matchIpAccessGroup:
         return FlowspecPolicyMatchIpv4RuleConfigMode
      elif matchOption == matchIpv6AccessGroup:
         return FlowspecPolicyMatchIpv6RuleConfigMode
      assert False, 'unknown matchOption ' + matchOption
      return None

   def actionMode( self ):
      return FlowspecPolicyActionsConfigMode

   def addPrefixFromSf( self, prefixSf ):
      if prefixSf.source:
         self.filter.source.clear()
      if prefixSf.destination:
         self.filter.destination.clear()
      super().addPrefixFromSf( prefixSf )

   def setAction( self, actionType, actionValue=None, no=False, clearActions=None ):
      if no:
         self.matchRuleAction.delAction( actionType )
      else:
         if clearActions:
            for action in clearActions:
               self.matchRuleAction.delAction( action )
         actionsConfig = self.trafficPolicyContext.config().actions
         if actionType == ActionType.deny:
            action = actionsConfig.dropAction.newMember( self.trafficPolicyName,
                                                         UniqueId() )
         elif actionType == ActionType.police:
            action = actionsConfig.policeAction.newMember( self.trafficPolicyName,
                                                           UniqueId(),
                                                           actionValue[ 0 ],
                                                           actionValue[ 1 ] )
         elif actionType == ActionType.setNexthop:
            action = actionsConfig.setNexthopAction.newMember(
               self.trafficPolicyName, UniqueId() )
            action.nexthop[ IpGenAddr( str( actionValue ) ) ] = True
         elif actionType == ActionType.setDscp:
            action = actionsConfig.setDscpAction.newMember( self.trafficPolicyName,
                                                            UniqueId(),
                                                            actionValue )
         elif actionType == ActionType.redirectVrfRouteTarget:
            rtExtComm = getExtCommValue( 'rt ' + actionValue )
            routeTarget = getNumberFromCommValue( rtExtComm )
            action = actionsConfig.redirectVrfRtAction.newMember(
               self.trafficPolicyName,
               UniqueId(),
               routeTarget )
         else:
            assert False, 'unknown action ' + actionType
         self.matchRuleAction.addAction( actionType, action )

# -------------------------------------------------------------------------------
# [ no|default ] policies
# Flowspec policies configuration under flow-spec config mode
# -------------------------------------------------------------------------------
class FlowspecPoliciesConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'policies'
   noOrDefaultSyntax = syntax
   data = {
      'policies' : 'Configure flow-spec policies'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( FlowspecPoliciesConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeAllFlowspecPolicies( mode )

def flowspecPolicyNames( mode ):
   if not gv.flowspecPolicyConfig:
      return []
   return iter( gv.flowspecPolicyConfig.pmapType.pmap )

flowspecPolicyNameMatcher = CliMatcher.DynamicNameMatcher( flowspecPolicyNames,
                                                           "Policy name" )

# -------------------------------------------------------------------------------
# [ no ] policy < policy-name >
# Flowspec policy configuration under flow-spec-policies config mode
# -------------------------------------------------------------------------------
class FlowspecPolicyConfigCmd( PolicyConfigCmdBase ):
   syntax = 'policy POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'policy' : 'Configure flow-spec policy',
         'POLICY_NAME' : flowspecPolicyNameMatcher,
   }

   @staticmethod
   def _feature():
      return POLICYMAP_FEATURE

   @classmethod
   def _context( cls, name, mode ):
      return FlowspecPolicyContext( gv.flowspecPolicyConfig,
                                    gv.flowspecPolicyStatusReqDir,
                                    gv.flowspecPolicyStatus, name )

# -------------------------------------------------------------------------------
# [ no ] match <match-rule-name> ( ipv4 | ipv6 )
# Flowspec policy match rule configuration under flow-spec-policy config mode
# -------------------------------------------------------------------------------
class FlowspecPolicyMatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = 'match RULE_NAME ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 ]'
   data = {
      'match' : 'Configure match rule',
      'RULE_NAME' : matchRuleNameMatcher,
      'ipv4' : 'IPv4 match rule',
      'ipv6' : 'IPv6 match rule',
   }

   @staticmethod
   def _feature():
      return POLICYMAP_FEATURE

   @staticmethod
   # pylint: disable-next=arguments-renamed
   def _context( policyContext, ruleName, matchOption ):
      return FlowspecPolicyMatchRuleContext( policyContext, ruleName, matchOption )

# ---------------------------------------------------------
# [ no ] source prefix <A.B.C.D/E>
# Source prefix configuration under flowspec match IPv4 config mode
# ---------------------------------------------------------
class FlowspecPolicySourcePrefixIpv4Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixCmdMatcher( addrType="ipv4", allowMultiple=False,
                                       allowSource=True, allowDestination=False )
   }

# ---------------------------------------------------------
# [ no ] source prefix <A:B:C:D:E:F:G:H/I>
# Source prefix configuration under flowspec match IPv6 config mode
# ---------------------------------------------------------
class FlowspecPolicySourcePrefixIpv6Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixCmdMatcher( addrType="ipv6", allowMultiple=False,
                                       allowSource=True, allowDestination=False )
   }

# ---------------------------------------------------------
# [ no ] destination prefix <A.B.C.D/E>
# Destination prefix configuration under flowspec match IPv4 config mode
# ---------------------------------------------------------
class FlowspecPolicyDestPrefixIpv4Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixCmdMatcher( addrType="ipv4", allowMultiple=False,
                                       allowSource=False, allowDestination=True )
   }

# ---------------------------------------------------------
# [ no ] destination prefix <A:B:C:D:E:F:G:H/I>
# Destination prefix configuration under flowspec match IPv6 config mode
# ---------------------------------------------------------
class FlowspecPolicyDestPrefixIpv6Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR" : PrefixCmdMatcher( addrType="ipv6", allowMultiple=False,
                                       allowSource=False, allowDestination=True )
   }

# --------------------------------------------------------------------------------
# The "actions" sub-prompt mode command under flowspec match IPv4/IPv6 rule
# config mode
# ---------------------------------------------------------------------------------
class FlowspecPolicyActionsConfigCmd( ActionsConfigCmdBase ):
   data = ActionsConfigCmdBase.data.copy()

   @staticmethod
   def _feature():
      return POLICYMAP_FEATURE

# Flowspec max rate is actually much larger since it is an F32. However, we are
# storing the rate in a U64 so limit to that max. This will not overflow the F32
# even considering the gpps multiplier.
flowspecMaxRateValue = 0xFFFFFFFFFFFFFFFF
ppsRateUnits = {
   RateUnit.pps : 'Rate in pps',
   RateUnit.kpps : 'Rate in kpps',
   RateUnit.mpps : 'Rate in mpps',
   RateUnit.gpps : 'Rate in gpps',
}
flowspecRateUnitTokens = rateUnitTokens.copy()
flowspecRateUnitTokens.update( ppsRateUnits )

# ---------------------------------------------------------------------------------
# [ no ] police rate RATE_VALUE [ RATE_UNIT ]
# Police rate configuration under flowspec action config mode
# ---------------------------------------------------------------------------------
class FlowspecPoliceActionCmd( ActionCmdBase ):
   _actionType = ActionType.police

   syntax = '''police rate RATE_VALUE [ RATE_UNIT ]'''
   noOrDefaultSyntax = '''police ...'''
   data = {
      'police' : 'Configure policer parameters',
      'rate' : 'Set rate limit',
      'RATE_VALUE' : CliMatcher.IntegerMatcher( 1, flowspecMaxRateValue,
         helpdesc='Specify rate limit value (Default Unit is Kbps)' ),
      'RATE_UNIT' : CliMatcher.EnumMatcher( flowspecRateUnitTokens ),
   }

   @classmethod
   def handler( cls, mode, args ):
      rateValue = args[ 'RATE_VALUE' ]
      rateUnit = args.get( 'RATE_UNIT', 'kbps' )
      # Burst rate is not needed for flowspec, set to defaults
      policeRate = Tac.newInstance( "PolicyMap::PoliceRate", rateValue, rateUnit )
      policeBurstSize = Tac.newInstance( "PolicyMap::PoliceBurstSize", 0, 'bytes' )
      policeAction = [ policeRate, policeBurstSize ]
      mode.context.setAction( cls._actionType, policeAction, no=False )

# ---------------------------------------------------------------------------
# [ no ] redirect next-hop NEXTHOP
# Redirect next-hop configuration under flowspec action config mode
# ----------------------------------------------------------------------------
class FlowspecRedirectNexthopCmd( ActionCmdBase ):
   _actionType = ActionType.setNexthop

   syntax = '''redirect next-hop NEXTHOP'''
   noOrDefaultSyntax = "redirect next-hop ..."
   data = {
      'redirect' : 'Redirect packet flow',
      'next-hop' : 'Redirect to next-hop',
      'NEXTHOP' : IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IPv4 or IPv6 nexthop address',
         helpdesc4='IPv4 nexthop address',
         helpdesc6='IPv6 nexthop address' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      nhAddr = args[ 'NEXTHOP' ]
      mode.context.setAction( cls._actionType, nhAddr, no=False )

# ---------------------------------------------------------------------------
# [ no ] redirect route-target RT
# Redirect route-target configuration under flowspec action config mode
# ----------------------------------------------------------------------------
class FlowspecRedirectVrfRtActionCmd( ActionCmdBase ):
   _actionType = ActionType.redirectVrfRouteTarget

   syntax = '''redirect route-target RT'''
   noOrDefaultSyntax = '''redirect route-target ...'''
   data = {
      'redirect' : 'Redirect packet flow',
      'route-target' : 'Redirect to VRF route-target',
      'RT' : RtSooExtCommCliMatcher( 'VRF route-target' )
   }

   @classmethod
   def handler( cls, mode, args ):
      routeTarget = args[ 'RT' ]
      mode.context.setAction( cls._actionType, routeTarget, no=False )

if RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
   for _mode in [ FlowspecPolicyConfigMode,
                  FlowspecPolicyMatchIpv4RuleConfigMode,
                  FlowspecPolicyMatchIpv6RuleConfigMode,
                  FlowspecPolicyActionsConfigMode, ]:
      _mode.addModelet( CommitAbortModelet )

   FlowspecConfigMode.addCommandClass( FlowspecPoliciesConfigCmd )
   FlowspecPoliciesConfigMode.addCommandClass( FlowspecPolicyConfigCmd )
   FlowspecPolicyConfigMode.addCommandClass( FlowspecPolicyMatchRuleConfigCmd )
   # IPv4 match rule commands
   for v4MatchCmd in [ FlowspecPolicySourcePrefixIpv4Cmd,
                       FlowspecPolicyDestPrefixIpv4Cmd,
                       ProtocolIcmpV4ConfigCmd,
                       ProtocolIcmpV4TypeCodeConfigCmd,
                       ProtocolIpv4ListConfigCmd,
                       ProtocolTcpFlagsOnlyCmd,
                       DscpConfigCmd,
                       MatchAllFragmentConfigCmd,
                       IpLengthConfigCmd, ]:
      FlowspecPolicyMatchIpv4RuleConfigMode.addCommandClass( v4MatchCmd )
   # IPv4 action command
   FlowspecPolicyMatchIpv4RuleConfigMode.addCommandClass(
      FlowspecPolicyActionsConfigCmd )
   # IPv6 match rule commands
   for v6MatchCmd in [ FlowspecPolicySourcePrefixIpv6Cmd,
                       FlowspecPolicyDestPrefixIpv6Cmd,
                       ProtocolIcmpV6ConfigCmd,
                       ProtocolIcmpV6TypeCodeConfigCmd,
                       ProtocolIpv6ListConfigCmd,
                       ProtocolTcpFlagsOnlyCmd,
                       DscpConfigCmd,
                       IpLengthConfigCmd, ]:
      FlowspecPolicyMatchIpv6RuleConfigMode.addCommandClass( v6MatchCmd )
   # IPv6 action command
   FlowspecPolicyMatchIpv6RuleConfigMode.addCommandClass(
      FlowspecPolicyActionsConfigCmd )

   # Actions (common to both IPv4 and IPv6)
   FlowspecPolicyActionsConfigMode.addCommandClass( DropActionWithoutNotifCmd )
   FlowspecPolicyActionsConfigMode.addCommandClass( SetDscpActionCmd )
   FlowspecPolicyActionsConfigMode.addCommandClass( FlowspecPoliceActionCmd )
   FlowspecPolicyActionsConfigMode.addCommandClass( FlowspecRedirectNexthopCmd )
   FlowspecPolicyActionsConfigMode.addCommandClass( FlowspecRedirectVrfRtActionCmd )

#------------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHwStatus

   routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )
   gv.cliConfig = ConfigMount.mount( entityManager,
                                     "flowspec/cliConfig",
                                     "Flowspec::CliConfig", "w" )
   gv.flowspecPbrConfig = ConfigMount.mount( entityManager,
                                             "flowspec/input/cli",
                                             "Pbr::PbrConfig", "w" )
   IntfCli.Intf.registerDependentClass( FlowspecIntf )

   if RoutingLibToggleLib.toggleBgpFlowspecStaticPolicyEnabled():
      mountGroup = entityManager.mountGroup()
      gv.flowspecPolicyStatus = mountGroup.mount( 'flowspec/policy/status',
                                                  'Tac::Dir', 'ri' )
      mountGroup.close( callback=None, blocking=False )
      gv.flowspecPolicyConfig = ConfigMount.mount(
         entityManager,
         'flowspec/policy/config',
         'Flowspec::FlowspecPolicyConfig',
         'wi' )
      gv.flowspecPolicyStatusReqDir = LazyMount.mount(
         entityManager, 'flowspec/policy/statusRequest',
         'PolicyMap::PolicyMapStatusRequestDir', 'w' )
