#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliToken.Flowspec
import CliToken.Ip
import CliToken.Ipv4
import CliToken.Ipv6
import CliMatcher
import Tac
import Tracing
import LazyMount
import SharkLazyMount
import SmashLazyMount
import sys
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin.FlowspecCliModel import FlowspecModel, FlowspecSummaryModel
from CliPlugin.FlowspecMatcher import ruleStringMatcher
import ShowCommand
from Toggles.RoutingLibToggleLib import toggleBgpFlowspecInterfaceSetEnabled

t0 = Tracing.trace0

config = None
cliConfig = None
status = None
counter = None
counterStatus = None
l3IntfConfig = None
l3IntfStatus = None
vrfSmash = None

JSON = 2

flowSpecMatcherForShow = CliMatcher.KeywordMatcher(
      'flow-spec', helpdesc='Show flow-spec rules' )

infeasibleMatcher = CliMatcher.KeywordMatcher(
      'infeasible', helpdesc='Infeasible flow-spec rule' )
summaryMatcherForShow = CliMatcher.KeywordMatcher(
      'summary', helpdesc='Summary of flow-spec rules' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Show flow-spec in a VRF',
      inclDefaultVrf=True )

ipv4MatcherForShow = CliToken.Ipv4.ipv4MatcherForShow
ipv6MatcherForShow = CliToken.Ipv6.ipv6MatcherForShow

# Returns the latest counter snapshot by looking at global snapshot and
# session snapshot timestamps.
def countersSnapshot( mode ):
   # get session snapshot
   sessionSnapshot = mode.session.sessionData( 'FlowspecSessionCounter', None )

   # get global snapshot
   LazyMount.force( counter )
   globalSnapshot = counter

   if sessionSnapshot:
      Tracing.trace8( 'session snapshot was at', sessionSnapshot.timestamp )

   if globalSnapshot:
      Tracing.trace8( 'global snapshot was at', globalSnapshot.timestamp )

   # compare session and global snapshots, return newest
   return globalSnapshot if sessionSnapshot is None or \
          globalSnapshot.timestamp > sessionSnapshot.timestamp else sessionSnapshot

def showFlowspec( mode, filterAf, filterVrfName, filterRuleId, filterRuleStr,
                  destPrefix, srcPrefix, infeasible, summary, interfaceSet ):
   LazyMount.force( config )
   LazyMount.force( cliConfig )
   LazyMount.force( status )
   LazyMount.force( l3IntfConfig )
   LazyMount.force( l3IntfStatus )
   counterStatusEntity = SharkLazyMount.force( counterStatus )
   # Fetch counters only for platforms which support it and we are
   # interested in seeing complete rule status ( summary is False ).
   if status.flowspecSupported and not summary:
      updateFlowspecCounters( mode )
   fd = sys.stdout.fileno()
   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()
   counters = countersSnapshot( mode )
   instance = Tac.newInstance( 'Flowspec::Cli::FlowspecShow', config, cliConfig,
                               status, vrfSmash, filterAf, filterVrfName,
                               filterRuleId, filterRuleStr, destPrefix, srcPrefix,
                               infeasible, summary, interfaceSet, counters,
                               l3IntfConfig, l3IntfStatus, counterStatusEntity )
   instance.render( fd, fmt, revision )
   return FlowspecSummaryModel if summary else FlowspecModel

def updateFlowspecCounters( mode ):
   LazyMount.force( cliConfig )
   LazyMount.force( status )
   reqTime = Tac.now() # monotonic time
   cliConfig.counterUpdateRequestTime = reqTime
   # For JSON, we dont want the error/warning spewed on stderr by Tac.waitFor
   # and CLI.
   jsonFmt = ( mode.session_.outputFormat() == JSON )
   warnAfter = None if jsonFmt else 1.0

   def countersUpdated():
      # wait until the time has been updated, indicating all the counters have been
      return reqTime <= status.counterUpdateTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, warnAfter=warnAfter, timeout=30.0 )
   except Tac.Timeout:
      if not jsonFmt:
         mode.addWarning( "displaying stale counters" )

def showFlowspecIpv4v6( mode, args ):
   af = 'ipv4' if 'ipv4' in args else 'ipv6'

   vrfName = args.get( 'VRF', "default" )
   filterRuleStr = args.get( 'STRING', '' )
   filterRuleId = args.get( 'RULEID', 0 )
   infeasible = bool( args.get( 'infeasible', False ) )
   interfaceSet = args.get( 'SETID', -1 )

   destPrefix = args.get( 'DSTPREFIX', "" )
   srcPrefix = args.get( 'SRCPREFIX', "" )
   if af == 'ipv6':
      destPrefix = destPrefix.stringValue if destPrefix else ""
      srcPrefix = srcPrefix.stringValue if srcPrefix else ""
   destPrefixTac = Tac.Value( "Arnet::IpGenPrefix", destPrefix )
   srcPrefixTac = Tac.Value( "Arnet::IpGenPrefix", srcPrefix )

   return showFlowspec( mode, af, vrfName,
                        filterRuleId, filterRuleStr,
                        destPrefixTac, srcPrefixTac,
                        infeasible, False, interfaceSet )

def showFlowspecSummary( mode, af, vrfName ):
   vrfName = vrfName or "default"
   filterRuleStr = ''
   filterRuleId = 0
   infeasible = False
   interfaceSet = -1
   destPrefixTac = Tac.Value( "Arnet::IpGenPrefix", "" )
   srcPrefixTac = Tac.Value( "Arnet::IpGenPrefix", "" )
   return showFlowspec( mode, af, vrfName,
                        filterRuleId, filterRuleStr,
                        destPrefixTac, srcPrefixTac,
                        infeasible, True, interfaceSet )

def showFlowspecIpv4v6Summary( mode, args ):
   vrfName = args.get( 'VRF', "default" )
   af = 'ipv4' if 'ipv4' in args else 'ipv6'
   return showFlowspecSummary( mode, af, vrfName )

#-------------------------------------------------------------------------------
# show flow-spec ( ipv4 | ipv6 )
#                     [ vrf <vrfName> ]
#                     [ [ rule ( <ruleId> | identifier <ruleString> ) ]
#                       | [ infeasible ] ]
#                     [ ( [ destination <prefix> ] [ source <prefix> ] )
#                       | ( source <prefix> destination <prefix> ) ]
#                     [ interface-set <setid> ]
#-------------------------------------------------------------------------------
flowspecShowSyntax = 'show flow-spec %s [ VRF ] ' \
         '[ ( rule ( STRING | ( identifier RULEID ) ) )' \
         '  | infeasible ]' \
         '[ ( [ destination DSTPREFIX ] [ source SRCPREFIX ] )' \
         '  | ( source SRCPREFIX destination DSTPREFIX ) ] '\
         '[ interface-set SETID ]'

flowspecShowData = {
   'flow-spec' : flowSpecMatcherForShow,
   # af : needs to be updated in each specific show command,
   'VRF' : vrfExprFactory,
   'source' : CliToken.Flowspec.sourceMatcher,
   'destination' : CliToken.Flowspec.destinationMatcher,
   # 'SRCPREFIX' : needs to be updated in each specific show command,
   # 'DSTPREFIX' : needs to be updated in each specific show command,
   'rule' : CliToken.Flowspec.ruleMatcher,
   'STRING' : ruleStringMatcher,
   'identifier' : CliToken.Flowspec.identifierMatcher,
   'RULEID' : CliToken.Flowspec.ruleIdMatcher,
   'infeasible' : infeasibleMatcher,
   'interface-set' : CliToken.Flowspec.interfaceSetMatcherForShow,
   'SETID' : CliToken.Flowspec.interfaceSetIdMatcher,
}
if not toggleBgpFlowspecInterfaceSetEnabled():
   flowspecShowSyntax = flowspecShowSyntax.replace(
      ' [ interface-set SETID ]', '' )
   for kw in [ 'interface-set', 'SETID' ]:
      flowspecShowData.pop( kw )

class FlowSpecIpv4ShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = flowspecShowSyntax % 'ipv4' # pylint: disable=consider-using-f-string

   data = flowspecShowData.copy()
   data.update( {
      'ipv4' : ipv4MatcherForShow,
      'SRCPREFIX' : CliToken.Flowspec.ipv4AddrMatcherForShow,
      'DSTPREFIX' : CliToken.Flowspec.ipv4AddrMatcherForShow,
   } )

   handler = showFlowspecIpv4v6
   cliModel = FlowspecModel

BasicCli.addShowCommandClass( FlowSpecIpv4ShowCmd )

class FlowSpecIpv6ShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = flowspecShowSyntax % 'ipv6' # pylint: disable=consider-using-f-string

   data = flowspecShowData.copy()
   data.update( {
      'ipv6' : ipv6MatcherForShow,
      'SRCPREFIX' : CliToken.Flowspec.ipv6AddrMatcherForShow,
      'DSTPREFIX' : CliToken.Flowspec.ipv6AddrMatcherForShow,
   } )

   handler = showFlowspecIpv4v6
   cliModel = FlowspecModel

BasicCli.addShowCommandClass( FlowSpecIpv6ShowCmd )

#--------------------------------------------------------------------------------
# show flow-spec ipv4 [ vrf VRF ] summary
# show flow-spec ipv6 [ vrf VRF ] summary
#--------------------------------------------------------------------------------
class FlowSpecIpv4v6SummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow-spec ( ipv4 | ipv6 ) [ VRF ] summary'
   data = {
      'flow-spec' : flowSpecMatcherForShow,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : vrfExprFactory,
      'summary' : summaryMatcherForShow,
   }
   handler = showFlowspecIpv4v6Summary
   cliModel = FlowspecSummaryModel

BasicCli.addShowCommandClass( FlowSpecIpv4v6SummaryCmd )

#------------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   global status
   global cliConfig
   global counter
   global counterStatus
   global l3IntfConfig
   global l3IntfStatus
   global vrfSmash

   config = LazyMount.mount( entityManager, "flowspec/config",
                             "Flowspec::Config", "r" )
   status = LazyMount.mount( entityManager, "flowspec/status",
                             "Flowspec::Status", "r" )
   cliConfig = LazyMount.mount( entityManager, "flowspec/cliConfig",
                                "Flowspec::CliConfig", "r" )
   counter = LazyMount.mount( entityManager, "flowspec/counter",
                              "Flowspec::Counter", "r" )
   sharkMountInfo = SharkLazyMount.mountInfo( 'shadow' )
   counterStatus = SharkLazyMount.mount(
      entityManager,
      "policyMap/counters/flowspec",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount=True )
   l3IntfConfig = LazyMount.mount( entityManager, "l3/intf/config",
                                   "L3::Intf::ConfigDir", "r" )
   l3IntfStatus = LazyMount.mount( entityManager, "l3/intf/status",
                                   "L3::Intf::StatusDir", "r" )
   vrfSmash = SmashLazyMount.mount( entityManager, "vrf/vrfIdMapStatus",
                                    "Vrf::VrfIdMap::Status",
                                    SmashLazyMount.mountInfo( 'reader' ),
                                    autoUnmount=True )
