#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin import ForwardingDestinationCommon
import CliToken
import LazyMount
import ShowCommand

gv = CliGlobal.CliGlobal( packetTracerHwStatus=None )

# -----------------------------------------------------------------------------------
# Guard functions
# -----------------------------------------------------------------------------------

def forwardingDestinationCliGuard( mode, token ):
   if gv.packetTracerHwStatus.tracingSupported:
      return None
   return CliParser.guardNotThisPlatform

def rawPacketCliGuard( mode, token ):
   if gv.packetTracerHwStatus.rawPacketSupported:
      return None
   return CliParser.guardNotThisPlatform

def extendedTunnelTypesCliGuard( mode, token ):
   if gv.packetTracerHwStatus.extendedTunnelTypesSupported:
      return None
   return CliParser.guardNotThisPlatform

def forwardingDestinationDetailCliGuard( mode, token ):
   if gv.packetTracerHwStatus.detailTracingSupported:
      return None
   return CliParser.guardNotThisPlatform

def keywordWithMaxMatches( keyword, helpdesc, maxMatches=1, guard=None ):
   return CliCommand.Node(
         CliMatcher.KeywordMatcher( keyword=keyword, helpdesc=helpdesc ),
         maxMatches=maxMatches,
         guard=guard )

def keywordWithMaxMatchesExtendedTypesGuard( keyword, helpdesc, maxMatches=1 ):
   return keywordWithMaxMatches( keyword, helpdesc, maxMatches,
                                 extendedTunnelTypesCliGuard )

# -----------------------------------------------------------------------------------
# show forwarding destination
# -----------------------------------------------------------------------------------
keywords = {
   'forwarding': 'Show forwarding information',
   'destination': keywordWithMaxMatches(
      'destination',
      helpdesc='Show packet forwarding destinations',
      guard=forwardingDestinationCliGuard ),
   'detail': keywordWithMaxMatches(
      'detail', helpdesc='Show packet forwarding destination detail',
      guard=forwardingDestinationDetailCliGuard ),
   'ingress-interface': keywordWithMaxMatches( 'ingress-interface',
                                               helpdesc='Ingress Interface' ),
   'edit': 'Edit the configured packet',
   'src-mac': keywordWithMaxMatches( 'src-mac',
                                     helpdesc='MAC address of the source' ),
   'dst-mac': keywordWithMaxMatches( 'dst-mac',
                                     helpdesc='MAC address for the destination' ),
   'eth-type': keywordWithMaxMatches( 'eth-type', helpdesc='Ethertype' ),
   'vlan': keywordWithMaxMatches( 'vlan', helpdesc='Identifier for a Virtual LAN' ),
   'inner-vlan': keywordWithMaxMatches( 'inner-vlan', helpdesc='Inner VLAN ID' ),
   'src-ipv4': keywordWithMaxMatches( 'src-ipv4',
                                      helpdesc='Source IPv4 Address' ),
   'dst-ipv4': keywordWithMaxMatches( 'dst-ipv4',
                                      helpdesc='Destination IPv4 Address' ),
   'ip-ttl': keywordWithMaxMatches( 'ip-ttl', helpdesc='IP TTL' ),
   'ip-protocol': keywordWithMaxMatches( 'ip-protocol', helpdesc='IP Protocol' ),
   'src-l4-port': keywordWithMaxMatches( 'src-l4-port',
                                         helpdesc='L4 Source Port' ),
   'dst-l4-port': keywordWithMaxMatches( 'dst-l4-port',
                                         helpdesc='L4 Destination Port' ),
   'src-ipv6': keywordWithMaxMatches( 'src-ipv6',
                                      helpdesc='Source IPv6 Address' ),
   'dst-ipv6': keywordWithMaxMatches( 'dst-ipv6',
                                      helpdesc='Destination IPv6 Address' ),
   'hop-limit': keywordWithMaxMatches( 'hop-limit', helpdesc='IPv6 Hop Limit' ),
   'next-header': keywordWithMaxMatches( 'next-header',
                                         helpdesc='IPv6 Next Header' ),
   'flow-label': keywordWithMaxMatches( 'flow-label',
                                        helpdesc='IPv6 Flow Label' ),
   'inner': CliCommand.Node(
      CliMatcher.KeywordMatcher( keyword='inner',
                                 helpdesc='Inner packet configuration' ),
      guard=extendedTunnelTypesCliGuard ),
   'INNER_SRC_MAC_KW': keywordWithMaxMatches( 'src-mac',
                                              helpdesc='MAC address of the source' ),
   'INNER_DST_MAC_KW': keywordWithMaxMatches(
      'dst-mac', helpdesc='MAC address of the destination' ),
   'INNER_ETH_TYPE_KW': keywordWithMaxMatches( 'eth-type', helpdesc='Ethertype' ),
   'INNER_SRC_IPV4_KW': keywordWithMaxMatches( 'src-ipv4',
                                               helpdesc='Source IPv4 Address' ),
   'INNER_DST_IPV4_KW': keywordWithMaxMatches( 'dst-ipv4',
                                               helpdesc='Destination IPv4 Address' ),
   'INNER_IP_TTL_KW': keywordWithMaxMatches( 'ip-ttl', helpdesc='IP TTL' ),
   'INNER_IP_PROTO_KW': keywordWithMaxMatches( 'ip-protocol',
                                               helpdesc='IP Protocol' ),
   'INNER_SRC_IPV6_KW': keywordWithMaxMatches( 'src-ipv6',
                                               helpdesc='Source IPv6 Address' ),
   'INNER_DST_IPV6_KW': keywordWithMaxMatches( 'dst-ipv6',
                                               helpdesc='Destination IPv6 Address' ),
   'INNER_HOP_LIMIT_KW': keywordWithMaxMatches( 'hop-limit',
                                                helpdesc='IPv6 Hop Limit' ),
   'INNER_NEXT_HEADER_KW': keywordWithMaxMatches( 'next-header',
                                              helpdesc='IPv6 Next Header' ),
   'INNER_FLOW_LABEL_KW': keywordWithMaxMatches( 'flow-label',
                                                 helpdesc='IPv6 Flow Label' ),
   'gre-key': keywordWithMaxMatchesExtendedTypesGuard( 'gre-key',
                                                       helpdesc='GRE Key' ),
   'gre-sequence': keywordWithMaxMatchesExtendedTypesGuard(
      'gre-sequence', helpdesc='GRE Sequence' ),
   'gre-checksum': keywordWithMaxMatchesExtendedTypesGuard(
      'gre-checksum', helpdesc='Generate GRE Checksum field' ),
   'nvgre-virtual-subnet-id': keywordWithMaxMatchesExtendedTypesGuard(
      'nvgre-virtual-subnet-id', helpdesc='NVGRE Virtual Subnet ID' ),
   'nvgre-flow-id': keywordWithMaxMatchesExtendedTypesGuard(
      'nvgre-flow-id', helpdesc='NVGRE Flow ID' ),
   'packet-type': keywordWithMaxMatchesExtendedTypesGuard( 'packet-type',
                                                           helpdesc='Packet type' ),
   'INNER_PACKET_TYPE_KW': keywordWithMaxMatchesExtendedTypesGuard(
      'packet-type', helpdesc='Packet type' ),
   'gre-type': keywordWithMaxMatchesExtendedTypesGuard( 'gre-type',
                                                        helpdesc='GRE type' ),
}

# The keyword matchers are defined in ForwardingDestinationCommon
keywords.update( ForwardingDestinationCommon.matchers )
del keywords[ '<rawPacket>' ]

class ShowForwardingDestinationCommand( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show forwarding destination [ detail ] [ edit ] '
              '[ ( { ( ingress-interface <ingressIntf> ) |'
              '( src-mac <srcMac> ) | ( dst-mac <dstMac> ) | '
              '( eth-type <etherType> ) | ( vlan <vlan> ) | '
              '( inner-vlan <innerVlan> ) | ( src-ipv4 <srcIpv4> ) | '
              '( dst-ipv4 <dstIpv4> ) | ( ip-ttl <ipTtl> ) | '
              '( ip-protocol <ipProto> ) | ( src-ipv6 <srcIpv6> ) | '
              '( dst-ipv6 <dstIpv6> ) | ( hop-limit <hopLimit> ) | '
              '( next-header <nextHeader> ) | ( flow-label <flowLabel> ) | '
              '( src-l4-port <srcL4Port> ) | ( dst-l4-port <dstL4Port> ) | '
              '( gre-key <greKey> ) | gre-checksum | '
              '( gre-sequence <greSequence> ) | '
              '( nvgre-virtual-subnet-id <nvgreVirtualSubnetId> ) | '
              '( nvgre-flow-id <nvgreFlowId> ) | '
              '( inner ( INNER_SRC_MAC_KW <innerL2SrcMac> ) | '
              '( INNER_DST_MAC_KW <innerL2DstMac> ) | '
              '( INNER_ETH_TYPE_KW <innerL2EtherType> ) | '
              '( INNER_SRC_IPV4_KW <innerSrcIpv4> ) | '
              '( INNER_DST_IPV4_KW <innerDstIpv4> ) | '
              '( INNER_IP_TTL_KW <innerIpTtl> ) | '
              '( INNER_IP_PROTO_KW <innerIpProto> ) | '
              '( INNER_SRC_IPV6_KW <innerSrcIpv6> ) | '
              '( INNER_DST_IPV6_KW <innerDstIpv6> ) | '
              '( INNER_HOP_LIMIT_KW <innerHopLimit> ) | '
              '( INNER_NEXT_HEADER_KW <innerNextHeader> ) | '
              '( INNER_FLOW_LABEL_KW <innerFlowLabel> ) | '
              '( INNER_PACKET_TYPE_KW <innerPacketType> ) ) | '
              '( packet-type <packetType> ) | ( gre-type <greType> ) } ) ]' )
   data = keywords

   cliModel = 'ForwardingDestinationModel.PacketDestination'
   noMore = True
   handler = 'ForwardingDestinationCliHandlers.commandHandler'

BasicCli.addShowCommandClass( ShowForwardingDestinationCommand )

class ShowRawForwardingDestinationCommand( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show forwarding destination [ detail ] '
              'ingress-interface <ingressIntf> raw <rawPacket>' )
   data = {
      'forwarding': 'Show forwarding information',
      'destination': keywordWithMaxMatches(
         'destination',
         helpdesc='Show packet forwarding destinations',
         guard=forwardingDestinationCliGuard ),
      'detail': keywordWithMaxMatches(
         'detail', helpdesc='Show packet forwarding destination detail',
         guard=forwardingDestinationDetailCliGuard ),
      'ingress-interface': keywordWithMaxMatches( 'ingress-interface',
                                                  helpdesc='Ingress Interface' ),
      '<ingressIntf>': ForwardingDestinationCommon.matchers[ '<ingressIntf>' ],
      'raw': keywordWithMaxMatches( 'raw', helpdesc='Raw Packet',
                                    guard=rawPacketCliGuard ),
      '<rawPacket>': ForwardingDestinationCommon.matchers[ '<rawPacket>' ],
   }

   cliModel = 'ForwardingDestinationModel.PacketDestination'
   privileged = True
   handler = 'ForwardingDestinationCliHandlers.commandHandler'

BasicCli.addShowCommandClass( ShowRawForwardingDestinationCommand )

# -----------------------------------------------------------------------------------
# clear forwarding destination packet
# -----------------------------------------------------------------------------------
class ClearForwardingDestinationPacketCommand( CliCommand.CliCommandClass ):
   syntax = 'clear forwarding destination packet'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'forwarding': 'Forwarding information',
      'destination': keywordWithMaxMatches(
         'destination',
         helpdesc='Packet forwarding destination',
         guard=forwardingDestinationCliGuard ),
      'packet': 'Saved packet configuration',
   }

   handler = 'ForwardingDestinationCliHandlers.clearPacketHandler'

BasicCli.ExecMode.addCommandClass( ClearForwardingDestinationPacketCommand )

def Plugin( entityManager ):
   gv.packetTracerHwStatus = LazyMount.mount(
      entityManager,
      'packettracer/hwstatus',
      'PacketTracer::HwStatus', 'r' )
