# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple, OrderedDict
import CliExtensions
import CliMatcher
from CliPlugin import EthIntfCli, SwitchIntfCli
import CliPlugin.IpAddrMatcher
import CliPlugin.Ip6AddrMatcher
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.VirtualIntfRule import IntfMatcher

showForwardingDestinationHook = CliExtensions.CliHook()

# Client name for packettracer config/status in sysdb
CLI_CLIENT_NAME = 'TracerCli'
# In the agent is not running we will wait AGENT_START_DELAY seconds until it is
AGENT_START_DELAY = 10
# Permit some delay in processing the request
PROCESSING_DELAY = 5
# For better interactivity, use a lower maxDelay in our calls to waitFor
MAX_WAITFOR_DELAY = 0.5

packetTypeKeywords = {
   'ethernet': 'Ethernet packet',
   'gre': 'Generic Routing Encapsulation packet',
   'ipv4': 'IPv4 packet',
   'ipv6': 'IPv6 packet',
   'raw': 'Raw packet',
}

innerPacketTypeKeywords = dict( packetTypeKeywords )
del innerPacketTypeKeywords[ 'gre' ]
del innerPacketTypeKeywords[ 'raw' ]

greTypeKeywords = {
   'gre': 'Generic Routing Encapsulation',
   'nvgre': 'Network Virtualization Generic Routing Encapsulation',
}

intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= SwitchIntfCli.SwitchIntf.matcher

matchers = {
   '<ingressIntf>': intfMatcher,
   '<rawPacket>': CliMatcher.StringMatcher( helpdesc='Hexadecimal String',
                                            pattern=r'[a-fA-F0-9 ]+' ),
   '<srcMac>': MacAddrMatcher( helpdesc='MAC address of the source' ),
   '<dstMac>': MacAddrMatcher( helpdesc='MAC address of the destination' ),
   '<etherType>': CliMatcher.IntegerMatcher( 0, 0xffff, helpdesc='Ethertype' ),
   '<vlan>': CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' ),
   '<innerVlan>': CliMatcher.IntegerMatcher( 1, 4094,
      helpdesc='Identifier for a Virtual LAN' ),
   '<srcIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher( 'Source IPv4 Address' ),
   '<dstIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher( 'Destination IPv4 Address' ),
   '<ipTtl>': CliMatcher.IntegerMatcher( 1, 0xff, helpdesc='IP TTL' ),
   '<ipProto>': CliMatcher.IntegerMatcher( 0, 0xff, helpdesc='IP Protocol' ),
   '<srcL4Port>': CliMatcher.IntegerMatcher( 0, 0xffff,
                                             helpdesc='Source TCP/UDP Port' ),
   '<dstL4Port>': CliMatcher.IntegerMatcher( 0, 0xffff,
                                             helpdesc='Destination TCP/UDP Port' ),
   '<srcIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher( 'Source IPv6 Address' ),
   '<dstIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher(
         'Destination IPv6 Address' ),
   '<hopLimit>': CliMatcher.IntegerMatcher( 1, 0xff, helpdesc='IPv6 Hop Limit' ),
   '<nextHeader>': CliMatcher.IntegerMatcher( 0, 0xff,
                                              helpdesc='IPv6 Next Header' ),
   '<flowLabel>': CliMatcher.IntegerMatcher( 0, 0xfffff,
                                             helpdesc='IPv6 Flow Label' ),
   '<innerL2SrcMac>': MacAddrMatcher( helpdesc='MAC address of the source' ),
   '<innerL2DstMac>': MacAddrMatcher( helpdesc='MAC address of the destination' ),
   '<innerL2EtherType>': CliMatcher.IntegerMatcher( 0, 0xffff,
                                                    helpdesc='Ethertype' ),
   '<innerSrcIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher( 'Source IPv4 Address' ),
   '<innerDstIpv4>': CliPlugin.IpAddrMatcher.IpAddrMatcher(
      'Destination IPv4 Address' ),
   '<innerIpTtl>': CliMatcher.IntegerMatcher( 1, 0xff, helpdesc='IP TTL' ),
   '<innerIpProto>': CliMatcher.IntegerMatcher( 0, 0xff, helpdesc='IP Protocol' ),
   '<innerSrcIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher(
      'Source IPv6 Address' ),
   '<innerDstIpv6>': CliPlugin.Ip6AddrMatcher.Ip6AddrMatcher(
         'Destination IPv6 Address' ),
   '<innerHopLimit>': CliMatcher.IntegerMatcher( 1, 0xff,
                                                 helpdesc='IPv6 Hop Limit' ),
   '<innerNextHeader>': CliMatcher.IntegerMatcher( 0, 0xff,
                                                   helpdesc='IPv6 Next Header' ),
   '<innerFlowLabel>': CliMatcher.IntegerMatcher( 0, 0xfffff,
                                                  helpdesc='IPv6 Flow Label' ),
   '<greKey>': CliMatcher.IntegerMatcher( 0, 0xffffffff, helpdesc='GRE Key' ),
   '<greSequence>': CliMatcher.IntegerMatcher( 0, 0xffffffff,
                                               helpdesc='GRE Sequence' ),
   '<nvgreVirtualSubnetId>': CliMatcher.IntegerMatcher( 0, 0xffffff,
      helpdesc='NVGRE Virtual Subnet ID' ),
   '<nvgreFlowId>': CliMatcher.IntegerMatcher( 0, 0xff, helpdesc='NVGRE Flow ID' ),
   '<packetType>': CliMatcher.EnumMatcher( packetTypeKeywords ),
   '<innerPacketType>': CliMatcher.EnumMatcher( innerPacketTypeKeywords ),
   '<greType>': CliMatcher.EnumMatcher( greTypeKeywords ),
}

BaseFields = [ '<ingressIntf>' ]
L2Fields = [ '<srcMac>', '<dstMac>', '<etherType>' ]
OptionalL2Fields = [ '<vlan>', '<innerVlan>' ]
IPv4Fields = [ '<srcIpv4>', '<dstIpv4>', '<ipTtl>', '<ipProto>' ]
IPv6Fields = [ '<srcIpv6>', '<dstIpv6>', '<hopLimit>', '<nextHeader>',
               '<flowLabel>' ]
InnerPacketTypeFields = [ '<innerPacketType>' ]
# Note that all GRE fields are optional
GreFields = [ '<greKey>', 'gre-checksum', '<greSequence>' ]
NvgreFields = [ '<nvgreVirtualSubnetId>', '<nvgreFlowId>' ]
GreTypeFields = [ '<greType>' ]
InnerL2Fields = [ '<innerL2SrcMac>', '<innerL2DstMac>', '<innerL2EtherType>' ]
InnerIPv4Fields = [ '<innerSrcIpv4>', '<innerDstIpv4>', '<innerIpTtl>',
                    '<innerIpProto>' ]
InnerIPv6Fields = [ '<innerSrcIpv6>', '<innerDstIpv6>', '<innerHopLimit>',
                    '<innerNextHeader>', '<innerFlowLabel>' ]
L4Fields = [ '<srcL4Port>', '<dstL4Port>' ]

ArgToLabel = {
   '<ingressIntf>': 'ingress-interface',
   '<rawPacket>': 'raw',
   '<srcMac>': 'src-mac',
   '<dstMac>': 'dst-mac',
   '<etherType>': 'eth-type',
   '<vlan>': 'vlan',
   '<innerVlan>': 'inner-vlan',
   '<srcIpv4>': 'src-ipv4',
   '<dstIpv4>': 'dst-ipv4',
   '<ipTtl>': 'ip-ttl',
   '<ipProto>': 'ip-protocol',
   '<srcL4Port>': 'src-l4-port',
   '<dstL4Port>': 'dst-l4-port',
   '<srcIpv6>': 'src-ipv6',
   '<dstIpv6>': 'dst-ipv6',
   '<hopLimit>': 'hop-limit',
   '<nextHeader>': 'next-header',
   '<flowLabel>': 'flow-label',
   '<innerL2SrcMac>': 'Inner src-mac',
   '<innerL2DstMac>': 'Inner dst-mac',
   '<innerL2EtherType>': 'Inner eth-type',
   '<innerSrcIpv4>': 'Inner src-ipv4',
   '<innerDstIpv4>': 'Inner dst-ipv4',
   '<innerIpTtl>': 'Inner ip-ttl',
   '<innerIpProto>': 'Inner ip-protocol',
   '<innerSrcIpv6>': 'Inner src-ipv6',
   '<innerDstIpv6>': 'Inner dst-ipv6',
   '<innerHopLimit>': 'Inner hop-limit',
   '<innerNextHeader>': 'Inner next-header',
   '<innerFlowLabel>': 'Inner flow-label',
   '<greKey>': 'gre-key',
   'gre-checksum': 'gre-checksum',
   '<greSequence>': 'gre-sequence',
   '<nvgreVirtualSubnetId>': 'nvgre-virtual-subnet-id',
   '<nvgreFlowId>': 'nvgre-flow-id',
   '<packetType>': 'packet-type',
   '<innerPacketType>': 'Inner packet-type',
   '<greType>': 'gre-type',
}

# These fields don't correspond to one specific packet attribute, but are used to
# determine the packet type in the interactive prompt.
PacketInfoFields = [ '<ipVersion>', '<greType>', '<innerPacketType>', '<l4Type>' ]

PacketInfo = namedtuple( 'PacketInfo', [ 'packetType', 'innerPacketType',
                                         'l4Type', 'ipVersion', 'greType' ] )

# Maintain a list of packet attribute fields that are incompatible with the provided
# packet type. Note that the "raw" type is omitted here since it has separate error
# handling
IncompatiblePacketTypeFields = {
   'ethernet': ( IPv4Fields + IPv6Fields + InnerPacketTypeFields + GreTypeFields +
                 GreFields + NvgreFields + InnerL2Fields + InnerIPv4Fields +
                 InnerIPv6Fields + L4Fields ),
   'ipv4': IPv6Fields + GreTypeFields + GreFields + NvgreFields + InnerL2Fields,
   'ipv6': IPv4Fields + GreTypeFields + GreFields + NvgreFields + InnerL2Fields,
}

IncompatibleInnerPacketTypeFields = {
   'ethernet': GreFields + InnerIPv4Fields + InnerIPv6Fields + L4Fields,
   'ipv4': InnerIPv6Fields,
   'ipv6': InnerIPv4Fields,
}

IncompatibleGreTypeFields = {
   'gre': NvgreFields + InnerL2Fields,
   'nvgre': GreFields,
}

# Dict of dicts to specify which packet information fields are incompatible with
# other packet attribute fields. For example, if the packet type "ipv4" is
# selected, all of the GRE packet attribute fields will be invalid. These fields will
# always be invalid regardless of their configured value.
IncompatibleFields = OrderedDict( [
   ( '<packetType>', IncompatiblePacketTypeFields ),
   ( '<greType>', IncompatibleGreTypeFields ),
   ( '<innerPacketType>', IncompatibleInnerPacketTypeFields ),
] )

IncompatiblePacketTypeFieldValues = {
   'ipv4': { '<innerPacketType>': 'ethernet' },
   'ipv6': { '<innerPacketType>': 'ethernet' },
}

IncompatibleGreTypeFieldValues = {
   'gre': { '<innerPacketType>': 'ethernet' },
}

# Specifies which packet information field *values* are incompatible with selected
# packet attribute fields. For example, if the packet type is selected as "ipv4",
# while the "inner packet-type" field is valid, it cannot have the value of
# "ethernet" (since only "ipv4" and "ipv6" would be valid packet types).
IncompatibleFieldValues = OrderedDict( [
   ( '<packetType>', IncompatiblePacketTypeFieldValues ),
   ( '<greType>', IncompatibleGreTypeFieldValues ),
] )
