#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import functools
import os
import sys
import time
from collections import defaultdict

import ArPyUtils
from CliParser import guardNotThisPlatform
import CliMatcher
from CliToken.Hardware import hardwareMatcherForConfig
from CliToken.Hardware import hardwareMatcherForShow
import CliCommand
import BasicCliModes
from CliModel import Model, Dict, Float, Int, Enum, Str
import ShowCommand
import BasicCli
import Tac
import Cell
import ConfigMount, LazyMount
from CliPlugin import FruCli
from CliPlugin.TechSupportCli import registerShowTechSupportCmd
import TableOutput
import FpgaUtil
import ProductAttributes


fpgaRecModeEnum = Tac.Type( "Hardware::Fpga::FpgaCliConfig::FpgaCrcRecoveryMode" )
systemCpldStatusDir = None
systemCpldConfigDir = None
standbyCpldStatusDir = None
standbyCpldConfigDir = None
fpgaCliConfig = None
fpgaCrcSliceDir = None
fpgaFruConfigCellDir = None
fpgaCrcCellDir = None
fpgaFruConfigSliceDir = None
scdConfigCellDir = None
scdConfigSliceDir = None
scdStatusCellDir = None
scdStatusSliceDir = None
entityMib = None

naturalsorted = functools.partial( ArPyUtils.naturalsorted, caseSensitive=False )

def getPrefdl():
   if os.path.getsize( '/etc/prefdl' ) == 0:
      try:
         Tac.run( [ "sh", "-c", "genprefdl 2>/dev/null >/etc/prefdl" ] )
      except Tac.SystemCommandError as e:
         print( time.asctime(),
                "getPrefdl: failed to exec genprefdl (%s)" % e, file=sys.stderr )
   with open( "/etc/prefdl", "rb" ) as f:
      prefdl = f.read()
   return prefdl

def isModular():
   return entityMib.root and \
          entityMib.root.tacType.fullTypeName != "EntityMib::FixedSystem"

def fpgaFixedGuard( mode, token ):
   if mode.session_.startupConfig():
      # disable the guard during startup-config since Sysdb is not properly
      # initialized yet.  Thus, the other conditions will be invalid.
      return None

   # Prevent the fixed system commands from combining with the corresponding
   # modular system commands.  Also, not all fixed systems support the
   # ability to enable/disable power-cycle upon FPGA CRC-Error.
   if( FruCli.modularSystemGuard( mode, token ) is not None and
       fpgaFruConfigCellDir and
       fpgaFruConfigCellDir.pciFpgaConfig ):
      return None
   return guardNotThisPlatform

def fpgaModularGuard( mode, token ):
   if mode.session_.startupConfig():
      # disable the guard during startup-config since Sysdb is not properly
      # initialized yet.  Thus, the other conditions will be invalid.
      return None
   return FruCli.modularSystemGuard( mode, token )

def fpgaShowGuard( mode, token ):
   if fpgaModularGuard( mode, token ) is None:
      return None
   return fpgaFixedGuard( mode, token )

fpgaMatcher = CliMatcher.KeywordMatcher(
   'fpga', helpdesc='Field-programmable gate array' )

fpgaMatcherForFixed = CliCommand.Node( matcher=fpgaMatcher, guard=fpgaFixedGuard )

fpgaMatcherForModular = CliCommand.Node( matcher=fpgaMatcher,
                                         guard=fpgaModularGuard )

fpgaMatcherForShow = CliCommand.Node( matcher=fpgaMatcher, guard=fpgaShowGuard )

errorMatcher = CliMatcher.KeywordMatcher(
   'error', helpdesc='Configuration memory errors' )

actionMatcher = CliMatcher.KeywordMatcher(
   'action', helpdesc='Action taken upon error detection' )

logMatcher = CliMatcher.KeywordMatcher(
   'log', helpdesc='Log and ignore error' )

powerCycleMatcher = CliMatcher.KeywordMatcher(
   'power-cycle', helpdesc='Power-cycle on error`' )

repairMatcher = CliMatcher.KeywordMatcher(
   'repair', helpdesc='Repair the FPGA - several agents may restart,' +
                      'traffic not impacted' )

detailMatcher = CliMatcher.KeywordMatcher(
   'detail', helpdesc='Display FPGA information' )

def cliParserError( mode, args ):
   mode.addError( "Parser internal error tokens:" )
   for token in args:
      mode.addMessage( "%s" % token )

def setModularFpgaRecoveryMode( mode, args, recoveryMode ):
   if 'linecard' in args:
      fpgaCliConfig.linecardFpgaCrcRecoveryMode = recoveryMode
   elif 'supervisor' in args:
      fpgaCliConfig.fpgaCrcRecoveryMode = recoveryMode
   else:
      cliParserError( mode, args )

#--------------------------------------------------------------------------------
# For modular systems:
#
# [ no | default ] hardware fpga error ( linecard | supervisor )
#                                action ( log | power-cycle )
#--------------------------------------------------------------------------------
class ModularHardwareFpgaErrorActionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'hardware fpga error ( linecard | supervisor ) action '
              '( log | power-cycle )' )
   noOrDefaultSyntax = 'hardware fpga error ( linecard | supervisor ) action ...'
   data = {
      'hardware': hardwareMatcherForConfig,
      'fpga': fpgaMatcherForModular,
      'error': errorMatcher,
      'linecard': 'Linecard configuration',
      'supervisor': 'Supervisor configuration',
      'action': actionMatcher,
      'log': logMatcher,
      'power-cycle': powerCycleMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      crcErrorRecoveryMode = None

      if 'log' in args:
         crcErrorRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryNone
      elif 'power-cycle' in args:
         crcErrorRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryPowerCycle
      else:
         cliParserError( mode, args )
         return

      setModularFpgaRecoveryMode( mode, args, crcErrorRecoveryMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setModularFpgaRecoveryMode( mode, args, fpgaRecModeEnum.fpgaCrcRecoveryNone )

BasicCliModes.GlobalConfigMode.addCommandClass( ModularHardwareFpgaErrorActionCmd )

#-------------------------------------------------------------------------------
# For fixed systems
#
# "[ default | no ] hardware fpga error action [ log | power-cycle ]"
#-------------------------------------------------------------------------------
def setFixedFpgaErrorRecoveryMode( mode, args ):
   crcErrorRecoveryMode = None

   if 'repair' in args:
      crcErrorRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryAutoRepair
   elif 'log' in args:
      crcErrorRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryNone
   elif 'power-cycle' in args:
      crcErrorRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryPowerCycle
   else:
      cliParserError( mode, args )
      return

   fpgaCliConfig.fpgaCrcRecoveryMode = crcErrorRecoveryMode

def noFixedFpgaErrorRecoveryMode( mode, args ):
   fpgaCliConfig.fpgaCrcRecoveryMode = fpgaRecModeEnum.fpgaCrcRecoveryNone

pa = ProductAttributes.productAttributes().productAttributes
pba = ProductAttributes.productAttributes().bootAttributes 

if ( pa.scdAutoRepairSupported and
     pba.asuHitlessSupported ):

   class FixedRepairFpgaErrorActionCmd( CliCommand.CliCommandClass ):
      syntax = 'hardware fpga error action ( log | power-cycle | repair )'
      noOrDefaultSyntax = 'hardware fpga error action ...'
      data = {
         'hardware': hardwareMatcherForConfig,
         'fpga': fpgaMatcherForFixed,
         'error': errorMatcher,
         'action': actionMatcher,
         'log': logMatcher,
         'power-cycle': powerCycleMatcher,
         'repair': repairMatcher,
      }

      handler = setFixedFpgaErrorRecoveryMode
      noOrDefaultHandler = noFixedFpgaErrorRecoveryMode

   BasicCliModes.GlobalConfigMode.addCommandClass( FixedRepairFpgaErrorActionCmd )
else:
   class FixedHardwareFpgaErrorActionCmd( CliCommand.CliCommandClass ):
      syntax = 'hardware fpga error action ( log | power-cycle )'
      noOrDefaultSyntax = 'hardware fpga error action ...'
      data = {
         'hardware': hardwareMatcherForConfig,
         'fpga': fpgaMatcherForFixed,
         'error': errorMatcher,
         'action': actionMatcher,
         'log': logMatcher,
         'power-cycle': powerCycleMatcher,
      }

      handler = setFixedFpgaErrorRecoveryMode
      noOrDefaultHandler = noFixedFpgaErrorRecoveryMode

   BasicCliModes.GlobalConfigMode.addCommandClass( FixedHardwareFpgaErrorActionCmd )


#-------------------------------------------------------------------------------
# "show hardware fpga error" command
#-------------------------------------------------------------------------------
modeEnumToCapiEnum = { 'fpgaCrcRecoveryNone' : 'log',
                       'fpgaCrcRecoveryPowerCycle' : 'powerCycle',
                       'fpgaCrcRecoveryAutoRepair' : 'autoRepair' }

modeCapiEnumToText = { 'log' : 'log',
                       'powerCycle' : 'power-cycle',
                       'autoRepair' : 'repair' }

class FpgaCrcErrorDetails( Model ):
   count = Int( help="Number of CRC-Error occurrences" )
   firstOccurrence = Float( help="First CRC-Error occurrence", optional=True )
   lastOccurrence = Float( help="Last CRC-Error occurrence", optional=True )

class FpgaCrcErrors( Model ):
   linecardFpgaCrcRecoveryMode = Enum( values=( "log", "powerCycle", ),
         help="Linecard action on FPGA CRC-Error",
         optional=True )
   fpgaCrcRecoveryMode = Enum( values=( "log", "powerCycle", "autoRepair" ),
         help="System action on FPGA CRC-Error" )
   uncorrectableCrcs = Dict( keyType=str, valueType=FpgaCrcErrorDetails,
         help="Maps FPGA name to its uncorrectable CRC-Error details. " +
         "On modular systems the FPGA name contains the slot number." )
   correctableCrcs = Dict( keyType=str, valueType=FpgaCrcErrorDetails,
         help="Maps FPGA name to its correctable CRC-Error details. " +
         "On modular systems the FPGA name contains the slot number." )
   softwareCorrectedCrcs = Dict( keyType=str, valueType=FpgaCrcErrorDetails,
         help="Maps FPGA name to its software-repaired CRC-Error details. " +
         "On modular systems the FPGA name contains the slot number." )

   def render( self ):
      def utcFloatToString( ftime ):
         if ftime:
            return time.strftime( "%Y-%m-%d %H:%M:%S", time.localtime( ftime ) )
         else:
            return ''

      self.renderRecoveryMode()

      if not self.uncorrectableCrcs:
         print( "No FPGAs found" )
         return

      # Table Headings
      tableHeadings = ( "FPGA", "Errors", "First Occurrence", "Last Occurrence", )

      # Table Column Formats
      tcfFpga = TableOutput.Format( justify="left" )
      tcfFpga.padLimitIs( True )
      tcfErrors = TableOutput.Format( minWidth=11, maxWidth=11 )
      tcfErrors.padLimitIs( True )
      tcfOccurrence = TableOutput.Format( justify="left", minWidth=19, maxWidth=19 )
      tcfOccurrence.padLimitIs( True )

      print( "\nUncorrected FPGA CRC-Errors:" )
      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( tcfFpga, tcfErrors, tcfOccurrence, tcfOccurrence )
      for fpga in naturalsorted( self.uncorrectableCrcs ):
         crcDetail = self.uncorrectableCrcs.get( fpga )
         table.newRow( fpga, crcDetail.count,
               utcFloatToString( crcDetail.firstOccurrence ),
               utcFloatToString( crcDetail.lastOccurrence ) )
      print( table.output() )

      print( "\nCorrected FPGA CRC-Errors:" )
      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( tcfFpga, tcfErrors, tcfOccurrence, tcfOccurrence )
      for fpga in naturalsorted( self.correctableCrcs ):
         crcDetail = self.correctableCrcs.get( fpga )
         table.newRow( fpga, crcDetail.count,
               utcFloatToString( crcDetail.firstOccurrence ),
               utcFloatToString( crcDetail.lastOccurrence ) )
      print( table.output() )

      print( "\nSoftware-repaired FPGA CRC-Errors:" )
      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( tcfFpga, tcfErrors, tcfOccurrence, tcfOccurrence )
      for fpga in naturalsorted( self.softwareCorrectedCrcs ):
         crcDetail = self.softwareCorrectedCrcs.get( fpga )
         table.newRow( fpga, crcDetail.count,
               utcFloatToString( crcDetail.firstOccurrence ),
               utcFloatToString( crcDetail.lastOccurrence ) )
      print( table.output() )

class FpgaCrcErrorsModular( FpgaCrcErrors ):

   def renderRecoveryMode( self ):
      print( "Supervisor Action: %s" % modeCapiEnumToText.get(
            self.fpgaCrcRecoveryMode ) )
      print( "Linecard Action: %s" % modeCapiEnumToText.get(
            self.linecardFpgaCrcRecoveryMode ) )

class FpgaCrcErrorsFixed( FpgaCrcErrors ):

   def renderRecoveryMode( self ):
      print( "System Action: %s" % modeCapiEnumToText.get(
            self.fpgaCrcRecoveryMode ) )

def crcErrorDetails( crcTally ):
   errorCount = crcTally.count
   errorFirst = crcTally.firstOccurrence
   if errorFirst == 0:
      errorFirst = None
   errorLast = crcTally.lastOccurrence
   if errorLast == 0:
      errorLast = None
   return FpgaCrcErrorDetails( count=errorCount, firstOccurrence=errorFirst,
                                                 lastOccurrence=errorLast )

def crcErrorDetailsCollection( fpgaStatusCollection ):
   uncorrectableCrcs = {}
   correctableCrcs = {}
   softwareCorrectedCrcs = {} 
   for fpga in fpgaStatusCollection:
      fpgaStatus = fpgaStatusCollection[ fpga ]
      uncorrectableCrcs[ fpga ] = crcErrorDetails( fpgaStatus.uncorrectedCrcTally )
      correctableCrcs[ fpga ] = crcErrorDetails( fpgaStatus.correctedCrcTally )
      softwareCorrectedCrcs[ fpga ] = ( 
         crcErrorDetails( fpgaStatus.softwareCorrectedCrcTally )
      )
   return uncorrectableCrcs, correctableCrcs, softwareCorrectedCrcs

def doShowModularFpgaErrors():
   fpgaCrcRecoveryMode = modeEnumToCapiEnum.get(
      fpgaCliConfig.fpgaCrcRecoveryMode )
   linecardRecoveryMode = modeEnumToCapiEnum.get(
         fpgaCliConfig.linecardFpgaCrcRecoveryMode )

   uncorrectable = {}
   correctable = {}
   softwareCorrected = {}
   uncorrectableCrcs = {}
   correctableCrcs = {}
   softwareCorrectedCrcs = {}
   # Supervisor
   uncorrectableCrcs, correctableCrcs, softwareCorrectedCrcs = (
      crcErrorDetailsCollection( fpgaCrcCellDir.pciFpgaCrcStatus ) 
   )
   # Linecard
   for fpgaSlice in fpgaCrcSliceDir:
      uncorrectable, correctable, softwareCorrected = crcErrorDetailsCollection(
            fpgaCrcSliceDir[ fpgaSlice ].pciFpgaCrcStatus )
      uncorrectableCrcs.update( uncorrectable )
      correctableCrcs.update( correctable )
      softwareCorrectedCrcs.update( softwareCorrected )
   return FpgaCrcErrorsModular( linecardFpgaCrcRecoveryMode=linecardRecoveryMode,
                                fpgaCrcRecoveryMode=fpgaCrcRecoveryMode,
                                uncorrectableCrcs=uncorrectableCrcs,
                                correctableCrcs=correctableCrcs,
                                softwareCorrectedCrcs=softwareCorrectedCrcs )

def doShowFixedFpgaErrors():
   fpgaCrcRecoveryMode = modeEnumToCapiEnum.get(
      fpgaCliConfig.fpgaCrcRecoveryMode )

   uncorrectableCrcs = {}
   correctableCrcs = {}
   softwareCorrectedCrcs = {}
   uncorrectableCrcs, correctableCrcs, softwareCorrectedCrcs = (
      crcErrorDetailsCollection( fpgaCrcCellDir.pciFpgaCrcStatus ) )

   return FpgaCrcErrorsFixed( fpgaCrcRecoveryMode=fpgaCrcRecoveryMode,
                              uncorrectableCrcs=uncorrectableCrcs,
                              correctableCrcs=correctableCrcs,
                              softwareCorrectedCrcs=softwareCorrectedCrcs )

class ShowHardwareFpgaError( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware fpga error'
   data = { 'hardware': hardwareMatcherForShow,
            'fpga': fpgaMatcherForShow,
            'error': errorMatcher,
            }
   privileged = True
   cliModel = FpgaCrcErrors

   @staticmethod
   def handler( mode, args ):
      if isModular():
         return doShowModularFpgaErrors()
      else:
         return doShowFixedFpgaErrors()

BasicCli.addShowCommandClass( ShowHardwareFpgaError )

#-------------------------------------------------------------------------------
#  "show hardware fpga detail" command
#-------------------------------------------------------------------------------
class FpgaHardwareDetail( Model ):
   __revision__ = 2
   fpgaClassName = Str( help="Name of FPGA" )
   majorVersion = Int( help="Major version number of FPGA" )
   minorVersion = Int( help="Minor version number of FPGA" )
   isHitlessResetSupported = Enum( values=( "True", "False", "N/A" ),
                                   help="FPGA support for hitless resets" )
   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr[ "isHitlessResetSupported" ] = \
                        self.isHitlessResetSupported == "True"
      return dictRepr

class FpgaHardwareDetails( Model ):
   __revision__ = 2
   fpgas = Dict( keyType=str, valueType=FpgaHardwareDetail,
                 help="Detailed information for the FPGAs in the system" )

class FpgaHardwareGroup( Model ):
   __revision__ = 2
   systemComponents = Dict( keyType=str, valueType=FpgaHardwareDetails,
               help="A mapping of system devices to its FPGAs" )

   def render( self ):
      def getVersionString( major, minor ):
         if major == 0 and minor == 0:
            return ""
         else:
            return "%d.%d" % ( major, minor )

      def naturalSort( l ):
         def addLen( l ):
            '''Items with more groups of letters or numbers rank higher.'''
            ls = ArPyUtils.naturalOrderIgnoreCaseKey( l )
            return [ len( ls ) ] + ls
         return sorted( l, key=addLen )

      if not self.systemComponents:
         print( "No FPGAs found" )
         return

      # Table Headings
      tableHeadings = ( "FPGA", "Name", "Version", "Hitless Reset Supported" )

      # Table Column Formats
      tcfFpga = TableOutput.Format( justify="left" )
      tcfFpga.padLimitIs( True )
      tcfName = TableOutput.Format( justify="left" )
      tcfName.padLimitIs( True )
      tcfVersion = TableOutput.Format( justify="left" )
      tcfVersion.padLimitIs( True )
      tcfHitlessSupport = TableOutput.Format( justify="left" )
      tcfHitlessSupport.padLimitIs( True )

      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( tcfFpga,
                           tcfName,
                           tcfVersion,
                           tcfHitlessSupport )
      # We only care about the component groupings in JSON format.
      # Remove the grouping so we can output the table appropriately
      fpgaDict = {}
      for group in self.systemComponents.values():
         fpgaDict.update( group.fpgas )

      for fpga in naturalSort( fpgaDict ):
         fpgaDetail = fpgaDict[ fpga ]
         table.newRow( fpga,
               fpgaDetail.fpgaClassName,
               getVersionString( fpgaDetail.majorVersion,
                                 fpgaDetail.minorVersion ),
               fpgaDetail.isHitlessResetSupported )

      print( table.output() )

def fpgaVersion( scdConfigDict, fpgaName, scdStatus ):
   try:
      scdConfigName = scdConfigDict[ fpgaName ].name
      status = scdStatus[ scdConfigName ]
      return FpgaUtil.FpgaVersion( status.firmwareRev, status.hardwareMajorRev )
   except KeyError:
      return FpgaUtil.FpgaVersion( 0, 0 )

def createFpgaName( fpgaName, cardSlot=None ):
   if cardSlot:
      nameStart = fpgaName.find( ':' )
      if nameStart != -1:
         return f"{cardSlot}:{fpgaName[ nameStart + 1: ]}"

   return fpgaName

def hardwareFpgaDetailCollection( pciFpgaConfig, scdConfig,
                                  scdStatus, cardSlot=None ):
   fpgaGroup = defaultdict( dict )
   scdConfigDict = \
      { scdConfig[ scdName ].fpgaName : scdConfig[ scdName ]
                                        for scdName in scdConfig }

   for fpgaName in pciFpgaConfig:
      fpgaConfig = pciFpgaConfig[ fpgaName ]
      version = fpgaVersion( scdConfigDict, fpgaName, scdStatus )
      name = createFpgaName( fpgaName, cardSlot=cardSlot )
      try:
         slotId = scdConfigDict[ fpgaName ].slotId
      except KeyError:
         slotId = -1

      fpgaGroup[ slotId ][ name ] = FpgaHardwareDetail(
                         fpgaClassName=fpgaConfig.fpgaClassName,
                         majorVersion=version.major,
                         minorVersion=version.minor,
                         isHitlessResetSupported= \
                                    str( fpgaConfig.isHitlessResetSupported ) )

   return fpgaGroup

def hardwareCpldDetailCollection( systemCpldStatus, systemCpldConfig,
                                  standbyCpldStatus, standbyCpldConfig,
                                  modular ):
   fpgaGroup = defaultdict( dict )
   for fpgaName in systemCpldStatus:
      status = systemCpldStatus[ fpgaName ]
      config = systemCpldConfig[ fpgaName ]
      slotName = config.customerName
      fpgaGroup[ slotName ][ fpgaName ] = (
         FpgaHardwareDetail( fpgaClassName=fpgaName,
                             majorVersion=status.firmwareRev,
                             minorVersion=status.hardwareMajorRev,
                             isHitlessResetSupported='N/A' ) )

   for fpgaName in standbyCpldStatus:
      status = standbyCpldStatus[ fpgaName ]
      config = standbyCpldConfig[ fpgaName ]
      slotName = config.customerName

      # Because all the StandbyCplds in a system are stored in a single collection,
      # they are given a numerical prefix in the FruPlugin for uniqueness. We don't
      # want this prefix to cause issues with how the Fpgas are represented in Rdam,
      # so we strip it off before we put it in the model.
      fpgaName = fpgaName[ fpgaName.find( "-" ) + 1 : ]
      fullFpgaName = f"{slotName}:{fpgaName}" if modular else fpgaName
      fpgaGroup[ slotName ][ fullFpgaName ] = (
         FpgaHardwareDetail( fpgaClassName=fpgaName,
                             majorVersion=status.revision,
                             minorVersion=status.minorRev,
                             isHitlessResetSupported='N/A' ) )

   return fpgaGroup

def doShowFpgaFixedFpgaDetail() :
   fpgaGroup = { 'FixedSystem' : {} }
   scdFpgaDetail = hardwareFpgaDetailCollection( fpgaFruConfigCellDir.pciFpgaConfig,
                                                 scdConfigCellDir.scdConfig,
                                                 scdStatusCellDir.scdStatus )
   cpldDetail = hardwareCpldDetailCollection( systemCpldStatusDir.cpldStatus,
                                              systemCpldConfigDir.cpldConfig,
                                              standbyCpldStatusDir.cpldStatus,
                                              standbyCpldConfigDir.cpldConfig,
                                              modular=False )
   tmpGroup = {}
   for slot in scdFpgaDetail:
      tmpGroup.update( scdFpgaDetail[ slot ] )

   for slot in cpldDetail:
      tmpGroup.update( cpldDetail[ slot ] )
   fpgaGroup[ 'FixedSystem' ] = FpgaHardwareDetails( fpgas=tmpGroup )
   return FpgaHardwareGroup( systemComponents=fpgaGroup )

def _getAllSlots():
   if entityMib.root and entityMib.root.initStatus == 'ok':
      slotDict = { slot.relPos: f"{slot.tag}{slot.label}"
                   for slot in entityMib.root.cardSlot.values() }
      slotDict[ 0 ] = "Cpld"
      return slotDict
   else:
      return {}

def _createFpgaHardwareDetails( fpgaDetail, fpgaGroup, unknownSlot ):
   slots = _getAllSlots()
   for slotId in fpgaDetail:
      try:
         slotName = slots[ slotId ]
         fpgaGroup[ slotName ] = FpgaHardwareDetails(
                                    fpgas=fpgaDetail[ slotId ] )
      except KeyError:
         unknownSlot.update( fpgaDetail[ slotId ] )

def doShowFpgaModularDetail():
   # Supervisor
   fpgaDetailSup = hardwareFpgaDetailCollection( fpgaFruConfigCellDir.pciFpgaConfig,
                                                 scdConfigCellDir.scdConfig,
                                                 scdStatusCellDir.scdStatus )

   # Linecards
   fpgaDetailCard = {}
   for fpgaSlice in fpgaFruConfigSliceDir:
      if fpgaSlice not in scdConfigSliceDir or fpgaSlice not in scdStatusSliceDir:
         # Some switchcard scds like Anaconda have fru config but status is in supe
         # for crc error propogation and fwd agent operational reasons
         # Skip getting details for them
         # BUG862002 : tracking fix for displaying Anaconda Switchcard scd
         continue

      fpgaDetail = {}
      fpgaDetail = hardwareFpgaDetailCollection(
                              fpgaFruConfigSliceDir[ fpgaSlice ].pciFpgaConfig,
                              scdConfigSliceDir[ fpgaSlice ].scdConfig,
                              scdStatusSliceDir[ fpgaSlice ].scdStatus,
                              cardSlot=fpgaSlice )

      if -1 in fpgaDetail and -1 in fpgaDetailCard:
         # Make sure the unknown slot FPGAs are merged
         fpgaDetailCard[ -1 ].update( fpgaDetail[ -1 ] )
         del fpgaDetail[ -1 ]
      fpgaDetailCard.update( fpgaDetail )

   # Cplds
   cpldDetail = hardwareCpldDetailCollection( systemCpldStatusDir.cpldStatus,
                                              systemCpldConfigDir.cpldConfig,
                                              standbyCpldStatusDir.cpldStatus,
                                              standbyCpldConfigDir.cpldConfig,
                                              modular=True )

   # Organize details from supervisor and cards
   systemComponents = {}
   unknownSlot = {}
   _createFpgaHardwareDetails( fpgaDetailSup, systemComponents, unknownSlot )
   _createFpgaHardwareDetails( fpgaDetailCard, systemComponents, unknownSlot )
   for slotName in cpldDetail:
      if slotName in systemComponents:
         systemComponents[ slotName ].fpgas.update( cpldDetail[ slotName ] )
      else:
         systemComponents[ slotName ] = FpgaHardwareDetails(
                                          fpgas=cpldDetail[ slotName ] )

   # Finally, handle for fpgas with unidentifiable slots.
   if unknownSlot:
      systemComponents[ "Unknown Slot" ] = FpgaHardwareDetails( fpgas=unknownSlot )

   return FpgaHardwareGroup( systemComponents=systemComponents )

class ShowHardwareFpgaDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware fpga detail'
   data = { 'hardware': hardwareMatcherForShow,
            'fpga': fpgaMatcherForShow,
            'detail': detailMatcher,
            }
   privileged = True
   cliModel = FpgaHardwareGroup

   @staticmethod
   def handler( mode, args ):
      if isModular():
         return doShowFpgaModularDetail()
      else:
         return doShowFpgaFixedFpgaDetail()

BasicCli.addShowCommandClass( ShowHardwareFpgaDetail )

def fpgaShowTechGuard():
   return fpgaFruConfigCellDir and fpgaFruConfigCellDir.pciFpgaConfig

registerShowTechSupportCmd( '2021-03-22 11:50:00',
      cmds=[ 'show hardware fpga error' ],
      cmdsGuard=fpgaShowTechGuard )

#-------------------------------------------------------------------------------
# Plugin
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global fpgaCliConfig, fpgaCrcSliceDir, fpgaFruConfigCellDir, fpgaCrcCellDir
   global fpgaFruConfigSliceDir, scdConfigCellDir, scdConfigSliceDir
   global scdStatusCellDir, scdStatusSliceDir, entityMib
   global systemCpldStatusDir, systemCpldConfigDir
   global standbyCpldStatusDir, standbyCpldConfigDir
   fpgaCliConfig = ConfigMount.mount( entityManager,
      'hardware/fpga/cliConfig', 'Hardware::Fpga::FpgaCliConfig', 'w' )
   fpgaCrcSliceDir = LazyMount.mount( entityManager,
      'hardware/fpga/crcStatus/slice', 'Tac::Dir', 'ri' )
   cellId = Cell.cellId()
   fpgaFruConfigCellDir = LazyMount.mount( entityManager,
                                           'hardware/cell/%s/fpga/config' % cellId,
                                           'Hardware::Fpga::PciFpgaConfigDir', 'r' )
   fpgaFruConfigSliceDir = LazyMount.mount( entityManager,
         'hardware/fpga/config/slice', 'Tac::Dir', 'ri' )
   fpgaCrcCellDir = LazyMount.mount( entityManager,
      'hardware/fpga/crcStatus/cell/%d' % cellId,
      'Hardware::Fpga::PciFpgaCrcStatusDir', 'r' )

   scdConfigCellDir = LazyMount.mount( entityManager,
                                       'hardware/cell/%s/scd/config' % cellId,
                                       'Hardware::Scd::Config', 'r' )
   scdConfigSliceDir = LazyMount.mount( entityManager,
                                       'hardware/scd/config/slice',
                                       'Tac::Dir', 'ri' )
   scdStatusCellDir = LazyMount.mount( entityManager,
                                       'hardware/archer/scd/status/cell/%s' % cellId,
                                       'Hardware::Scd::Status', 'r' )
   scdStatusSliceDir = LazyMount.mount( entityManager,
                                       'hardware/archer/scd/status/slice',
                                       'Tac::Dir', 'ri' )
   systemCpldConfigDir = LazyMount.mount( entityManager,
      'hardware/cell/%s/cpld/config' % cellId,
      'Hardware::StandbyCpld::SystemConfig', 'r' )
   systemCpldStatusDir = LazyMount.mount( entityManager,
      'hardware/cpld/status/cell/%s' % cellId,
      'Hardware::StandbyCpld::SystemStatus', 'r' )
   standbyCpldStatusDir = LazyMount.mount( entityManager,
                                           'hardware/standbyCpld/status',
                                           'Hardware::StandbyCpld::Status', 'r' )
   standbyCpldConfigDir = LazyMount.mount( entityManager,
                                           'hardware/standbyCpld/config',
                                           'Hardware::StandbyCpld::Config', 'r' )

   entityMib = LazyMount.mount( entityManager,
                                'hardware/entmib',
                                'EntityMib::Status', 'r' )
