# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function

import os
import six

from CliPlugin.FruModel import Inventory
from CliPlugin.MacAddr import macAddrMatcher, compareMacs
import CliPlugin.FruModel as FruModel # pylint: disable=consider-using-from-import
from CliToken.System import systemMatcherForConfig
import Arnet
import BasicCli
import BasicCliModes
import BasicCliSession
import Cell
import CliCommand
import CliExtensions
import CliGlobal
import CliMatcher
import CliParserCommon
import CommonGuards
import ConfigMount
import Fru
import Fru.CliUtilities
import LazyMount
import ShowCommand
import Tac
import CliParser
import Tracing
import collections
import re
import string

gv = CliGlobal.CliGlobal( dict( chassiStatus=None, ) )

# ------------------------------------------------------
# Registration system for fixed vs modular system rules.
# CliPlugins register callbacks with the SystemTypeReactor
# to be called once the system type has been determined
# (ie the entity mib root has been set).
# ------------------------------------------------------


__defaultTraceHandle__ = Tracing.Handle( "FruCli" )
t0 = Tracing.trace0

# -----------------------------------------------------
# Maximum number of cards of each type
# -----------------------------------------------------
maxNumOfSups = 2
maxNumOfLinecards = 16
maxNumOfFabricCards = 8
maxNumOfSwitchcards = 2

systemTypeReactorInitialized_ = False
modularSystemCallbacksRegisteredWithReadline_ = False
fixedSystemCallbacksRegisteredWithReadline_ = False

gv = CliGlobal.CliGlobal( dict( powerFuseConfig=None, redundancyConfig=None,
                                entmib=None, epochStatus=None, allLeds=None ) )
   
EthAddr = Tac.Type( 'Arnet::EthAddr' )
macAddrConfig = None
hwEntMib = None

class SystemTypeReactor( Tac.Notifiee ):

   notifierTypeName = "EntityMib::Status"

   def __init__( self, entmib ):
      Tac.Notifiee.__init__( self, entmib )
      self.systemType_ = None
      self.handleSystemType()

   def verifySystemType( self, expectedSystemType ):
      if 'SIMULATION_PLATFORM' not in os.environ:
         assert self.systemType_ == expectedSystemType

   @Tac.handler( 'root' )
   def handleSystemType( self ):
      """ React to the entity mib root being set by registering the
      appropriate callbacks to be invoked by the readline thread. We
      cannot simply invoke the callbacks ourselves, since we're
      in the activities thread, and the readline thread may be in
      the process of iterating through rules lists, etc """

      systemType = (
         self.notifier_.root.tacType.fullTypeName if self.notifier_.root else None )
      if systemType is None:
         # System type not yet set
         self.verifySystemType( None )
      elif systemType == "EntityMib::Chassis":
         if self.systemType_ is None:
            while modularSystemCallbacks_:
               callback = modularSystemCallbacks_.pop( 0 )
               BasicCliSession.registerReadlineThreadWork( callback )
            self.systemType_ = systemType
            global modularSystemCallbacksRegisteredWithReadline_
            modularSystemCallbacksRegisteredWithReadline_ = True
         else:
            self.verifySystemType( systemType )
      elif systemType == "EntityMib::FixedSystem":
         if self.systemType_ is None:
            while fixedSystemCallbacks_:
               callback = fixedSystemCallbacks_.pop( 0 )
               BasicCliSession.registerReadlineThreadWork( callback )
            self.systemType_ = systemType
            global fixedSystemCallbacksRegisteredWithReadline_
            fixedSystemCallbacksRegisteredWithReadline_ = True
         else:
            self.verifySystemType( systemType )

fixedSystemCallbacks_ = []
modularSystemCallbacks_ = []

# calls to this should be replaced with fixedSystemGuard
def registerFixedSystemCallback( callback ):
   """ Register a callback to be invoke by the readline thread
   at the point that we determine that we're in a fixed system."""

   assert not systemTypeReactorInitialized_
   fixedSystemCallbacks_.append( callback )

# Will be replaced by registration using modularSystemGuard. ref BUG 23010.
def registerModularSystemCallback( callback ):
   """ Register a callback to be invoke by the readline thread
   at the point that we determine that we're in a module system."""
   t0( 'Registering Modular Callback', callback )

   assert not systemTypeReactorInitialized_
   modularSystemCallbacks_.append( callback )

def modularSystemCallbacksRegisteredWithReadline():
   return modularSystemCallbacksRegisteredWithReadline_

def fixedSystemCallbacksRegisteredWithReadline():
   return fixedSystemCallbacksRegisteredWithReadline_

def _defaultTags( root ):
   if root is None:
      return {}
   cardSlot = root.defaultChildTags.get( 'CardSlot' )
   if cardSlot:
      return [ Fru.CliUtilities.normalizeTag( t ) for t in cardSlot.tag.values() ]
   return []

# Named Tuple that is written if slot specified has not been discovered yet.
SlotDescription = collections.namedtuple( "SlotDescription",
      "slot, tag, label, secondary" )

class SlotMatcher( CliMatcher.Matcher ):
   """Matches a token either in the form of tag+number or just number.
   If forceTags is specified, only matches the number corresponding to
   the specified tags (no tag part).
   """
   # regexs for matching
   labelRegEx = re.compile( '^([0-9]+)$' )
   tagLabelRegEx = re.compile( '^([a-zA-Z]+)([0-9]+)$' )

   def __init__( self, forceTags=None, overriddenDefaultTags=None,
                 includeSecondary=False, **kargs ):
      # pylint: disable-next=super-with-arguments
      super( SlotMatcher, self ).__init__( **kargs )
      self.forceTags_ = forceTags
      # _defaultTags returns [ 'Linecard', 'Supervisor' ] on modular systems.
      # We should not be considering supervisor when we are only
      # matching (or completing) linecard slots. This attr overrides that.
      self.overriddenDefaultTags_ = overriddenDefaultTags
      self.includeSecondary_ = includeSecondary

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root

      t0( "guardsEnabled is ", mode.session.guardsEnabled() )
      # try to match tag+number
      m = self.tagLabelRegEx.match( token )

      if m:
         ( tag, label ) = m.groups()

      if root is None:
         # Hardware has not yet been discovered.
         # Any well formed token will be accepted in Cli when Guards are disabled,
         # one such case being startup.
         if not mode.session.guardsEnabled() and m:
            result = SlotDescription( None, tag, label, None )
            return CliParserCommon.MatchResult( result, token )
         else:
            return CliParserCommon.noMatch

      if self.forceTags_ is None: # pylint: disable=too-many-nested-blocks
         # All possible tags are to be considered
         if m:
            taglabels = Fru.CliUtilities.tagLabels(
               root, tag, includeSecondary=self.includeSecondary_ )
            # return the dictionary of tags with 'tag' as the prefix
            # it has to contain exactly one item
            if len( taglabels ) == 1:
               for tag in taglabels:
                  for slot in root.cardSlot.values():
                     _tag = Fru.CliUtilities.normalizeTag( slot.tag )
                     if _tag == tag:
                        if slot.label == label:
                           result = SlotDescription( slot, tag, label, None )
                           return CliParserCommon.MatchResult( result, token )
                        elif slot.card and slot.card.secondaryCard and \
                              slot.card.secondaryCard.label == label:
                           result = SlotDescription( slot, tag, label,
                                 slot.card.secondaryCard )
                           return CliParserCommon.MatchResult( result, token )
            return CliParserCommon.noMatch

      m = self.labelRegEx.match( token )
      if m:
         # number only
         label = m.group( 1 )
         # Consider forceTags over overriddenDefaultTags_ and _defaultTags
         validTags = _defaultTags( root )
         if self.forceTags_:
            validTags = self.forceTags_
         elif self.overriddenDefaultTags_:
            validTags = self.overriddenDefaultTags_

         for slot in root.cardSlot.values():
            _tag = Fru.CliUtilities.normalizeTag( slot.tag )
            if _tag in validTags:
               if slot.label == label:
                  result = SlotDescription( slot, _tag, label, None )
                  return CliParserCommon.MatchResult( result, token )
               elif slot.card and slot.card.secondaryCard and \
                     slot.card.secondaryCard.label == label:
                  result = SlotDescription( slot, _tag, label,
                     slot.card.secondaryCard )
                  return CliParserCommon.MatchResult( result, token )
         return CliParserCommon.noMatch
      return CliParserCommon.noMatch

   def _slotNumberCompletions( self, root, number, taglabels ):
      # Return completion for a partially input number. If the number is None or
      # falls into the valid range, add the range a non-literal completion.
      completions = []

      for ( tag, label ) in six.iteritems( taglabels ):
         intLabel = [ int( d ) for d in label ]
         minLabel = min( intLabel )
         maxLabel = max( intLabel )

         if number is None or \
                CliParser.isPrefixOfValueInRange( number, minLabel, maxLabel ):
            completions.append( CliParser.Completion( '<%d-%d>'
                                                      %( minLabel, maxLabel ),
                                                      "%s Number" % tag,
                                                      literal=False ) )
      return completions

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None or not hasattr( root, "cardSlot" ):
         # Hardware has not yet been discovered
         return []

      if token and not self.forceTags_:
         # try to match tag+number
         m = self.tagLabelRegEx.match( token )
         if m:
            taglabels = Fru.CliUtilities.tagLabels(
               root, m.group( 1 ), includeSecondary=self.includeSecondary_ )
            if len( taglabels ) == 1:
               return self._slotNumberCompletions( root, int( m.group( 2 ) ),
                                                   taglabels )
            return []

      # just return the number range
      m = self.labelRegEx.match( token )
      if m or not token:
         validTags = _defaultTags( root )
         # Consider forceTags over overriddenDefaultTags_ and _defaultTags
         if self.forceTags_:
            validTags = self.forceTags_
         elif self.overriddenDefaultTags_:
            validTags = self.overriddenDefaultTags_

         taglabels = Fru.CliUtilities.tagLabels(
            root, '', validTags=validTags, includeSecondary=self.includeSecondary_ )
         number = int( m.group( 1 ) ) if m else None
         return self._slotNumberCompletions( root, number, taglabels )
      # nothing matches
      return []
         
   def __str__( self ):
      return "<slot>"

class Card( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, mode, name, entmibRoot ):
      self.mode = mode
      self.name = name
      self.entmibRoot = entmibRoot
      self.tag = None
      self.label = None
      match = SlotMatcher.tagLabelRegEx.match( self.name )
      if match:
         self.tag, self.label = match.groups()
      else:
         assert SlotMatcher.labelRegEx.match( self.name )
         tagLabels = Fru.CliUtilities.tagLabels(
            self.entmibRoot, validTags=[ 'Linecard', 'Supervisor' ] )
         for tag in tagLabels : # pylint: disable=consider-using-dict-items
            if self.name in tagLabels[ tag ]:
               self.tag = tag
               self.label = self.name
               break
         self.name = '%s%s' % ( self.tag, self.label )
      assert self.tag
      assert self.name

      self.isFanSpinner = False
      # Fan spinners are a kind of fabric cards that don't have any chips
      # They don't support powering off
      if self.tag == 'Fabric' and self.entmibRoot: # temporary hack of BUG923857
         for slot in self.entmibRoot.cardSlot.values():
            if slot.tag == self.tag and slot.label == self.label:
               self.isFanSpinner = not slot.card.chip
               break
 
      self.sysdbName = self.name

      if Cell.product().startswith( 'tundra' ):
         # Set the name of Tundra switchcards to SwitchcardCes.
         self.sysdbName = self.name.replace( 'Switchcard', 'SwitchcardCes' )

   def powerOff( self ):
      if self.name == ( "Supervisor%d" % Fru.slotId() ):
         self.mode.addError( "Powering off the current supervisor is not supported. "
                             "Use 'reload power' instead." )
         return
      if self.isFanSpinner:
         self.mode.addError( "Powering off this module is not supported" )
         return

      t0( 'powering off', self.name )
      gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ self.sysdbName ] = True
      gv.powerFuseConfig.powerOffRequested[ self.sysdbName ] = "user configuration"

      # Power off uplink, is a noop on supervisors without an uplink
      if self.tag == 'Supervisor':
         uplinkName = 'Linecard{}'.format( self.label )
         gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ uplinkName ] = True
         gv.powerFuseConfig.powerOffRequested[ uplinkName ] = "user configuration"

      # See BUG8484 for why this sleep is required
      if not self.mode.session.inConfigSession():
         import time # pylint: disable=import-outside-toplevel
         time.sleep( 0.5 )

   def powerOn( self ):
      t0( 'powering on', self.name )
      del gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ self.sysdbName ]
      del gv.powerFuseConfig.powerOffRequested[ self.sysdbName ]

      # Power on uplink, is a noop on supervisors without an uplink
      if self.tag == 'Supervisor':
         uplinkName = 'Linecard{}'.format( self.label )
         del gv.powerFuseConfig.ignorePowerOffUntilRemovalRequested[ uplinkName ]
         del gv.powerFuseConfig.powerOffRequested[ uplinkName ]

def cardTypeGuard( mode, token, includeSecondary=False ):
   # Dont show a card type if it's not present on the System
   # Test only for card labels and not slot ranges

   if not token or token.title() not in [ 'Linecard', 'Supervisor',
                                          'Fabric', 'Switchcard' ]:
      return None

   if not gv.entmib or not gv.entmib.root:
      return CliParser.guardNotThisPlatform
   else:
      slotLabels = Fru.CliUtilities.tagLabels(
         gv.entmib.root, '', includeSecondary=includeSecondary )
      if token.title() not in slotLabels:
         return CliParser.guardNotThisPlatform

   return None

def modularSystemGuard( mode, token ):
   # etba dut has no entmib.root, considered as fixed system
   if ( not gv.entmib.root or
        gv.entmib.root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
      return CliParser.guardNotThisPlatform
   return None

def fixedSystemGuard( mode, token ):
   if ( gv.entmib.root and
        gv.entmib.root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
      return None
   return CliParser.guardNotThisPlatform

def powerSupplyGuard( mode, token ):
   if gv.entmib and gv.entmib.root and gv.entmib.root.powerSupplySlot:
      return None
   return CliParser.guardNotThisPlatform

def cardSlotGuard( mode, token ):
   if ( gv.entmib and
        gv.entmib.root and
        gv.entmib.root.cardSlot ):
      return None
   return CliParser.guardNotThisPlatform

# pylint: disable-next=pointless-string-statement
"""The SlotRule allows the following input:
1. a tag followed by a space and a number, such as 'Linecard 1'
2. a tag followed immediately by a number, such as 'Linecard1'
3. a number without a tag for default tags, such as 1

1 is handled by a series of ConcatRule, and 2/3 are handled by a TokenRule
with SlotMatcher.

Partial tokens are allowed as long as it's not ambiguous, such as 'lin1'
"""
# new parser
class SlotExpressionFactory( CliCommand.CliExpressionFactory ):
   moduleTypes_ = Fru.CliUtilities.moduleTypes

   def __init__( self, includeSecondary=False ):
      # pylint: disable-next=super-with-arguments
      super( SlotExpressionFactory, self ).__init__()
      self.includeSecondary_ = includeSecondary
      if "slotMatcher_" not in self.__class__.__dict__:
         self.slotMatcher_ = SlotMatcher( includeSecondary=includeSecondary )

   @staticmethod
   def cardSlotAndCardTypeGuard( mode, token ):
      return cardSlotGuard( mode, token ) or cardTypeGuard( mode, token )

   def generate( self, name ):
      expression_ = name
      data_ = { name : self.slotMatcher_ }
      names = []
      for k in self.moduleTypes_:
         kwName = name + '_' + k.lower()
         slotName = name + '_' + k.upper()
         expression_ += '| ( %s %s )' % ( kwName, slotName )
         data_[ kwName ] = CliCommand.guardedKeyword(
            k, "%s modules" % k, SlotExpressionFactory.cardSlotAndCardTypeGuard )
         data_[ slotName ] = SlotMatcher( forceTags=[ k ],
                                          includeSecondary=self.includeSecondary_ )
         names.append( slotName )

      class _SlotExpression( CliCommand.CliExpression ):
         expression = expression_
         data = data_
         slotNames_ = names

         @staticmethod
         def adapter( mode, args, argsList ):
            for n in _SlotExpression.slotNames_:
               val = args.pop( n, None )
               if val is not None:
                  if isinstance( val, list ):
                     # the expression is used in a list
                     args.setdefault( name, [] )
                     args[ name ].extend( val )
                  else:
                     args[ name ] = val
                     # only one arg is possible, just stop
                     break
      return _SlotExpression

def getCardsFromIdList( mode, idList, tagLong ):
   '''
   idList is a list of card Ids ( labels ) such as [ 3, 4, 5, 7 ]
   tagLong could be 'Supervisor', 'Linecard', 'Fabric' or '' ( as in the case
   of commands without tags such as 'power enable module 3' )
   '''
   return [ Card( mode, '%s%s' % ( tagLong, cardId ), gv.entmib.root ) \
            for cardId in idList ]

# pylint: disable-next=inconsistent-return-statements
def modularRprActiveSupeGuard( mode, token ):
   if gv.redundancyConfig:
      if gv.redundancyConfig.protocol == "rpr":
         return CommonGuards.standbyGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def cardTypeOrModularRprActiveSupeGuard( model, token ):
   return cardTypeGuard( model, token ) or modularRprActiveSupeGuard( model, token )

def rangeFn( *args, **kwargs ):
   return Fru.CliUtilities.rangeFn( gv.entmib.root, *args, **kwargs )

class PowerSupplyMatcher( CliMatcher.Matcher ):
   """Matches a power supply token in the form of a number.
   If fixedConfig is specified, the command works on both modular
   and fixed config switches, otherwise only on modular.
   """
   # regexs for matching
   labelRegEx = re.compile( '^([0-9]+)$' )

   def __init__( self, fixedConfig=True, **kargs ):
      # pylint: disable-next=super-with-arguments
      super( PowerSupplyMatcher, self ).__init__( **kargs )
      self.fixedConfig_ = fixedConfig

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return CliParserCommon.noMatch

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
         m = self.labelRegEx.match( token )
         if m:
            # ok, the token is a number
            label = m.group( 1 )
            for powersupply in root.powerSupplySlot.values():
               if powersupply.label == label:
                  return CliParserCommon.MatchResult( powersupply, token )
            return CliParserCommon.noMatch
         else:
            return CliParserCommon.noMatch
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return []

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):

         # just return the number range
         m = self.labelRegEx.match( token )
         if m or not token:
            # Return completion for a partially input number. If the number is
            # None or falls into the valid range, add the range a non-literal
            # completion.
            completions = []

            minLabel = min( root.powerSupplySlot )
            maxLabel = max( root.powerSupplySlot )

            number = int( m.group( 1 ) ) if m else None

            if number is None or \
                   CliParser.isPrefixOfValueInRange( number, minLabel, maxLabel ):
               completions.append( CliParser.Completion( '<%d-%d>'
                                                         %( minLabel, maxLabel ),
                                                         "Power Supply Number",
                                                         literal=False ) )
            return completions

         # nothing matches
         return []
      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )

class FanTrayMatcher( CliMatcher.Matcher ):
   """Matches a fan tray token in the form of a number.
   If fixedConfig is specified, the command works on both modular
   and fixed config switches, otherwise only on modular.
   """

   def __init__( self, fixedConfig=True, **kargs ):
      # pylint: disable-next=super-with-arguments
      super( FanTrayMatcher, self ).__init__( **kargs )
      self.fixedConfig_ = fixedConfig

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return CliParserCommon.noMatch

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):

         # Determine if FanX/Y system. The slot configuration of a modular system is
         # not a reliable indicator of this, we have to check manually,
         # unfortunately.

         # If they pass X for a FanX/Y system, there is no match. We have to check
         # for this because some X/Y systems have FanX listed even though it is not
         # not correlated with a physical LED or fan.
         if self.isSlashedSystem() and "/" not in token:
            return CliParserCommon.noMatch

         fanName = f"Fan{token}"
         if fanName in gv.allLeds.leds:
            return CliParserCommon.MatchResult( fanName, token )
         else:
            return CliParserCommon.noMatch

      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         # Hardware has not yet been discovered
         return []

      if root.tacType.fullTypeName == "EntityMib::Chassis" or \
         ( self.fixedConfig_ and \
           root.tacType.fullTypeName == "EntityMib::FixedSystem" ):
         def getKeyRange( regexStr: str ):
            """ Takes in a string to be used for regex. Its first group must
            capture an integer. Then, it returns the range of the numbers that it 
            matches on in the list of leds at /hardware/led/configInit/leds. E.g.,
            we can get the range of X in for LEDs named FanX/Y. Returns a string,
            in the format <{low}-{high}>.
            """
            numsSeen = []
            # Match strings beginning with "Fan", and grab the first integer.
            reg = re.compile( regexStr )
            for led in gv.allLeds.leds.keys():
               cur = reg.match( led )
               if cur:
                  numsSeen.append( int( cur.group( 1 ) ) )
            if not numsSeen:
               return "<none matched>"
            return f"<{min( numsSeen )}-{max( numsSeen)}>"

         completions = []
         result = ""
         helper = ""

         # First regex retrieves the first number after a Fan LED,
         # the second retrieves the second number (this means there must be a first
         # number, as well as a slash).
         regExGetFirst = r"^Fan(\d+)"
         regExGetSecond = r"^Fan\d+\/(\d+)$"

         # Overall function returns "<int>/<int>" if FanX/Y system, "<int>" if FanX
         if self.isSlashedSystem():
            trayRange = getKeyRange( regExGetFirst )
            numRange = getKeyRange( regExGetSecond )
            result = f"{trayRange}/{numRange}"
            helper = "fantray number/fan number"
         else:
            numRange = getKeyRange( regExGetFirst )
            result = numRange
            helper = "fan number"

         completions.append( CliParser.Completion( result,
                                                   helper,
                                                   literal=False ) )
         return completions

      else:
         raise Tac.InternalException(
            "Invalid system root: %s" % root.tacType.fullTypeName )

   def isSlashedSystem( self ):
      """ Returns True if there exist fans in the system which have slashes in them, 
      and False otherwise. E.g. if FanX/Y exists, return True.
      """
      for led in gv.allLeds.leds.keys():
         if r"Fan" in led and r"/" in led:
            return True
      return False


class ModuleExpression( CliCommand.CliExpression ):
   expression = 'MODULE | ( PowerSupply POWER_SUPPLY )'
   data = {
      'MODULE' : SlotExpressionFactory(),
      'PowerSupply' : CliCommand.guardedKeyword( 'PowerSupply',
                                                 'Power Supply modules',
                                                 guard=powerSupplyGuard ),
      'POWER_SUPPLY' : PowerSupplyMatcher(),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args.pop( 'PowerSupply', None )
      ps = args.pop( 'POWER_SUPPLY', None )
      if ps is not None:
         args[ 'MODULE' ] = ps
         return

class SwitchOrModuleExpression( CliCommand.CliExpression ):
   expression = 'Switch | Chassis | MODULE'
   data = {
      'Switch' : CliCommand.guardedKeyword( 'Switch', 'The switch',
                                            guard=fixedSystemGuard ),
      'Chassis' : CliCommand.guardedKeyword( 'Chassis', 'The chassis',
                                             guard=modularSystemGuard ),
      'MODULE' : ModuleExpression,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      for key in ( 'Switch', 'Chassis' ):
         module = args.pop( key, None )
         if module is not None:
            args[ 'MODULE' ] = gv.entmib.root
            break

def _strip( s ):
   if not s:
      return ""
   return six.ensure_str( s.strip(), errors="replace" )

def _safe( s ):
   # escape any weird non-printable characters that may have come out
   # of a corrupted serial eeprom on some device
   return six.ensure_str( s, errors="replace" )

showInvCallbackList = []
def registerShowInventoryCallback( fn ):
   showInvCallbackList.append( fn )

def invokeShowInventoryCallbacks( mode, model ):
   entMib = mode.entityManager.lookup( 'hardware/entmib' )
   for fn in showInvCallbackList:
      fn( entMib, model )

def _populatePowerSupplyInfo( powerSupplySlots, model ):
   for ( i, slot ) in sorted( powerSupplySlots.items() ):
      ps = slot.powerSupply
      if ps:
         serialNum = _safe( ps.serialNum ) or "Unknown"
         name = _safe( ps.modelName ) or "Unknown"
         model.powerSupplySlots[ i ] = FruModel.PowerSupplySlot( 
               name=name, serialNum=serialNum )
      else:
         model.powerSupplySlots[ i ] = FruModel.PowerSupplySlot()
   
def _populateFanTrayInfo( fanTraySlots, model ):
   for ( i, fanTraySlot ) in sorted( fanTraySlots.items() ):
      fanTray = fanTraySlot.fanTray
      if fanTray:
         serialNum = _safe( fanTray.serialNum ) or "Unknown"
         name = _safe( fanTray.modelName ) or "Unknown"
         fans = len( fanTray.fan )
         model.fanTraySlots[ i ] = FruModel.FanTraySlot( 
            numFans=fans, name=name, serialNum=serialNum,
            secondarySerialNum=fanTraySlot.secondarySerialNum )
         for fan in fanTray.fan.values():
            if fan.secondary:
               model.fanTraySlots[ i ].secondaryFans[ fan.relPos ] = fan.fanName
      else:
         model.fanTraySlots[ i ] = FruModel.FanTraySlot()

def _populatePortInfo( portSlotDir, model ):
   roleCounts = collections.defaultdict( int )
   for cardName in portSlotDir:
      port = portSlotDir[ cardName ]
      model.portCount += len( port )
      for p in port.values():
         if p.role:
            if p.role in { "Internal", "FpgaManagement", "WatchPtp",
                           "MuxInput", "MuxOutput", "TrafficGenerator",
                           "Downstream", "Upstream", "WhiteRabbit",
                           "WatchInput", "WatchOutput", "WatchPassthrough" }:
               # BUG5354 workaround - do not display internal interfaces
               # Also, do not display functional interfaces
               model.portCount -= 1
            elif p.role == "SwitchedCpu":
               roleCounts[ "SwitchManagement" ] += 1
            else:
               if p.tag.startswith( "UnconnectedEthernet" ):
                  assert p.role in ( "Switched", "SwitchedBootstrap" )
                  roleCounts[ "Unconnected" ] += 1
               else:
                  roleCounts[ p.role ] += 1

   def firstCharToLower( s ):
      # Convert only the 1st letter of the word into lower case
      return s[ :1 ].lower() + s[ 1: ]

   for ( portType, portCount ) in six.iteritems( roleCounts ):
      portName = "%sPortCount" % firstCharToLower( portType )
      setattr( model, portName, portCount )

def _populateXcvrInfo( xcvrSlotDir, model ):
   def _cleanStr( xcvrStr ):
      """Helper to remove unprintable chars and leading and trailing
         whitespaces from a string.
      """
      return ''.join( x for x in xcvrStr if x in string.printable ).strip()

   def _getMgmtXcvrName( xcvr ):
      """
      Given a management interface returns the xcvr name, else returns None.
      e.g. Returns "1/2" for a xcvr with description "Management1/2"
      """
      # Check the first port since we don't expect there to be multiple for a
      # management interface.  Note: it is possible the port collection is
      # empty, in which case this is not a management transceiver.
      if not xcvr.port:
         return None
      port = list( xcvr.port.values() )[ 0 ]
      if port.role == 'Management':
         m = re.match( r"Management([\d/]+)", port.description )
         return m.group( 1 ) if m else None
      return None

   pos = 0

   def _cardSort( cardName ):
      # Use 0 for sorting on empty cardName (fixed system)
      return cardName if cardName else 0
   for cardName in sorted( xcvrSlotDir, key=_cardSort ):
      xcvrSlots = xcvrSlotDir[ cardName ]
      # Some fixed-systems have card slots, for example, the Caravan
      # platforms support NIM cards.  These platforms may have
      # transceiver slots fixed on the front panel.  For those fixed
      # transceiver slots, we give them a placeholder cardName of "1"
      # even though they aren't really on a card.  This is to reduce confusion
      # between the fixed transceivers slots and NIM card transceiver slots
      # when referring to transceiver slots by name.
      if ( Cell.cellType() == "fixed" and Cell.cardSlotsSupported() ):
         cardName = "1" if not cardName else cardName
      for i in sorted( xcvrSlots ):
         pos += 1
         xcvrSlot = xcvrSlots[ i ]
         xcvr = xcvrSlot.xcvr

         mgmtXcvrName = _getMgmtXcvrName( xcvrSlot )
         if mgmtXcvrName:
            xcvrName = mgmtXcvrName
            modelDict = model.mgmtXcvrSlots
         else:
            # For modular, prepend card number and "/"
            xcvrName = "%s/%s" % ( cardName, i ) if cardName else str( i )
            # Some systems have special slots called Auxiliary slots.
            # Use the EntityMib's description to delineate between normal
            # transceiver slots and Auxiliary slots.
            if xcvrSlot.description.lower().startswith( "auxiliary" ):
               modelDict = model.auxiliaryXcvrSlots
            elif xcvrSlot.description.lower().startswith( "fabric" ):
               modelDict = model.fabricXcvrSlots
            else:
               modelDict = model.xcvrSlots
         if xcvr:
            modelDict[ xcvrName ] = FruModel.XcvrSlot(
                 mfgName=_cleanStr( xcvr.mfgName ),
                 modelName=_cleanStr( xcvr.modelName ),
                 serialNum=_cleanStr( xcvr.serialNum ),
                 hardwareRev=_cleanStr( xcvr.hardwareRev ) )
         else:
            modelDict[ xcvrName ] = FruModel.XcvrSlot()
         modelDict[ xcvrName ]._pos = pos # pylint: disable=protected-access

def _populateCardInfo( cardSlots, model, secondary ):
   for i in cardSlots:
      slot = cardSlots[ i ]
      pos = slot.relPos
      if secondary:
         if not slot.card:
            continue
         card = slot.card.secondaryCard
         if not card:
            continue
         pos = card.relPos
         name = "%s%s" % ( Fru.CliUtilities.normalizeTag( card.tag ), card.label )
      else:
         name = "%s%s" % ( Fru.CliUtilities.normalizeTag( slot.tag ), slot.label )
         card = slot.card
      if card:
         # Until we have RedSup SSO the standby supervisor is shown as 'Inserted'
         # with the rest of the fields blank
         if slot.tag == 'Supervisor' and _safe( card.modelName ) == 'Unknown':
            model.cardSlots[ name ] = FruModel.CardSlot( modelName="Inserted" )
         else:
            hwEpoch = gv.epochStatus.card.get( name ).hwEpoch if \
                      gv.epochStatus.card.get( name ) else _safe( card.hwEpoch )
            model.cardSlots[ name ] = FruModel.CardSlot( 
                  modelName=_safe( card.modelName ), 
                  hardwareRev=_safe( card.hardwareRev ),
                  serialNum=_safe( card.serialNum ), 
                  mfgDate=_safe( card.mfgDate ),
                  hwEpoch=_safe( hwEpoch ) )
      else:
         model.cardSlots[ name ] = FruModel.CardSlot()
      model.cardSlots[ name ]._pos = pos # pylint: disable=protected-access

def _getCardPortSlots( cardSlots ):
   portSlotDir = {}
   for cardSlotId in cardSlots:
      if cardSlots[ cardSlotId ].card:
         portSlotDir[ cardSlotId ] = cardSlots[ cardSlotId ].card.port
   return portSlotDir

def _getCardXcvrSlots( cardSlots ):
   xcvrSlotDir = {}
   for cardSlotId in cardSlots:
      # This assert is necessary, because the code that prints out the
      # xcvrs prefixes xcvrIds with "cardId/" when this is non-false.
      assert cardSlotId
      if cardSlots[ cardSlotId ].card:
         xcvrSlotDir[ cardSlotId ] = cardSlots[ cardSlotId ].card.xcvrSlot
   return xcvrSlotDir

def doShowModularInventory( chassis, model ):
   _populateCardInfo( chassis.cardSlot, model, False )
   _populateCardInfo( chassis.cardSlot, model, True )
   # BUG5354: as-is, this does not count the management ports on the
   #          redundant supervisor, but does count (and displays) the
   #          "Internal" ports.
   # We currently work around the "Internal" ports. We cannot fix the
   # redundant supervisor part until BUG9772 is fixed.

   portSlotDir = _getCardPortSlots( chassis.cardSlot )
   _populatePortInfo( portSlotDir, model )

   xcvrSlotDir = _getCardXcvrSlots( chassis.cardSlot )
   _populateXcvrInfo( xcvrSlotDir, model )

def doShowFixedInventory( fs, model ):
   _populateCardInfo( fs.cardSlot, model, False )
   _populateCardInfo( fs.cardSlot, model, True )

   portSlotDir = _getCardPortSlots( fs.cardSlot )
   portSlotDir[ '' ] = fs.port
   _populatePortInfo( portSlotDir, model )

   xcvrSlotDir = _getCardXcvrSlots( fs.cardSlot )
   xcvrSlotDir[ '' ] = fs.xcvrSlot
   _populateXcvrInfo( xcvrSlotDir, model )

def doShowPrecisionClock( fs, model ):
   if fs.modelName:
      if re.search( '-?CL-?', fs.modelName ):
         model.precisionClock = FruModel.Clock()

def doShowInventory( mode, args ):
   model = FruModel.Inventory()
   sys = mode.entityManager.lookup( 'hardware/entmib' ).fixedSystem or \
       mode.entityManager.lookup( 'hardware/entmib' ).chassis
   cardStatus = gv.epochStatus.card.get( 'FixedSystem' ) or \
       gv.epochStatus.card.get( 'Chassis' )
   if sys:
      model.systemInformation = FruModel.SystemInformation( 
         name=_safe( sys.modelName ), description=_safe( sys.description ),
         serialNum=_safe( sys.serialNum ), hardwareRev=_safe( sys.hardwareRev ),
         mfgDate=_safe( sys.mfgDate ),
         hwEpoch=_safe( cardStatus.hwEpoch if cardStatus else '' ) )
      _populatePowerSupplyInfo( sys.powerSupplySlot, model )
      _populateFanTrayInfo( sys.fanTraySlot, model )
      if mode.entityManager.lookup( 'hardware/entmib' ).fixedSystem:
         doShowFixedInventory( sys, model )
      elif mode.entityManager.lookup( 'hardware/entmib' ).chassis:
         doShowModularInventory( sys, model )
      # Invoke Registered Callback functions
      invokeShowInventoryCallbacks( mode, model )
      doShowPrecisionClock( sys, model )
   else:
      model.systemInformation = FruModel.SystemInformation()
   return model

def prepareSlotRule( entityManager ):
   mg = entityManager.mountGroup()
   gv.entmib = mg.mount( "hardware/entmib", "EntityMib::Status", "r" )
   gv.epochStatus = mg.mount( 'hwEpoch/status', 'HwEpoch::Status', 'r' )
   def _mountsComplete():
      global systemTypeReactorInitialized_
      systemTypeReactorInitialized_ = True
      entityManager.systemTypeReactor = SystemTypeReactor( gv.entmib )
   mg.close( _mountsComplete )

#--------------------------------------------------------------------------------
# show inventory
#--------------------------------------------------------------------------------
class InventoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show inventory'
   data = {
      'inventory': 'Hardware components of the system',
   }
   handler = doShowInventory
   cliModel = Inventory
   privileged = False

BasicCli.addShowCommandClass( InventoryCmd )

#------------------------------------------------------------------------------------
# [no|default] system mac-address <mac_addr>
#------------------------------------------------------------------------------------
# canSetSystemMacAddrHook allows other packages to check if a proposed system mac
# address is acceptable.
canSetSystemMacAddrHook = CliExtensions.CliHook()

def setMacAddress( mode, args ):
   macAddr = args.get( 'MACADDR', EthAddr.ethAddrZero )
   for hook in canSetSystemMacAddrHook.extensions():
      [ accept, message ] = hook( macAddr )
      if message:
         if accept:
            mode.addWarning( message )
         else:
            mode.addError( message )
      if not accept:
         return
   if not Arnet.EthAddr( macAddr ).isUnicast:
      mode.addError( 'System mac-address must be a unicast address.' )
      return

   if not compareMacs( macAddr, hwEntMib.systemMacAddr ):
      mode.addWarning(
              "Change will take effect only after switch reboot." )
   macAddrConfig.macAddr = macAddr

class SystemMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'system mac-address MACADDR'
   noOrDefaultSyntax = 'system mac-address ...'
   data = {
      'system': systemMatcherForConfig,
      'mac-address': CliCommand.guardedKeyword( 'mac-address', 
                                                'Configure MAC address',
                                                guard=fixedSystemGuard ),
      'MACADDR': macAddrMatcher,
   }
   handler = setMacAddress
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( SystemMacAddressCmd )

#--------------------------------------------------
# Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global macAddrConfig, hwEntMib
   gv.powerFuseConfig = ConfigMount.mount( entityManager,
                                           "power/fuse/config/admin",
                                           "Power::SoftwareFuse", "w" )
   gv.redundancyConfig = LazyMount.mount( entityManager, "redundancy/config",
                                          "Redundancy::RedundancyConfig", "r" )
   gv.allLeds = LazyMount.mount( entityManager, "hardware/led/configInit",
                                "Hardware::Led::LedSystemConfigDir", "r" )
   macAddrConfig = ConfigMount.mount( entityManager,
                                      "sys/macAddress/config",
                                      "Fru::MacAddressConfig", "w" )
   hwEntMib = LazyMount.mount( entityManager,
                               "hardware/entmib", 
                               "EntityMib::Status", "r" )

   # Call and set global variable _slotId once
   Fru.slotId()
   # Call and set global variable fruUtil_ once
   Fru.fruUtil()

   prepareSlotRule( entityManager )
