#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function

from collections import defaultdict
import re
import six

from CliModel import (
      Dict,
      Enum,
      Int,
      Model,
      Str,
      Submodel
)
import HumanReadable
from TableOutput import Format, createTable

# Helper Functions

def printDashes( formatStr ):
   # This prints out a series of dashes to match a format string that
   # looks like "%-16s %-12s %-16s %-20s %-10s".

   out = ""
   while formatStr:
      c = formatStr[ 0 ]
      if c == ' ':
         out += c
         formatStr = formatStr[ 1: ]
         continue
      if c == '%':
         spec = formatStr.split()[ 0 ]
         formatStr = formatStr[ len( spec ): ]
         # Match optional '-', number, type specifier.
         match = re.search( r'-?(\d+).*', spec )
         if not match:
            # Not good, but what else can we do?
            continue
         count = int( match.group( 1 ) )
         out += '-' * count
         continue
      out += c
      formatStr = formatStr[ 1: ]
   print( out )

# Eapi Models for the Fru package

class SystemInformation( Model ):
   name = Str( help="Model name", default="Unknown" )
   description = Str( help="Model Description", default="" )
   hardwareRev = Str( help="Hardware Revision", default="" )
   serialNum = Str( help="Serial Number", default="" )
   mfgDate = Str( help="Manufacture Date ( YYYY-MM-DD )", default="" )
   hwEpoch = Str( help="Hardware Epoch", default="" )

storageTypes = {
   'UNKNOWN' : 'unknown',
   'EMMC'    : 'eMMC',
   'SSD'     : 'SSD',
   'EUSB'    : 'eUSB',
   'NVME'    : 'NVME',
}

class StorageDevice( Model ):
   model = Str( help="Model Name of the Storage Device" )
   mount = Str( help="Mount Point of the Storage Device" )
   storageType = Enum( values=list( storageTypes.values() ),
                       help="Storage Type of the Storage Device" )
   serialNum = Str( help="Serial Number of the Storage Device" )
   firmwareRev = Str( help="Firmware Revision of the Storage Device" )
   storageSize = Int( help="Size in GB of the Storage Device" )

   supervisor = Int( help="The supervisor that this device is associated with",
                     optional=True )

   def storageTypeCompat( self ):
      return self.storageType

# For CAPI backward compatibility, this model is retained. The elements
# are logically just different storage devices.
class EmmcFlashDevice( Model ):
   model = Str( help="Model Name of the Flash Device" )
   mount = Str( help="Mount point of the Flash Device" )
   serialNum = Str( help="Serial Number of the Flash Device" )
   firmwareRev = Str( help="Firmware Revision of the Flash Device" )
   storageSize = Int( help="Size in GB of the Flash Device" )

   def storageTypeCompat( self ):
      return "emmc"


class PowerSupplySlot( Model ):
   name = Str( help="Model name of the power supply slot", default="Not Inserted" )
   serialNum = Str( help="Serial number for the power supply slot", default="" )

class FanTraySlot( Model ):
   numFans = Int( help="Number of fans in the fan tray slot.", default=0 )
   name = Str( help="Model name of the fan tray slot", default="" )
   serialNum = Str( help="Serial number for the fan tray slot",
                    default="" )
   secondarySerialNum = Str(
         help="Serial number for the fan tray slot on secondary card", default="" )
   secondaryFans = Dict( keyType=int, valueType=str,
                         help="Secondary fans, keyed by their slot number" )

class XcvrSlot( Model ):
   _pos = Int( help="Relative position of the xcvr slot" )
   mfgName = Str( help="Transceiver Manufacturer name", default="Not Present" )
   modelName = Str( help="Transceiver model name", default="" )
   serialNum = Str( help="Transceiver serial number", default="" )
   hardwareRev = Str( help="Transceiver hardware revision number", default="" )

class CardSlot( Model ):
   _pos = Int( help="Relative position of the card slot" )
   modelName = Str( help="Model name of the card slot", default="Not Inserted" )
   hardwareRev = Str( help="Hardware version of the card slot", default="" )
   serialNum = Str( help="Serial Number of the card slot", default="" )
   mfgDate = Str( help="Manufacture Date of the card slot (YYYY-MM-DD)", default="" )
   hwEpoch = Str( help="Hardware Epoch", default="" )

class Fpga( Model ):
   fpgaRev = Str( help="Revision of the FPGA if present", default="Not Programmed" )

class Clock( Model ):
   pass

class Inventory( Model ):
   __revision__ = 2

   systemInformation = Submodel( valueType=SystemInformation,
                                 help="System Information" )
   # Additional system information, may not be present for namespace duts
   powerSupplySlots = Dict( keyType=int, valueType=PowerSupplySlot,
                            help="Dictionary of power supply slots" )
   fanTraySlots = Dict( keyType=int, valueType=FanTraySlot,
                        help="Dictionary of fan tray slots" )
   portCount = Int( help="Total count of ports", default=0 )
   internalPortCount = Int( help="Total count of Internal ports", default=0 )
   cpuPortCount = Int( help="Total count of CPU ports", default=0 )
   dataLinkPortCount = Int( # not used, here for backwards compatibility
      help="Total count of data link ports", default=0 )
   switchedPortCount = Int( help="Total count of Switched ports", default=0 )
   frontPanelFabricPortCount = Int(
         help="Total count of Front Panel Fabric front ports", default=0 )
   fabricPortCount = Int( help="Total count of Fabric front ports", default=0 )
   switchManagementPortCount = Int( help="Total count of Switch management ports",
                                    default=0 )
   switchedBootstrapPortCount = Int( help="Total count of Switched/Bootstrap ports",
                                     default=0 )
   unconnectedPortCount = Int( help="Total count of unconnected ports", default=0,
                               optional=True )
   switchedFortyGOnlyPortCount = Int( # not used, here for backwards compatibility
      help="Total count of Switched 40G Only ports", default=0 )
   managementPortCount = Int( help="Total count of Management ports", default=0 )
   xcvrSlots = Dict( valueType=XcvrSlot,
                     help="Dictionary of Transceiver slots indexed by name" )
   mgmtXcvrSlots = Dict( valueType=XcvrSlot,
                     help="Dictionary of Transceiver slots indexed by name" )
   # Auxiliary transceiver slots are slots which are not connected to the
   # data plane.  They are only on select platforms, so the dictionary is
   # optional.
   auxiliaryXcvrSlots = Dict( valueType=XcvrSlot, optional=True,
         help="Dictionary of Auxiliary Transceiver slots indexed by name" )
   # Fabric transceiver slots are slots which are connected to the fabric and are
   # used only to pass fabric traffic. They are only on select platforms, so the
   # dictionary is optional
   fabricXcvrSlots = Dict( valueType=XcvrSlot, optional=True,
         help="Dictionary of Fabric Transceiver slots indexed by name" )
   # This attribute is only populated for the modular systems
   cardSlots = Dict( valueType=CardSlot, help=
                     "Dictionary of Card slots, only present for modular systems" )

   storageSerials = Dict( valueType=StorageDevice,
         help="Dictionary of Storage Devices indexed by Serial Number" )

   # Some systems have subcomponents with their own serial numbers.
   subcompSerNums = Dict( keyType=str, valueType=str, optional=True,
                          help="Dictionary of subcomponent serial numbers" )

   # The 'fpgas' and 'precisionClock' attributes are not and should not be used.
   # The associated "show inventory" callback functions lived within the Stratix and
   # Ruby packages (respectively), which have now been deprecated. They're kept
   # because we cannot remove attributes from CLI models once they have shipped.
   fpgas = Dict( valueType=Fpga, help="Dictionary of fpgas indexed by fpga name" )
   precisionClock = Submodel( valueType=Clock, help="System Precision clock",
                              optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'storageDevices' ] = {}
         dictRepr[ 'emmcFlashDevices' ] = {}

         mntFlashKey = None
         for dev in dictRepr[ 'storageSerials' ].values():
            if dev[ 'mount' ] == '/mnt/flash':
               mntFlashKey = dev[ 'model' ]
               break
         suffix = 2

         for dev in dictRepr[ 'storageSerials' ].values():
            key = dev[ 'model' ]

            if ( dev[ 'mount' ] != '/mnt/flash' ) and ( key == mntFlashKey or key in
                  dictRepr[ 'emmcFlashDevices' ] ):
               key += '-%d' % suffix
               suffix += 1
            if dev[ 'storageType' ] == "eMMC":
               del dev[ 'storageType' ]
               dictRepr[ 'emmcFlashDevices' ][ key ] = dev
            else:
               dictRepr[ 'storageDevices' ][ key ] = dev
         del dictRepr[ 'storageSerials' ]
      return dictRepr


   def render( self ):
      # Split system inventory onto two lines for longer SKU/desc
      def makeTable( header, minWidths=None, padLimit=True ):
         if minWidths is None:
            minWidths = [ None ] * len( header )
         assert len( header ) == len( minWidths )
         t = createTable( header, indent=2 )
         formats = []
         for minWidth in minWidths:
            f = Format( justify="left", minWidth=minWidth )
            f.noPadLeftIs( True )
            f.padLimitIs( padLimit )
            formats.append( f )
         t.formatColumns( *formats )
         return t
      print( "System information" )
      sysTitleL1 = ( "Model", "Description" )
      sysTitleL2 = ( "HW Version", "Serial Number", "Mfg Date", "Epoch" )
      whatL1 = ( self.systemInformation.name, self.systemInformation.description )
      whatL2 = ( self.systemInformation.hardwareRev,
                 self.systemInformation.serialNum, self.systemInformation.mfgDate,
                 self.systemInformation.hwEpoch )
      t = makeTable( sysTitleL1, minWidths=[ 24, 52 ] )
      t.newRow( *whatL1 )
      print( t.output() )
      t = makeTable( sysTitleL2, minWidths=[ 11, 14, 10, 5 ] )
      t.newRow( *whatL2 )
      print( t.output() )

      if self.subcompSerNums:
         print( "Subcomponent serial numbers" )
         t = makeTable( ( "Component", "Serial Number" ), padLimit=False )
         for ( c, sn ) in six.iteritems( self.subcompSerNums ):
            t.newRow( c, sn )
         print( t.output() )

      if self.cardSlots:
         cs = len( self.cardSlots )
         print( "System has %s card slots" % cs )
         t = makeTable( ( "Module", "Model", "HW Version", "Serial Number",
                          "Mfg Date", "Epoch" ), padLimit=False )
         # pylint: disable-msg=W0212
         for ( name, card ) in sorted( self.cardSlots.items(),
                                       key=lambda x: x[ 1 ]._pos ):
            t.newRow( name, card.modelName, card.hardwareRev,
                      card.serialNum, card.mfgDate, card.hwEpoch )
         print( t.output() )

      if self.powerSupplySlots:
         ps = len( self.powerSupplySlots )
         print( "System has %s" % HumanReadable.plural( ps, "power supply slot" ) )
         psHeader = ( "Slot", "Model", "Serial Number" )
         t = makeTable( psHeader, minWidths=[ 4, 16, 16 ] )
         for ( i, ps ) in six.iteritems( self.powerSupplySlots ):
            t.newRow( i, ps.name, ps.serialNum )
         print( t.output() )

      if self.fanTraySlots:
         num = len( self.fanTraySlots )
         print( "System has %s" % HumanReadable.plural( num, "fan module" ) )
         fanHeader = ( "Module", "Number of Fans", "Model", "Serial Number" )
         t = makeTable( fanHeader, minWidths=[ 7, 15, 16, 16 ] )
         for ( i, ft ) in six.iteritems( self.fanTraySlots ):
            if ft.numFans > 0:
               # BUG683716: Print fans info for secondary and primary cards
               #            separately
               serialNum = ft.serialNum
               if ft.secondarySerialNum:
                  serialNum = ft.secondarySerialNum
               t.newRow( i, ft.numFans, ft.name, serialNum )
            else:
               t.newRow( i, "Not Inserted", ft.name, ft.serialNum )
         print( t.output() )

      if self.portCount:
         print( "System has %s" % HumanReadable.plural( self.portCount, "port" ) )
         portTypes = ( "management", "switched", "internal", "unconnected",
                       "switchedBootstrap", "cpu", "switchManagement",
                       "frontPanelFabric" )
         # We can have ports that have not been allocated any roles
         # (ie. switched, management etc), in which case pc will be 0 but
         # self.portCount will not be 0. Hence we need to make an
         # additional check here.
         pc = 0
         for portType in portTypes:
            pc += getattr( self, "%sPortCount" % portType )
         if pc:
            # Convert only the 1st alphabet of the word into upper case
            firstCharToUpper = lambda s : s[ :1 ].upper() + s[ 1: ]
            portFormat = "  %-18s %-4s"
            print( "  %-18s %-4s" % ( "Type", "Count" ) )
            printDashes( portFormat )
            for portType in portTypes:
               portCount = getattr( self, "%sPortCount" % portType )
               if portCount:
                  print( portFormat % ( firstCharToUpper( portType ), portCount ) )
         print()

      def printXcvrSlots( xcvrSlots ):
         xcvrFormat = "  %-4s %-16s %-16s %-16s %-4s"
         print( xcvrFormat %
                ( "Port", "Manufacturer", "Model", "Serial Number", "Rev" ) )
         printDashes( xcvrFormat )
         # pylint: disable-msg=W0212
         for ( xcvrName, xcvr ) in sorted( xcvrSlots.items(),
                                           key=lambda x: x[ 1 ]._pos ):
            print( xcvrFormat % ( xcvrName, xcvr.mfgName, xcvr.modelName,
                                 xcvr.serialNum, xcvr.hardwareRev ) )
         print()

      if self.mgmtXcvrSlots:
         print( "System has %s" % HumanReadable.plural( len( self.mgmtXcvrSlots ),
                "management transceiver slot" ) )
         printXcvrSlots( self.mgmtXcvrSlots )

      if self.xcvrSlots:
         print( "System has %s" % HumanReadable.plural( len( self.xcvrSlots ),
                "switched transceiver slot" ) )
         printXcvrSlots( self.xcvrSlots )

      if self.fabricXcvrSlots:
         fabricXcvrStr = HumanReadable.plural( len( self.fabricXcvrSlots ),
                                               "fabric transceiver slot" )
         print( "System has {}".format( fabricXcvrStr ) )
         printXcvrSlots( self.fabricXcvrSlots )

      if self.auxiliaryXcvrSlots:
         auxStr = HumanReadable.plural( len( self.auxiliaryXcvrSlots ),
                                        "auxiliary transceiver slot" )
         print( "System has {}".format( auxStr ) )
         printXcvrSlots( self.auxiliaryXcvrSlots )

      if self.fpgas:
         raise ValueError( "self.fpgas should be empty; see model definition" )

      # To maintain CAPI backward compatability, storageDevices and emmcFlashDevices
      # are modeled separately, but are combined in a single output here. They are
      # sorted by mount point, where /mnt/flash is displayed first, followed by
      # /mnt/drive. For future compatibility, any other mounts are displayed in
      # alphabetical order after that.

      def mountKey( elem ):
         if elem.mount == '/mnt/flash':
            return '.0'
         if elem.mount == '/mnt/drive':
            return '.1'
         return elem.mount

      # List of all devices and a dictionary keyed by supervisor
      dictDevices = defaultdict( list )
      listDevices = self.storageSerials.values()

      # Filling the dict with entries
      for dev in listDevices:
         key = dev.supervisor
         dictDevices[ key ].append( dev )

      # getting the total number of devices in the system
      lenn = sum( len( dev ) for dev in dictDevices.values() )
      if lenn:
         print( "System has %s" % HumanReadable.plural( lenn,
                "storage device" ) )

      for count, ( key, deviceIter ) in enumerate( sorted( dictDevices.items() ) ):
         if deviceIter:
            if count:
               print()
            if key != None: # pylint: disable=singleton-comparison
               print( "Supervisor {}: ".format( key ) )
            t = makeTable( ( "Mount", "Type", "Model", "Serial Number", "Rev",
                             "Size (GB)" ) )

            for dev in sorted( deviceIter, key=mountKey ):
               t.newRow( dev.mount, dev.storageTypeCompat(), dev.model,
                         dev.serialNum, dev.firmwareRev, dev.storageSize )
            print( t.output() )

      if self.precisionClock:
         raise ValueError(
            "self.precisionClock should be empty; see model definition" )
