# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import Subnet
import BasicCli
import BasicCliModes
from CliMode.FsfMode import FsfMode
import CliCommon
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.FsfModels as FsfModels # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliToken.Clear
import CliToken.Monitor
import ConfigMount
import LazyMount
import ShowCommand
import TableOutput
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'Fsf' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2

em = None
config = None
status = None
cmdRequest = None
hwStatus = None

def fsfGuard( mode, token ):
   return CliParser.guardNotThisPlatform if not hwStatus.supported  else None

class FsfConfigMode( FsfMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'server-failure'

   def __init__( self, parent, session ):
      FsfMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.enabled = False

def _subnetsOverlap( newSubnet ):
   new = Subnet( str( newSubnet ) )
   for subnet in config.subnet:
      if new.overlapsWith( Subnet( str( subnet ) ) ):
         return True
   return False

def noShutdown( mode, args ):
   config.enabled = True

def shutdown( mode, args ):
   config.enabled = False

def setNetwork( mode, args ):
   subnet = args[ 'SUBNET' ]
   if subnet in config.subnet:
      return
   if not subnet.validAsPrefix:
      mode.addError( 'Host part of the network must be zero' )
      return
   if _subnetsOverlap( subnet ):
      mode.addError( 'Network overlaps with an already configured one' )
      return
   config.subnet[ subnet ] = True

def noNetwork( mode, args ):
   subnet = args[ 'SUBNET' ]
   if subnet not in config.subnet:
      # pylint: disable-next=consider-using-f-string
      mode.addWarning( 'Network %s was not found for deletion' % subnet )
      return
   else:
      del config.subnet[ subnet ]

def setProxy( mode, args ):
   config.proxy = True
   if 'LIFETIME' in args:
      config.proxyLifetime = args[ 'LIFETIME' ]

def noProxy( mode, args ):
   if 'lifetime' not in args:
      config.proxy = False
   config.proxyLifetime = config.proxyLifetimeDefault

def gotoFsfMode( mode, args ):
   childMode = mode.childMode( FsfConfigMode )
   mode.session_.gotoChildMode( childMode )

def noFsf( mode, args ):
   config.enabled = False
   config.subnet.clear()
   config.proxy = False
   config.proxyLifetime = config.proxyLifetimeDefault

#-------------------------------------------------------------------------------
# [ no | default ] monitor server-failure
#-------------------------------------------------------------------------------
class GotoMonitorServerFailureModeCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor server-failure'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'server-failure': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'server-failure',
            helpdesc='Server-failure configuration' ),
         guard=fsfGuard ),
   }
   handler = gotoFsfMode
   noOrDefaultHandler = noFsf

BasicCliModes.GlobalConfigMode.addCommandClass( GotoMonitorServerFailureModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] network SUBNET
#--------------------------------------------------------------------------------
class NetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'network SUBNET'
   noOrDefaultSyntax = syntax
   data = {
      'network': 'IP network to monitor',
      'SUBNET': IpAddrMatcher.ipAddrWithMaskExpr( 'Network address', 'Network mask',
                                                  'Prefix' ),
   }
   handler = setNetwork
   noOrDefaultHandler = noNetwork

FsfConfigMode.addCommandClass( NetworkCmd )

#--------------------------------------------------------------------------------
# [ no | default ] proxy [ lifetime LIFETIME ]
#--------------------------------------------------------------------------------
class ProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'proxy [ lifetime LIFETIME ]'
   noOrDefaultSyntax = 'proxy [ lifetime [ LIFETIME ] ]'
   data = {
      'proxy': 'Proxy settings',
      'lifetime': 'Proxy lifetime',
      'LIFETIME': CliMatcher.IntegerMatcher( 1, 10080, helpdesc='Minutes' ),
   }
   handler = setProxy
   noOrDefaultHandler = noProxy

FsfConfigMode.addCommandClass( ProxyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable server-failure monitoring',
   }
   handler = shutdown
   noHandler = noShutdown
   defaultHandler = shutdown

FsfConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# show monitor server-failure
# show monitor server-failure history
# show monitor server-failure servers
# show monitor server-failure servers proxying
# show monitor server-failure servers inactive
# show monitor server-failure servers all
# show monitor server-failure servers A.B.C.D
#-------------------------------------------------------------------------------
def showFsf( mode, args ):
   fsfShow = FsfModels.MonitorServerFailure()
   fsfShow.enabled = status.enabled
   fsfShow.proxyService = config.proxy
   fsfShow.proxyLifetimeDefault = config.proxyLifetimeDefault
   fsfShow.proxyLifetime = config.proxyLifetime
   subnetServersCount = {}
   for server in status.server.values():
      if server.subnet not in subnetServersCount:
         subnetServersCount[ server.subnet ] = 0
      subnetServersCount[ server.subnet] += 1
   for subnet in config.subnet:
      fsfShow.networkServers[ subnet.stringValue ] = \
          subnetServersCount[ subnet ] if subnet in subnetServersCount else 0
   return fsfShow

def _fillServerDetails( modelServer, server ):
   modelServer.isValid = True
   modelServer.ipAddress = server.ipAddr
   modelServer.ethAddress = server.ethAddr
   modelServer.subnet = server.subnet.stringValue
   modelServer.state = server.state
   modelServer.intf = server.intf
   modelServer.discoveryTime = modelServer.toUtc( server.discoveryTime )
   modelServer.lastFailureTime = modelServer.toUtc( server.lastFailureTime )
   modelServer.lastProxyTime = modelServer.toUtc( server.lastProxyTime )
   modelServer.lastInactiveTime = modelServer.toUtc( server.lastInactive )
   modelServer.numFailed = server.numFailed
   modelServer.numProxied = server.numProxied
   modelServer.numInactive = server.numInactive

# This function returns a server to CAPI and render function in model actually 
# shows the server
def showServerWithIpAddr( mode, args ):
   ipAddr = args[ 'IP_ADDR' ]
   if ipAddr not in status.server:
      # pylint: disable-next=consider-using-f-string
      msg = 'Server with Ip address %s doesn\'t exist' % ipAddr
      raise CliCommon.ApiError( msg )
   modelServer = FsfModels.Server()
   server = status.server[ ipAddr ]
   _fillServerDetails( modelServer, server )
   return modelServer

serverStates = [ 'up', 'down', 'proxying', 'inactive' ]
summaryStatement = {
      'proxying' : 'Active failed servers being proxied:',
      'inactive' : 'Inactive servers:',
      'all' : 'Total servers monitored:',
      'active' : 'Active servers:' }

def _createFsfTable( headings ):
   table = TableOutput.createTable( headings )
   f = TableOutput.Format( justify='left' )
   f.noPadLeftIs( True )
   table.formatColumns( *( [ f ] * len( headings ) ) )
   return table

# This function returns a collection to CAPI and model's render function actually 
# shows the servers
def showServers( mode, args ):
   state = args.get( 'STATE' )
   servers = FsfModels.Servers()
   servers.fsfEnabled = status.enabled
   if not state:
      showStates = list( set( serverStates ) - {  'inactive'  } )
      state = 'active'
   elif state == 'all':
      showStates = serverStates
   else:
      assert state in serverStates
      showStates = [ state ]
   for server in sorted( status.server.values() ):
      if not server.state in showStates:
         continue
      modelServer = FsfModels.Server()
      _fillServerDetails( modelServer, server )
      servers.servers.append( modelServer )
   servers.summaryStatement = summaryStatement[ state ]
   return servers

# This function returns a collection to CAPI and model's render function actually 
# shows the history
def showHistory( mode, args ):
   history = FsfModels.ServerHistory()
   history.fsfEnabled = status.enabled
   for server in sorted( status.server.values() ):
      if not server.lastFailureTime:
         continue
      modelServer = FsfModels.Server()
      _fillServerDetails( modelServer, server )
      history.servers.append( modelServer )
   return history

matcherServers = CliMatcher.KeywordMatcher( 'servers', helpdesc='Servers monitored' )
nodeServerFailure = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'server-failure',
         helpdesc='Server-failure information' ), 
      guard=fsfGuard )

#--------------------------------------------------------------------------------
# show monitor server-failure
#--------------------------------------------------------------------------------
class MonitorServerFailureCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor server-failure'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'server-failure': nodeServerFailure,
   }
   handler = showFsf
   cliModel = FsfModels.MonitorServerFailure

BasicCli.addShowCommandClass( MonitorServerFailureCmd )

#--------------------------------------------------------------------------------
# show monitor server-failure history
#--------------------------------------------------------------------------------
class MonitorServerFailureHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor server-failure history'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'server-failure': nodeServerFailure,
      'history': 'History of failed servers',
   }
   handler = showHistory
   cliModel = FsfModels.ServerHistory

BasicCli.addShowCommandClass( MonitorServerFailureHistoryCmd )

#--------------------------------------------------------------------------------
# show monitor server-failure servers [ STATE ]
#--------------------------------------------------------------------------------
class MonitorServerFailureServersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor server-failure servers [ STATE ]'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'server-failure': nodeServerFailure,
      'servers': matcherServers,
      'STATE': CliMatcher.EnumMatcher( {
         'all': 'Show all servers',
         'proxying': 'Servers being proxied',
         'inactive': 'Inactive servers'
      } ),
   }
   handler = showServers
   cliModel = FsfModels.Servers

BasicCli.addShowCommandClass( MonitorServerFailureServersCmd )

#--------------------------------------------------------------------------------
# show monitor server-failure servers IP_ADDR
#--------------------------------------------------------------------------------
class MonitorServerFailureServersIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor server-failure servers IP_ADDR'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'server-failure': nodeServerFailure,
      'servers': matcherServers,
      'IP_ADDR': IpAddrMatcher.IpAddrMatcher(
                                            helpdesc='Show server with IP address' ),
   }
   handler = showServerWithIpAddr
   cliModel = FsfModels.Server

BasicCli.addShowCommandClass( MonitorServerFailureServersIpCmd )

#--------------------------------------------------------------------------------
# clear server-failure servers inactive
#--------------------------------------------------------------------------------
def clearInactiveServers( mode, args ):
   if not status.enabled:
      return
   requestTime = Tac.now()
   cmdRequest.clearInactiveServersRequest = requestTime
   try:
      Tac.waitFor( lambda: status.clearInactiveServersResponse > requestTime,
                   timeout=5, warnAfter=None, maxDelay=0.1, sleep=True )
   except Tac.Timeout:
      mode.addWarning( 'Inactive servers may not have cleared' )

class ClearServerFailureServersInactiveCmd( CliCommand.CliCommandClass ):
   syntax = 'clear server-failure servers inactive'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'server-failure': nodeServerFailure,
      'servers': matcherServers,
      'inactive': 'Inactive servers',
   }
   handler = clearInactiveServers

BasicCliModes.EnableMode.addCommandClass( ClearServerFailureServersInactiveCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em, config, status, hwStatus, cmdRequest

   em = entityManager
   config = ConfigMount.mount( entityManager, 'fastserverfailover/config',
                             'FastServerFailover::Config', 'w' )
   status = LazyMount.mount( entityManager, 'fastserverfailover/status',
                             'FastServerFailover::Status', 'r' )
   cmdRequest = LazyMount.mount( entityManager, 'fastserverfailover/cmdrequest',
                               'FastServerFailover::CmdRequest', 'w' )
   hwStatus = entityManager.mount( 'fastserverfailover/hardware/status',
                                   'FastServerFailover::Hardware::Status', 'r' )

