# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliPlugin
from CliPlugin import SwitchIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.GeneratorModel import (
   LinkQualificationProfileModel,
   LinkQualificationAllProfilesModel,
   LinkQualificationGeneratorIntfModel,
   LinkQualificationGeneratorAllIntfsModel,
)
from CliPlugin.GeneratorCliLib import (
   GeneratorsMode,
   rfc2544ConfigKwMatcher,
   satLinkQualificationSupportedGuard,
   guardLinkQualificationCommand,
   profileMatcher,
   linkQualProfileMatcher,
   generatorsNode,
   interfaceNode,
   isFeatureEnabled,
)
import CliMatcher
import CliCommand
from CliToken.Generator import(
   startKwForGenerator,
   stopKwForGenerator,
)
from CliToken.Clear import (
   clearKwNode,
)
from CliToken.Monitor import ( 
   monitorMatcher,
   monitorMatcherForShow,
   monitorMatcherForClear
)
import ShowCommand
import ConfigMount
import LazyMount
import Tac
from EoamTypes import (
   rateUnitToEnumMap,
   FeatureEnabledEnum,
)
from TypeFuture import TacLazyType
from GeneratorCliUtils import (
   generatorTypeKwLinkQualification,
   generatorTypeKwRFC2544,
)
from CliMode.Generator import (
   ProfilesModeBase,
)
import Intf
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType

# Assigning globals
satProfilesConfig = None
intfCliConfig = None
reconciledIntfConfig = None
intfStatus = None
satHwCapabilities = None
rfc2544ProfileConfigDir = None
featureConfig = None
featureStatus = None
rfc2544ExecConfigDir = None

IntfId = TacLazyType( 'Arnet::IntfId' )
pktSize = TacLazyType( 'Sat::PktSize' )
testStatus = TacLazyType( 'Sat::TestStatus' )
trafficDuration = TacLazyType( 'Sat::TrafficDuration' )
request = TacLazyType( 'FlowGenerator::FlowRequestType' )
rateUnit = TacLazyType( 'FlowGenerator::FlowRateUnit' )
featureEnabled = TacLazyType( 'FlowGenerator::FeatureEnabled' )

rfc2544EnabledWarning = 'Link qualification cannot run when rfc2544 is enabled\n'
linkQualDisabledWarning = 'Link qualification is currently disabled\n'

def pktSizeMax():
   if satHwCapabilities:
      return satHwCapabilities.maxPktSize
   return 2 ** 16 - 1 # max u16

def pktRateMaxValue():
   if satHwCapabilities:
      return satHwCapabilities.maxPacketRateAndUnit.packetRateValue
   return 2 ** 64 - 1 # max u64

def pktRateDefaultValue():
   if satHwCapabilities:
      return satHwCapabilities.minPacketRateAndUnit.packetRateValue
   return 10 # in kbps

linkNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'link',
   helpdesc='Physical link connecting devices' ),
   guard=satLinkQualificationSupportedGuard )
qualificationNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'qualification',
   helpdesc='Link qualification' ) )
profileInterfaceNode = CliCommand.guardedKeyword( 'interface',
   helpdesc='Interface to be tested',
   guard=guardLinkQualificationCommand )

gIntfMatcher = CliPlugin.VirtualIntfRule.IntfMatcher()
gIntfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher 
gIntfMatcher |= SwitchIntfCli.SwitchIntf.matcher

gIntfTypes = [ EthPhyAutoIntfType, SwitchIntfCli.SwitchAutoIntfType ]
gIntfRangeMatcher = Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=gIntfTypes )

#------------------------------------------------------------------
# (config)# monitor ( ( link qualification ) | rfc2544 ) generators
#------------------------------------------------------------------
class GeneratorsModeCommand( CliCommand.CliCommandClass ):
   syntax = 'monitor ( ( link qualification ) | rfc2544  ) generators'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' : monitorMatcher,
      'link' : linkNode,
      'qualification' : qualificationNode,
      'rfc2544' : rfc2544ConfigKwMatcher,
      'generators' : generatorsNode,
   }

   @staticmethod
   def handler( mode, args ):
      genType = None
      if "link" in args and "qualification" in args:
         genType = generatorTypeKwLinkQualification
      elif generatorTypeKwRFC2544 in args:
         genType = generatorTypeKwRFC2544
      childMode = mode.childMode( GeneratorsMode, genType=genType )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if "link" in args and "qualification" in args:
         satProfilesConfig.txConfig.clear()
         intfCliConfig.intfList.clear()
         del featureConfig.featureEnabled[ FeatureEnabledEnum.linkQualification ]
      elif generatorTypeKwRFC2544 in args:
         # Since RFC2544 is unconfigured altogether, we clear the following:
         # - all profile config
         # - all exec requests
         rfc2544ExecConfigDir.intfExecRequest.clear()
         rfc2544ProfileConfigDir.profileConfig.clear()
         del featureConfig.featureEnabled[ FeatureEnabledEnum.rfc2544 ]

BasicCli.GlobalConfigMode.addCommandClass( GeneratorsModeCommand )

#--------------------------------------------
# # (config-mon-<genType>-gen)# [no] disabled
#--------------------------------------------
class GeneratorsDisabledCommand( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled' : 'Disable generator' }

   # User can enable both the features using the CLI. In such cases,
   # as concluded in cli-review, only RFC2544 will be operationally active.
   # SAT link qualification will be inactive and show the reason in one of
   # the show commands.
   @staticmethod
   def handler( mode, args ):
      if mode.genType == generatorTypeKwLinkQualification:
         del featureConfig.featureEnabled[ FeatureEnabledEnum.linkQualification ]
      elif mode.genType == generatorTypeKwRFC2544:
         # Since RFC2544 is disabled, but it is still configured, we should not
         # delete the profile config. However, we should clear the exec requests
         # because when the feature is disabled, the agent will go ahead and delete
         # all execStatus entries. When RFC2544 generator is enabled again,
         # Eoam agent walks over existing exec requests to reconcile. At this point,
         # if we find there are no testExecStatus entries, we create one on the fly
         # to report test failed because of eoamAgentRestart. This can be misleading
         # output and can cause confusion where test-reports are presented to
         # requests before the feature was enabled.
         rfc2544ExecConfigDir.intfExecRequest.clear()
         del featureConfig.featureEnabled[ FeatureEnabledEnum.rfc2544 ]

   @staticmethod
   def noHandler( mode, args ):
      if mode.genType == generatorTypeKwLinkQualification:
         featureConfig.featureEnabled[ FeatureEnabledEnum.linkQualification ] = True
      elif mode.genType == generatorTypeKwRFC2544:
         featureConfig.featureEnabled[ FeatureEnabledEnum.rfc2544 ] = True

   @staticmethod
   def defaultHandler( mode, args ):
      if mode.genType == generatorTypeKwLinkQualification:
         del featureConfig.featureEnabled[ FeatureEnabledEnum.linkQualification ]
      elif mode.genType == generatorTypeKwRFC2544:
         rfc2544ExecConfigDir.intfExecRequest.clear()
         del featureConfig.featureEnabled[ FeatureEnabledEnum.rfc2544 ]

GeneratorsMode.addCommandClass( GeneratorsDisabledCommand )

def setProfileToDefault( profileNames ):
   for profileName in profileNames:
      config = satProfilesConfig.txConfig.newMember( profileName )
      config.pktSize = pktSize.defaultSize
      config.duration = trafficDuration.defaultDuration
      config.packetRateAndUnit = Tac.newInstance(
                                    "FlowGenerator::PacketRateAndUnit",
                                    pktRateDefaultValue(),
                                    rateUnit.rateUnitKbps )

class LinkQualificationProfilesMode( ProfilesModeBase, BasicCli.ConfigModeBase ):
   name = 'link qualification profiles'

   def __init__( self, parent, session, genType, profileName ):
      self.genType = genType
      ProfilesModeBase.__init__( self, ( genType, profileName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.profile = self.getOrCreateProfileConfig( profileName )

   def getOrCreateProfileConfig( self, profileName ):
      return satProfilesConfig.txConfig.newMember( profileName )

class Rfc2544ProfilesMode( ProfilesModeBase, BasicCli.ConfigModeBase ):
   name = 'rfc2544 profiles'

   def __init__( self, parent, session, genType, profileName ):
      self.genType = genType
      ProfilesModeBase.__init__( self, ( genType, profileName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.profile = self.getOrCreateProfileConfig( profileName )

   def getOrCreateProfileConfig( self, profileName ):
      return rfc2544ProfileConfigDir.profileConfig.newMember( profileName )

#--------------------------------------------
# (config-mon-<genType>-gen)# profile PROFILE
#--------------------------------------------
class ProfilesModeCommand( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile { PROFILE }'
   data = { 'profile': 'Configure generator profile',
            'PROFILE': profileMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE' ]
      if len( profileName ) > 50:
         mode.addErrorAndStop( profileName + " too long: must be no more than 50 "\
                               "characters" )
      if mode.genType == generatorTypeKwLinkQualification:
         childMode = mode.childMode( LinkQualificationProfilesMode,
                                     genType=mode.genType,
                                     profileName=profileName )
      elif mode.genType == generatorTypeKwRFC2544:
         childMode = mode.childMode( Rfc2544ProfilesMode,
                                     genType=mode.genType,
                                     profileName=profileName )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateProfileConfig( profileName )

   @staticmethod
   def noHandler( mode, args ):
      profileNames = args[ 'PROFILE' ]
      if mode.genType == generatorTypeKwLinkQualification:
         for profileName in profileNames:
            del satProfilesConfig.txConfig[ profileName ]
      elif mode.genType == generatorTypeKwRFC2544:
         for profileName in profileNames:
            del rfc2544ProfileConfigDir.profileConfig[ profileName ]

   @staticmethod
   def defaultHandler( mode, args ):
      profileNames = args[ 'PROFILE' ]
      if mode.genType == generatorTypeKwLinkQualification:
         setProfileToDefault( profileNames )
      elif mode.genType == generatorTypeKwRFC2544:
         for profileName in profileNames:
            del rfc2544ProfileConfigDir.profileConfig[ profileName ]

GeneratorsMode.addCommandClass( ProfilesModeCommand )

#----------------------------------------------------
# (config-mon-lq-gen)# interface INTF profile PROFILE
#----------------------------------------------------
class IntfProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF profile PROFILE'
   noOrDefaultSyntax = 'interface INTF profile ...'
   data = { 'interface' : profileInterfaceNode,
            'INTF': gIntfMatcher,
            'profile': 'Profile describing link qualification test',
            'PROFILE': profileMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      profile = args[ 'PROFILE' ]
      intfCliConfig.intfList.newMember( IntfId( intf.name ), profile )
      if profile not in satProfilesConfig.txConfig:
         # We allow profile association before creation/configuration
         mode.addWarning( profile + " profile has not been configured yet" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      del intfCliConfig.intfList[ IntfId( intf.name ) ]

GeneratorsMode.addCommandClass( IntfProfileCmd )

#-------------------------------------------------------------------------
# (config-mon-lq-gen-prof-<profile_name>)# [no|default] packet size <SIZE>
#-------------------------------------------------------------------------
class packetSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'packet size SIZE bytes'
   noOrDefaultSyntax = 'packet size ...'
   # TODO: add better helpdesc text
   data = { 'packet': 'Packet',
            'size': 'Packet size in bytes',
            'SIZE': CliMatcher.IntegerMatcher( pktSize.min, pktSizeMax(),
                        helpdesc='Packet size in bytes' ),
            'bytes' : 'Packet size unit',
          }

   @staticmethod
   def handler( mode, args ):
      size = args.get( 'SIZE', pktSize.defaultSize )
      mode.profile.pktSize = size

   noOrDefaultHandler = handler

LinkQualificationProfilesMode.addCommandClass( packetSizeCmd )

#----------------------------------------------------------------------------
# (config-mon-lq-gen-prof-<profile_name>)# [no|default] traffic duration TIME
#----------------------------------------------------------------------------
class trafficDurationCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic duration TIME seconds'
   noOrDefaultSyntax = 'traffic duration ...'
   # TODO: 1. add better helpdesc text
   #       2. TIME: what's the min and max?
   data = { 'traffic': 'Traffic',
            'duration': 'Traffic duration in seconds',
            'TIME': CliMatcher.IntegerMatcher( trafficDuration.min,
                       trafficDuration.max, helpdesc='Traffic duration in seconds' ),
            'seconds' : 'Traffic duration time unit',
          }

   @staticmethod
   def handler( mode, args ):
      # default traffic duration = 600 seconds.
      duration = args.get( 'TIME', trafficDuration.defaultDuration )
      mode.profile.duration = duration

   noOrDefaultHandler = handler

LinkQualificationProfilesMode.addCommandClass( trafficDurationCmd )

#--------------------------------------------------------------------------
# (config-mon-lq-gen-prof-<profile_name>)# [no|default] traffic rate <RATE>
#--------------------------------------------------------------------------
class trafficRateCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic rate RATE UNIT'
   noOrDefaultSyntax = 'traffic rate ...'
   # TODO: add better helpdesc
   data = { 'traffic': 'Traffic',
            'rate': 'Traffic rate',
            'RATE': CliMatcher.IntegerMatcher( 1,
                        pktRateMaxValue(),
                        helpdesc='Traffic rate' ),
            'UNIT': CliMatcher.EnumMatcher( {
                           'kbps' : 'Traffic unit in kbps',
                           'mbps' : 'Traffic unit in mbps',
                           'gbps' : 'Traffic unit in gbps',
                        } )
          }

   @staticmethod
   def handler( mode, args ):
      rate = args.get( 'RATE', pktRateDefaultValue() )
      unit = rateUnitToEnumMap[ args.get( 'UNIT', 'kbps' ) ]

      if unit is rateUnit.rateUnitKbps and rate < pktRateDefaultValue():
         mode.addErrorAndStop( 'Rate configured below supported minimum' )

      mode.profile.packetRateAndUnit = Tac.newInstance(
                                          "FlowGenerator::PacketRateAndUnit",
                                          rate, unit )
   noOrDefaultHandler = handler

LinkQualificationProfilesMode.addCommandClass( trafficRateCmd )

#------------------------------------------------------------------------------
# #( start | stop ) monitor link qualification generators interface INTFS
#------------------------------------------------------------------------------
class IntfProfileActionCmd( CliCommand.CliCommandClass ):
   syntax = '( start | stop ) monitor link qualification generators interface INTFS'
   data = { 'start' : startKwForGenerator,
            'stop' : stopKwForGenerator,
            'monitor': monitorMatcher,
            'link' : linkNode,
            'qualification' : qualificationNode,
            'generators' : generatorsNode,
            'interface' : interfaceNode,
            'INTFS': gIntfRangeMatcher,
          }

   @staticmethod
   def handler( mode, args ):  
      featureEnabledError = isFeatureEnabled( FeatureEnabledEnum.linkQualification )
      if featureEnabledError:
         mode.addErrorAndStop( featureEnabledError )

      start = 'start' in args

      for intf in args[ 'INTFS' ]:
         intfInfo = intfCliConfig.intfList.get( IntfId( intf ) )
         if intfInfo is None or intfInfo.profile == '':
            # pylint: disable-next=consider-using-f-string
            mode.addErrorAndStop( 'There is no profile associated with %s' % intf )

         if intfInfo.profile not in satProfilesConfig.txConfig and start:
            # we care about profile existence only in 'start'
            mode.addErrorAndStop( intfInfo.profile + " profile associated with " \
                                  + str( intf ) + " does not exist" )

         hwIntfStatus = intfStatus.hwIntfList.get( IntfId( intf ) )
         if hwIntfStatus and start and hwIntfStatus.started == testStatus.running:
            mode.addErrorAndStop( "Link qualification is already running on " \
                                  + str( intf ) )

         intfInfo.request = request.startFlow if start else request.stopFlow
         intfInfo.requestVersion += 1

BasicCli.EnableMode.addCommandClass( IntfProfileActionCmd )

#------------------------------------------------------------------------------
# #clear monitor link qualification generators interface INTFS
#------------------------------------------------------------------------------
class IntfProfileClearCmd( CliCommand.CliCommandClass ):
   syntax = 'clear monitor link qualification generators interface INTFS'
   data = { 'clear' : clearKwNode,
            'monitor' : monitorMatcherForClear,
            'link' : linkNode,
            'qualification' : qualificationNode,
            'generators' : generatorsNode,
            'interface' : interfaceNode,
            'INTFS': gIntfRangeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      featureEnabledError = isFeatureEnabled( FeatureEnabledEnum.linkQualification )
      if featureEnabledError:
         mode.addErrorAndStop( featureEnabledError )

      for intf in args[ 'INTFS' ]:
         intfInfo = intfCliConfig.intfList.get( IntfId( intf ) )
         if intfInfo is None:
            continue

         intfInfo.request = request.clearFlow
         intfInfo.requestVersion += 1
      
BasicCli.EnableMode.addCommandClass( IntfProfileClearCmd )

def populateLinkQualificationProfiles( profileName=None ):
   linkQualificationAllProfiles = LinkQualificationAllProfilesModel()

   profilesDict = { profileName : satProfilesConfig.txConfig.get( profileName ) } \
                  if profileName else satProfilesConfig.txConfig
   for key, profile in profilesDict.items():
      profileModel = LinkQualificationProfileModel(
                        pktSize=profile.pktSize,
                        duration=profile.duration,
                        rate=profile.packetRateAndUnit.packetRateValue,
                        rateUnit=profile.packetRateAndUnit.packetRateUnit )
      linkQualificationAllProfiles.profiles[ key ] = profileModel
   linkQualificationAllProfiles.enabled = \
         ( featureStatus.featureEnabled == FeatureEnabledEnum.linkQualification )

   return linkQualificationAllProfiles

class ShowLinkQualificationProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor link qualification generators profiles [ PROFILE ]'
   data = { 'monitor' : monitorMatcherForShow,
            'link' : linkNode,
            'qualification' : qualificationNode,
            'generators' : generatorsNode,
            'profiles' : 'Profiles',
            'PROFILE' : linkQualProfileMatcher,
          }

   cliModel = LinkQualificationAllProfilesModel

   @staticmethod
   def handler( mode, args ):
      profileName = args.get( 'PROFILE' )
      if profileName and profileName not in satProfilesConfig.txConfig:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "profile %s not configured" % profileName )
         return None

      if featureStatus.featureEnabled == FeatureEnabledEnum.rfc2544:
         mode.addWarning( rfc2544EnabledWarning )
      elif featureStatus.featureEnabled != FeatureEnabledEnum.linkQualification:
         mode.addWarning( linkQualDisabledWarning )

      return populateLinkQualificationProfiles( profileName=profileName )

BasicCli.addShowCommandClass( ShowLinkQualificationProfilesCmd )

def populateLinkQualificationGeneratorIntf( status ):
   interfaceModel = LinkQualificationGeneratorIntfModel(
                       state=status.started,
                       startTime=status.startTime,
                       endTime=status.endTime,
                       pktSize=status.pktSize,
                       duration=status.duration,
                       rate=status.actualPacketRateAndUnit.packetRateValue,
                       rateUnit=status.actualPacketRateAndUnit.packetRateUnit,
                       pktSent=status.txPktCount,
                       pktRcvd=status.rxPktCount,
                       pktDropped=status.dropPktCount,
                       txErrors=status.txErrorCount,
                       rxErrors=status.rxErrorCount,
                       txOctetCount=status.txOctetCount,
                       rxOctetCount=status.rxOctetCount )
   if status.errorCode:
      interfaceModel.errorCode = status.errorCode

   return interfaceModel

class ShowLinkQualificationGeneratorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor link qualification generators [ interface INTF ]'
   data = { 'monitor' : monitorMatcherForShow,
            'link' : linkNode,
            'qualification' : qualificationNode,
            'generators' : generatorsNode,
            'interface' : 'Interface',
            'INTF' : gIntfMatcher,
         }

   cliModel = LinkQualificationGeneratorAllIntfsModel

   @staticmethod
   def handler( mode, args ):
      if featureStatus.featureEnabled == FeatureEnabledEnum.rfc2544:
         mode.addWarning( rfc2544EnabledWarning )
      elif featureStatus.featureEnabled != FeatureEnabledEnum.linkQualification:
         mode.addWarning( linkQualDisabledWarning )

      allIntfsModel = LinkQualificationGeneratorAllIntfsModel()
      allIntfsModel.enabled = \
             ( featureStatus.featureEnabled == FeatureEnabledEnum.linkQualification )

      intf = args.get( 'INTF' )
      if intf and not intf.name in intfStatus.hwIntfList:
         mode.addError( intf.name + ' is not configured as a generator' )
         return None

      intfs = [ intf.name ] if intf else list( intfStatus.hwIntfList )
      for intf in intfs:
         hwIntfStatus = intfStatus.hwIntfList.get( IntfId( intf ) )
         allIntfsModel.interfaces[ intf ] = \
            populateLinkQualificationGeneratorIntf( hwIntfStatus )
      return allIntfsModel


BasicCli.addShowCommandClass( ShowLinkQualificationGeneratorsCmd )

# --------------------------------------------------------------------------------
# Register the CLI extension for LinkQual Generator
# -------------------------------------------------------------------------------
def linkQualificationGeneratorExplanation( intfName, mode ):
   if ( featureStatus.featureEnabled == featureEnabled.linkQualification and
        intfName in reconciledIntfConfig.intfList ):
      return ( 'generator', 'generator', 'generator configuration' )
   return ( None, None, None )

IntfCli.linkQualificationExplanationHook.addExtension(
      linkQualificationGeneratorExplanation )

#-------------------------------------------------------------------
# Plugin Func
#-------------------------------------------------------------------
def Plugin( entityManager ):
   global satProfilesConfig
   global intfCliConfig
   global reconciledIntfConfig
   global intfStatus
   global satHwCapabilities
   global rfc2544ProfileConfigDir
   global featureConfig
   global featureStatus
   global rfc2544ExecConfigDir

   satProfilesConfig = ConfigMount.mount( entityManager, 'sat/profile/cliConfig',
         'Sat::SatProfilesConfig', 'w' )
   intfCliConfig = ConfigMount.mount( entityManager, 'sat/intf/cliConfig',
         'Sat::IntfConfig', 'w' )
   reconciledIntfConfig = LazyMount.mount( entityManager, 'sat/intf/config',
         'Sat::IntfConfig', 'w' )
   intfStatus = LazyMount.mount( entityManager, 'sat/intf/status',
         'Sat::IntfStatus', 'r' )
   satHwCapabilities = LazyMount.mount( entityManager, 'sat/hardware/capabilities',
         'Sat::SatHwCaps', 'r' )
   rfc2544ProfileConfigDir = ConfigMount.mount(
         entityManager, 'rfc2544Initiator/profileConfigDir',
         'Rfc2544Initiator::ProfileConfigDir', 'w' )
   featureConfig = ConfigMount.mount(
         entityManager, 'generator/featureConfig',
         'FlowGenerator::FeatureConfig', 'w' )
   featureStatus = LazyMount.mount( entityManager, 'generator/featureStatus',
         'FlowGenerator::FeatureStatus', 'r' )
   rfc2544ExecConfigDir = ConfigMount.mount(
         entityManager, 'rfc2544Initiator/execConfig',
         'Rfc2544Initiator::ExecConfigDir', 'w' )
