# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliLib
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
from CliPlugin.TcpMssCli import CfgTcpMssCeilingCmdBase
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from ConfigConsistencyChecker import UndefinedReferenceChecker
import CliToken.Ip
import CliToken.Ipv6
import CliToken.System
import ConfigMount
import Plugins

cliCpConfig = None
paramConfig = None

tcpMssConfig4 = None
tcpMssConfig6 = None

#--------------------------------------------------------------------------------
# system control-plane
#--------------------------------------------------------------------------------
class SystemControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'system control-plane'
   noOrDefaultSyntax = syntax
   data = {
      'system': CliToken.System.systemMatcherForConfig,
      'control-plane': 'Configure control-plane features',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.gotoChildMode( mode.childMode( AclCli.CpConfigMode ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for config in cliCpConfig.cpConfig.values():
         config.globalCpAclDefault = ''
         config.globalCpAcl.clear()
      AclCli.CpConfigMode.clearHook.notifyExtensions( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( SystemControlPlaneCmd )

#--------------------------------------------------------------------------------
# control-plane (deprecated)
#--------------------------------------------------------------------------------
class ControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'control-plane'
   data = {
      'control-plane': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'control-plane',
            helpdesc='Configure control-plane features' ),
         deprecatedByCmd='system control-plane' )
   }

   handler = SystemControlPlaneCmd.handler

BasicCliModes.GlobalConfigMode.addCommandClass( ControlPlaneCmd )

def configureControlPlaneDefaultAcl( mode, args, aclType ):
   cliCpConfig.cpConfig[ aclType ].globalCpAclDefault = args.get( 'ACL_NAME', '' )
   AclCliLib.tryWaitForWarmup( mode )

def configureControlPlaneAcl( mode, args, aclType ):
   aclName = args[ 'ACL_NAME' ]
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   if aclName == paramConfig.cpAclNameDefault:
      del cliCpConfig.cpConfig[ aclType ].globalCpAcl[ vrfName ]
   else:
      cliCpConfig.cpConfig[ aclType ].globalCpAcl[ vrfName ] = aclName
   AclCliLib.tryWaitForWarmup( mode )

def unconfigureControlPlaneAcl( mode, args, aclType ):
   aclName = args.get( 'ACL_NAME' )
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   # Get the effective cpAcl name
   aclConfig = cliCpConfig.cpConfig[ aclType ].globalCpAcl
   currentAclName = aclConfig.get( vrfName, paramConfig.cpAclNameDefault )

   if aclName is None or aclName == currentAclName:
      del aclConfig[ vrfName ]
      AclCliLib.tryWaitForWarmup( mode )
   else:
      AclCli.printNotConfiguredError( mode, aclType, aclName,
                                      # pylint: disable-next=consider-using-f-string
                                      'control-plane(%s VRF)' % vrfName )

class CpIpChangeDefaultAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ingress default ACL_NAME'
   noOrDefaultSyntax = 'ip access-group ingress default ...'
   data = {
            'ip': CliToken.Ip.ipMatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ingress': AclCli.ingressKwMatcher,
            'default': AclCli.defaultKwMatcher,
            'ACL_NAME': AclCli.ipAclNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      configureControlPlaneDefaultAcl( mode, args, 'ip' )

   noOrDefaultHandler = handler

AclCli.CpConfigMode.addCommandClass( CpIpChangeDefaultAclCmd )

class CpIp6ChangeDefaultAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ingress default ACL_NAME'
   noOrDefaultSyntax = 'ipv6 access-group ingress default ...'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ingress': AclCli.ingressKwMatcher,
            'default': AclCli.defaultKwMatcher,
            'ACL_NAME': AclCli.ip6AclNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      configureControlPlaneDefaultAcl( mode, args, 'ipv6' )

   noOrDefaultHandler = handler

AclCli.CpConfigMode.addCommandClass( CpIp6ChangeDefaultAclCmd )

class CpIpAclCmd( CliCommand.CliCommandClass ):
   syntax =  'ip access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ip': CliToken.Ip.ipMatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ACL_NAME': AclCli.ipAclNameMatcher,
            'VRF': AclCli.vrfKwAndNameExprFactory,
            'in': AclCli.inKwMatcher
          }

   @staticmethod
   def handler( mode, args ):
      configureControlPlaneAcl( mode, args, 'ip' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unconfigureControlPlaneAcl( mode, args, 'ip' )
 
AclCli.CpConfigMode.addCommandClass( CpIpAclCmd )

class CpIp6AclCmd( CliCommand.CliCommandClass ):
   syntax =  'ipv6 access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigCp,
            'access-group': AclCli.accessGroupKwMatcher,
            'ACL_NAME': AclCli.ip6AclNameMatcher,
            'VRF': AclCli.vrfKwAndNameExprFactory,
            'in': AclCli.inKwMatcher
          }

   @staticmethod
   def handler( mode, args ):
      configureControlPlaneAcl( mode, args, 'ipv6' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      unconfigureControlPlaneAcl( mode, args, 'ipv6' )
 
AclCli.CpConfigMode.addCommandClass( CpIp6AclCmd )

class CfgTcpMssCeilingCmd( CfgTcpMssCeilingCmdBase ):
   # The control-plane version of this command does not support directionality
   # (ingress / egress).
   syntax = '''tcp mss ceiling { ( ipv4 MSS4 ) | ( ipv6 MSS6 ) }'''
   data = {
      'tcp': 'TCP',
   }
   data.update( CfgTcpMssCeilingCmdBase._baseData )

   @staticmethod
   def _getConfig( mode ):
      cpConfig = cliCpConfig.cpConfig
      ipv4Config = cpConfig[ 'ip' ].tcpMssConfig
      ipv6Config = cpConfig[ 'ipv6' ].tcpMssConfig
      assert ipv4Config and ipv6Config
      return ipv4Config, ipv6Config

   @staticmethod
   def handler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonHandler( mode, args, ipv4Config, ipv6Config )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4Config, ipv6Config = CfgTcpMssCeilingCmd._getConfig( mode )
      CfgTcpMssCeilingCmdBase._commonNoOrDefaultHandler( mode, ipv4Config,
                                                         ipv6Config )

AclCli.CpConfigMode.addCommandClass( CfgTcpMssCeilingCmd )

#-------------------------------------------------------------------------------
# Control plane references to access list configurations gatherer
#-------------------------------------------------------------------------------
aclTypes = {"IP access list": 'ip', "IPv6 access list": 'ipv6' }

class CpAclReferenceGatherer:
   """This class gathers access list references used by the control plane."""
   @staticmethod
   def gather( feature ):
      aclType = aclTypes[ feature ]
      references = set( cliCpConfig.cpConfig[ aclType ].globalCpAcl.values() )
      return references

UndefinedReferenceChecker.addReferenceGatherer( aclTypes, CpAclReferenceGatherer )

@Plugins.plugin()
def Plugin( entityManager ):
   global cliCpConfig
   global paramConfig
   cliCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                    'Acl::Input::CpConfig', 'w' )
   paramConfig = ConfigMount.mount( entityManager, 'acl/paramconfig',
                             'Acl::ParamConfig', 'w' )
