# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import TacSigint

from CliPlugin import TechSupportCli
from CliPlugin.VrfCli import getVrfNames
from CliMatcher import ( KeywordMatcher, DynamicNameMatcher )
import BasicCli
import ShowCommand

import enum
import re
import traceback
import Toggles.GmpToggleLib

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )

#------------------------------------------------------------------------------------
# The "show tech-support multicast" command
#------------------------------------------------------------------------------------

showTechMulticastCallbacks = []
showTechMulticastVrfCallbacks = []
showTechSummaryCallbacks = []

class ShowTechMulticastType( enum.Enum ):
   noVrf = 0
   vrf = 1
   summary = 2

def _getCmdCallback( cmds, guard ):
   def _listFunc():
      return cmds

   def _guardedListFunc( ):
      try:
         if guard() is None:
            return cmds
         return []
      except Exception as e: # pylint: disable=broad-except
         err = []
         callbackName = f'plugin for {guard.__module__}.{guard.__name__}'
         err.append( f'! {callbackName} raised an exception: {e!r}' )
         print( err )
         return []

   # there are no commands, don't return a function
   if not cmds:
      return None

   # if we have a cmds guard then return the function that invokes the guard.
   # if it is unconditional return function that always returns the cmds list
   retFunc = _listFunc
   if guard is not None:
      assert callable( guard )
      retFunc = _guardedListFunc

   return retFunc

def registerShowTechMulticast( cmds, showTechMcastType=ShowTechMulticastType.noVrf,
                               guard=None ):
   """
   API is used to register show tech commands
   """
   if not Toggles.GmpToggleLib.toggleMulticastShowTechEnabled():
      return

   callBack = _getCmdCallback( cmds, guard )
   assert callBack
   fileName, _, _functionName, _ = traceback.extract_stack( limit=2 )[ 0 ]
   m = re.search( 'CliPlugin/(.*).py', fileName )
   if m:
      module = "CliPlugin." + m.group( 1 )
   else:
      module = fileName
   callBack.__module__ = module
   callBack.__name__ = _functionName

   if showTechMcastType == ShowTechMulticastType.summary:
      showTechSummaryCallbacks.append( callBack )
   elif showTechMcastType == ShowTechMulticastType.vrf:
      showTechMulticastVrfCallbacks.append( callBack )
   else:
      showTechMulticastCallbacks.append( callBack )

#------------------------------------------------------------------------------------
# show tech-support multicast [vrfName]
#  - display all registered pim show tech commands under each vrf
#  - when vrf is not specified, display membership for all vrfs
#------------------------------------------------------------------------------------
vrfMatcher = KeywordMatcher( 'vrf', helpdesc='Display VRF state' )
vrfNameMatcher = DynamicNameMatcher( getVrfNames, 'VRF name' )

class ShowTechMulticastSupport( ShowCommand.ShowCliCommandClass ):
   syntax = "show tech-support multicast [ vrf VRF_NAME ]"
   data = {
            'tech-support': TechSupportCli.techSupportKwMatcher,
            'multicast': 'Show detailed state for Multicast',
            'vrf': vrfMatcher,
            'VRF_NAME': vrfNameMatcher,
          }
   handler = "GmpShowTechMcastHandler.handlerShowTechMulticastSupport"
   cliModel = "GmpShowTechMcastHandler.ShowTechMulticastModel"
   hidden = True

if Toggles.GmpToggleLib.toggleMulticastShowTechEnabled():
   BasicCli.addShowCommandClass( ShowTechMulticastSupport )
