# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliMode.Gnmi import MgmtGnmiMode, GnmiTransportMode, GrpctunnelTransportMode
import CliMatcher
from CliPlugin import OpenConfigCliLib
import ConfigMount
import DscpCliLib
from IpLibConsts import DEFAULT_VRF
import LazyMount
import Tac
from CliCommon import InvalidInputError
import AuthnUserPriorityCli

matcherGnmi = CliMatcher.KeywordMatcher( 'gnmi', helpdesc='Show gNMI information' )
gnmiSocket = 'unix:///var/run/gnmiServer.sock'
sslConfig = None
gnmiConfig = None
octaConfig = None
gnsiConfig = None

# ------------------------------------------------------
# GNMI config commands
# ------------------------------------------------------

class MgmtGnmiConfigMode( MgmtGnmiMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api gnmi'."""

   name = "GNMI configuration"

   def __init__( self, parent, session ):
      self.config_ = gnmiConfig
      MgmtGnmiMode.__init__( self, "api-gnmi" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtGnmiConfigMode( mode, args ):
   childMode = mode.childMode( MgmtGnmiConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtGnmiConfigMode( mode, args ):
   """Resets OpenConfig agent configuration to default."""
   gnmiConfig.enabled = False
   gnmiConfig.setPersistence = False
   octaConfig.enabled = False
   for name in gnmiConfig.endpoints:
      noGnmiTransportConfigMode( mode, { 'TRANSPORT_NAME': name } )
   for name in gnmiConfig.grpctunnelClients:
      noGrpctunnelTransportConfigMode( mode, { 'TRANSPORT': name } )

# ------------------------------------------------------
# switch(config)# management api openconfig
#
# made obsolete by "management api gnmi" but still
# available as hidden command for ease of transitions
# ------------------------------------------------------
def gotoMgmtOpenConfigConfigMode( mode, args ):
   mode.addWarning(
      'This command is deprecated and will be saved as \'management api gnmi\'.' )
   gotoMgmtGnmiConfigMode( mode, args )

class GnmiTransportConfigMode( GnmiTransportMode,
                                     BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc <name>'."""

   name = 'Transport for GNMI'

   def __init__( self, parent, session, name ):
      self.config_ = gnmiConfig
      self.name = name

      GnmiTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# ------------------------------------------------------
# switch(config-mgmt-api-gnmi)#transport grpc <name>
# ------------------------------------------------------
def gotoGnmiTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   if name not in gnmiConfig.endpoints:
      endpoint = gnmiConfig.newEndpoints( name )
      endpoint.transport = 'grpc'
      endpoint.vrfName = DEFAULT_VRF
      # since 'initially' attributes don't get attrlogged, be explicit for now
      endpoint.port = endpoint.portDefault
      endpoint.enabled = True
      endpoint.useNotifSendTime = False
      endpoint.certUsernameAuthn = False
      endpoint.listenAddrs.clear()
      endpoint.gnsiEnabled = name in gnsiConfig.endpoints

      for elm in AuthnUserPriorityCli.defaultAuthnUserPriority:
         endpoint.authnUsernamePriority.push( elm )

      serviceStatusName = "gnmi-" + name
      OpenConfigCliLib.createSslServiceStatusEntity( serviceStatusName )

      OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

   childMode = mode.childMode( GnmiTransportConfigMode, name=name )
   mode.session_.gotoChildMode( childMode )

def noGnmiTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT_NAME' ]
   endpoint = gnmiConfig.endpoints.get( name )
   if endpoint is not None:
      if endpoint.enabled:
         endpoint.enabled = False
      endpoint.port = endpoint.portDefault

      serviceStatusName = "gnmi-" + name
      OpenConfigCliLib.deleteSslServiceStatusEntity( serviceStatusName )

      try:
         del gnmiConfig.endpoints[ name ]
      except KeyError:
         pass

   OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

def shutdown( mode, args ):
   endpoint = mode.config_.endpoints[ mode.name ]
   if endpoint.enabled:
      endpoint.enabled = False
      OpenConfigCliLib.updateLevelEnabledFlag( mode.config_ )

def noShutdown( mode, args ):
   # This stanza can only be enabled if there is no other
   # stanza with same 'transport' type already enabled
   endpoint = mode.config_.endpoints.get( mode.name )
   endpoint.enabled = True
   mode.config_.enabled = True

def setSslProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   mode.config_.endpoints[ mode.name ].sslProfile = profileName

def noSslProfile( mode, args ):
   mode.config_.endpoints[ mode.name ].sslProfile = ''

profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: sslConfig.profileConfig,
      'Profile name')

def setVrfName( mode, args ):
   endpoint = mode.config_.endpoints[ mode.name ]
   vrfName = args.get( 'VRFNAME', DEFAULT_VRF )
   if vrfName == endpoint.vrfName:
      return
   if endpoint.serviceAcl:
      if OpenConfigCliLib.existsOtherTransportSameVrfWithServiceAcl( mode, vrfName ):
         return
      OpenConfigCliLib.noServiceAcl( mode, 'gnmi', endpoint )
   endpoint.vrfName = vrfName
   if endpoint.serviceAcl:
      OpenConfigCliLib.setServiceAcl( mode, 'gnmi', endpoint )

def setPort( mode, args ):
   endpoint = mode.config_.endpoints[ mode.name ]
   port = args.get( 'PORT', endpoint.portDefault )
   if port == endpoint.port:
      return
   endpoint.port = port
   if endpoint.serviceAcl:
      OpenConfigCliLib.setServiceAcl( mode, 'gnmi', endpoint )

def setConnectionLimit( mode, args ):
   endpoint = mode.config_.endpoints[ mode.name ]
   connectionLimit = args.get( 'CONNECTION_LIMIT', endpoint.connectionLimitDefault )
   if connectionLimit == endpoint.connectionLimit:
      return
   endpoint.connectionLimit = connectionLimit

# ---------------------------------------------------------------------
# switch(config-mgmt-api-gnmi-transport-<name>)# certificate username auhentication
# ---------------------------------------------------------------------
def setCertificateUsernameAuthentication( mode, args ):
   authnUsernamePriority = mode.config_.endpoints[ mode.name ].authnUsernamePriority
   authnUsernamePriority.clear()
   for elm in AuthnUserPriorityCli.defaultAuthnUserPriorityWithCUA:
      authnUsernamePriority.push( elm )

def noCertificateUsernameAuthentication( mode, args ):
   return

# ---------------------------------------------------------------------
# switch(config-mgmt-api-gnmi-transport-<name>)# authorization requests
# ---------------------------------------------------------------------
def setAuthorization( mode, args ):
   mode.config_.endpoints[ mode.name ].authorization = True

def noAuthorization( mode, args ):
   mode.config_.endpoints[ mode.name ].authorization = False

# ---------------------------------------------------------------------
# switch(config-mgmt-api-gnmi-transport-<name>)# qos dscp <dscpValue>
# ---------------------------------------------------------------------
def setDscp( mode, args ):
   mode.config_.endpoints[ mode.name ].qosDscp = args[ 'DSCP' ]

def noDscp( mode, args ):
   mode.config_.endpoints[ mode.name ].qosDscp = 0

DscpCliLib.addQosDscpCommandClass( GnmiTransportConfigMode, setDscp, noDscp )

# ---------------------------------------------------------------------
# [ no | default ] transport grpc-tunnel TRANSPORT
# ---------------------------------------------------------------------
class GrpctunnelTransportConfigMode( GrpctunnelTransportMode,
                                     BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc-tunnel TRANSPORT'."""

   name = 'gRPC-tunnel dial-out transport for gNMI'

   def __init__( self, parent, session, name ):
      self.config_ = gnmiConfig
      self.name = name

      GrpctunnelTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoGrpctunnelTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT' ]
   if name not in gnmiConfig.grpctunnelClients:
      gnmiConfig.newGrpctunnelClients( name )
   childMode = mode.childMode( GrpctunnelTransportConfigMode, name=name )
   OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )
   mode.session_.gotoChildMode( childMode )

def noGrpctunnelTransportConfigMode( mode, args ):
   name = args[ 'TRANSPORT' ]
   del gnmiConfig.grpctunnelClients[ name ]
   OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )
   
# -------------------------------------------------------------------------------
# The "show gnmi get { PATH }" command
# -------------------------------------------------------------------------------
def showGnmiGet( mode, args ):
   shellCmd = [ 'gnmi', '-addr', gnmiSocket, 'get' ]
   paths = args[ 'PATH' ]

   for path in paths:
      if "/" not in path and path != '...':
         raise InvalidInputError( f"Invalid path: {path}" )

   err = Tac.run( shellCmd + paths,
                  asRoot=True,
                  ignoreReturnCode=True,
                  stderr=Tac.CAPTURE )
   if err:
      for line in err.strip().split( '\n' ):
         mode.addError( line )

def Plugin( entityManager ):
   global gnmiConfig, sslConfig, octaConfig, gnsiConfig

   gnmiConfig = ConfigMount.mount( entityManager, "mgmt/gnmi/config",
                                   "Gnmi::Config", "w" )
   gnsiConfig = ConfigMount.mount( entityManager, "mgmt/gnsi/config",
                                   "Gnsi::Config", "w" )
   sslConfig = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config",
                                "r" )
   octaConfig = ConfigMount.mount( entityManager, "mgmt/octa/config",
                                   "Octa::Config", "w" )
