# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AuthnUserPriorityCli
import BasicCli
import BasicCliModes
from CliMode.GribiMode import MgmtGribiMode, GribiTransportMode
import CliParserCommon
from CliPlugin import ConfigMgmtMode
from CliPlugin import VrfCli
from CliPlugin import AclCli
from CliToken.Clear import clearKwNode
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
from IpLibConsts import DEFAULT_VRF
import AclCliLib
from socket import IPPROTO_TCP
import Toggles.gribiToggleLib

gribiConfig = None
sslConfig = None
gribiStatus = None
sessionParams = None
aclConfig = None
aclCpConfig = None
aclCheckpoint = None
aclStatus = None

matcherAccessList = AclCli.accessListKwMatcherForServiceAcl
matcherCert = CliMatcher.KeywordMatcher( 'certificate',
      helpdesc='Configure certificate options' )
matcherUsername = CliMatcher.KeywordMatcher( 'username',
      helpdesc='Configure certificate username options' )
matcherAuthentication = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Configure certificate username authentication' )
authenticationMatcher = matcherAuthentication
matcherAccounting = CliMatcher.KeywordMatcher( 'accounting',
      helpdesc='Configure accounting' )
matcherRequests = CliMatcher.KeywordMatcher( 'requests',
      helpdesc='Configure RPC requests accounting' )

if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
   authenticationMatcher = CliCommand.Node( matcher=matcherAuthentication,
         deprecatedByCmd='authentication username priority' )

# ------------------------------------------------------
# gRIBI config commands
# ------------------------------------------------------
class MgmtGribiConfigMode( MgmtGribiMode, BasicCli.ConfigModeBase ):
   """CLI configuration mode 'management api gribi'"""

   name = "gRIBI configuration"

   def __init__( self, parent, session ):
      self.config_ = gribiConfig
      MgmtGribiMode.__init__( self, "api-gribi" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMgmtGribiConfigMode( mode, args ):
   childMode = mode.childMode( MgmtGribiConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMgmtGribiConfigMode( mode, args ):
   """Resets gRIBI agent configuration to default."""
   gribiConfig.enabled = False
   for name in gribiConfig.endpoints:
      noGribiTransportConfigMode( mode, { 'TRANSPORT_NAME': name } )

# ------------------------------------------------------
# [ no | default ] management api gribi
# ------------------------------------------------------
class GribiMgmtApiConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'management api gribi'
   noOrDefaultSyntax = syntax
   data = {
         'management': 'Configure management services',
         'api': 'Configure management APIs for the switch',
         'gribi': 'Configure gRIBI',
   }

   handler = gotoMgmtGribiConfigMode
   noOrDefaultHandler = noMgmtGribiConfigMode

BasicCliModes.GlobalConfigMode.addCommandClass( GribiMgmtApiConfigCmd )

class GribiTransportConfigMode( GribiTransportMode, BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc <name>'."""

   name = 'Transport for gRIBI'

   def __init__( self, parent, session, name ):
      self.config_ = gribiConfig
      self.name = name

      GribiTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoGribiTransportConfigMode( mode, args ):
   transportName = args.get( 'TRANSPORT_NAME' )
   if transportName not in gribiConfig.endpoints:
      endpointName = next( iter( gribiConfig.endpoints ), None )
      if endpointName is None:
         endpoint = gribiConfig.newEndpoints( transportName, )
         endpoint.transport = 'grpc'
         endpoint.port = endpoint.portDefault
         endpoint.vrfName = DEFAULT_VRF
         endpoint.enabled = True
         endpoint.accountingRequests = False
         endpoint.dscp = 0
         endpoint.serviceAcl = ''
         endpoint.serviceAclV6 = ''
         gribiConfig.enabled = True
         for elm in AuthnUserPriorityCli.defaultAuthnUserPriority:
            endpoint.authnUsernamePriority.push( elm )
      else:
         mode.addErrorAndStop( "transport '%s' of type 'grpc' already enabled; "
                               "can not enable another" % endpointName )
   childMode = mode.childMode( GribiTransportConfigMode, name=transportName )
   mode.session_.gotoChildMode( childMode )

def noGribiTransportConfigMode( mode, args ):
   name = args.get( 'TRANSPORT_NAME' )
   endpoint = gribiConfig.endpoints.get( name )
   if endpoint is not None:
      endpoint.enabled = False
      endpoint.port = endpoint.portDefault
      gribiConfig.enabled = False
      del gribiConfig.endpoints[ name ]

# ------------------------------------------------------
# [ no | default ] transport grpc <name>
# ------------------------------------------------------
class GribiTransportConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'transport grpc TRANSPORT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'transport': 'Configure a transport',
         'grpc': 'Configure grpc transport for gRIBI',
         'TRANSPORT_NAME': CliMatcher.DynamicNameMatcher(
            lambda mode: gribiConfig.endpoints,
            'Transport name',
            helpname='WORD',
            extraEmptyTokenCompletionFn=lambda mode, context:
               [ CliParserCommon.Completion( endpoint, endpoint )
               for endpoint in gribiConfig.endpoints ] ),
   }

   handler = gotoGribiTransportConfigMode
   noOrDefaultHandler = noGribiTransportConfigMode

MgmtGribiConfigMode.addCommandClass( GribiTransportConfigCmd )

def shutdown( mode, args ):
   mode.config_.endpoints[ mode.name ].enabled = False
   mode.config_.enabled = False

def noShutdown( mode, args ):
   mode.config_.endpoints[ mode.name ].enabled = True
   mode.config_.enabled = True

# ------------------------------------------------------
# [ no | default ] shutdown
# ------------------------------------------------------
class GribiShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
         'shutdown': 'Disable protocol',
   }

   handler = shutdown
   noOrDefaultHandler = noShutdown

GribiTransportConfigMode.addCommandClass( GribiShutdownCmd )

def setPort( mode, args ):
   endpoint = mode.config_.endpoints.get( mode.name )
   port = endpoint.port
   endpoint.port = args.get( 'PORT', endpoint.portDefault )
   if port == endpoint.port:
      return
   if endpoint.serviceAcl:
      setServiceAclV4( mode, endpoint )
   if endpoint.serviceAclV6:
      setServiceAclV6( mode, endpoint )

# ------------------------------------------------------
# [ no | default ] port <port number>
# ------------------------------------------------------
class GribiPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': 'The port number to listen on',
      'PORT': CliMatcher.IntegerMatcher( 1, 65535,
                                         helpdesc='Port number to use' ),
   }
   handler = setPort
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiPortCmd )

def setSslProfile( mode, args ):
   profileName = args.get( 'PROFILENAME', '' )
   mode.config_.endpoints[ mode.name ].sslProfile = profileName

# ------------------------------------------------------
# [ no | default ] ssl profile <profile name>
# ------------------------------------------------------
class GribiSslProfileNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': 'Configure SSL related options',
      'profile': 'Configure SSL profile',
      'PROFILENAME': CliMatcher.DynamicNameMatcher(
                        lambda mode: sslConfig.profileConfig,
                        'Profile name' ),
   }
   handler = setSslProfile
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiSslProfileNameCmd )

def setCertificateUsernameAuthentication( mode, args ):
   if not Toggles.gribiToggleLib.toggleGribiAUPEnabled():
      mode.config_.endpoints[ mode.name ].certUsernameAuthn = True
      return
   authnUsernamePriority = mode.config_.endpoints[ mode.name ].authnUsernamePriority
   authnUsernamePriority.clear()
   for elm in AuthnUserPriorityCli.defaultAuthnUserPriorityWithCUA:
      authnUsernamePriority.push( elm )

def noCertificateUsernameAuthentication( mode, args ):
   if not Toggles.gribiToggleLib.toggleGribiAUPEnabled():
      mode.config_.endpoints[ mode.name ].certUsernameAuthn = False
      return
   authnUsernamePriority = mode.config_.endpoints[ mode.name ].authnUsernamePriority
   authnUsernamePriority.clear()
   for elm in AuthnUserPriorityCli.defaultAuthnUserPriority:
      authnUsernamePriority.push( elm )

# --------------------------------------------------------------------------------
# [ no | default ] certificate username authentication
# --------------------------------------------------------------------------------
class CertUsernameAuthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate username authentication'
   noOrDefaultSyntax = syntax
   data = {
      'certificate': matcherCert,
      'username': matcherUsername,
      'authentication': authenticationMatcher,
   }
   handler = setCertificateUsernameAuthentication
   noOrDefaultHandler = noCertificateUsernameAuthentication

GribiTransportConfigMode.addCommandClass( CertUsernameAuthenticationCmd )

if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
   GribiTransportConfigMode.addCommandClass(
         AuthnUserPriorityCli.authnUsernamePriorityNoOidCmd )

def setTransportVrfName( mode, args ):
   endpoint = mode.config_.endpoints.get( mode.name )
   vrfName = endpoint.vrfName
   endpoint.vrfName = args.get( 'VRFNAME', DEFAULT_VRF )
   if vrfName == endpoint.vrfName:
      return
   if endpoint.serviceAcl:
      setServiceAclV4( mode, endpoint )
      noServiceAcl( mode, 'ip', vrfName )
   if endpoint.serviceAclV6:
      setServiceAclV6( mode, endpoint )
      noServiceAcl( mode, 'ipv6', vrfName )

# ------------------------------------------------------
# [ no | default ] vrf <vrfName>
# ------------------------------------------------------
class GribiTransportVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRFNAME': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = setTransportVrfName
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiTransportVrfCmd )

def setAccountingRequests( mode, args ):
   mode.config_.endpoints[ mode.name ].accountingRequests = True

def noAccountingRequests( mode, args ):
   mode.config_.endpoints[ mode.name ].accountingRequests = False

# ------------------------------------------------------
# [ no | default ] accounting requests
# ------------------------------------------------------
class GribiTransportAccountingRequestsCmd( CliCommand.CliCommandClass ):
   syntax = 'accounting requests'
   noOrDefaultSyntax = syntax
   data = {
      'accounting': matcherAccounting,
      'requests': matcherRequests
   }
   handler = setAccountingRequests
   noOrDefaultHandler = noAccountingRequests
if Toggles.gribiToggleLib.toggleGribiAccountingRequestsEnabled():
   GribiTransportConfigMode.addCommandClass( GribiTransportAccountingRequestsCmd )

dscpRangeMatcher = CliMatcher.IntegerMatcher(
      0,
      63,
      helpdesc='DSCP value'
)

def setTransportDscp( mode, args ):
   mode.config_.endpoints[ mode.name ].dscp = args.get( 'DSCP', 0 )

# ------------------------------------------------------
# [ no | default ] qos dscp <dscp>
# ------------------------------------------------------
class GribiTransportDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos dscp DSCP'
   noOrDefaultSyntax = 'qos dscp ...'
   data = {
      'qos': CliMatcher.KeywordMatcher( 'qos',
                                        'Set QoS parameters for gRPC traffic' ),
      'dscp': CliMatcher.KeywordMatcher( 'dscp', 'Set DSCP value for gRPC traffic' ),
      'DSCP': dscpRangeMatcher
   }
   handler = setTransportDscp
   noOrDefaultHandler = handler

GribiTransportConfigMode.addCommandClass( GribiTransportDscpCmd )

class GribiAccessGroupExpr( CliCommand.CliExpression ):
   expression = '( ( ip access-group V4ACL ) | ( ipv6 access-group V6ACL ) )'

   data = {
      'ip': 'IP config commands',
      'ipv6': 'IPv6 config commands',
      'access-group': AclCli.accessGroupKwMatcher,
      'V4ACL': AclCli.ipAclNameMatcher,
      'V6ACL': AclCli.ip6AclNameMatcher,
   }

class GribiAccessGroupNoOrDefExpr( CliCommand.CliExpression ):
   expression = '( ( ip access-group [ V4ACL ] ) '\
         '| ( ipv6 access-group [ V6ACL ] ) )'

   data = {
      'ip': 'IP config commands',
      'ipv6': 'IPv6 config commands',
      'access-group': AclCli.accessGroupKwMatcher,
      'V4ACL': AclCli.ipAclNameMatcher,
      'V6ACL': AclCli.ip6AclNameMatcher,
   }

# --------------------------------------------------------------------------------
# [ no | default ] (ip |ipv6) access-group ACLNAME [ in ]
# --------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ACL_EXPR [ in ]'
   noOrDefaultSyntax = 'ACL_NOORDEF_EXPR [ in ]'
   data = {
      'ACL_EXPR': GribiAccessGroupExpr,
      'ACL_NOORDEF_EXPR': GribiAccessGroupNoOrDefExpr,
      'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      endpoint = mode.config_.endpoints.get( mode.name )
      if 'ipv6' in args:
         endpoint.serviceAclV6 = args[ 'V6ACL' ]
         setServiceAclV6( mode, endpoint )
      else:
         endpoint.serviceAcl = args[ 'V4ACL' ]
         setServiceAclV4( mode, endpoint )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      endpoint = mode.config_.endpoints.get( mode.name )
      if 'ipv6' in args:
         endpoint.serviceAclV6 = ''
         noServiceAcl( mode, 'ipv6', endpoint.vrfName )
      else:
         endpoint.serviceAcl = ''
         noServiceAcl( mode, 'ip', endpoint.vrfName )

GribiTransportConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

def noServiceAcl( mode, aclType, vrfName ):
   AclCliLib.noServiceAcl( mode, 'gribi', aclConfig, aclCpConfig,
         None, aclType, vrfName )

def setServiceAclV4( mode, endpoint ):
   AclCliLib.setServiceAcl( mode, 'gribi', IPPROTO_TCP, aclConfig, aclCpConfig,
         endpoint.serviceAcl, 'ip', endpoint.vrfName, port=[ endpoint.port ] )

def setServiceAclV6( mode, endpoint ):
   AclCliLib.setServiceAcl( mode, 'gribi', IPPROTO_TCP, aclConfig, aclCpConfig,
         endpoint.serviceAclV6, 'ipv6', endpoint.vrfName, port=[ endpoint.port ] )

def clearGribiAclCounters( mode, args ):
   for aclType in [ 'ip', 'ipv6' ]:
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

# --------------------------------------------------------------------------------
# clear management api gribi counters access-list
# --------------------------------------------------------------------------------
class ClearManagementApiGribiCountersAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'clear management api gribi counters access-list'
   data = {
      'clear': clearKwNode,
      'management': ConfigMgmtMode.managementClearKwMatcher,
      'api': ConfigMgmtMode.apiKwMatcher,
      'gribi': CliCommand.guardedKeyword( 'gribi',
         helpdesc='Clear gRIBI statistics', guard=AclCli.serviceAclGuard ),
      'counters': AclCli.countersKwMatcher,
      'access-list': matcherAccessList,
   }
   handler = clearGribiAclCounters

BasicCliModes.EnableMode.addCommandClass(
   ClearManagementApiGribiCountersAccessListCmd )

def Plugin( entityManager ):
   global gribiConfig, gribiStatus, sslConfig
   global sessionParams
   global aclConfig, aclCpConfig
   global aclCheckpoint, aclStatus

   gribiConfig = ConfigMount.mount( entityManager, "mgmt/gribi/config",
                                    "Gribi::Config", "w" )
   gribiStatus = LazyMount.mount( entityManager, "mgmt/gribi/status",
                                    "Gribi::Status", "r" )
   sslConfig = LazyMount.mount( entityManager, "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config", "r" )
   sessionParams = LazyMount.mount( entityManager, "mgmt/gribi/sessionParams",
                                    "Gribi::SessionParams", "r" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
