# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time
from Ark import timestampToStr, utcTimeRelativeToNowStr
from ArnetModel import (
   IpGenericAddress,
   IpGenericPrefix,
   MacAddress,
)
from CliModel import ( Bool,
                       Dict,
                       Enum,
                       Float,
                       GeneratorDict,
                       Int,
                       List,
                       Model,
                       Str,
                       Submodel )
from IntfModels import Interface
from CliPlugin.SrTePolicyLibModel import SrTeSegmentListVia
from CliPlugin.AclCliModel import AllAclList
from operator import attrgetter
from TableOutput import (
   Format,
   createTable,
)
import Ethernet
import Toggles.gribiToggleLib

class GribiIpInIpEncap ( Model ):
   dstIp = IpGenericAddress( help="Destination address" )
   srcIp = IpGenericAddress( help="Source address" )

class GribiNhgNhAftType( Model ):
   nhId = Int( help='Nexthop ID' )
   weight = Int( help='Weight of the nexthop' )
   ipAddress = IpGenericAddress( optional=True, help="Next hop address" )
   interface = Interface( optional=True, help="Next hop interface" )
   macAddress = MacAddress( optional=True, help="Next hop MAC address" )
   decapActionType = Enum( values=[ 'ipip' ],
                           help="Decapsulation type", optional=True )
   fallbackVrf = Str( help="Fallback VRF name", optional=True )
   encapActionType = Enum( values=[ 'ipip' ],
                           help="Encapsulation type", optional=True )
   ipInIpEncap = Submodel( valueType=GribiIpInIpEncap, optional=True,
                           help='IpInIp Encapsulation' )
   mplsVias = List( valueType=SrTeSegmentListVia,
                    optional=True,
                    help='Segment Routing Traffic Engineering'
                         ' policy information' )

   def render( self, detail, extraTab='' ): # pylint: disable=arguments-differ
      print( f"{extraTab}\tNexthop ID: {self.nhId}, Weight: {self.weight}" )
      if detail:
         ipAddrIntf = ""
         if self.ipAddress:
            ipAddrIntf += f"\t\tIP address: {self.ipAddress}"
         if self.interface:
            if self.ipAddress:
               ipAddrIntf += ", "
            else:
               ipAddrIntf += "\t\t"
            ipAddrIntf += f"Interface: {self.interface.stringValue}"
         if self.macAddress:
            if self.ipAddress or self.interface:
               ipAddrIntf += ", "
            else:
               ipAddrIntf += "\t\t"
            macAddrStr = self.macAddress.stringValue
            macAddrStr = Ethernet.convertMacAddrCanonicalToDisplay( macAddrStr )
            ipAddrIntf += f"MAC address: {macAddrStr}"
         if ipAddrIntf:
            print( extraTab + ipAddrIntf )
         if self.decapActionType == "ipip":
            decapInfo = "\t\tIP-in-IP decapsulation"
            fallbackTxt = f", fallback VRF {self.fallbackVrf}" \
                  if self.fallbackVrf else ""
            print( extraTab + decapInfo + fallbackTxt )
         elif self.fallbackVrf:
            print( extraTab + f"\tFallback VRF: {self.fallbackVrf}" )
         if self.encapActionType == "ipip":
            encapInfo = "\t\tIP-in-IP encapsulation: "
            if self.ipInIpEncap:
               encapInfo += "dest: {} source: {}".format(
                  self.ipInIpEncap.dstIp, self.ipInIpEncap.srcIp )
            print( extraTab + encapInfo )
         if self.mplsVias:
            for mplsVia in self.mplsVias:
               print( extraTab + "\t\tResolved nexthop: {}, Interface: {}".format(
                  mplsVia.nexthop, mplsVia.interface.stringValue ) )
               labelStackStr = "\t\t\tResolved label stack: ["
               labelStackStr += ' '.join( '%u' % label
                                          for label in mplsVia.mplsLabels )
               labelStackStr += "]"
               print( extraTab + labelStackStr )
         elif not ipAddrIntf:
            if not ( self.decapActionType and self.encapActionType
                  and self.fallbackVrf ):
               print( extraTab + "\t\tunresolved" )

class AcknowledgementFecVersionId( Model ):
   versionId = Int( help="FEC version ID to be acknowledged", default=0 )
   requestedDelete = Bool( help="FEC due to be deleted" )
   failed = Bool( help="FEC install failed" )

class GribiNhgAftType( Model ):
   nhgId = Int( help='Nexthop Group ID' )
   nhgNhs = List( valueType=GribiNhgNhAftType,
                  help='List of nexthops in a Nexthop Group AFT entry' )
   backupNhgId = Int( help='Backup nexthop group ID', optional=True )
   fibAckRequested = Submodel(
      valueType=AcknowledgementFecVersionId, optional=True,
      help='Requested FIB-ACK FEC version ID' )
   fibAckAcknowledged = Submodel(
      valueType=AcknowledgementFecVersionId, optional=True,
      help='Acknowledged FIB-ACK FEC version ID' )
   fecId = Int( help='FEC ID', optional=True )

def fibAckOpStr( requestedDelete, failed ):
   # similar to `operationStr` in AleL3CliModel.py, ie, what is shown by
   # show ip hardware ale adj acknowledgement all
   operStr = []
   if failed:
      operStr.append( "failed" )
   if requestedDelete:
      operStr.append( "delete" )
   # ale uses `modify`. We use `replace` because that is the AFT operation name and
   # `modify` is the RPC name.
   return "/".join( operStr ) or "add/replace"

class GribiNhgAftVersionType( Model ):
   nhgVersionAft = Dict( keyType=int, valueType=GribiNhgAftType,
                  help='Map of Nexthop Group AFT entries, keyed by'
                  ' version ID' )

class GribiPendingNhgAftTable( Model ):
   unacknowledgedNhgs = Dict( keyType=int, valueType=GribiNhgAftVersionType,
                  help='Map of versioned Nexthop Group AFT entries, keyed by'
                  ' Nexthop Group ID' )
   details_ = Bool( default=False, help='gRIBI Nexthop Group AFT Detail' )

   def render( self ):
      lastId = None
      for _, data in sorted( self.unacknowledgedNhgs.items() ):
         for _, aft in sorted( data.nhgVersionAft.items() ):
            if lastId != aft.nhgId:
               print( "Nexthop Group ID: %u" % aft.nhgId )
               lastId = aft.nhgId
            op = fibAckOpStr( aft.fibAckRequested.requestedDelete,
                              aft.fibAckRequested.failed )
            print( "\tFIB-ACK version: {}, FIB-ACK operation: {}".format(
               aft.fibAckRequested.versionId, op ) )
            if aft.fibAckAcknowledged:
               op = fibAckOpStr( aft.fibAckAcknowledged.requestedDelete,
                                 aft.fibAckAcknowledged.failed )
               print( "\t\tAcknowledged version: {}, "
                      "Acknowledged operation: {}".format(
                         aft.fibAckAcknowledged.versionId, op ) )
            if self.details_ and aft.fecId:
               print( "\t\tgRIBI FEC ID: %u" % aft.fecId )
            for nh in sorted( aft.nhgNhs, key=attrgetter( 'nhId' ) ):
               nh.render( self.details_, extraTab='\t' )
            if aft.backupNhgId:
               print( "\t    Backup nexthop group ID: %u" % aft.backupNhgId )

class GribiNhgAftTable( Model ):
   nhgAft = Dict( keyType=int, valueType=GribiNhgAftType,
                  help='Map of Nexthop Group AFT entries, keyed by'
                       ' Nexthop Group ID' )
   details_ = Bool( default=False, help='gRIBI Nexthop Group AFT Detail' )

   def render( self ):
      for _, aft in sorted( self.nhgAft.items() ):
         print( "Nexthop Group ID: %u" % aft.nhgId )
         # Only show the fib-ack requested line if fib-ack is enabled (ie.,
         # non zero versionId)
         if aft.fibAckRequested is not None and aft.fibAckRequested.versionId:
            op = fibAckOpStr( aft.fibAckRequested.requestedDelete,
                              aft.fibAckRequested.failed )
            print( "\tFIB-ACK version: {}, FIB-ACK operation: {}".format(
               aft.fibAckRequested.versionId, op ) )
            if aft.fibAckAcknowledged:
               op = fibAckOpStr( aft.fibAckAcknowledged.requestedDelete,
                                 aft.fibAckAcknowledged.failed )
               print( "\tAcknowledged version: {}, "
                      "Acknowledged operation: {}".format(
                         aft.fibAckAcknowledged.versionId, op ) )
         if self.details_ and aft.fecId:
            print( "\tgRIBI FEC ID: %u" % aft.fecId )
         for nh in sorted( aft.nhgNhs, key=attrgetter( 'nhId' ) ):
            nh.render( self.details_ )
         if aft.backupNhgId:
            print( "\tBackup nexthop group ID: %u" % aft.backupNhgId )

class GribiFecVia( Model ):
   tunnelIdx = Int( help="Tunnel index " )
   weight = Int( "Weight of the nexthop", optional=True )
   mplsVias = List( valueType=SrTeSegmentListVia,
                    optional=True,
                    help="Segment Routing Traffic Engineering"
                         " policy information" )

class GribiFecDetail( Model ):
   fecId = Int( help='FEC ID' )
   fecVias = List( valueType=GribiFecVia,
                   optional=True,
                   help="List of FEC vias" )

class GribiMplsAftDetail( Model ):
   fecDetail = Submodel( valueType=GribiFecDetail, optional=True,
                         help='gRIBI FEC detail' )

   def render( self ):
      if self.fecDetail:
         print( "\tgRIBI FEC ID: %u" % ( self.fecDetail.fecId ) )
         for fecVia in self.fecDetail.fecVias:
            print( "\t\tTunnel ID: gRIBI tunnel index %u, Weight: %u" % (
                  fecVia.tunnelIdx, fecVia.weight ) )
            for mplsVia in fecVia.mplsVias:
               viaStr = "\t\t\tResolved nexthop: {}, Interface: {}\n".format(
                     mplsVia.nexthop, mplsVia.interface.stringValue )
               viaStr += "\t\t\t\tResolved label stack: ["
               viaStr += ' '.join( '%u' % label for label in mplsVia.mplsLabels )
               viaStr += "]"
               print( viaStr )
      else:
         print( "\tunresolved" )

class GribiNhAftDetail( Model ):
   tunnelDetail = Submodel( valueType=GribiFecVia, optional=True,
                            help='Tunnel detail' )

   # pylint: disable=W0221
   def render( self, labelStackString ):
      if self.tunnelDetail and labelStackString:
         print( "%s, Tunnel ID: gRIBI tunnel index %u" % ( labelStackString,
                                                   self.tunnelDetail.tunnelIdx ) )
         for mplsVia in self.tunnelDetail.mplsVias:
            viaStr = "\t\tResolved nexthop: {}, Interface: {}\n".format(
                  mplsVia.nexthop, mplsVia.interface.stringValue )
            viaStr += "\t\t\tResolved label stack: ["
            viaStr += ' '.join( '%u' % label for label in mplsVia.mplsLabels )
            viaStr += "]"
            print( viaStr )
      else:
         if labelStackString:
            print( labelStackString )
            print( "\t\tunresolved" )

class GribiMplsAftType( Model ):
   label = Int( help='Label' )
   nhgId = Int( help='Nexthop Group ID' )
   mplsAftDetail = Submodel( valueType=GribiMplsAftDetail, optional=True,
                              help='gRIBI MPLS AFT detail' )

class GribiMplsAftTable( Model ):
   mplsAft = Dict( keyType=int, valueType=GribiMplsAftType,
                   help='Map of MPLS AFT entries, keyed by Label' )

   def render( self ):
      for key in sorted( self.mplsAft ):
         aft = self.mplsAft[ key ]
         print( "Label: %u Nexthop Group ID: %u" % ( aft.label, aft.nhgId ) )
         if aft.mplsAftDetail:
            aft.mplsAftDetail.render()

class AcknowledgementRouteVersionId( Model ):
   versionId = Int( help="Route version ID to be acknowledged", default=0 )
   requestedDelete = Bool( help="Route due to be deleted" )
   failed = Bool( help="Route failed installation in hardware" )

class GribiIpAftEntry( Model ):
   prefix = IpGenericPrefix( help='Prefix' )
   nhgId = Int( help='Nexthop Group ID' )
   metadata = Str( optional=True, help="Metadata" )
   fibAckRequested = Submodel(
      valueType=AcknowledgementRouteVersionId, optional=True,
      help='Requested FIB-ACK route version ID' )
   fibAckAcknowledged = Submodel(
      valueType=AcknowledgementRouteVersionId, optional=True,
      help='Acknowledged FIB-ACK route version ID' )

class GribiIpAftVersionType( Model ):
   ipVersionAft = Dict( keyType=int, valueType=GribiIpAftEntry,
                         help='Map of Route AFT entries, keyed by version ID' )

class GribiPendingIpAft( Model ):
   _vrf = Str( help="VRF name" )
   unacknowledgedPrefixes = GeneratorDict( keyType=str,
                                    valueType=GribiIpAftVersionType,
                             help='Map of versioned Route AFT entries, keyed by'
                             ' Prefix' )
   _tableFmt = Bool( help='Output in table format' )

   def renderNonTable( self ):
      print( f"VRF: {self._vrf}" )
      for _, data in self.unacknowledgedPrefixes:
         for _, entry in sorted( data.ipVersionAft.items() ):
            print( f"{entry.prefix} via NHG ID {entry.nhgId}" )
            if entry.metadata:
               print( f"\tMetadata: 0x{entry.metadata}" )
            fibAckReq = entry.fibAckRequested
            if fibAckReq is not None:
               version = fibAckReq.versionId
               if version != 0:
                  op = fibAckOpStr( fibAckReq.requestedDelete, fibAckReq.failed )
                  print( "\tFIB-ACK version: {}, FIB-ACK operation: {}"
                          .format( version, op ) )

   def renderTable( self ):
      print( f"VRF: {self._vrf}" )
      pfxColFormat = Format( justify='left' )
      pfxColFormat.noPadLeftIs( True )
      nhgColFormat = Format( justify='right' )
      nhgColFormat.noTrailingSpaceIs( True )
      metadataFormat = Format( justify='left' )
      metadataFormat.noPadLeftIs( True )
      reqAckVersionColFormat = Format( justify='right' )
      reqAckVersionColFormat.noTrailingSpaceIs( True )
      reqAckOpVersionColFormat = Format( justify='left' )
      reqAckOpVersionColFormat.noPadLeftIs( True )
      reqAckOpVersionColFormat.padLimitIs( True )
      t = createTable( ( 'Prefix', 'Next Hop Group ID', 'Metadata',
                         'FIB-ACK\nVersion', 'FIB-ACK\nOperation' ),
                       tableWidth=120 )
      t.formatColumns( pfxColFormat, nhgColFormat, metadataFormat,
                       reqAckVersionColFormat, reqAckOpVersionColFormat )
      for _, data in self.unacknowledgedPrefixes:
         for _, entry in sorted( data.ipVersionAft.items() ):
            version = 0
            op = 'n/a'
            metadata = '-'
            pfx = entry.prefix
            if entry.metadata:
               metadata = '0x' + entry.metadata
            fibAckReq = entry.fibAckRequested
            if fibAckReq is not None:
               version = fibAckReq.versionId
               if version != 0:
                  op = fibAckOpStr( fibAckReq.requestedDelete, fibAckReq.failed )
            t.newRow( pfx, entry.nhgId, metadata, version, op )
      print( t.output() )

   def render( self ):
      if self._tableFmt:
         self.renderTable( )
      else:
         self.renderNonTable( )

class GribiIpAft( Model ):
   _vrf = Str( help="VRF name" )
   prefixes = GeneratorDict( keyType=IpGenericPrefix, valueType=GribiIpAftEntry,
                             help='AFT entries keyed by prefix' )
   _tableFmt = Bool( help='Output in table format' )

   def renderNonTable( self ):
      print( f"VRF: {self._vrf}" )
      for prefix, entry in self.prefixes:
         print( f"{prefix} via NHG ID {entry.nhgId}" )
         if entry.metadata:
            print( f"\tMetadata: 0x{entry.metadata}" )
         fibAckReq = entry.fibAckRequested
         if fibAckReq is not None:
            version = fibAckReq.versionId
            if version != 0:
               op = fibAckOpStr( fibAckReq.requestedDelete, fibAckReq.failed )
               print( "\tFIB-ACK version: {}, FIB-ACK operation: {}"
                       .format( version, op ) )
         fibAckAcked = entry.fibAckAcknowledged
         if fibAckAcked is not None:
            ackedVersion = fibAckAcked.versionId
            if ackedVersion and ackedVersion != 0:
               ackedOp = fibAckOpStr( fibAckAcked.requestedDelete,
                                      fibAckAcked.failed )
               print( "\tAcknowledged version: {}, Acknowledged operation: {}"
                      .format( ackedVersion, ackedOp ) )

   def renderTable( self ):
      print( f"VRF: {self._vrf}" )
      pfxColFormat = Format( justify='left' )
      pfxColFormat.noPadLeftIs( True )
      nhgColFormat = Format( justify='right' )
      nhgColFormat.noTrailingSpaceIs( True )
      metadataFormat = Format( justify='left' )
      metadataFormat.noPadLeftIs( True )
      reqAckVersionColFormat = Format( justify='right' )
      reqAckVersionColFormat.noTrailingSpaceIs( True )
      reqAckOpVersionColFormat = Format( justify='left' )
      reqAckOpVersionColFormat.noPadLeftIs( True )
      reqAckOpVersionColFormat.padLimitIs( True )
      ackedAckVersionColFormat = Format( justify='right' )
      ackedAckVersionColFormat.noTrailingSpaceIs( True )
      ackedAckOpVersionColFormat = Format( justify='left' )
      ackedAckOpVersionColFormat.noPadLeftIs( True )
      ackedAckOpVersionColFormat.padLimitIs( True )
      t = createTable( ( 'Prefix', 'Next Hop Group ID', 'Metadata',
                         'FIB-ACK\nVersion', 'FIB-ACK\nOperation',
                         'Acknowledged\nVersion', 'Acknowledged\nOperation' ),
                       tableWidth=120 )
      t.formatColumns( pfxColFormat, nhgColFormat, metadataFormat,
                       reqAckVersionColFormat, reqAckOpVersionColFormat,
                       ackedAckVersionColFormat, ackedAckOpVersionColFormat )

      for prefix, entry in self.prefixes:
         version = ackedVersion = 0
         op = ackedOp = 'n/a'
         metadata = '-'
         if entry.metadata:
            metadata = '0x' + entry.metadata
         fibAckReq = entry.fibAckRequested
         if fibAckReq is not None:
            version = fibAckReq.versionId
            if version != 0:
               op = fibAckOpStr( fibAckReq.requestedDelete, fibAckReq.failed )
         fibAckAcked = entry.fibAckAcknowledged
         if fibAckAcked is not None:
            ackedVersion = fibAckAcked.versionId
            if ackedVersion != 0:
               ackedOp = fibAckOpStr( fibAckAcked.requestedDelete,
                                      fibAckAcked.failed )
         t.newRow(
            prefix, entry.nhgId, metadata, version, op, ackedVersion, ackedOp )
      print( t.output() )

   def render( self ):
      if self._tableFmt:
         self.renderTable( )
      else:
         self.renderNonTable( )

class GribiNhAftType( Model ):
   __revision__ = 2
   nhId = Int( help='Nexthop ID' )
   pushMplsLabels = List( valueType=int,
                          help='Push MPLS Label Stack Data' )
   ipAddress = IpGenericAddress( optional=True, help="Next hop address" )
   interface = Interface( optional=True, help="Next hop interface" )
   macAddress = MacAddress( optional=True, help="Next hop MAC address" )
   decapActionType = Enum( values=[ 'ipip' ],
                           help="Decapsulation type", optional=True )
   vrf = Str( help="VRF name", optional=True )
   encapActionType = Enum( values=[ 'ipip' ],
                           help="Encapsulation type", optional=True )
   ipInIpEncap = Submodel( valueType=GribiIpInIpEncap, optional=True,
                        help='IpInIp Encapsulation' )
   nhAftDetail = Submodel( valueType=GribiNhAftDetail, optional=True,
                           help='gRIBI Nexthop AFT detail' )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # optional 'vrf' attribute is named 'fallbackVrf' in revisions < 2
         vrf = dictRepr.get( 'vrf' )
         if vrf is not None:
            dictRepr[ 'fallbackVrf' ] = vrf
            del dictRepr[ 'vrf' ]
      return dictRepr

class GribiNhAftVersionType( Model ):
   nhVersionAft = Dict( keyType=int, valueType=GribiNhAftType,
                  help='Map of Nexthop AFT entries, keyed by versionId' )

def renderNhAft( aftEntry ):
   print( "Nexthop ID: %u" % aftEntry.nhId )
   ipAddrIntf = ""
   if aftEntry.ipAddress:
      ipAddrIntf += f"\tIP address: {aftEntry.ipAddress}"
   if aftEntry.interface:
      if aftEntry.ipAddress:
         ipAddrIntf += ", "
      else:
         ipAddrIntf += "\t"
      ipAddrIntf += f"Interface: {aftEntry.interface.stringValue}"
   if aftEntry.macAddress:
      if aftEntry.ipAddress or aftEntry.interface:
         ipAddrIntf += ", "
      else:
         ipAddrIntf += "\t"
      macAddrStr = aftEntry.macAddress.stringValue
      macAddrStr = Ethernet.convertMacAddrCanonicalToDisplay( macAddrStr )
      ipAddrIntf += f"MAC address: {macAddrStr}"
   if ipAddrIntf:
      print( ipAddrIntf )
   if aftEntry.vrf:
      print( f"\tVRF: {aftEntry.vrf}" )
   if aftEntry.decapActionType == "ipip":
      print( "\tIP-in-IP decapsulation" )
   if aftEntry.encapActionType == "ipip":
      encapInfo = "\tIP-in-IP encapsulation: "
      if aftEntry.ipInIpEncap:
         encapInfo += "dest: {} source: {}".format(
            aftEntry.ipInIpEncap.dstIp, aftEntry.ipInIpEncap.srcIp )
      print( encapInfo )

   labelStackString = ""
   if aftEntry.pushMplsLabels:
      labelStackString += "\tLabel stack: [%s]" % \
                          ' '.join( '%u' % l for l in aftEntry.pushMplsLabels )
   if aftEntry.nhAftDetail:
      aftEntry.nhAftDetail.render( labelStackString )
   else:
      if labelStackString:
         print( labelStackString )

class GribiPendingNhAftTable( Model ):
   unacknowledgedNhs = Dict( keyType=int, valueType=GribiNhAftVersionType,
                  help='Map of versioned Nexthop AFT entries, keyed by'
                  ' Nexthop ID' )

   def render( self ):
      for _, data in sorted( self.unacknowledgedNhs.items() ):
         for _, aft in sorted( data.nhVersionAft.items() ):
            renderNhAft( aft )

class GribiNhAftTable( Model ):
   __revision__ = 2
   nhAft = Dict( keyType=int, valueType=GribiNhAftType,
                 help='Map of Nexthop AFT entries, keyed by Nexthop ID' )

   def render( self ):
      for _, aftEntry in sorted( self.nhAft.items() ):
         renderNhAft( aftEntry )

   def degrade( self, dictRepr, revision ):
      if revision >= 2:
         return dictRepr
      for nhId in dictRepr:
         aftEntry = dictRepr[ nhId ]
         # optional 'vrf' attribute is named 'fallbackVrf' in revisions < 2
         vrf = aftEntry.get( 'vrf' )
         if vrf is not None:
            aftEntry[ 'fallbackVrf' ] = vrf
            del aftEntry[ 'vrf' ]
            dictRepr[ nhId ] = aftEntry
      return dictRepr

class GribiElectionId( Model ):
   high = Int( help="High 64 bits of the 128-bit election ID" )
   low = Int( help="Low 64 bits of the 128-bit election ID" )

redundancyEnumToStr = { 'allPrimary': 'all primary',
      'singlePrimary': 'single primary',
      'invalid': 'invalid' }
persistenceEnumToStr = { 'delAft': 'delete',
      'preserve': 'preserve',
      'invalid': 'invalid' }
ackTypeEnumToStr = { 'ribAck': 'RIB',
      'ribAndFibAck': 'RIB and FIB',
      'invalid': 'invalid' }

class GribiModifyRpcStreamStatus( Model ):
   connectionTimeStamp = Float( help="Timestamp when the stream connected" )
   electionId = Submodel( optional=True, valueType=GribiElectionId,
         help="Election ID received" )
   electionIdTimeStamp = Float( optional=True,
         help="Timestamp when election ID is received" )
   clientRedundancy = Enum( values=redundancyEnumToStr, optional=True,
         help="Client redundancy" )
   aftPersistence = Enum( values=persistenceEnumToStr, optional=True,
         help="AFT persistence" )
   aftOpAckTypeRequested = Enum( values=ackTypeEnumToStr, optional=True,
         help="AFT operation acknowledgement type requested" )
   aftOpAckTypeApplied = Enum( values=ackTypeEnumToStr, optional=True,
         help="AFT operation acknowledgement type applied" )
   sessionParamsTimeStamp = Float(
         help="Timestamp when session parameters are received" )
   lastAftOpTimeStamp = Float( help="Timestamp of last AFT operation received" )

   def render( self ):
      print( "Connected: {} ({})".format(
         timestampToStr( self.connectionTimeStamp, relative=False, now=time.time() ),
         utcTimeRelativeToNowStr( self.connectionTimeStamp ) ) )

      electionIdStr = "none"
      electionIdTimeStampStr = "never"
      if self.electionIdTimeStamp != 0:
         electionIdStr = "High: {}, Low: {}".format( self.electionId.high,
               self.electionId.low )
         electionIdTimeStampStr = "{} ({})".format(
            timestampToStr( self.electionIdTimeStamp, relative=False,
               now=time.time() ),
            utcTimeRelativeToNowStr( self.electionIdTimeStamp ) )
      print( "Election ID:", electionIdStr )
      print( "Election ID event:", electionIdTimeStampStr )

      if self.sessionParamsTimeStamp == 0:
         print( "Session parameters: none" )
         print( "Session parameters event: never" )
      else:
         print( "Session parameters:" )
         print( "Client redundancy:", redundancyEnumToStr[ self.clientRedundancy ] )
         print( "AFT persistence:", persistenceEnumToStr[ self.aftPersistence ] )
         print( "AFT operation acknowledgement type: " +
            "Requested: {}, Applied: {}".format(
               ackTypeEnumToStr[ self.aftOpAckTypeRequested ],
               ackTypeEnumToStr[ self.aftOpAckTypeApplied ] ) )
         print( "Session parameters event: {} ({})".format(
            timestampToStr( self.sessionParamsTimeStamp, relative=False,
               now=time.time() ),
            utcTimeRelativeToNowStr( self.sessionParamsTimeStamp ) ) )

      lastAftOpTimeStampStr = "never"
      if self.lastAftOpTimeStamp != 0:
         lastAftOpTimeStampStr = "{} ({})".format(
            timestampToStr( self.lastAftOpTimeStamp, relative=False,
               now=time.time() ),
            utcTimeRelativeToNowStr( self.lastAftOpTimeStamp ) )
      print( "Last AFT operation event:", lastAftOpTimeStampStr )

class GribiModifyRpcClientIpPortStatus( Model ):
   streams = Dict( keyType=int, valueType=GribiModifyRpcStreamStatus,
         help="Map of Modify RPC stream status, keyed by stream ID" )

   def render( self, primaryStreamId ): # pylint: disable=arguments-differ
      yetToRenderCount = len( self.streams )
      # show primary stream information first
      if primaryStreamId in self.streams:
         print( "Internal ID:", primaryStreamId )
         self.streams[ primaryStreamId ].render()
         yetToRenderCount -= 1
         if yetToRenderCount > 0:
            print( "" )
      # show rest of the streams sorted by internal ID (or connection time)
      for streamId, status in sorted( self.streams.items() ):
         if streamId != primaryStreamId:
            print( "Internal ID:", streamId )
            status.render()
            yetToRenderCount -= 1
            if yetToRenderCount > 0:
               print( "" )

class GribiModifyRpcClientIpStatus( Model ):
   ports = Dict( keyType=int, valueType=GribiModifyRpcClientIpPortStatus,
         help="Map of Modify RPC status, keyed by port" )

   # pylint: disable=arguments-differ
   def render( self, clientIp, primaryClientIp, primaryClientPort, primaryStreamId ):
      # show primary port information first
      if clientIp == primaryClientIp and primaryClientPort in self.ports:
         portStatus = self.ports[ primaryClientPort ]
         if primaryStreamId in portStatus.streams:
            print( f"\nClient: {clientIp}:{primaryClientPort}" )
            portStatus.render( primaryStreamId )
      # show rest of the ports
      for port, status in sorted( self.ports.items() ):
         if port != primaryClientPort:
            print( f"\nClient: {clientIp}:{port}" )
            status.render( primaryStreamId )
   # pylint: enable=arguments-differ

class GribiRpcStatus( Model ):
   totalClients = Int( help="Total number of Modify RPC client connections" )
   primaryClientIp = IpGenericAddress( optional=True,
         help="Current Modify RPC primary client IP address" )
   primaryClientPort = Int( optional=True,
         help="Current Modify RPC primary client port" )
   primaryClientId = Int( help="Current Modify RPC primary client internal ID" )
   highestElectionId = Submodel( valueType=GribiElectionId,
         help="Highest election ID received so far" )
   highestElectionIdTimeStamp = Float(
         help="Timestamp when highest election ID is received" )
   modifyRpcClients = Dict( keyType=IpGenericAddress,
         valueType=GribiModifyRpcClientIpStatus,
         help="Map of Modify RPC client status, keyed by client IP" )

   def render( self ):
      print( "Total Modify RPC clients:", self.totalClients )
      primaryClientStr = "none"
      primaryClientIdStr = "none"
      if self.primaryClientId != 0:
         primaryClientStr = "{}:{}".format( self.primaryClientIp,
               self.primaryClientPort )
         primaryClientIdStr = f"{self.primaryClientId}"
      print( "Primary Modify RPC client:", primaryClientStr )
      print( "Primary Modify RPC client internal ID:", primaryClientIdStr )

      highestElectionIdStr = "none"
      highestElectionIdTimeStampStr = "never"
      if self.highestElectionIdTimeStamp != 0:
         highestElectionIdStr = "High: {}, Low: {}".format(
            self.highestElectionId.high, self.highestElectionId.low )
         highestElectionIdTimeStampStr = "{} ({})".format(
            timestampToStr( self.highestElectionIdTimeStamp, relative=False,
               now=time.time() ),
            utcTimeRelativeToNowStr( self.highestElectionIdTimeStamp ) )
      print( "Highest election ID received:", highestElectionIdStr )
      print( "Highest election ID event:", highestElectionIdTimeStampStr )

      # show the primary modify RPC client information first
      if self.primaryClientIp in self.modifyRpcClients:
         self.modifyRpcClients[ self.primaryClientIp ].render( self.primaryClientIp,
               self.primaryClientIp, self.primaryClientPort, self.primaryClientId )

      # show the rest of the modify RPC clients
      for ip, status in sorted( self.modifyRpcClients.items() ):
         if ip != str( self.primaryClientIp ):
            status.render( ip, self.primaryClientIp, self.primaryClientPort,
                  self.primaryClientId )

class GribiEndpointStatus( Model ):
   __revision__ = 2
   enabled = Bool( help="The endpoint is enabled" )
   started = Bool( help="gRIBI server is started" )
   if Toggles.gribiToggleLib.toggleGribiAccountingRequestsEnabled():
      accountingRequests = Bool( help="gRIBI RPC accounting requests is enabled" )
   port = Int( help="The port this protocol's server is listening on" )
   sslProfile = Str( help="SSL profile name", optional=True )
   mTls = Bool( help="gRIBI server is running mTLS" )
   authzEnabled = Bool( help="policy based gRPC authorization is enabled" )
   if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
      authnUsernamePriority = List( valueType=str,
            help="Authentication username extraction priority" )
   else:
      certUsernameAuthn = Bool( help="Use the username from the client TLS"
                                " certificate for AAA authentication" )
   error = Str( help="Error which occurred while starting the endpoint",
                optional=True )
   vrfName = Str( help="The VRF this protocol's server is listening in",
                  default="default" )
   serviceAcl = Str( help="The access control group to filter which"
                  " gRIBI clients can connect via IPv4", optional=True )
   serviceAclV6 = Str( help="The access control group to filter which"
                  " gRIBI clients can connect via IPv6", optional=True )
   dscp = Int( help="The DSCP value set for outgoing packets" )
   lastServiceStartTimeStamp = Float( help="Timestamp of last grpc service start" )
   rpcStatus = Submodel( valueType=GribiRpcStatus, optional=True,
                         help="gRIBI RPC server and client status" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'certUsernameAuthn' ] = False
      return dictRepr

   def render( self ):
      print( "Enabled:", ( "yes" if self.enabled else "no" ) )
      if self.enabled and self.started:
         print( "Server: running on port {}, in {} VRF with DSCP {}".
                format( self.port, self.vrfName, self.dscp ) )
      else:
         print( "Server: not yet running" )
      if self.error:
         print( "Error:", self.error )
      print( "SSL profile:", ( self.sslProfile or "none" ) )
      if Toggles.gribiToggleLib.toggleGribiAccountingRequestsEnabled():
         print( "Accounting requests:", ( "yes" if self.accountingRequests else
                                          "no" ) )
      if Toggles.gribiToggleLib.toggleGribiAUPEnabled():
         print( "Authentication username priority:",
               ", ".join( self.authnUsernamePriority )
               if self.authnUsernamePriority else "none" )
      else:
         print( "Certificate username authentication:",
               "yes" if self.certUsernameAuthn else "no" )
      print( "IPv4 Access-group:", ( self.serviceAcl or "none" ) )
      print( "IPv6 Access-group:", ( self.serviceAclV6 or "none" ) )
      if self.rpcStatus:
         self.rpcStatus.render()

class GribiAclStatus( Model ):
   transports = Dict( valueType=AllAclList,
                      help="Transport ACLs indexed by transport name" )

   def render( self ):
      for name, acl in sorted( self.transports.items() ):
         print( 'Transport:', name )
         acl.render()
