# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliToken.Clear
import CliToken.Monitor
import ConfigMount
from CliPlugin import HadoopTracerModel
from CliPlugin.HadoopTracerModel import _intfName
import HostnameCli
import LazyMount
import Tac

from CliMode.HadoopTracer import MonitorHadoopMode
from CliMode.HadoopTracer import MonitorHadoopClusterMode

config = None
status = None
cmdRequest = None

class MonitorHadoopConfigMode( MonitorHadoopMode, BasicCli.ConfigModeBase ):
   name = "monitor-hadoop"

   def __init__( self, parent, session ):
      MonitorHadoopMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def noShutdown( self, args ):
      config.enabled = True

   def shutdown( self, args ):
      config.enabled = False

def clusterConfigHandler( mode, onSessionCommit ):

   # Walk through each cluster in the config and make sure there's a cmdRequest
   # for them
   for cluster in config.clusterConfig:
      if cluster not in cmdRequest.clusterCmdRequest:
         cmdRequest.newClusterCmdRequest( cluster )

   # Now go through each key in cmdRequest and delete the ones for which there's
   # no config
   for cluster in cmdRequest.clusterCmdRequest:
      if cluster not in config.clusterConfig:
         del cmdRequest.clusterCmdRequest[ cluster ]

class ClusterConfigMode( MonitorHadoopClusterMode, BasicCli.ConfigModeBase ):
   name = "monitor-hadoop-cluster"

   def __init__( self, parent, session, clusterName ):
      MonitorHadoopClusterMode.__init__( self, clusterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.clusterName = clusterName
      config.newClusterConfig( clusterName )
      session.maybeCallConfigSessionOnCommitHandler( "hadooptracer",
                                                     clusterConfigHandler )

   def clusterConfig( self ):
      return config.clusterConfig.get( self.clusterName )

   def isMR1Configured( self ):
      return bool( self.clusterConfig().httpPort != 
                   self.clusterConfig().httpPortDefault
                   or self.clusterConfig().host 
                   or self.clusterConfig().rpcPort != 
                   self.clusterConfig().rpcPortDefault
                   or self.clusterConfig().user )

   def isMR2Configured( self ):
      return bool( self.clusterConfig().resourceManagerHost 
                   or self.clusterConfig().resourceManagerPort != 
                   self.clusterConfig().resourceManagerPortDefault
                   or self.clusterConfig().jobHistoryHost 
                   or self.clusterConfig().jobHistoryPort !=
                   self.clusterConfig().jobHistoryPortDefault )

   def setVersion( self, version ):
      if version == "MR1":
         self.clusterConfig().mapReduceVersion = version         
         # Clear MR2 attributes so the configure replace doesnt throw an error.
         if self.isMR2Configured():
            self.clusterConfig().resourceManagerHost = ""
            self.clusterConfig().resourceManagerPort = \
                self.clusterConfig().resourceManagerPortDefault
            self.clusterConfig().jobHistoryHost = ""
            self.clusterConfig().jobHistoryPort = \
                self.clusterConfig().jobHistoryPortDefault
      elif version == "MR2":
         self.clusterConfig().mapReduceVersion = version         
         # Clear MR1 attributes so the configure replace doesnt throw an error.
         if self.isMR1Configured():
            self.clusterConfig().host = ""
            self.clusterConfig().rpcPort = self.clusterConfig().rpcPortDefault
            self.clusterConfig().user = ""
            self.clusterConfig().httpPort = self.clusterConfig().httpPortDefault
      
   def setHttpPort( self, args ):
      self.setVersion( "MR1" )
      self.clusterConfig().httpPort = args[ 'HTTP_PORT' ]

   def noHttpPort( self, args ):
      self.clusterConfig().httpPort = self.clusterConfig().httpPortDefault
      if not ( self.isMR1Configured() or self.isMR2Configured() ):
         # if none of the  attribute is configured then set the version to None
         self.clusterConfig().mapReduceVersion = "None"

   def setInterval( self, args ):
      self.clusterConfig().interval = args[ 'INTERVAL' ]

   def noInterval( self, args ):
      self.clusterConfig().interval = self.clusterConfig().intervalDefault
      
   def setJobTracker( self, args ):
      self.setVersion( "MR1" )
      if 'host' in args:
         self.clusterConfig().host = args[ 'HOST' ][ 0 ]
      if 'rpc-port' in args:
         self.clusterConfig().rpcPort = args[ 'RPC_PORT' ][ 0 ]
      if 'username' in args:
         self.clusterConfig().user = args[ 'USERNAME' ][ 0 ]

   def noJobTracker( self, args ):
      if 'host' in args:
         self.clusterConfig().host = ''
      if 'rpc-port' in args:
         self.clusterConfig().rpcPort = self.clusterConfig().rpcPortDefault
      if 'username' in args:
         self.clusterConfig().user = ''

      if not ( self.isMR1Configured() or self.isMR2Configured() ):
         # if none of the  attribute is configured then set the version to None
         self.clusterConfig().mapReduceVersion = "None"

   def setResourceManager( self, args ):
      self.setVersion( "MR2" )
      if 'host' in args:
         self.clusterConfig().resourceManagerHost = args[ 'HOST' ][ 0 ]
      if 'port' in args:
         self.clusterConfig().resourceManagerPort = args[ 'PORT' ][ 0 ]

   def noResourceManager( self, args ):
      if 'host' in args:
         self.clusterConfig().resourceManagerHost = ''
      if 'port' in args:
         self.clusterConfig().resourceManagerPort = \
               self.clusterConfig().resourceManagerPortDefault

      if not ( self.isMR1Configured() or self.isMR2Configured() ):
         # if none of the  attribute is configured then set the version to None
         self.clusterConfig().mapReduceVersion = "None"

   def setJobHistoryServer( self, args ):
      self.setVersion( "MR2" )
      if 'host' in args:
         self.clusterConfig().jobHistoryHost = args[ 'HOST' ][ 0 ]
      if 'port' in args:
         self.clusterConfig().jobHistoryPort = args[ 'PORT' ][ 0 ]

   def noJobHistoryServer( self, args ):
      if 'host' in args:
         self.clusterConfig().jobHistoryHost = ''
      if 'port' in args:
         self.clusterConfig().jobHistoryPort = \
               self.clusterConfig().jobHistoryPortDefault

      if not ( self.isMR1Configured() or self.isMR2Configured() ):
         # if none of the  attribute is configured then set the version to None
         self.clusterConfig().mapReduceVersion = "None"

   def setDescription( self, args ):
      self.clusterConfig().description = args[ 'DESCRIPTION' ]

   def noDescription( self, args ):
      self.clusterConfig().description = ""

   def shutdown( self, args ):
      self.clusterConfig().enabled = False

   def noShutdown( self, args ):
      self.clusterConfig().enabled = True

def gotoMonitorHadoopMode( mode, args ):
   childMode = mode.childMode( MonitorHadoopConfigMode )
   mode.session_.gotoChildMode( childMode )

def gotoClusterConfigMode( mode, args ):
   # if new cluster and max already configured return error
   clusterName = args[ 'CLUSTER_NAME' ]
   if ( not config.clusterConfig.get( clusterName ) and
        len( config.clusterConfig ) >= 5 ):
      mode.addError( "Maximum number of clusters supported is 5" )
      return

   # Do not allow cluster name to be 'all' as it will conflict with show commands
   # that take cluster name as an option and have 'all' keyword as well
   if clusterName == "all":
      mode.addError( "Cluster name 'all' is not allowed" )
      return

   childMode = mode.childMode( ClusterConfigMode,
                               clusterName=clusterName )
   mode.session_.gotoChildMode( childMode )

def noMonitorHadoop( mode, args ):
   config.enabled = False
   config.clusterConfig.clear()
   mode.session_.maybeCallConfigSessionOnCommitHandler( 
      "hadooptracer", clusterConfigHandler )

def noCluster( mode, args ):
   clusterName = args[ 'CLUSTER_NAME' ]
   if config.clusterConfig.get( clusterName ):
      del config.clusterConfig[ clusterName ]
   mode.session_.maybeCallConfigSessionOnCommitHandler( 
      "hadooptracer", clusterConfigHandler )

def clearJobHistory( mode, clusterName=None ):
   if clusterName and not config.clusterConfig.get( clusterName ):
      mode.addError( "This cluster does not exist" )
      return
   for cluster, clusterCmdRequest in cmdRequest.clusterCmdRequest.items():
      if clusterName and cluster != clusterName:
         continue
      clusterCmdRequest.clearJobHistory += 1
   print( "Cleared job history" )

def clearBurstCounters( mode, clusterName=None ):
   if clusterName and not config.clusterConfig.get( clusterName ):
      mode.addError( "This cluster does not exist" )
      return
   for cluster, clusterCmdRequest in cmdRequest.clusterCmdRequest.items():
      if clusterName and cluster != clusterName:
         continue
      clusterCmdRequest.clearBurstCounters += 1
   print( "Cleared burst counters" )

#--------------------------------------------------------------------------------
# [ no | default ] monitor hadoop
#--------------------------------------------------------------------------------
hadoopKwMatcher = CliMatcher.KeywordMatcher( "hadoop",
                                     helpdesc="Monitor Hadoop Configuration" )

class MonitorHadoopCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor hadoop'
   noOrDefaultSyntax = syntax
   data = {
      'monitor': CliToken.Monitor.monitorMatcher,
      'hadoop': hadoopKwMatcher,
   }
   handler = gotoMonitorHadoopMode
   noOrDefaultHandler = noMonitorHadoop

BasicCliModes.GlobalConfigMode.addCommandClass( MonitorHadoopCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class GlobalShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable MapReduce Tracer globally',
   }
   handler = MonitorHadoopConfigMode.shutdown
   noHandler = MonitorHadoopConfigMode.noShutdown
   defaultHandler = MonitorHadoopConfigMode.shutdown

MonitorHadoopConfigMode.addCommandClass( GlobalShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cluster CLUSTERNAME
#--------------------------------------------------------------------------------
# We have decided to shorten the cluster name or actually make it an Id so that it
# will fit well in the tabular output of the commands. However, we were using cluster
# name all over the code. So, the easiest way to change this to id is to allow only
# digits in our regex and limit them to 4. Nothing else needs to change
clusterMatcher = CliMatcher.PatternMatcher(
   "(?!all$)[0-9A-Za-z_.]+",
   helpname="WORD",
   helpdesc="Cluster Name" )

class ClusterClusternameCmd( CliCommand.CliCommandClass ):
   syntax = 'cluster CLUSTER_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'cluster': 'Configure cluster',
      'CLUSTER_NAME': clusterMatcher
   }
   handler = gotoClusterConfigMode
   noOrDefaultHandler = noCluster

MonitorHadoopConfigMode.addCommandClass( ClusterClusternameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] jobtracker [ { ( host HOST ) | ( rpc-port RPC_PORT ) |
#                                                ( username USERNAME ) } ]
#--------------------------------------------------------------------------------
class JobtrackerCmd( CliCommand.CliCommandClass ):
   syntax = ( 'jobtracker [ { ( host HOST ) | ( rpc-port RPC_PORT ) | '
                            ' ( username USERNAME ) } ]' )
   noOrDefaultSyntax = syntax
   data = {
      'jobtracker': 'JobTracker configuration',
      'host': CliCommand.singleKeyword( 'host', helpdesc='JobTracker Host' ),
      'HOST': HostnameCli.IpAddrOrHostnameMatcher(
         helpname='Hostname or A.B.C.D',
         helpdesc='IP address or hostname of the JobTracker' ),
      'rpc-port': CliCommand.singleKeyword( 'rpc-port',
         helpdesc='JobTracker RPC Port' ),
      'RPC_PORT': CliMatcher.IntegerMatcher( 1024, 65535,
         helpdesc='RPC Port Number' ),
      'username': CliCommand.singleKeyword( 'username',
         helpdesc='Configure the username to use to connect to the JobTracker' ),
      'USERNAME': CliMatcher.PatternMatcher( pattern=r'[.A-Za-z0-9_]+',
         helpdesc='Username to use to connect to the JobTracker', helpname='WORD' ),
   }
   handler = ClusterConfigMode.setJobTracker
   noOrDefaultHandler = ClusterConfigMode.noJobTracker

ClusterConfigMode.addCommandClass( JobtrackerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] resource-manager [ { ( host HOST ) | ( port PORT ) } ]
#--------------------------------------------------------------------------------
class ResourceManagerCmd( CliCommand.CliCommandClass ):
   syntax = 'resource-manager [ { ( host HOST ) | ( port PORT ) } ]'
   noOrDefaultSyntax = syntax
   data = {
      'resource-manager': 'Resource Manager Configuration',
      'host': CliCommand.singleKeyword( 'host', helpdesc='Resource Manager Host' ),
      'HOST': HostnameCli.IpAddrOrHostnameMatcher(
         helpname='Hostname or A.B.C.D',
         helpdesc='IP address or hostname of the Resource-Manager' ),
      'port': CliCommand.singleKeyword( 'port', helpdesc='Resource Manager Port' ),
      'PORT': CliMatcher.IntegerMatcher( 1024, 65535,
         helpdesc='Resource Manager Port Number' ),
   }
   handler = ClusterConfigMode.setResourceManager
   noOrDefaultHandler = ClusterConfigMode.noResourceManager

ClusterConfigMode.addCommandClass( ResourceManagerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] job-history-server [ { ( host HOST ) | ( port PORT ) } ]
#--------------------------------------------------------------------------------
class JobHistoryServerCmd( CliCommand.CliCommandClass ):
   syntax = 'job-history-server [ { ( host HOST ) | ( port PORT ) } ]'
   noOrDefaultSyntax = syntax
   data = {
      'job-history-server': 'Job History Server Configuration',
      'host': CliCommand.singleKeyword( 'host', helpdesc='JobTracker Host' ),
      'HOST': HostnameCli.IpAddrOrHostnameMatcher(
         helpname='Hostname or A.B.C.D',
         helpdesc='IP address or hostname of the Job History Server' ),
      'port': CliCommand.singleKeyword( 'port', helpdesc='Job History Server Port' ),
      'PORT': CliMatcher.IntegerMatcher( 1024, 65535,
         helpdesc='Job History Server Port Number' ),
   }

   handler = ClusterConfigMode.setJobHistoryServer
   noOrDefaultHandler = ClusterConfigMode.noJobHistoryServer

ClusterConfigMode.addCommandClass( JobHistoryServerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description DESCRIPTION
#--------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = syntax
   data = {
      'description': 'Cluster description',
      'DESCRIPTION': CliMatcher.StringMatcher( helpdesc='Description of the cluster',
         helpname='description' ),
   }
   handler = ClusterConfigMode.setDescription
   noOrDefaultHandler = ClusterConfigMode.noDescription

ClusterConfigMode.addCommandClass( DescriptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tasktracker http-port HTTP_PORT
#--------------------------------------------------------------------------------
class TasktrackerHttpPortCmd( CliCommand.CliCommandClass ):
   syntax = 'tasktracker http-port HTTP_PORT'
   noOrDefaultSyntax = 'tasktracker http-port ...'
   data = {
      'tasktracker': 'TaskTracker configuration',
      'http-port': 'TaskTracker HTTP port',
      'HTTP_PORT': CliMatcher.IntegerMatcher( 1024, 65535,
         helpdesc='HTTP port number' ),
   }
   handler = ClusterConfigMode.setHttpPort
   noOrDefaultHandler = ClusterConfigMode.noHttpPort

ClusterConfigMode.addCommandClass( TasktrackerHttpPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interval INTERVAL
#--------------------------------------------------------------------------------
class IntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'interval INTERVAL'
   noOrDefaultSyntax = 'interval ...'
   data = {
      'interval': 'JobTracker polling interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 600, helpdesc='Seconds' ),
   }
   handler = ClusterConfigMode.setInterval
   noOrDefaultHandler = ClusterConfigMode.noInterval

ClusterConfigMode.addCommandClass( IntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable MapReduce Tracer',
   }
   handler = ClusterConfigMode.shutdown
   noHandler = ClusterConfigMode.noShutdown
   defaultHandler = ClusterConfigMode.shutdown

ClusterConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# clear monitor hadoop ( burst-counters | job-history ) [ cluster CLUSTER_NAME ]
#--------------------------------------------------------------------------------
class ClearMonitorHadoopBurstCountersCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear monitor hadoop ( burst-counters | job-history ) '
                                                         '[ cluster CLUSTER_NAME ]' )
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'monitor': CliToken.Monitor.monitorMatcherForClear,
      'hadoop': hadoopKwMatcher,
      'burst-counters': 'Clear all burst counters',
      'job-history': 'Clear all job history',
      'cluster': 'Clear for a cluster',
      'CLUSTER_NAME': clusterMatcher
   }

   @staticmethod
   def handler( mode, args ):
      if 'burst-counters' in args:
         clearBurstCounters( mode, args.get( 'CLUSTER_NAME' ) )
      elif 'job-history' in args:
         clearJobHistory( mode, args.get( 'CLUSTER_NAME' ) )

BasicCliModes.EnableMode.addCommandClass( ClearMonitorHadoopBurstCountersCmd )

# Check error for given cluster. Return True after printing an error
# if there's an error
def _clusterError( mode, cluster, operInfo=True ):
   assert cluster, "Please provide a cluster name"
   if cluster not in config.clusterConfig:
      mode.addError( "Cluster %s does not exist" % cluster )
      return True
   if ( operInfo and ( cluster not in status.clusterStatus or
                       not status.clusterStatus.get( cluster ).operState ) ):
      mode.addError( "Cluster %s is operationally disabled" % cluster )
      return True
   return False

# Returns a dict of clusterStatus. If cluster is given then the dict will have
# just one clusterStatus for that given cluster or will have all clusterStatus
def _getClusterStatus( cluster=None ):
   if cluster:
      clusterStatus = {}
      value = status.clusterStatus.get( cluster )
      if value:
         clusterStatus[ cluster ] = value
   else:
      clusterStatus = status.clusterStatus
   return clusterStatus

# See if given cluster is of interest for a given cluster status. Basically,
# we want to ignore clusters whose oper state is false or whose name doesn't
# match the given cluster name
def _clusterOfInterest( cluster, clusterStatus, givenTt=None ):
   if not clusterStatus.operState:
      return False
   # figure out if given TT is in the given cluster status
   if givenTt:
      for ttName in clusterStatus.localTaskTrackerInfo:
         if givenTt and givenTt == ttName:
            return True
      return False
   return True

# return the latest update time from all clusters. Used when not getting any
# cluster specific information
def _getLastUpdateTime( cluster=None ):
   t = 0.0
   allClusterStatus = _getClusterStatus( cluster )
   for clusterName, clusterStatus in allClusterStatus.items():
      if not clusterStatus.operState:
         continue
      if cluster and cluster == clusterName:
         return clusterStatus.lastUpdate
      if clusterStatus.lastUpdate > t:
         t = clusterStatus.lastUpdate
   return t

def _countersPerInterface( intfId, ttNames, counters ):
   cntr = HadoopTracerModel.CountersPerInterface( interface=intfId )
   cntr.hdfsBytesWritten = counters.hdfsBytesWritten
   cntr.reduceShuffleBytes = counters.reduceShuffleBytes
   cntr.hdfsBytesRead = counters.hdfsBytesRead
   cntr.taskTrackers = ttNames
   return cntr

# return all running jobs on a given cluster and given TT. If no cluster or
# TT is given then return all locally running jobs
def _getRunningJobs( cluster=None, givenTt=None, jobId=None ):
   jobsDict = {}
   allClusterStatus = _getClusterStatus( cluster )
   for clusterName, clusterStatus in allClusterStatus.items():
      if not _clusterOfInterest( cluster, clusterStatus, givenTt):
         continue
      mapReduceVersion = config.clusterConfig.get( clusterName ).mapReduceVersion
      ttJobs = {} #stores map and reduce count per jobId
      if givenTt:
         allTts = [ clusterStatus.localTaskTrackerInfo.get( givenTt ) ]
      else:
         allTts = list( clusterStatus.localTaskTrackerInfo.values() )
      for tt in allTts:
         for jobKey, ttJob in tt.runningJob.items():
            maps = sum( 1 for task in ttJob.runningTask if task.isMap )
            reduces = len( ttJob.runningTask ) - maps
            tts = [ tt.name ]
            if jobKey.jobId in ttJobs:
               maps += ttJobs[ jobKey.jobId ][ 0 ] 
               reduces += ttJobs[ jobKey.jobId ][ 1 ]
               tts += ttJobs[ jobKey.jobId ][ 2 ]
            ttJobs[ jobKey.jobId ] = ( maps, reduces, tts )
      for jobKey, job in clusterStatus.runningJob.items():
         # if details of specific job is needed then return JobInfo otherwise
         # collection of BasicJobInfo
         if not job.isLocal:
            continue
         if jobId:
            if jobId != jobKey.jobId:
               continue
            jobInfo = HadoopTracerModel.JobInfo()
         else:
            jobInfo = HadoopTracerModel.BasicJobInfo()
         jobInfo.jobId = jobKey.jobId
         jobInfo.jtId = jobKey.jtId
         jobInfo.name = job.jobInfo.name
         jobInfo.user = job.jobInfo.user
         jobInfo.cluster = clusterName
         jobInfo.startTime = job.jobInfo.startTime
         jobInfo.mapProgress = job.jobInfo.mapProgress
         jobInfo.reduceProgress = job.jobInfo.reduceProgress
         jobInfo.lastUpdated = _getLastUpdateTime( cluster=cluster )
         maps, reduces, tts = ttJobs.get( jobKey.jobId, ( 0, 0, [] ) )
         jobInfo.maps = maps
         jobInfo.taskTrackers = tts
         jobInfo.reduces = reduces
         for intfId, localJobCounter in job.localJobCounter.items():
            jobInfo.countersPerInterface[ intfId ] = \
                _countersPerInterface( intfId, list( localJobCounter.taskTracker ),
                                       localJobCounter.accumBytes )

         # If both cluster and jobId are given, it means we are looking for one
         # specific job. So, return just that
         if jobId:
            if mapReduceVersion == "MR1":
               jobInfo.cleanupProgress = job.jobInfo.cleanupProgress
               jobInfo.setupProgress = job.jobInfo.setupProgress
               jobInfo.priority = job.jobInfo.priority
               jobInfo.runState = job.jobInfo.runState
               jobInfo.failureInfo = job.jobInfo.failureInfo
               jobInfo.schedulingInfo = job.jobInfo.schedulingInfo
               jobInfo.queueName = job.jobInfo.queueName
               jobInfo.url = job.jobInfo.url
            elif mapReduceVersion == "MR2":
               # non-optional args has to be filled.
               jobInfo.cleanupProgress = -1.0
               jobInfo.setupProgress = -1.0
               jobInfo.priority = "none"
               jobInfo.runState = job.jobInfo.runState
               jobInfo.failureInfo = job.jobInfo.failureInfo
               jobInfo.schedulingInfo = "none"
               jobInfo.queueName = job.jobInfo.queueName
               jobInfo.url = job.jobInfo.url               
            return jobInfo
         dictKey = jobKey.jtId + "-" + jobInfo.cluster + "-" + str( jobKey.jobId )
         jobsDict[ dictKey ] = jobInfo
   # if we are here that means we did not find job info for the given jobId. This
   # could happen if isLocal is not set for some reason
   if jobId:
      return None
   return jobsDict

# Show commands from here onwards
def showStatus( mode, args ):
   hadoopStatus = HadoopTracerModel.HadoopStatus()
   hadoopStatus.adminStatus = config.enabled
   hadoopStatus.operStatus = status.running
   hadoopStatus.numClusters = len( config.clusterConfig )
   hadoopStatus.numTaskTrackers = sum( len( c.localTaskTrackerInfo )
                                  for c in status.clusterStatus.values() )
   hadoopStatus.clusterErrors = {}

   numRunningJobs = 0
   for cluster, clusterStatus in status.clusterStatus.items():
      numRunningJobs += sum( 1 for job in clusterStatus.runningJob.values()
                             if job.isLocal )
      if clusterStatus.error:
         clusterError = HadoopTracerModel.ClusterError()
         clusterError.errorSeenAt = clusterStatus.lastErrorUpdate
         clusterError.error = clusterStatus.error
         hadoopStatus.clusterErrors[ cluster ] = clusterError
   hadoopStatus.numRunningJobs = numRunningJobs
   hadoopStatus.lastUpdated = _getLastUpdateTime()
   return hadoopStatus

def showRunningJobs( mode, args ):
   runningJobs = HadoopTracerModel.RunningJobs()
   runningJobs.jobs = _getRunningJobs()
   runningJobs.lastUpdated = _getLastUpdateTime()
   return runningJobs

def showCounters( mode, args ):
   counters = HadoopTracerModel.HadoopCounters()
   counters.jobs = _getRunningJobs()
   counters.lastUpdated = _getLastUpdateTime()
   return counters

# return single job if jobId is given or a list of jobs
def _getJobHistory( cluster=None, jobId=None ):
   jobsDict = {}
   allClusterStatus = _getClusterStatus( cluster )
   for clusterName, clusterStatus in allClusterStatus.items():
      if not _clusterOfInterest( cluster, clusterStatus):
         continue
      for origHistory in clusterStatus.jobHistory.values():
         jobKey = origHistory.jobKey
         if jobId and jobKey.jobId != jobId:
            continue
         data = HadoopTracerModel.JobHistoryData()
         data.jobId = jobKey.jobId
         data.jtId = jobKey.jtId
         data.jobName = origHistory.jobName
         data.cluster = clusterName
         data.user = origHistory.user
         data.startTime = origHistory.startTime
         data.endTime = origHistory.endTime
         for intfId, intfCounter in origHistory.countersPerIntfId.items():
            data.countersPerInterface[ intfId ] = \
                _countersPerInterface( intfId, list( intfCounter.taskTracker ),
                                       intfCounter.counters )
         # If both cluster and jobId are given, it means we are looking for one
         # specific job. Even then return the dictionary as calling function will
         # handle this case
         dictKey = jobKey.jtId + "-" + data.cluster + "-" + str( jobKey.jobId )
         jobsDict[ dictKey ] = data

   return jobsDict

def showHistory( mode, args ):
   jobHistory = HadoopTracerModel.JobHistory()
   jobHistory.jobHistory = _getJobHistory()
   return jobHistory

# check if there's a TT with given host name of given set of interfaces. These
# interfaces are given the the CLI as intfId objects. We return True if there's
# just one TT for given set of interfaces. We don't have to have all of the intfs
# have TT on them
def _tTOfInterest( tt, hostName, intfs ):
   if hostName:
      # Let partial names also be matched. If there are multiple hosts for this is
      # called then all will match but the calling function will bail out after the
      # first match so in that sense only the first one will be matched
      return hostName in tt.name
   if intfs:
      if tt.intfId:
         return tt.intfId in intfs
      return False
   # if hostname or interface not given then we say TT is of interest
   return True

def _getTaskTrackers( hostName=None, intfs=(), cluster=None ):
   tts = []
   lastUpdated = 0
   intfSet = set( intfs )
   allClusterStatus = _getClusterStatus( cluster )
   for clusterName, clusterStatus in allClusterStatus.items():
      if not _clusterOfInterest( cluster, clusterStatus):
         continue
      for ttName, tt in clusterStatus.localTaskTrackerInfo.items():
         if not _tTOfInterest( tt, hostName, intfSet ):
            continue
         # return best lastUpdate time based on clusters to which the TT belongs
         if clusterStatus.lastUpdate > lastUpdated:
            lastUpdated = clusterStatus.lastUpdate
         newTt = HadoopTracerModel.TaskTracker()
         newTt.name = ttName
         newTt.cluster = clusterName
         newTt.ipAddress = tt.ipAddress
         newTt.state = tt.state
         newTt.stateDetail = tt.stateDetail
         newTt.runningJobs = HadoopTracerModel.RunningJobs()
         newTt.runningJobs.jobs = _getRunningJobs( givenTt=tt.name, cluster=cluster )
         newTt.runningTasks = _getTtRunningTasks( hostName=ttName,
                                                  cluster=cluster )[ 0 ]
         newTt.interface = tt.intfId
         newTt.lastUpdated = lastUpdated
         tts.append( newTt )
         # if hostname is given then return list of just one TT
         if hostName:
            return tts, float( lastUpdated )
   return tts, float( lastUpdated )

def taskTrackerExists( hostName, intfs, cluster ):
   intfSet = set( intfs )
   allClusterStatus = _getClusterStatus( cluster )
   for clusterStatus in allClusterStatus.values():
      if not _clusterOfInterest( cluster, clusterStatus):
         continue
      for ttName, tt in clusterStatus.localTaskTrackerInfo.items():
         if _tTOfInterest( tt, ttName, intfSet ):
            return True
   return False

def showTaskTrackers( mode, args ):
   taskTrackers = HadoopTracerModel.TaskTrackers()
   tts, lastUpdated = _getTaskTrackers()
   taskTrackers.taskTrackers = { tt.name: tt for tt in tts }
   taskTrackers.lastUpdated = lastUpdated
   if 'status' in args:
      # pylint: disable=protected-access
      taskTrackers._printDetail = True
   return taskTrackers

def showAllTaskTrackerCounters( mode, args ):
   tts, lastUpdated = _getTaskTrackers()
   taskTrackerCounters = HadoopTracerModel.TaskTrackerCounters()
   for tt in tts:
      taskTrackerCounter = HadoopTracerModel.TaskTrackerCounter()
      taskTrackerCounter.name = tt.name
      taskTrackerCounter.ipAddress = tt.ipAddress
      taskTrackerCounter.interface = tt.interface
      taskTrackerCounter.bytesIn, taskTrackerCounter.bytesOut = \
          tt.getBytesReadWritten()
      taskTrackerCounters.counters[ tt.name ] = taskTrackerCounter
   taskTrackerCounters.lastUpdated = lastUpdated
   return taskTrackerCounters

def showTTStatus( mode, args ):
   # _getTaskTrackers returns one TaskTracker if hostname is given
   tts, _ = _getTaskTrackers( hostName=args[ 'HOST' ] )
   if tts:
      return tts[ 0 ]
   mode.addError( "Node not found" )
   return None

def showIntfTTStatus( mode, args ):
   intfs = args[ 'INTFS' ]
   tts, lastUpdated = _getTaskTrackers( intfs=intfs )
   if not tts:
      mode.addError( "Node not found" )
      return None
   taskTrackers = HadoopTracerModel.InterfaceTaskTrackers()
   taskTrackers.taskTrackers = { _intfName( tt.interface ): tt
                                 for tt in tts }
   taskTrackers.lastUpdated = lastUpdated
   # pylint: disable=protected-access
   taskTrackers._printDetail = True
   return taskTrackers

def _logErrorTaskTrackerNotFound( mode, cluster ):
   if cluster:
      msg = "Node not found on given cluster: %s"\
          % cluster
   else:
      msg = "Node not found"
   mode.addError( msg )

# Show running jobs for each TT in given list of TTs
def showTTJobs( mode, cluster, hostName, intfs ):
   if cluster and _clusterError( mode, cluster ):
      return None
   tts, _ = _getTaskTrackers( hostName=hostName, intfs=intfs, cluster=cluster )
   if not tts:
      _logErrorTaskTrackerNotFound( mode, cluster )
      return None
   ttJobs = HadoopTracerModel.TaskTrackerRunningJobs()
   ttJobs.runningJobs = {}
   # There can be multiple TTs for multiple interfaces
   for tt in tts:
      ttJobs.runningJobs[ tt.name ] = tt.runningJobs
   ttJobs.lastUpdated = _getLastUpdateTime( cluster=cluster )
   return ttJobs

# Show counters per running job for each TT in given list of TTs
def showTTCounters( mode, hostName, intfs ):
   tts, _ = _getTaskTrackers( hostName=hostName, intfs=intfs )
   if not tts:
      _logErrorTaskTrackerNotFound( mode, None )
      return None
   ttJobs = HadoopTracerModel.TaskTrackerPerJobCounters()
   ttJobs.runningJobs = {}
   # There can be multiple TTs for multiple interfaces
   for tt in tts:
      counters = HadoopTracerModel.HadoopCounters()
      counters.jobs = tt.runningJobs.jobs
      ttJobs.runningJobs[ tt.name ] = counters
   ttJobs.lastUpdated = _getLastUpdateTime()
   return ttJobs

def _getTtRunningTasks( hostName=None, intfs=(), cluster=None, jobId=None ):
   runningTasksList = []
   intfSet = set( intfs )
   allClusterStatus = _getClusterStatus( cluster )
   for clusterName, clusterStatus in allClusterStatus.items():
      if not _clusterOfInterest( cluster, clusterStatus):
         continue
      for ttName, tt in clusterStatus.localTaskTrackerInfo.items():
         if not _tTOfInterest( tt, hostName, intfSet ):
            continue
         runningTasks = HadoopTracerModel.RunningTasks()
         runningTasks.taskTrackerName = ttName
         runningTasks.interface = tt.intfId
         runningTasks.taskReports = []
         runningTasksList.append( runningTasks )
         for jobKey, ttJob in tt.runningJob.items():
            if jobId and jobId != jobKey.jobId:
               continue
            for task in ttJob.runningTask.values():
               tr = HadoopTracerModel.TaskReport()
               tr.jobId = jobKey.jobId
               tr.taskId = task.taskId
               tr.cluster = clusterName
               tr.attemptId = task.attemptId
               tr.isMap = task.isMap
               tr.status = task.status
               tr.state = task.state
               tr.startTime = task.startTime
               tr.endTime = task.endTime
               tr.taskTrackerName = ttName
               tr.interface = tt.intfId
               tr.progress = task.progress
               tr.hdfsBytesRead = task.counter.hdfsBytesRead
               tr.hdfsBytesWritten = task.counter.hdfsBytesWritten
               tr.reduceShuffleBytes = task.counter.reduceShuffleBytes
               runningTasks.taskReports.append( tr )
            # If one job was given and is taken care of then move on
            if jobId:
               break
   return runningTasksList

def showTTTasks( mode, intfs, hostName, cluster, jobId ):
   if cluster and _clusterError( mode, cluster ):
      return None

   if not taskTrackerExists( hostName, intfs, cluster ):
      _logErrorTaskTrackerNotFound( mode, cluster )
      return None

   runningTasksList = _getTtRunningTasks( hostName, intfs, cluster, jobId )
   taskTrackerRunningTasks = HadoopTracerModel.TaskTrackerRunningTasks()
   taskTrackerRunningTasks.lastUpdated = _getLastUpdateTime( cluster )
   taskTrackerRunningTasks.runningTasksList = runningTasksList
   return taskTrackerRunningTasks

def showOneTask( mode, cluster, jobId, taskId, intfs, hostName ):
   if _clusterError( mode, cluster ):
      return None

   if not taskTrackerExists( hostName, intfs, cluster ):
      mode.addError( "Node not found" )
      return None

   for runningTasks in _getTtRunningTasks( hostName, intfs, cluster, jobId ):
      for tr in runningTasks.taskReports:
         if tr.jobId == jobId and tr.cluster == cluster and tr.taskId == taskId:
            tr.lastUpdated = _getLastUpdateTime()
            return tr
   mode.addError( "Task was not found" )
   return None

def showClusterBursts( mode, intfs, cluster ):
   def _newBurst( burst ):
      newBurst = HadoopTracerModel.BurstCounter()
      newBurst.value = burst.value
      newBurst.jobId = burst.jobKey.jobId
      newBurst.jobName = burst.jobName
      newBurst.timestamp = burst.timestamp
      return newBurst

   intfSet = set( intfs )
   intfBursts = HadoopTracerModel.InterfaceBursts()
   allClusterStatus = _getClusterStatus( cluster )
   for clust, clusterStatus in allClusterStatus.items():
      for intfId, topBurstCounter in clusterStatus.topBurstCounter.items():
         # If given interface does not match the interface then continue
         # pylint: disable-next=unsupported-membership-test
         if intfSet and intfId not in intfSet:
            continue

         intfBurst = HadoopTracerModel.InterfaceBurst()
         intfBurst.interface = intfId
         intfBurst.cluster = clust

         inBursts = []
         for idx in range( topBurstCounter.numBurstInElements ):
            burst = _newBurst( topBurstCounter.burstInCounter[ idx ] )
            inBursts.append( burst )
         intfBurst.inBursts = inBursts
         # No need to sort the burts as they are stored sorted by the agent
         outBursts = []
         for idx in range( topBurstCounter.numBurstOutElements ):
            burst = _newBurst( topBurstCounter.burstOutCounter[ idx ] )
            outBursts.append( burst )
         intfBurst.outBursts = outBursts

         # Append only if there's some data
         if inBursts or outBursts:
            intfBursts.bursts.append( intfBurst )
   intfBursts.lastUpdated = _getLastUpdateTime()
   return intfBursts

# fill clusterInfo for the given cluster and return True if filled otherwise
# return False
def _fillClusterInfo( clusterInfo, cluster ):
   clusterStatus = status.clusterStatus.get( cluster )
      # we dont have status for this cluster so just return
   if not clusterStatus:
      return False
   clusterConfig = config.clusterConfig.get( cluster )
   clusterInfo.operState = clusterStatus.operState
   clusterInfo.numActiveTrackers = clusterStatus.clusterInfo.activeTrackers
   clusterInfo.numDecommissionedNodes = \
       clusterStatus.clusterInfo.decommissionedNodes   
   clusterInfo.status = clusterStatus.clusterInfo.status
   clusterInfo.numMapTasksRunning = clusterStatus.clusterInfo.runningMapTasks
   clusterInfo.numReduceTasksRunning = clusterStatus.clusterInfo.runningReduceTasks

   if clusterConfig.mapReduceVersion == "MR1":
      clusterInfo.numBlacklistedTrackers = \
          clusterStatus.clusterInfo.blackListedTrackers
      clusterInfo.trackerExpiryInterval = \
          clusterStatus.clusterInfo.trackerExpireInterval
      clusterInfo.mapSlots = clusterStatus.clusterInfo.totalMapSlots
      clusterInfo.reduceSlots = clusterStatus.clusterInfo.totalReduceSlots
      clusterInfo.jobTrackerHeapSize = clusterStatus.clusterInfo.jobTrackerHeapSize
      clusterInfo.jobTrackerMaxHeapSize = \
          clusterStatus.clusterInfo.jobTrackerMaxHeapSize
   elif clusterConfig.mapReduceVersion == "MR2":
      clusterInfo.allocatedMemory = clusterStatus.clusterInfo.allocatedMB
      clusterInfo.totalMemory = clusterStatus.clusterInfo.totalMB
   return True

def _fillClusterConfig( clusterConfig, cluster ):
   origConfig = config.clusterConfig.get( cluster )
   clusterConfig.adminStatus = origConfig.enabled
   clusterConfig.mapReduceVersion = origConfig.mapReduceVersion
   clusterConfig.interval = origConfig.interval
   if origConfig.mapReduceVersion == "MR1":
      clusterConfig.host = origConfig.host
      clusterConfig.rpcPort = origConfig.rpcPort
      clusterConfig.user = origConfig.user
      clusterConfig.httpPort = origConfig.httpPort
   elif origConfig.mapReduceVersion == "MR2":
      clusterConfig.host = ""
      clusterConfig.rpcPort = -1
      clusterConfig.user = ""
      clusterConfig.httpPort = -1
      clusterConfig.resourceManagerHost = origConfig.resourceManagerHost
      clusterConfig.resourceManagerPort = origConfig.resourceManagerPort
      clusterConfig.jobHistoryHost = origConfig.jobHistoryHost
      clusterConfig.jobHistoryPort = origConfig.jobHistoryPort
   else:
      # Fill these because these are non-optional
      clusterConfig.host = ""
      clusterConfig.rpcPort = -1
      clusterConfig.user = ""
      clusterConfig.httpPort = -1

def showAllClusters( mode, args ):
   clusters = HadoopTracerModel.Clusters()
   clusters.clusters = {}
   for cluster, origConfig in config.clusterConfig.items():
      clusterConfig = HadoopTracerModel.ClusterConfig()
      clusterInfo = HadoopTracerModel.ClusterInfo()
      clusterInfo.name = cluster
      _fillClusterConfig( clusterConfig, cluster )
      clusterInfo.clusterConfig = clusterConfig
      _fillClusterInfo( clusterInfo, origConfig.name )
      clusters.clusters[ cluster ] = clusterInfo
   clusters.lastUpdated = _getLastUpdateTime()
   return clusters

def showOneCluster( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   if _clusterError( mode, cluster, operInfo=False ):
      return None
   clusterStatus = HadoopTracerModel.ClusterStatus()
   lastUpdated = _getLastUpdateTime( cluster=cluster )
   clusterStatus.lastUpdated = lastUpdated

   clusterInfo = HadoopTracerModel.ClusterInfo()
   clusterInfo.name = cluster
   _fillClusterInfo( clusterInfo, cluster )
   clusterStatus.clusterInfo = clusterInfo
   clusterConfig = HadoopTracerModel.ClusterConfig()
   _fillClusterConfig( clusterConfig, cluster )
   clusterInfo.clusterConfig = clusterConfig

   taskTrackers = HadoopTracerModel.TaskTrackers()
   tts, lastUpdated = _getTaskTrackers( cluster=cluster)
   taskTrackers.lastUpdated = lastUpdated
   taskTrackers.taskTrackers = { tt.name: tt for tt in tts }
   clusterStatus.localTaskTrackers = taskTrackers

   runningJobs = HadoopTracerModel.RunningJobs()
   runningJobs.jobs = _getRunningJobs( cluster=cluster )
   runningJobs.lastUpdated = lastUpdated
   clusterStatus.runningJobs = runningJobs

   origStatus = status.clusterStatus.get( cluster )
   if origStatus and origStatus.error:
      clusterError = HadoopTracerModel.ClusterError()
      clusterError.error = origStatus.error
      clusterError.errorSeenAt = origStatus.lastErrorUpdate
      clusterStatus.clusterError = clusterError

   return clusterStatus

def showClusterJobs( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   if _clusterError( mode, cluster ):
      return None
   clusterRunningJobs = HadoopTracerModel.ClusterRunningJobs()
   clusterRunningJobs.name = cluster
   clusterRunningJobs.jobs = _getRunningJobs( cluster=cluster )
   clusterRunningJobs.lastUpdated = _getLastUpdateTime( cluster=cluster )
   return clusterRunningJobs

def showClusterJob( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   jobId = args[ 'JOB' ]
   if _clusterError( mode, cluster ):
      return None
   # _getRunningJobs will return JobInfo for the job with given jobId
   jobInfo = _getRunningJobs( cluster=cluster, jobId=jobId )
   if jobInfo:
      jobInfo.lastUpdated = _getLastUpdateTime( cluster=cluster )
      return jobInfo
   mode.addError( "Job not found" )
   return None


def showClusterJobCounters( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   jobId = args[ 'JOB' ]
   if _clusterError( mode, cluster ):
      return None
   # _getRunningJobs will return JobInfo for the job with given jobId
   jobInfo = _getRunningJobs( cluster=cluster, jobId=jobId )
   counter = HadoopTracerModel.PerJobPerInterfaceCounters()
   counter.jobInfo = jobInfo
   counter.lastUpdated = _getLastUpdateTime( cluster=cluster )
   return counter

def showClusterCounters( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   if _clusterError( mode, cluster ):
      return None
   counters = HadoopTracerModel.ClusterHadoopCounters()
   counters.jobs = _getRunningJobs( cluster=cluster )
   counters.name = cluster
   counters.lastUpdated = _getLastUpdateTime()
   return counters

def showClusterTaskTrackers( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   if _clusterError( mode, cluster ):
      return None
   taskTrackers = HadoopTracerModel.ClusterTaskTrackers()
   taskTrackers.name = cluster
   tts, lastUpdated = _getTaskTrackers( cluster=cluster )
   taskTrackers.taskTrackers = { tt.name: tt for tt in tts }
   taskTrackers.lastUpdated = lastUpdated
   if 'status' in args:
      # pylint: disable=protected-access
      taskTrackers._printDetail = True
   return taskTrackers

def showClusterHistory( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   if _clusterError( mode, cluster, operInfo=False ):
      return None
   clusterJobHistory = HadoopTracerModel.ClusterJobHistory()
   clusterJobHistory.name = cluster
   clusterJobHistory.jobHistory = _getJobHistory( cluster )
   clusterJobHistory.lastUpdated = _getLastUpdateTime( cluster=cluster )
   return clusterJobHistory

def showClusterHistoricalJob( mode, args ):
   cluster = args[ 'CLUSTER_NAME' ]
   jobId = args[ 'JOB' ]
   if _clusterError( mode, cluster, operInfo=False ):
      return None
   # _getJobHistory will return a dict even when there's jobid given because there
   # could be more than one jobs with that id when JT restarts
   clusterJobHistory = HadoopTracerModel.ClusterJobHistory()
   clusterJobHistory.name = cluster
   history = _getJobHistory( cluster, jobId )
   clusterJobHistory.jobHistory = history
   clusterJobHistory.lastUpdated = _getLastUpdateTime( cluster=cluster )

   # Let the model know that it has to print details of each job
   # pylint: disable=protected-access
   clusterJobHistory._printDetail = True
   return clusterJobHistory

def allTtHostNames( mode ):
   """ Return a collection of all Node names that can be used in auto completion 
   of possible host names for commands that show Nodes for their host names
   """
   ttNames = []
   for clusterStatus in status.clusterStatus.values():
      for ttName in clusterStatus.localTaskTrackerInfo:
         # Make sure there are no duplicates
         if ttName not in ttNames:
            ttNames.append( ttName )
   return ttNames

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   global status
   global cmdRequest

   config = ConfigMount.mount( entityManager, "hadooptracer/config",
                             "HadoopTracer::Config", "w" )
   status = LazyMount.mount( entityManager, "hadooptracer/status",
                             "HadoopTracer::Status", "r" )
   cmdRequest = LazyMount.mount( entityManager, "hadooptracer/cmdrequest",
                              "HadoopTracer::CmdRequest", "w" )

