# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliParserCommon
from CliPlugin import EthIntfCli
from CliPlugin import HadoopTracerCli
from CliPlugin import HadoopTracerModel
from CliPlugin import LagIntfCli
import CliToken.Monitor
from Intf import IntfRange
import ShowCommand

hadoopKwMatcher = CliMatcher.KeywordMatcher( 'hadoop',
                                         helpdesc='Show monitor hadoop Information' )
statusKwMatcher = CliMatcher.KeywordMatcher( 'status', helpdesc='Show Node status' )
countersKwMatcher = CliMatcher.KeywordMatcher( 'counters',
                                          helpdesc='Show counters for running jobs' )
historyKwMatcher = CliMatcher.KeywordMatcher( 'history',
                                                 helpdesc='Show history of job run' )
burstKwMatcher = CliMatcher.KeywordMatcher( 'burst',
                                            helpdesc='Top bursts per Interface' )
trafficKwMatcher = CliMatcher.KeywordMatcher( 'traffic',
                                              helpdesc='Show traffic data' )
nodeKwMatcher = CliMatcher.KeywordMatcher( 'node', helpdesc='Show Nodes',
                                           alternates=[ "tasktracker" ] )
allKwMatcher = CliMatcher.KeywordMatcher( 'all', helpdesc='Show all Nodes' )
hostNameMatcher = CliMatcher.DynamicNameMatcher( HadoopTracerCli.allTtHostNames,
                                   helpdesc='Name of the host',
                                   pattern=r'[\.A-Za-z0-9_-]+' )
clusterNameMatcher = CliMatcher.PatternMatcher( '(?!all$)[0-9A-Za-z_.]+',
                                                helpname='WORD',
                                                helpdesc='Cluster Name' )
jobIdMatcher = CliMatcher.IntegerMatcher( 0, 2**31 - 1, helpdesc='Job Id' )
hostKwMatcher = CliMatcher.KeywordMatcher( 'host',
                                           helpdesc='Show Node with given name' )
clusterKwMatcher = CliMatcher.KeywordMatcher( 'cluster',
                                              helpdesc='Cluster Information' )
jobsKwMatcher = CliMatcher.KeywordMatcher( 'jobs', helpdesc='Show jobs' )
jobKwMatcher = CliMatcher.KeywordMatcher( 'job', helpdesc='Show job' )
runningTasksKwMatcher = CliMatcher.KeywordMatcher( 'running-tasks',
                                              helpdesc='Show running tasks on Node' )
interfaceKwMatcher = CliMatcher.KeywordMatcher( 'interface',
                                            helpdesc='Show Node on given interface' )
intfRangeMatcher = IntfRange.IntfRangeMatcher(
      explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                          LagIntfCli.LagAutoIntfType ) )

#-----------------------------------------------------------------------------------
# show monitor hadoop
#-----------------------------------------------------------------------------------
class ShowMonitorHadoop( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
          }
   cliModel = HadoopTracerModel.RunningJobs
   handler = HadoopTracerCli.showRunningJobs

BasicCli.addShowCommandClass( ShowMonitorHadoop )

#-----------------------------------------------------------------------------------
# show monitor hadoop status
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop status'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'status': statusKwMatcher,
          }
   cliModel = HadoopTracerModel.HadoopStatus
   handler = HadoopTracerCli.showStatus

BasicCli.addShowCommandClass( ShowMonitorHadoopStatus )

#-----------------------------------------------------------------------------------
# show monitor hadoop counters
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop counters'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'counters': countersKwMatcher,
          }
   cliModel = HadoopTracerModel.HadoopCounters
   handler = HadoopTracerCli.showCounters

BasicCli.addShowCommandClass( ShowMonitorHadoopCounters )

#-----------------------------------------------------------------------------------
# show monitor hadoop history
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopHistory( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop history'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'history': historyKwMatcher,
          }
   cliModel = HadoopTracerModel.JobHistory
   handler = HadoopTracerCli.showHistory

BasicCli.addShowCommandClass( ShowMonitorHadoopHistory )

#-----------------------------------------------------------------------------------
# show monitor [ cluster CLUSTER_NAME ] hadoop traffic burst [ interface INTFS ]
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopTrafficBurst( ShowCommand.ShowCliCommandClass ):
   syntax = '''show monitor hadoop [ cluster CLUSTER_NAME ]
               traffic burst [ interface INTFS ]'''
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'traffic': trafficKwMatcher,
            'burst': burstKwMatcher,
            'interface': 'Show burst for Node on given interface',
            'INTFS': intfRangeMatcher,
          }
   cliModel = HadoopTracerModel.InterfaceBursts

   @staticmethod
   def handler( mode, args ):
      return HadoopTracerCli.showClusterBursts( mode,
                                                intfs=args.get( 'INTFS', () ),
                                                cluster=args.get( 'CLUSTER_NAME' ) )

BasicCli.addShowCommandClass( ShowMonitorHadoopTrafficBurst )

#-----------------------------------------------------------------------------------
# show monitor hadoop node all [ status ]
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeAll( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop node all [ status ]'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'all': allKwMatcher,
            'status': statusKwMatcher,
          }
   cliModel = HadoopTracerModel.TaskTrackers
   handler = HadoopTracerCli.showTaskTrackers

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeAll )

#-----------------------------------------------------------------------------------
# show monitor hadoop node all counters
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeAllCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop node all counters'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'all': allKwMatcher,
            'counters': 'Show counters for Nodes',
          }
   cliModel = HadoopTracerModel.TaskTrackerCounters
   handler = HadoopTracerCli.showAllTaskTrackerCounters

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeAllCounters )

#-----------------------------------------------------------------------------------
# show monitor hadoop node host HOST status
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeHostStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop node host HOST status'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'host': hostKwMatcher,
            'HOST': hostNameMatcher,
            'status': statusKwMatcher,
          }
   cliModel = HadoopTracerModel.TaskTracker
   handler = HadoopTracerCli.showTTStatus

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeHostStatus )

#-----------------------------------------------------------------------------------
# show monitor hadoop node ( host HOST ) | ( interface INTFS ) jobs
#                                                            [ cluster CLUSTER_NAME ]
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeHostJobs( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor hadoop node ( host HOST ) | ( interface INTFS ) jobs '
                                                         '[ cluster CLUSTER_NAME ]' )
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'host': hostKwMatcher,
            'HOST': hostNameMatcher,
            'interface': interfaceKwMatcher,
            'INTFS': intfRangeMatcher,
            'jobs': jobsKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
          }
   cliModel = HadoopTracerModel.TaskTrackerRunningJobs

   @staticmethod
   def handler( mode, args ):
      return HadoopTracerCli.showTTJobs( mode,
                                         cluster=args.get( 'CLUSTER_NAME' ),
                                         hostName=args.get( 'HOST' ),
                                         intfs=args.get( 'INTFS', () ) )

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeHostJobs )

#-----------------------------------------------------------------------------------
# show monitor hadoop node ( host HOST ) | ( interface INTFS ) counters
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeHostCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop node ( host HOST ) | ( interface INTFS ) counters'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'host': hostKwMatcher,
            'HOST': hostNameMatcher,
            'interface': interfaceKwMatcher,
            'INTFS': intfRangeMatcher,
            'counters': countersKwMatcher,
          }
   cliModel = HadoopTracerModel.TaskTrackerPerJobCounters

   @staticmethod
   def handler( mode, args ):
      return HadoopTracerCli.showTTCounters( mode,
                                             hostName=args.get( 'HOST' ),
                                             intfs=args.get( 'INTFS', () ) )

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeHostCounters )

#-----------------------------------------------------------------------------------
# show monitor hadoop node host HOST running-tasks
#                                               [ cluster CLUSTER_NAME [ job JOB  ] ]
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopNodeHostRunningTasks( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor hadoop node ( host HOST ) | ( interface INTFS ) '
               'running-tasks [ cluster CLUSTER_NAME [ job JOB ] ]' )
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'host': hostKwMatcher,
            'HOST': hostNameMatcher,
            'interface': interfaceKwMatcher,
            'INTFS': intfRangeMatcher,
            'running-tasks': runningTasksKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'job': jobKwMatcher,
            'JOB': jobIdMatcher,
          }
   cliModel = HadoopTracerModel.TaskTrackerRunningTasks

   @staticmethod
   def handler( mode, args ):
      return HadoopTracerCli.showTTTasks( mode,
                                          intfs=args.get( 'INTFS', () ),
                                          hostName=args.get( 'HOST' ),
                                          cluster=args.get( 'CLUSTER_NAME' ),
                                          jobId=args.get( 'JOB' ) )

BasicCli.addShowCommandClass( ShowMonitorHadoopNodeHostRunningTasks )

#-----------------------------------------------------------------------------------
# show monitor hadoop node ( host HOST ) | ( interface INTFS ) running-tasks
#                                             cluster CLUSTER_NAME job JOB task TASK
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopTaskReport( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show monitor hadoop node ( host HOST ) | ( interface INTFS ) '
              'running-tasks cluster CLUSTER_NAME job JOB task TASK' )
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'node': nodeKwMatcher,
            'host': hostKwMatcher,
            'HOST': hostNameMatcher,
            'interface': interfaceKwMatcher,
            'INTFS': intfRangeMatcher,
            'running-tasks': runningTasksKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'job': jobKwMatcher,
            'JOB': jobIdMatcher,
            'task': 'Task id',
            'TASK': CliMatcher.IntegerMatcher( 0, 2**31 - 1, helpdesc='Task Id' ),
          }
   cliModel = HadoopTracerModel.TaskReport

   @staticmethod
   def handler( mode, args ):
      return HadoopTracerCli.showOneTask( mode,
                                          cluster=args[ 'CLUSTER_NAME' ],
                                          jobId=args[ 'JOB' ],
                                          taskId=args[ 'TASK' ],
                                          intfs=args.get( 'INTFS', () ),
                                          hostName=args.get( 'HOST' ) )

BasicCli.addShowCommandClass( ShowMonitorHadoopTaskReport )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster all
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterAll( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster all'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'all': CliMatcher.KeywordMatcher( 'all', helpdesc='Show all clusters',
                                              priority=CliParserCommon.PRIO_HIGH ),
          }
   cliModel = HadoopTracerModel.Clusters
   handler = HadoopTracerCli.showAllClusters

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterAll )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME status
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME status'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'status': statusKwMatcher,
          }
   cliModel = HadoopTracerModel.ClusterStatus
   handler = HadoopTracerCli.showOneCluster

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterStatus )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME counters
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME counters'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'counters': countersKwMatcher,
          }
   cliModel = HadoopTracerModel.HadoopCounters
   handler = HadoopTracerCli.showClusterCounters

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterCounters )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME node [ status ]
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterNode( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME node [ status ]'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'node': nodeKwMatcher,
            'status': statusKwMatcher,
          }
   cliModel = HadoopTracerModel.ClusterTaskTrackers
   handler = HadoopTracerCli.showClusterTaskTrackers

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterNode )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME jobs
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterJobs( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME jobs'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'jobs': jobsKwMatcher,
          }
   cliModel = HadoopTracerModel.ClusterRunningJobs
   handler = HadoopTracerCli.showClusterJobs

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterJobs )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME jobs JOB
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterJob( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME jobs JOB'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'jobs': jobsKwMatcher,
            'JOB': jobIdMatcher,
          }
   cliModel = HadoopTracerModel.JobInfo
   handler = HadoopTracerCli.showClusterJob

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterJob )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME jobs JOB counters
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterJobsCounter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME jobs JOB counters'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'jobs': jobsKwMatcher,
            'JOB': jobIdMatcher,
            'counters': countersKwMatcher,
          }
   cliModel = HadoopTracerModel.PerJobPerInterfaceCounters
   handler = HadoopTracerCli.showClusterJobCounters

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterJobsCounter )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME history
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterHistory( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME history'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'history': historyKwMatcher,
          }
   cliModel = HadoopTracerModel.ClusterJobHistory
   handler = HadoopTracerCli.showClusterHistory

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterHistory )

#-----------------------------------------------------------------------------------
# show monitor hadoop cluster CLUSTER_NAME history job JOB
#-----------------------------------------------------------------------------------
class ShowMonitorHadoopClusterHistoryJob( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop cluster CLUSTER_NAME history job JOB'
   data = {
            'monitor': CliToken.Monitor.monitorMatcherForShow,
            'hadoop': hadoopKwMatcher,
            'cluster': clusterKwMatcher,
            'CLUSTER_NAME': clusterNameMatcher,
            'history': historyKwMatcher,
            'job': jobKwMatcher,
            'JOB': jobIdMatcher,
          }
   cliModel = HadoopTracerModel.ClusterJobHistory
   handler = HadoopTracerCli.showClusterHistoricalJob

BasicCli.addShowCommandClass( ShowMonitorHadoopClusterHistoryJob )

#--------------------------------------------------------------------------------
# show monitor hadoop node interface INTFS status
#--------------------------------------------------------------------------------
class MonitorHadoopNodeInterfaceIntfStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor hadoop node interface INTFS status'
   data = {
      'monitor': CliToken.Monitor.monitorMatcherForShow,
      'hadoop': hadoopKwMatcher,
      'node': nodeKwMatcher,
      'interface': interfaceKwMatcher,
      'INTFS': intfRangeMatcher,
      'status': statusKwMatcher,
   }
   handler = HadoopTracerCli.showIntfTTStatus
   cliModel = HadoopTracerModel.InterfaceTaskTrackers

BasicCli.addShowCommandClass( MonitorHadoopNodeInterfaceIntfStatusCmd )
