# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import ConfigMount
from CliPlugin import FruCli
from CliPlugin import HardwareLedMode as LedMode
from CliPlugin import LedCommon
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import LazyMount
import Tac
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Model
import Intf.IntfRange
import ShowCommand

#------------------------------------------------------------------------------------
# Constants declaration
#------------------------------------------------------------------------------------
SET_NAME = 'HardwareLedCliSet'
RAMP_RATE = 0.0104
MAX_BRIGHTNESS = 100

entityMib = None
hwLedConfig = None
ledConfig = None
plutoStatus = Tac.Type( 'Led::PlutoStatus' )

#------------------------------------------------------------------------------------
# Model classes for show command
#------------------------------------------------------------------------------------
class Led( Model ):
   color = Enum( values=( 'green', 'yellow', 'red', 'blue', 'off' ),
                 help='Color of LED' )
   flash = Bool( help='True if this LED flashes on and off' )

class LedModel( Model ):
   leds = Dict( keyType=str, valueType=Led,
                help='All LEDs; key is the display name of the LED' )
   def render( self ):
      print( 'User configured LEDs:' )
      print( '---------------------' )
      for led in sorted( self.leds ):
         # pylint: disable-next=consider-using-f-string
         print( '{} (color:{}{})'.format( led, self.leds[ led ].color,
                                     ' flashing' if self.leds[
                                           led ].flash else '' ) )
      print( '' )

class ChassisStatusMatcher( CliMatcher.Matcher ):
   '''
   Matches all linecards/fabrics/supervisors on modular systems.
   Matches the status on fixed.
   '''
   def __init__( self, **kwargs ):
      super().__init__( helpdesc='', **kwargs )

   def match( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return CliMatcher.noMatch
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         if token == 'Status1':
            return CliMatcher.MatchResult( 'Status1', 'Status1' )
      elif root.tacType.fullTypeName == 'EntityMib::Chassis':
         for slot in root.cardSlot.values():
            if f'{slot.tag}{slot.label}' == token:
               result = f'{slot.tag}{slot.label}'
               return CliMatcher.MatchResult( result, result )
      return CliMatcher.noMatch

   def completions( self, mode, context, token ):
      root = mode.entityManager.lookup( 'hardware/entmib' ).root
      if root is None:
         return []
      if root.tacType.fullTypeName == 'EntityMib::FixedSystem':
         return [ CliParser.Completion( 'Status1', 'Chassis Status LED' ) ]
      elif root.tacType.fullTypeName == "EntityMib::Chassis":
         completions = []
         for slot in root.cardSlot.values():
            # pylint: disable-next=consider-using-f-string
            completions.append( CliParser.Completion( '{}{}'.format( slot.tag,
                                                                 slot.label ), '' ) )
         return completions
      return []

#------------------------------------------------------------------------------------
# Helper functions
#------------------------------------------------------------------------------------
def setLed( mode, led=None, color=None, disable=None, flashing=None ):
   '''
   Setting the configuration for one LED
   '''
   ledSet = LedCommon.getLedSet( SET_NAME, ledConfig )

   # In case of Supervisors the LEDs are called 'Status1' and 'Status2'
   led = led.replace( 'Supervisor', 'Status' )

   if disable:
      if led in ledSet.led:
         del ledSet.led[ led ]
         return
      print( 'LED already not user configured!' )
      return

   if not color:
      print( 'Please enter a color!' )
      return

   ledSetting = Tac.Value( 'Led::LightSetting' )
   ledSetting.rampRate = RAMP_RATE
   ledSetting.maxBright = MAX_BRIGHTNESS
   ledSetting.flashRate = 1 if flashing else 0
   if flashing:
      ledSetting.plutoStatus = plutoStatus.plutoStatusBeacon
      if color == 'off':
         print( 'Cannot be flashing and be set to off!' )
         return
   elif color == 'green':
      ledSetting.plutoStatus = plutoStatus.plutoStatusGood
   elif color == 'yellow':
      ledSetting.plutoStatus = plutoStatus.plutoStatusInactive
   elif color == 'red':
      ledSetting.plutoStatus = plutoStatus.plutoStatusBad
   elif color == 'off':
      ledSetting.plutoStatus = plutoStatus.plutoStatusOff
   else:
      ledSetting.plutoStatus = plutoStatus.plutoStatusUnknown
   if color != 'off' and color != 'blue': # pylint: disable=consider-using-in
      setattr( ledSetting, color, True )
   if color == 'blue':
      setattr( ledSetting, color, True )
      if not getattr( hwLedConfig.leds.get( led ), 'ledAttribute', False ) \
         or not getattr( hwLedConfig.leds.get( led ).ledAttribute, 'blue', False ):
         print( 'WARNING: This LED does not support blue!' )
   if color in [ 'red', 'yellow' ]:
      ledAttribute = getattr( hwLedConfig.leds.get( led ), 'ledAttribute', False )
      if not ledAttribute or \
         ( not getattr( ledAttribute, 'red', False ) or \
           not getattr( ledAttribute, 'yellow', False ) ):
         print( 'WARNING: This LED may display red/yellow inaccurately!' )


   ledSet.led[ led ] = ledSetting

#--------------------------------------------------------------------------------
# [ no | default ] ( ( chassis ( CHASSIS | LED ) ) |
#                    ( fan-tray ( FAN_TRAY | LED ) ) |
#                    ( power ( POWER_SUPPLY | LED ) ) |
#                    ( interface ( INTFS | LED ) ) ) color COLOR [ flashing ]
#--------------------------------------------------------------------------------
class LedCmd( CliCommand.CliCommandClass ):
   syntax = ( '( ( chassis ( CHASSIS | LED ) ) | '
                '( fan-tray ( FAN_TRAY | LED ) ) | '
                '( power ( POWER_SUPPLY | LED ) ) | '
                '( interface ( INTFS | LED ) ) ) color COLOR [ flashing ]' )
   noOrDefaultSyntax = ( '( chassis CHASSIS ) |'
                         '( fan-tray FAN_TRAY ) |'
                         '( power POWER_SUPPLY ) |'
                         '( interface INTFS ) |'
                         '( [ interface | chassis | fan-tray | power ] LED )' )
   data = {
      'chassis' : 'Chassis LEDs',
      'CHASSIS' : ChassisStatusMatcher(),
      'fan-tray' : 'Fan-tray LEDs',
      'FAN_TRAY' : FruCli.FanTrayMatcher(),
      'power' : 'Power supply LEDs',
      'POWER_SUPPLY' : FruCli.PowerSupplyMatcher(),
      'interface' : 'Interface LEDs',
      'INTFS' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'LED' : CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern='.+',
            helpdesc='LED target', helpname='WORD',
            priority=CliParser.PRIO_LOW ),
         hidden=True # NOTE: See BUG254320 for more info
      ),
      'color' : 'LED color',
      'COLOR' : LedCommon.colorEnumMatcher,
      'flashing' : 'Flash this LED',
   }

   @staticmethod
   def handler( mode, args ):
      color = args.get( 'COLOR' )
      flashing = 'flashing' in args
      disable = CliCommand.isNoOrDefaultCmd( args )

      if chassis := args.get( 'CHASSIS' ):
         setLed( mode, led=chassis, color=color, disable=disable, flashing=flashing )
      elif fan := args.get( 'FAN_TRAY' ):
         # FanTrayMatcher returns the fan led name, we can just set it directly
         setLed( mode, fan, color, disable, flashing )
      elif ps := args.get( 'POWER_SUPPLY' ):
         # Need to get a sorted list for the powerSupplyRule to act correctly.
         powerLeds = [ power for power in hwLedConfig.leds
                       if power.startswith( 'Power' ) ]
         powerLeds = sorted( powerLeds )
         if len( powerLeds ) >= int( ps.label ):
            setLed( mode, powerLeds[ int( ps.label ) - 1 ], color, disable,
                  flashing )
         else:
            setLed( mode, ps.label, color, disable, flashing )
      elif intfs := args.get( 'INTFS' ):
         for intf in intfs:
            setLed( mode, intf, color, disable, flashing )
      elif led := args.get( 'LED' ):
         setLed( mode, led=led, color=color, disable=disable, flashing=flashing )
      else:
         assert disable, 'Unknown type'
         # Called via parent mode's `no/default`.
         if config := ledConfig.get( SET_NAME ):
            config.led.clear()

   noOrDefaultHandler = handler

LedMode.LedMode.addCommandClass( LedCmd )

#--------------------------------------------------------------------------------
# show led
#--------------------------------------------------------------------------------
def ledToShowColor( led, ledSet ):
   for color in [ 'green', 'yellow', 'red', 'blue' ]:
      if getattr( ledSet.led[ led ], color ):
         return color
   return 'off'

class ShowLedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led'
   data = {
      'led' : 'User configured LED',
   }
   cliModel = LedModel

   @staticmethod
   def handler( mode, args ):
      ledSet = LedCommon.getLedSet( SET_NAME, ledConfig )
      ret = LedModel()
      for led in ledSet.led:
         ledC = Led()
         ledC.color = ledToShowColor( led, ledSet )
         ledC.flash = ledSet.led[ led ].flashRate == 1
         ret.leds[ led ] = ledC
      return ret

LedMode.LedMode.addShowCommandClass( ShowLedCmd )

#------------------------------------------------------------------------------------
# Plugin initialization
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global entityMib, hwLedConfig, ledConfig
   
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
   hwLedConfig = LazyMount.mount( entityManager, 'hardware/led/configInit',
                                  'Hardware::Led::LedSystemConfigDir', 'r' )
   ledConfig = ConfigMount.mount( entityManager, 'led/config', 'Tac::Dir', 'w' )
