# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliMatcher
from CliPlugin.HealthMonitorCliModels import ( HealthMonitorStatusEntry,
                                     HealthMonitorProbeStatus,
                                     HealthMonitorStatus,
                                     HealthMonitorSummary,
                                     HealthMonitorCounters,
                                     HealthMonitorTimestamps,
                                     HealthMonitorProbeDelay,
                                     HealthMonitorStatusDetail
                                     )
import LazyMount
import ShowCommand
import CliModel
from AgentCommandRequest import ( RunSocketCommandException,
                                  runSocketCommand,
                                  runCliPrintSocketCommand )
import AgentDirectory

_hmServerConfig = None
_hmServerStatus = None
entityManager = None

class ShowHealthMonitorStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor status"
   data = {
         "health-monitor": "Show health-monitor information",
         "status": "Show health-monitor status",
   }

   privileged = True
   cliModel = HealthMonitorStatus

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorStatus()

      for nhgName, nhgProbeStatus in _hmServerStatus.nhgProbeStatus.items():
         model.probeStatus[ nhgName ] = HealthMonitorProbeStatus()
         for serverIp, status in nhgProbeStatus.serverProbeStatus.items():
            hmStatus = HealthMonitorStatusEntry()
            hmStatus.adminState = status.adminState
            hmStatus.operState = status.operState
            hmStatus.weight = status.weight
            model.probeStatus[ nhgName ].serverStatus[ serverIp.stringValue ] = \
                  hmStatus

      return model

BasicCli.addShowCommandClass( ShowHealthMonitorStatusCmd )

class ShowHealthMonitorStatusDetailCmd ( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor status detail"
   data = {
         "health-monitor": "Show health-monitor information",
         "status": "Show health-monitor status",
         "detail": "Show health-monitor status detail",
   }

   privileged = True
   cliModel = HealthMonitorStatusDetail

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorStatusDetail
      command = "show health-monitor status detail"
      runCliPrintSocketCommand( mode.entityManager, "HealthMonitor",
            "HmCliF", command, mode,
            asyncCommand=True )
      return CliModel.cliPrinted( model )

BasicCli.addShowCommandClass( ShowHealthMonitorStatusDetailCmd )

class ShowHealthMonitorSummaryCmd ( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor summary"
   data = {
         "health-monitor": "Show health-monitor information",
         "summary": "Show health-monitor summary",
   }

   privileged = True
   cliModel = HealthMonitorSummary

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorSummary
      command = "show health-monitor summary"
      if not AgentDirectory.agent( mode.entityManager.sysname(), "HealthMonitor" ):
         mode.addError( "HealthMonitor agent is not running" )
         return None
      try:
         outputFormat = mode.session_.outputFormat_
         errors = [ "HealthMonitor is not initialized\n" ]
         runSocketCommand( mode.entityManager, "HealthMonitor",
               "HmCliF", command, stringBuff=None, outputFormat=outputFormat,
               throwException=True, errors=errors )
         return CliModel.cliPrinted( model )
      except RunSocketCommandException as e:
         mode.addError( str( e ) )
      return None

BasicCli.addShowCommandClass( ShowHealthMonitorSummaryCmd )

probeIntervalRule = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc="Probe interval in seconds" )

class ShowHealthMonitorProbeDelayCmd ( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor probe delay [ interval INTERVAL ]"
   data = {
         "health-monitor": "Show health-monitor information",
         "probe": "Show health-monitor probe information",
         "delay": "Show health-monitor probe delay",
         "interval": "Show probe delay for interval",
         "INTERVAL": probeIntervalRule,
   }
   privileged = True
   cliModel = HealthMonitorProbeDelay

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorProbeDelay
      probeInterval = args.get( "INTERVAL" )
      command = "show health-monitor probe delay"
      if probeInterval:
         command += " interval %d" % probeInterval
      if not AgentDirectory.agent( mode.entityManager.sysname(), "HealthMonitor" ):
         mode.addError( "HealthMonitor agent is not running" )
         return None
      try:
         outputFormat = mode.session_.outputFormat_
         errors = [ "HealthMonitor is not initialized\n" ]
         runSocketCommand( mode.entityManager, "HealthMonitor",
               "HmCliF", command, stringBuff=None, outputFormat=outputFormat,
               throwException=True, errors=errors )
         return CliModel.cliPrinted( model )
      except RunSocketCommandException as e:
         mode.addError( str( e ) )
      return None

BasicCli.addShowCommandClass( ShowHealthMonitorProbeDelayCmd )

class ShowHealthMonitorCountersCmd ( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor counters"
   data = {
         "health-monitor": "Show health-monitor information",
         "counters": "Show health-monitor counters",
   }

   privileged = True
   cliModel = HealthMonitorCounters

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorCounters
      command = "show health-monitor counters"
      runCliPrintSocketCommand( mode.entityManager, "HealthMonitor",
            "HmCliF", command, mode,
            asyncCommand=True )
      return CliModel.cliPrinted( model )

BasicCli.addShowCommandClass( ShowHealthMonitorCountersCmd )

class ShowHealthMonitorTimestampsCmd ( ShowCommand.ShowCliCommandClass ):
   syntax = "show health-monitor timestamps"
   data = {
         "health-monitor": "Show health-monitor information",
         "timestamps": "Show health-monitor timestamps",
   }

   privileged = True
   cliModel = HealthMonitorTimestamps

   @staticmethod
   def handler( mode, args ):
      model = HealthMonitorTimestamps
      command = "show health-monitor timestamps"
      runCliPrintSocketCommand( mode.entityManager, "HealthMonitor",
            "HmCliF", command, mode,
            asyncCommand=True )
      return CliModel.cliPrinted( model )

BasicCli.addShowCommandClass( ShowHealthMonitorTimestampsCmd )

def Plugin( em ):
   global _hmServerConfig, _hmServerStatus

   _hmServerConfig = LazyMount.mount( em,
      'healthMonitor/config/serverConfig', 'HmState::ServerConfig', 'r' )
   _hmServerStatus = LazyMount.mount( em,
      'healthMonitor/status/serverStatus', 'HmState::ServerStatus', 'r' )
