# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
import Tracing
import TableOutput
import ArnetModel
from IntfModels import Interface
import Toggles.HostInjectToggleLib
enableHostInjectPortChannelandEthernet = \
       Toggles.HostInjectToggleLib.toggleEthernetAndPortchannelHostInjectEnabled()

__defaultTraceHandle__ = Tracing.Handle( 'HostInjectCliModels' )

#-----------------------------------------------------------------------------------
#  Models used for "show ip attached-hosts route export [ vlan < vlanId > ]"
#-----------------------------------------------------------------------------------

class AttachedHostEntry( CliModel.Model ):
   address = ArnetModel.IpGenericAddress( help="IP Address" )
   interface = Interface( help="Interface on which host is attached to" )
   prefix = ArnetModel.IpGenericPrefix( help="Injected prefix" )

class AttachedHostList( CliModel.Model ):
   hosts = CliModel.List( help="List of all the injected hosts",
                          valueType=AttachedHostEntry )
   def setAttrFromData( self, data ):
      ahe = AttachedHostEntry()
      setattr( ahe, 'address', data[ 'ipAddr' ] )
      setattr( ahe, 'interface', data[ 'interface' ] )
      setattr( ahe, 'prefix', data[ 'prefix' ] )
      self.hosts.append( ahe )

class AttachedHostVlan( CliModel.Model ):
   vlans = CliModel.Dict( valueType=AttachedHostList, keyType=str,
         help="A mapping of vlan to hosts information" )

class AttachedHostPrefix( CliModel.Model ):
   address = ArnetModel.IpGenericAddress( help="IP Address" )
   # l2Interface is available for Vlan Only
   l2Interface = Interface( help="L2 interface on which host is learnt on",
         optional=True )
   prefix = ArnetModel.IpGenericPrefix(
         help="Prefix added corresponding to attached host routes" )

class AttachedHostInterfaceEntry( CliModel.Model ):
   hosts = CliModel.List( help="List of all hosts learnt on the L3 interface",
                          valueType=AttachedHostPrefix )

   def setAttrFromData( self, data ):
      ahe = AttachedHostPrefix()
      setattr( ahe, 'address', data[ 'ipAddr' ] )
      if 'l2Interface' in data:
         setattr( ahe, 'l2Interface', data[ 'l2Interface' ] )
      setattr( ahe, 'prefix', data[ 'prefix' ] )
      self.hosts.append( ahe )

class AttachedHostInterfaces( CliModel.Model ):
   l3Interfaces = CliModel.Dict( valueType=AttachedHostInterfaceEntry, keyType=str,
         help="A dictionary of L3 interface to hosts learnt" )
   
class AttachedHostModel( CliModel.Model ):
   if enableHostInjectPortChannelandEthernet:
      __revision__ = 3
      vrfs = CliModel.Dict( valueType=AttachedHostInterfaces, keyType=str,
            help="A dictionary of VRF to corresponding L3 interfaces" )
   else:
      __revision__ = 2
      vrfs = CliModel.Dict( valueType=AttachedHostVlan, keyType=str,
            help="A mapping of VRF to vlan information" )

   def setAttachedHosts( self, data ):
      for item in data:
         vrf = item[ 'vrf' ]
         if enableHostInjectPortChannelandEthernet:
            ahVrf = self.vrfs[ vrf ] if vrf in self.vrfs \
                  else AttachedHostInterfaces()
            self.vrfs[ vrf ] = ahVrf
            vId = item[ 'l3Interface' ]
            ahVlan = ahVrf.l3Interfaces[ vId ] if \
                  vId in ahVrf.l3Interfaces else AttachedHostInterfaceEntry()
            ahVlan.setAttrFromData( item )
            ahVrf.l3Interfaces[ vId ] = ahVlan
         else:
            ahVrf = self.vrfs[ vrf ] if vrf in self.vrfs else AttachedHostVlan()
            self.vrfs[ vrf ] = ahVrf
            vId = item[ 'vlan' ]
            ahVlan = ahVrf.vlans[ vId ] if vId in ahVrf.vlans else AttachedHostList()
            ahVlan.setAttrFromData( item )
            ahVrf.vlans[ vId ] = ahVlan

   def render( self ):
      header = [ 'IP Address', 'VLAN', "Interface", "Prefix", "VRF" ]
      if enableHostInjectPortChannelandEthernet:
         header = [ 'IP Address', 'L3 Interface', 'L2 Interface', 'Prefix', 'VRF' ]
      table = TableOutput.createTable( ( h, 'l' ) for h in header )

      if not enableHostInjectPortChannelandEthernet:
         for vrf, aheVrf in self.vrfs.items():
            for vlan, aheList in aheVrf.vlans.items():
               for ahe in aheList.hosts:
                  table.newRow( ahe.address, vlan, ahe.interface.stringValue,
                                ahe.prefix.stringValue, vrf )
      else:
         for vrf, aheVrf in self.vrfs.items():
            for vlan, aheList in aheVrf.l3Interfaces.items():
               for ahe in aheList.hosts:
                  l2Interface = \
                        ahe.l2Interface.stringValue if ahe.l2Interface else "n/a"
                  table.newRow( ahe.address, vlan, l2Interface,
                                ahe.prefix.stringValue, vrf )
         fTable = TableOutput.Format( justify="left" )
         fTable.noPadLeftIs( True )
         fTable.padLimitIs( True )
         table.formatColumns( fTable, fTable,
               fTable, fTable, fTable )

      print( table.output() )

   def degrade( self, dictRepr, revision ):
      # The new revision is introduced to accomodate the vrf changes.
      if not enableHostInjectPortChannelandEthernet:
         if revision == 1:
            # this newDict will be populated from dictRepr to represent revision 1.
            dictRepr = self.degradeFromVerision2To1( dictRepr )
      else:
         if revision == 1:
            # this newDict will be populated from dictRepr to represent revision 1.
            dictRepr = self.degradeFromVerision3To1( dictRepr )
         elif revision == 2:
            # this newDict will be populated from dictRepr to represent revision 2.
            dictRepr = self.degradeFromVerision3To2( dictRepr )
      return dictRepr

   def degradeFromVerision2To1( self, dictRepr ):
      newDict = {}
      newDict[ "attachedHostEntries" ] = {} # default entry
      allVlans = {}
      for vrf in dictRepr[ "vrfs" ]:
         if vrf != "default":
            continue
         for vlan in dictRepr[ "vrfs" ][ vrf ][ "vlans" ]:
            allVlans[ vlan ] = {}
            hostList = dictRepr[ "vrfs" ][ vrf ][ "vlans" ][ vlan ][ "hosts" ]
            allVlans[ vlan ][ "attachedHosts" ] = hostList
      newDict[ "attachedHostEntries" ] = allVlans
      return newDict

   def degradeFromVerision3To2( self, dictRepr ):
      newDict = {}
      allVlans = {}
      newDict[ "vrfs" ] = {} # default entry
      for vrf in dictRepr[ "vrfs" ]:
         allVlans[ vrf ] = {}
         allVlans[ vrf ][ "vlans" ] = {}
         for intfId in dictRepr[ "vrfs" ][ vrf ][ "l3Interfaces" ]:
            if "Vlan" not in intfId:
               continue
            vlan = int( intfId[ 4 : ] )
            allVlans[ vrf ][ "vlans" ][ vlan ] = {}
            hostList = dictRepr[ "vrfs" ][ vrf ][ "l3Interfaces" ]
            hostList = hostList[ intfId ][ "hosts" ]
            modifiedHostList = []
            for host in hostList:
               modifiedHost = {}
               modifiedHost[ "address" ] = host[ "address" ]
               modifiedHost[ "interface" ] = host[ "l2Interface" ]
               modifiedHost[ "prefix" ] = host[ "prefix" ]
               modifiedHostList.append( modifiedHost )
            allVlans[ vrf ][ "vlans" ][ vlan ] = modifiedHostList
         newDict[ "vrfs" ] = allVlans
      return newDict

   def degradeFromVerision3To1( self, dictRepr ):
      newDict = {}
      newDict[ "attachedHostEntries" ] = {} # default entry
      allVlans = {}
      for vrf in dictRepr[ "vrfs" ]:
         if vrf != "default":
            continue
         for intfId in dictRepr[ "vrfs" ][ vrf ][ "l3Interfaces" ]:
            if "Vlan" not in intfId:
               continue
            vlan = int( intfId[ 4 : ] )
            allVlans[ vlan ] = {}
            hostList = dictRepr[ "vrfs" ][ vrf ][ "l3Interfaces" ]
            hostList = hostList[ intfId ][ "hosts" ]
            modifiedHostList = []
            for host in hostList:
               modifiedHost = {}
               modifiedHost[ "address" ] = host[ "address" ]
               modifiedHost[ "interface" ] = host[ "l2Interface" ]
               modifiedHost[ "prefix" ] = host[ "prefix" ]
               modifiedHostList.append( modifiedHost )
            allVlans[ vlan ][ "attachedHosts" ] = modifiedHostList
      newDict[ "attachedHostEntries" ] = allVlans
      return newDict

