# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from HttpServiceConstants import ServerConstants
import CliCommand
import CliMatcher
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import ConfigMgmtMode
from CliPlugin import HttpServiceModel
from CliPlugin import TechSupportCli
from CliPlugin.VrfCli import VrfNameExprFactory
import CliGlobal
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import CliToken
import DefaultSslLib


# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig = None,
      serverStatus = None,
      sslConfig = None,
      sslStatus = None,
   )
)

def getHttpServices( mode ):
   serviceDict = {}
   for service in gv.capiConfig.service:
      # pylint: disable-next=consider-using-f-string
      serviceDict[ service ] = "Configure %s service settings" % service
   return serviceDict


capiConstants = Tac.Type( "HttpService::Constants" )
protocolHelpdesc = 'Configure server options'
httpHelpdesc = 'Configure HTTP server options'
httpsHelpdesc = 'Configure HTTPs server options'
sslHelpdesc = 'Configure SSL options'
profileHelpdesc = 'Configure SSL profile'
vrfHelpdesc = 'Configure server options on a VRF'
vrfShutdownHelpdesc = 'Disable server access in this VRF'
portHelpdesc = 'Specify the TCP port to serve on'
httpServerShowHelpDesc = 'Show Http Server VRF configuration details'
aclServiceName = ServerConstants.serviceName

#
# Management config mode for http server
#
class HttpServerConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "HTTP server configuration"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "http-server" )

   def enterCmd( self ):
      return "management http-server"

#-------------------------------------------------------------------------------
# The "[no|default] management http-server" mode command.
#-------------------------------------------------------------------------------
class EnterHttpServerMode( CliCommand.CliCommandClass ):
   syntax = '''management http-server'''
   noOrDefaultSyntax = '''management http-server'''

   data = { 'management': ConfigMgmtMode.managementKwMatcher,
            'http-server': 'Configure the HTTP server' }

   handler = "HttpServiceHandler.enterHttpServerModeHandler"
   noOrDefaultHandler = "HttpServiceHandler.enterHttpServerModeNoOrDefaultHandler"

BasicCli.GlobalConfigMode.addCommandClass( EnterHttpServerMode )

class EnterVrfMode( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF'''
   noOrDefaultSyntax = '''vrf [ VRF ]'''
   data = { 'vrf': vrfHelpdesc,
            'VRF': VrfNameExprFactory( inclDefaultVrf=True ) }

   handler = "HttpServiceHandler.enterVrfConfigHandler"
   noOrDefaultHandler = "HttpServiceHandler.enterVrfConfigNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( EnterVrfMode )

#-------------------------------------------------------------------------------
# The "[no|default] protocol ((http [localhost])|https) [port <number>]" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
class HttpProtocolCmd( CliCommand.CliCommandClass ):
   syntax = """protocol ((http[localhost])|https) [port <number>]"""
   noOrDefaultSyntax = """protocol ((http[localhost])|https) [port [<number>]]"""

   data = { "protocol": protocolHelpdesc,
            "http": httpHelpdesc,
            "https": httpsHelpdesc,
            "localhost": "Server bound on localhost",
            "port": portHelpdesc,
            "<number>": CliMatcher.IntegerMatcher( capiConstants.minPort,
                                                   capiConstants.maxPort,
                                                   helpdesc="TCP port" )
          }

   handler = "HttpServiceHandler.httpProtocolCmdHandler"
   noOrDefaultHandler = "HttpServiceHandler.httpProtocolCmdNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( HttpProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol unix-socket" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
class UnixProtocolCmd( CliCommand.CliCommandClass ):
   syntax = """protocol unix-socket"""
   noOrDefaultSyntax = """protocol unix-socket"""

   data = { "protocol": protocolHelpdesc,
            "unix-socket": "Configure Unix Domain Socket" }

   handler = "HttpServiceHandler.unixProtocolCmdHandler"
   noHandler = "HttpServiceHandler.unixProtocolCmdNoHandler"
   defaultHandler = "HttpServiceHandler.unixProtocolCmdDefaultHandler"

HttpServerConfigMode.addCommandClass( UnixProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol https ssl profile" command,
# in "management http-server" mode.
#-------------------------------------------------------------------------------
profileNameRule = CliMatcher.DynamicNameMatcher(
      lambda mode: gv.sslConfig.profileConfig,
      'Profile name' )

class HttpsProfile( CliCommand.CliCommandClass ):
   syntax = """protocol https ssl profile <profileName>"""
   noOrDefaultSyntax = """protocol https ssl profile ..."""

   data = { "protocol": protocolHelpdesc,
            "https": httpsHelpdesc,
            "ssl": sslHelpdesc,
            "profile": profileHelpdesc,
            "<profileName>": profileNameRule }

   handler = "HttpServiceHandler.httpsProfileHandler"
   noOrDefaultHandler = "HttpServiceHandler.httpsProfileNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( HttpsProfile )

#-------------------------------------------------------------------------------
# The "[ no | default ] qos dscp <dscpValue>" command
#-------------------------------------------------------------------------------
singleDscpValueMatcher = CliMatcher.IntegerMatcher( 0, 63,
                                            helpdesc='DSCP value between 0 and 63' )

class QosDscp( CliCommand.CliCommandClass ):
   syntax = """qos dscp <dscpValue>"""
   noOrDefaultSyntax = """qos dscp [<dscpValue>]"""

   data = { "qos": "Configure QoS parameters",
            "dscp": "Set the DSCP value for eAPI",
            "<dscpValue>": singleDscpValueMatcher }

   handler = "HttpServiceHandler.qosDscpHandler"
   noOrDefaultHandler = "HttpServiceHandler.qosDscpNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( QosDscp )

#-------------------------------------------------------------------------------
# The "[ no | default ] log-level <severity>" command
#-------------------------------------------------------------------------------
logLevels = Tac.Type( 'HttpService::LogLevel' )
nginxLogLevels = { s: s.upper() + ' log level'
                   for s in logLevels.attributes if s != logLevels.none }

class NginxSyslog( CliCommand.CliCommandClass ):
   syntax = """log-level <severity>"""
   noOrDefaultSyntax = """log-level [<severity>]"""

   data = { "log-level": "Configure nginx logging level",
            "<severity>": CliMatcher.DynamicKeywordMatcher(
                                                      lambda mode: nginxLogLevels ) }

   handler = "HttpServiceHandler.nginxSyslogHandler"
   noOrDefaultHandler = "HttpServiceHandler.nginxSyslogNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( NginxSyslog )

#-------------------------------------------------------------------------------
# The "[ no | default ] default-services" command
#-------------------------------------------------------------------------------
class DefaultServices( CliCommand.CliCommandClass ):
   syntax = """default-services"""
   noSyntax = """default-services"""
   defaultSyntax = """default-services"""

   data = { "default-services": "Enable default services: capi-doc and tapagg" }

   handler = "HttpServiceHandler.defaultServicesHandler"
   noHandler = "HttpServiceHandler.defaultServicesNoHandler"
   defaultHandler = "HttpServiceHandler.defaultServicesHandler"

HttpServerConfigMode.addCommandClass( DefaultServices )

#-------------------------------------------------------------------------------
# The "[ no | default ] content frame ancestors <uri>" command
#-------------------------------------------------------------------------------
URI_REGEX = r"^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?"
tokenUri = CliMatcher.PatternMatcher( URI_REGEX,
                           helpname="URI",
                           helpdesc='URI of allowed host' )
class FrameAncestors( CliCommand.CliCommandClass ):
   syntax = """header csp frame-ancestors <uri>"""
   noOrDefaultSyntax = """header csp frame-ancestors [ <uri> ]"""

   data = { "header": "Additional headers",
            "csp": "Content Security Policy Headers",
            "frame-ancestors": "CSP directive frame-ancestors",
            "<uri>": tokenUri }

   handler = "HttpServiceHandler.frameAncestorsHandler"
   noOrDefaultHandler = "HttpServiceHandler.frameAncestorsNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( FrameAncestors )

#-------------------------------------------------------------------------------
# The "cors allowed domian [ALL | URL]" command
#-------------------------------------------------------------------------------
corsRegex = r'^(?:http)s?://(?:(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}'\
      r'[a-zA-Z0-9])?\.)+(?:[a-zA-Z]{2,6}\.?)(?::\d+)?$|'\
      r'(localhost)(?::\d+)?$|'\
      r'(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})(?::\d+)?$|'\
      r'(\[?[A-Fa-f0-9]*:[A-Fa-f0-9:]+\]?)(?::\d+)?$)'

corsRegexPartial = '(http)*'

# tokens for CORS ( cross origin request service )
tokenOrigin = CliMatcher.PatternMatcher(
    corsRegex,
    helpname='Origin',
    helpdesc='Origin of form http[s]://<hostname>[:<port>]',
    partialPattern=corsRegexPartial )

corsNameRule = CliMatcher.DynamicNameMatcher(
    lambda mode: gv.capiConfig.corsAllowedOrigins,
    'Origin',
    pattern='.*' )

class CorsOrigins( CliCommand.CliCommandClass ):
   syntax = """cors allowed-origin ( all | * | <originString> )"""
   noOrDefaultSyntax = """cors allowed-origin [ <corsName> ]"""

   data = { "cors": "Configure CORS functionality",
            "allowed-origin": "Enter allowed origins",
            "all": "Allow all Origins",
            "*": "Allow all Origins",
            "<originString>": tokenOrigin,
            "<corsName>": corsNameRule }

   handler = "HttpServiceHandler.corsOriginsHandler"
   noOrDefaultHandler = "HttpServiceHandler.corsOriginsNoOrDefaultHandler"

HttpServerConfigMode.addCommandClass( CorsOrigins )

#-------------------------------------------------------------------------------
# show management http-server ( ip|ipv6 ) access-list [<acl>] [summary]
#-------------------------------------------------------------------------------
class ShowManagementHttpServerAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management http-server'
              '('
              ' ( ip access-list [ IP_ACL_NAME ] ) | '
              ' ( ipv6 access-list [ IPV6_ACL_NAME ] ) '
              ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc, 
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'IP_ACL_NAME': AclCli.ipAclNameExpression,
            'IPV6_ACL_NAME': AclCli.ip6AclNameExpression
          }

   cliModel = AclCliModel.AllAclList
   handler = "HttpServiceHandler.showManagementHttpServerAclHandler"

BasicCli.addShowCommandClass( ShowManagementHttpServerAcl ) 

#----------------------------------------------------------------
# "clear management http-server counters ( ip | ipv6 ) access-list"
#----------------------------------------------------------------
class ClearHttpServerAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management http-server counters ( ip | ipv6 ) access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'http-server': httpServerShowHelpDesc,
            'counters': AclCli.countersKwMatcher,
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl }

   handler = "HttpServiceHandler.clearHttpServerAclCountersHandler"

BasicCli.EnableMode.addCommandClass( ClearHttpServerAclCounters ) 

#-------------------------------------------------------------------------------
# The "show management http-server" command
#-------------------------------------------------------------------------------
def getCommonServerInfo( ret, config ):
   ret.fipsEnabled = False
   ret.tlsProtocol = ServerConstants.DEFAULT_TLS_PROTOCOLS

   profile = config.httpsConfig.sslProfile
   if profile:
      status = gv.sslStatus.profileStatus.get( profile )
      if status == None: # pylint: disable=singleton-comparison
         ret.sslProfile = HttpServiceModel.SslProfile( name=profile,
                                                       configured=False )
      else:
         profileState = status.state
         ret.sslProfile = HttpServiceModel.SslProfile( name=profile,
                                                       configured=True,
                                                       state=profileState )
         ret.fipsEnabled = status.fipsMode
         ret.tlsProtocol = DefaultSslLib.tlsVersionMaskToStrList( status.tlsVersion )
   ret.dscpValue = config.qosDscp
   ret.logLevel = config.syslogLevel
   if gv.capiConfig.contentFrameAncestor:
      ret.iframeAncestors.append( config.contentFrameAncestor )

def showHttpServer( mode, args ):
   ret = HttpServiceModel.HttpServiceStatus()
   getCommonServerInfo( ret, gv.capiConfig )

   for ( vrf, status ) in gv.serverStatus.vrfStatus.items():
      # check vrf error first
      for error in status.vrfError.values():
         if error:
            mode.addWarning( error )
      # only show enabled VRFs
      if status.enabled:
         currVrf = HttpServiceModel.HttpServiceVrf()
         currVrf.httpsServer = HttpServiceModel.HttpProtocolStatus(
               configured=gv.capiConfig.httpsConfig.enabled,
               running=status.httpsEnabled, port=gv.capiConfig.httpsConfig.port )
         currVrf.httpServer = HttpServiceModel.HttpProtocolStatus(
               configured=gv.capiConfig.httpConfig.enabled,
               running=status.httpEnabled, port=gv.capiConfig.httpConfig.port )
         currVrf.localHttpServer = HttpServiceModel.HttpProtocolStatus(
               configured=gv.capiConfig.localHttpConfig.enabled,
               running=status.httpLocalEnabled, 
               port=gv.capiConfig.localHttpConfig.port)
         currVrf.unixSocketServer = HttpServiceModel.UnixProtocolStatus(
               configured=gv.capiConfig.unixConfig.enabled, 
               running=status.unixEnabled )

         # only show services that enabled in this VRF
         vrfService = []
         for ( service, serviceStatus )in status.vrfService.items():
            if serviceStatus.enabled:
               vrfService.append( gv.capiConfig.service[ service ].serviceName )
         currVrf.services = vrfService
         ret.vrfs[ vrf ] = currVrf

   return ret

class ShowManagementHttpServer( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management http-server'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc }
   handler = showHttpServer
   cliModel = HttpServiceModel.HttpServiceStatus

BasicCli.addShowCommandClass( ShowManagementHttpServer )
#-------------------------------------------------------------------------------
# The "show management http-server counters" command
#-------------------------------------------------------------------------------
class ShowManagementHttpServerCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management http-server counters'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'http-server': httpServerShowHelpDesc,
            'counters': 'Show http connection stats (per vrf)' }
   handler = "HttpServiceHandler.showHttpServerStatsHandler"
   cliModel = HttpServiceModel.HttpServerStats

BasicCli.addShowCommandClass( ShowManagementHttpServerCounters )

#-------------------------------------------------------------------------------
# Command-API commands in "show tech-support"
#-------------------------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmd(
   '2022-09-02 16:43:57',
   cmds=[ 'show management http-server' ] )

# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                      "mgmt/capi/config",
                                      "HttpService::Config",
                                      "w" )
   gv.serverStatus = LazyMount.mount( entityManager,
                                      "mgmt/httpserver/status",
                                      "HttpService::Status",
                                      "r" )
   gv.sslConfig = LazyMount.mount( entityManager,
                                   "mgmt/security/ssl/config",
                                   "Mgmt::Security::Ssl::Config",
                                   "r" )
   gv.sslStatus = LazyMount.mount( entityManager,
                                   "mgmt/security/ssl/status",
                                   "Mgmt::Security::Ssl::Status",
                                   "r" )
