# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Model, Submodel
from CliModel import Bool, Int, Str, List, Dict, Enum
import TableOutput
import Tac

ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
LogLevel = Tac.Type( "HttpService::LogLevel" )

class ProtocolStatus( Model ):
   configured = Bool( help="Whether this protocol is configured to be enabled" )
   running = Bool( help="Whether this protocol is currently serving and "
                   "accepting requests" )

class UnixProtocolStatus( ProtocolStatus ):
   # We don't add any additional attributes here, but separate it out for 
   # documentation reasons 
   pass

class HttpProtocolStatus( ProtocolStatus ):
   port = Int( help="The port this protocol's server is listening on" )

class SslProfile( Model ):
   name = Str( help="SSL profile name" )
   configured = Bool( help="Whether this profile is configured in SSL Manager" )
   state = Enum( optional=True, help="State of the SSL profile",
                 values=ProfileState.attributes )

class HttpServiceVrf( Model ):
   httpServer = Submodel( valueType=HttpProtocolStatus,
                          help="The status of the HTTP server in vrf" )
   localHttpServer = Submodel( valueType=HttpProtocolStatus, optional=True,
                               help="The status of localhost HTTP server in vrf" )
   httpsServer = Submodel( valueType=HttpProtocolStatus,
                           help="The status of the HTTPS server in vrf" )
   unixSocketServer = Submodel( valueType=UnixProtocolStatus, optional=True,
                                help="The status of the Unix Domain Socket server" )
   services = List( valueType=str, help="The list of enabled http services in VRF" )

def printLineItem( label, content ):
   print( '{:<20}{:}'.format( '%s:' % label, content ) )

class HttpServiceStatus( Model ):   
   vrfs = Dict( keyType=str, valueType=HttpServiceVrf,
                help="A mapping between the VRF name and http server status" )
   sslProfile = Submodel( optional=True, valueType=SslProfile,
                          help="The configuration and status of the SSL profile" )
   fipsEnabled = Bool( optional=True, help="Whether FIPS mode is enabled for "
                                           "HTTPS algorithms" )
   dscpValue = Int( help="The QoS DSCP value set in outgoing IP packets",
                    optional=True )
   logLevel = Enum( help="Logging level for the HTTP server",
                    values=LogLevel.attributes )
   iframeAncestors = List( valueType=str, help="Allowed Iframe ancestors" )
   tlsProtocol = List( valueType=str, help="Allowed TLS protocol versions" )
 
   def render( self ):
      if self.sslProfile:
         state = 'not configured'
         if self.sslProfile.state:
            state = self.sslProfile.state
         printLineItem( 'SSL Profile', '%s, %s' % ( self.sslProfile.name, state ) )
      else:
         printLineItem( 'SSL Profile', '%s' % 'none' )
      printLineItem( "FIPS Mode", "Yes" if self.fipsEnabled else "No" )

      printLineItem( 'QoS DSCP', '%s' % self.dscpValue )
      printLineItem( 'Log Level', '%s' % self.logLevel )
      iframeAncestors = ( ' '.join( self.iframeAncestors ) if self.iframeAncestors 
                          else 'None' )
      printLineItem( 'CSP Frame Ancestor', '%s' % iframeAncestors )

      printLineItem( 'TLS Protocols', '%s' % ' '.join( self.tlsProtocol ) )

      headings = ( 'VRF', 'Server Status', 'Enabled Services' )
      table = TableOutput.createTable( headings )
      f = TableOutput.Format( justify="left" )
      table.formatColumns( *( [ f ] * len( headings ) ) )
      for vrf in sorted( self.vrfs ):
         vrfStatus = ''
         vrfService = ''
         serviceStatus = self.vrfs[ vrf ]
         if serviceStatus.httpsServer.running:
            vrfStatus += 'HTTPS: port %d\n' % serviceStatus.httpsServer.port
         if serviceStatus.httpServer.running:
            vrfStatus += 'HTTP: port %d\n' % serviceStatus.httpServer.port
         if serviceStatus.localHttpServer.running:
            vrfStatus += 'Local HTTP: port %d\n' % serviceStatus.localHttpServer.port
         if serviceStatus.unixSocketServer.running:
            vrfStatus += 'Unix Socket: running\n'
         for s in serviceStatus.services:
            vrfService += '%s\n' % s 
         table.newRow( vrf, vrfStatus, vrfService )
      print( table.output() )

class VrfStats( Model ):
   maxConnections = Int( help="The maximum number of simultaneous connections" )
   activeConnections = Int( help="The current number of active client connections" )
   acceptConnections = Int( help="The total number of accepted client connections." )
   handledConnections = Int( help="The total number of handled connections." )
   refusedConnections = Int( help="The total number of refused connections." )
   requests = Int( help="The total number of client requests." )
   readingConnections = Int( help="The current number of connections "
                             "where nginx is reading the request header." )
   writingConnections = Int( help="The current number of connections "
                                  "where nginx is writing the response back "
                                  "to the client." ) 
   waitingConnections = Int( help="The current number of connections "
                                  "waiting for a request." )
   
class HttpServerStats( Model ):
   vrfs = Dict( keyType=str, valueType=VrfStats,
                help="A mapping between VRF name and running nginx statistics" )

   def render( self ):
      headingsStats = (  'VRF', 'Request', 'Connection\nMax', 
                           'Connection\nAccepted', 'Connection\nHandled', 
                           'Connection\nRefused' )
      headingsStatus = ( 'VRF', 'Connection\nActive', 'Connection\nReading', 
                         'Connection\nWriting', 'Connection\nWaiting' )
      tableStats = TableOutput.createTable( headingsStats )
      tableStatus = TableOutput.createTable( headingsStatus )
      f = TableOutput.Format( justify="left" )
      tableStats.formatColumns( *( [ f ] * len( headingsStats ) ) )
      tableStatus.formatColumns( *( [ f ] * len( headingsStatus ) ) )

      for vrf in sorted( self.vrfs ):
         stats = self.vrfs[ vrf ]
         tableStats.newRow( vrf, stats.requests, 
                            stats.maxConnections, stats.acceptConnections, 
                            stats.handledConnections, stats.refusedConnections )
         tableStatus.newRow( vrf, stats.activeConnections, stats.readingConnections,
                             stats.writingConnections, stats.waitingConnections )
      print( tableStats.output() )
      print( tableStatus.output() )
