# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount

from CliMode.IcmpErrorResponderMode import (
      IcmpErrExtBaseMode,
      IcmpErrExtVrfBaseMode )
from CliPlugin.AclCli import (
      standardIpAclNameMatcher,
      standardIp6AclNameMatcher )
from CliPlugin.IcmpResponderCliLib import (
   icmpErrExtPlatformGuard,
   matcherIcmp,
   matcherIcmpError,
   matcherIcmpErrorExtensions,
)
from CliPlugin.IntfCli import Intf
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.RouterGeneralCli import (
      routerGeneralCleanupHook,
      routerGeneralVrfCleanupHook,
      RouterGeneralMode,
      RouterGeneralVrfMode )
from IpLibConsts import DEFAULT_VRF
from Toggles.IcmpResponderToggleLib import (
      toggleIcmpExtendedErrorResponderEnabled,
      toggleIcmpExtendedErrorResponderIpv6Enabled,
      toggleIcmpExtendedErrorResponderNdVrfEnabled )
from TypeFuture import TacLazyType
import IcmpErrorCliHooks

TristateBool = TacLazyType( 'Ark::TristateBoolean' )
NodeIdCodePoint = TacLazyType( 'IcmpResponder::ErrorResponder::NodeIdCodePoint' )
# ----------------------------------------------------------------------------------
#                                Sysdb mounts
# ----------------------------------------------------------------------------------
configDir = None # IcmpResponder::ErrorResponder::Config

# ----------------------------------------------------------------------------------
#                                CLI MODES
# - Global mode applicable to all vrfs.
# - Per vrf mode, specific to vrfs
#   (default vrf config will be under 'icmp error extensions'
#    under 'vrf default' mode under 'router general')
# ----------------------------------------------------------------------------------

class IcmpErrExtMode( IcmpErrExtBaseMode, BasicCli.ConfigModeBase ):
   '''
   This mode holds global icmp error extensions configuration
   common to all VRFs.
   '''
   name = 'ICMP Error Extension Configuration'

   def __init__( self, parent, session ):
      self.session_ = session
      IcmpErrExtBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class IcmpErrExtVrfMode( IcmpErrExtVrfBaseMode, BasicCli.ConfigModeBase ):
   '''
   This mode holds global icmp error extensions configuration
   specific to a VRF. vrf default also has a corresponding instance.
   '''
   name = 'ICMP Error Extension Configuration for VRF'

   def __init__( self, parent, session ):
      self.session_ = session
      IcmpErrExtVrfBaseMode.__init__( self, parent.vrfName )
      self.vrfName = parent.vrfName
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterGeneralVrfIcmpExtErrModelet( CliParser.Modelet ):
   '''
   This modelet allows supporting non-default VRF conditional on
   IcmpExtendedErrorResponderNdVrf toggle state.
   '''
   @staticmethod
   def shouldAddModeletRule( mode ):
      assert isinstance( mode, RouterGeneralVrfMode )
      return ( mode.vrfName == DEFAULT_VRF or
               toggleIcmpExtendedErrorResponderNdVrfEnabled() )

RouterGeneralVrfMode.addModelet( RouterGeneralVrfIcmpExtErrModelet )

# ----------------------------------------------------------------------------------
#                                K E Y W O R D S
# ----------------------------------------------------------------------------------
matcherExtensions = CliMatcher.KeywordMatcher(
      'extensions',
      'Enable/Disable rfc5837 ICMP/ICMPv6 error extensions' )
matcherExtensionsVrf = CliMatcher.KeywordMatcher(
      'vrf',
      'Enable/Disable rfc5837 ICMP/ICMPv6 error extensions on VRF' )
matcherExtensionsVrfAll = CliMatcher.KeywordMatcher(
      'all',
      'Enable/Disable rfc5837 ICMP/ICMPv6 error extensions on all VRFs' )
matcherExtensionsDisabled = CliMatcher.KeywordMatcher(
      'disabled',
      'Disable rfc5837 ICMP/ICMPv6 error extensions.' )
matcherIp = CliMatcher.KeywordMatcher(
      'ip',
      'IPv4 configuration' )
matcherIp6 = CliMatcher.KeywordMatcher(
      'ip6',
      'IPv6 configuration' )
matcherIcmpErrorExtensionsAccessGroup = CliMatcher.KeywordMatcher(
      'access-group',
      'Match the source IP address of original packet that '
      'triggered the ICMP error' )
matcherIcmpErrorExtensionsParams = CliMatcher.KeywordMatcher(
      'params',
      'Configure parameters for the extended error response' )
matcherIcmpErrorExtensionsParamsInclude = CliMatcher.KeywordMatcher(
      'include',
      'Include parameter in the extended error response' )
matcherIcmpErrorExtensionsIncludeHostname = CliMatcher.KeywordMatcher(
      'hostname',
      'Include hostname along with interface name in interface name '
      'sub-object part of the interface information object '
      'in incoming interface role.' )

matcherNodeid = CliMatcher.KeywordMatcher(
      'nodeid',
      'Configure parameters for Node Identification Object' )
matcherNodeidCodepoint = CliMatcher.KeywordMatcher(
      'codepoint',
      'Configure ClassNum for Node Identification Object' )
matcherNodeidIpv4 = CliMatcher.KeywordMatcher(
      'ipv4',
      'Configure IPv4 parameters for Node Identification Object' )
matcherNodeidIpv6 = CliMatcher.KeywordMatcher(
      'ipv6',
      'Configure IPv6 parameters for Node Identification Object' )
matcherNodeidIpAddress = CliMatcher.KeywordMatcher(
      'address',
      "Configuration for IP address sub-object" )
matcherNodeidInterface = CliMatcher.KeywordMatcher(
      'interface',
      'Configure interface for selecting address for '
      'IP address sub-object' )

# ----------------------------------------------------------------------------------
#                               C O M M A N D S
# ----------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# [ no | default ] icmp error extensions
# in "router general" global and per-VRF modes
# --------------------------------------------------------------------------------
class IcmpErrExtCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp error extensions'
   noOrDefaultSyntax = syntax

   data = {
      'icmp': matcherIcmp,
      'error': CliCommand.Node( matcher=matcherIcmpError,
                                guard=icmpErrExtPlatformGuard ),
      'extensions': matcherIcmpErrorExtensions,
   }

   handler = 'IcmpErrorResponderHandler.doIcmpErrExt'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noIcmpErrExt'

# --------------------------------------------------------------------------------
# [ no | default ] extensions vrf all [ disabled ]
# in "icmp error extensions" global mode
# --------------------------------------------------------------------------------
class ExtensionsVrfAllCmd( CliCommand.CliCommandClass ):
   syntax = 'extensions vrf all [disabled]'
   noOrDefaultSyntax = 'extensions vrf all ...'

   data = {
      'extensions': matcherExtensions,
      'vrf': matcherExtensionsVrf,
      'all': matcherExtensionsVrfAll,
      'disabled': matcherExtensionsDisabled,
   }

   handler = 'IcmpErrorResponderHandler.doExtensionsVrfAll'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noExtensionsVrfAll'

# --------------------------------------------------------------------------------
# [ no | default ] params include hostname
# in "icmp error extensions" global mode
# --------------------------------------------------------------------------------
class ParamsIncludeHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'params include hostname'
   noOrDefaultSyntax = syntax

   data = {
         'params': matcherIcmpErrorExtensionsParams,
         'include': matcherIcmpErrorExtensionsParamsInclude,
         'hostname': matcherIcmpErrorExtensionsIncludeHostname,
   }

   handler = 'IcmpErrorResponderHandler.doParamsIncludeHostname'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noParamsIncludeHostname'

# --------------------------------------------------------------------------------
# [ no | default ] nodeid codepoint CODEPOINT
# in "icmp error extensions" global mode
# --------------------------------------------------------------------------------
class NodeidCodepointCmd( CliCommand.CliCommandClass ):
   syntax = 'nodeid codepoint CODEPOINT'
   noOrDefaultSyntax = 'nodeid codepoint ...'

   data = {
         'nodeid': matcherNodeid,
         'codepoint': matcherNodeidCodepoint,
         'CODEPOINT': CliMatcher.IntegerMatcher(
            NodeIdCodePoint.min,
            NodeIdCodePoint.max,
            helpdesc='Class number' ),
   }

   handler = 'IcmpErrorResponderHandler.doNodeidCodepoint'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noNodeidCodepoint'

def icmpErrNodeIdClassNum():
   if not toggleIcmpExtendedErrorResponderEnabled():
      return None
   codePoint = configDir.nodeIdCodePoint
   if codePoint != NodeIdCodePoint.invalid:
      return codePoint
   else:
      return None


# --------------------------------------------------------------------------------
# [ no | default ] extensions [ disabled ]
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class ExtensionsCmd( CliCommand.CliCommandClass ):
   syntax = 'extensions [disabled]'
   noOrDefaultSyntax = 'extensions ...'

   data = {
      'extensions': matcherExtensions,
      'disabled': matcherExtensionsDisabled,
   }

   handler = 'IcmpErrorResponderHandler.doExtensions'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noExtensions'

# --------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class IpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME'
   noOrDefaultSyntax = 'ip access-group ...'

   data = {
      'ip': matcherIp,
      'access-group': matcherIcmpErrorExtensionsAccessGroup,
      'ACLNAME': standardIpAclNameMatcher,
   }

   handler = 'IcmpErrorResponderHandler.doIpAccessGroup'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noIpAccessGroup'

# --------------------------------------------------------------------------------
# [ no | default ] ip6 access-group ACLNAME
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class Ip6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip6 access-group ACLNAME'
   noOrDefaultSyntax = 'ip6 access-group ...'

   data = {
      'ip6': matcherIp6,
      'access-group': matcherIcmpErrorExtensionsAccessGroup,
      'ACLNAME': standardIp6AclNameMatcher,
   }

   handler = 'IcmpErrorResponderHandler.doIp6AccessGroup'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noIp6AccessGroup'

# --------------------------------------------------------------------------------
# [ no | default ] nodeid ip address IPADDR
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class NodeidIpv4AddressCmd( CliCommand.CliCommandClass ):
   syntax = 'nodeid ipv4 address IPADDR'
   noOrDefaultSyntax = 'nodeid ipv4 address ...'

   data = {
         'nodeid': matcherNodeid,
         'ipv4': matcherNodeidIpv4,
         'address': matcherNodeidIpAddress,
         'IPADDR': IpAddrMatcher( 'IPv4 address' ),
   }

   handler = 'IcmpErrorResponderHandler.doNodeidIpv4Address'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noNodeidIpv4Address'

# --------------------------------------------------------------------------------
# [ no | default ] nodeid ip address IPADDR
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class NodeidIpv6AddressCmd( CliCommand.CliCommandClass ):
   syntax = 'nodeid ipv6 address IP6ADDR'
   noOrDefaultSyntax = 'nodeid ipv6 address ...'

   data = {
         'nodeid': matcherNodeid,
         'ipv6': matcherNodeidIpv6,
         'address': matcherNodeidIpAddress,
         'IP6ADDR': Ip6AddrMatcher( 'IPv6 address' ),
   }

   handler = 'IcmpErrorResponderHandler.doNodeidIpv6Address'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noNodeidIpv6Address'

# --------------------------------------------------------------------------------
# [ no | default ] nodeid interface INTF
# in "icmp error extensions" per-VRF mode
# --------------------------------------------------------------------------------
class NodeidInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'nodeid interface INTF'
   noOrDefaultSyntax = 'nodeid interface ...'

   data = {
         'nodeid': matcherNodeid,
         'interface': matcherNodeidInterface,
         'INTF': Intf.matcherWithIpSupport,
   }

   handler = 'IcmpErrorResponderHandler.doNodeidInterface'
   noOrDefaultHandler = 'IcmpErrorResponderHandler.noNodeidInterface'

def cleanupAllIcmpErrConfig( mode ):
   configDir.vrfConfig.clear()
   configDir.enabled = TristateBool()
   configDir.addHostName = False
   configDir.nodeIdCodePoint = NodeIdCodePoint()

def cleanupVrfIcmpErrConfig( vrfName ):
   del configDir.vrfConfig[ vrfName ]

if toggleIcmpExtendedErrorResponderEnabled():
   RouterGeneralMode.addCommandClass( IcmpErrExtCmd )

   RouterGeneralVrfIcmpExtErrModelet.addCommandClass( IcmpErrExtCmd )

   if toggleIcmpExtendedErrorResponderNdVrfEnabled():
      IcmpErrExtMode.addCommandClass( ExtensionsVrfAllCmd )
   IcmpErrExtMode.addCommandClass( ParamsIncludeHostnameCmd )
   IcmpErrExtMode.addCommandClass( NodeidCodepointCmd )

   IcmpErrExtVrfMode.addCommandClass( ExtensionsCmd )
   IcmpErrExtVrfMode.addCommandClass( IpAccessGroupCmd )
   if toggleIcmpExtendedErrorResponderIpv6Enabled():
      IcmpErrExtVrfMode.addCommandClass( Ip6AccessGroupCmd )
   IcmpErrExtVrfMode.addCommandClass( NodeidIpv4AddressCmd )
   # not under ipv6 toggle because ICMPv4 error messages
   # can have IPv6 address sub-objects for nodeid object.
   IcmpErrExtVrfMode.addCommandClass( NodeidIpv6AddressCmd )
   IcmpErrExtVrfMode.addCommandClass( NodeidInterfaceCmd )

   routerGeneralCleanupHook.addExtension( cleanupAllIcmpErrConfig )
   routerGeneralVrfCleanupHook.addExtension( cleanupVrfIcmpErrConfig )
   IcmpErrorCliHooks.icmpErrorExtensionNodeIdClassNum.addExtension(
                icmpErrNodeIdClassNum )

def Plugin( entityManager ):
   global configDir
   configDir = ConfigMount.mount( entityManager,
                                  'icmp/errorResponder/config',
                                  'IcmpResponder::ErrorResponder::Config', 'w' )
