# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Cell
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
from CliPlugin.IcmpResponderCliLib import (
   icmpErrExtPlatformGuard,
   matcherIcmp,
   matcherIcmpError,
   matcherIcmpErrorExtensions,
)
from CliPlugin.IraIpCli import vrfRoutingSupportedGuard
from CliPlugin.VrfCli import VrfExprFactory
from CliToken import Clear
from Toggles.IcmpResponderToggleLib import (
      toggleIcmpExtendedErrorResponderEnabled )

# ----------------------------------------------------------------------------------
#                                Sysdb mounts
# ----------------------------------------------------------------------------------
statusDir = None # IcmpResponder::ErrorResponder::Status
aclCliConfig = None # Acl::Input::Config
allVrfStatusLocal = None # Ip::AllVrfStatusLocal

# ----------------------------------------------------------------------------------
#                                K E Y W O R D S
# ----------------------------------------------------------------------------------
matcherVrf = VrfExprFactory(
      helpdesc='Show only specific VRF',
      inclAllVrf=False,
      inclDefaultVrf=True,
      guard=vrfRoutingSupportedGuard )
matcherIcmpErrorExtensionsCounters = CliMatcher.KeywordMatcher(
   'counters',
   'Show counters' )
# ----------------------------------------------------------------------------------
#                               C O M M A N D S
# ----------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# show icmp error extensions [ vrf VRFNAME ]
# --------------------------------------------------------------------------------
class ShowIcmpErrExtCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show icmp error extensions [ VRF ]'

   data = {
      'icmp': matcherIcmp,
      'error': CliCommand.Node( matcher=matcherIcmpError,
                                guard=icmpErrExtPlatformGuard ),
      'extensions': matcherIcmpErrorExtensions,
      'VRF': matcherVrf,
   }
   cliModel = 'IcmpErrorResponderModel.ErrorResponderModel'
   handler = 'IcmpErrorResponderHandler.showIcmpErrExt'

if toggleIcmpExtendedErrorResponderEnabled():
   BasicCli.addShowCommandClass( ShowIcmpErrExtCmd )

class ShowIcmpErrExtCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show icmp error extensions counters [ VRF ]'

   data = {
      'icmp': matcherIcmp,
      'error': CliCommand.Node( matcher=matcherIcmpError,
                                guard=icmpErrExtPlatformGuard ),
      'extensions': matcherIcmpErrorExtensions,
      'counters': matcherIcmpErrorExtensionsCounters,
      'VRF': matcherVrf,
   }
   cliModel = 'IcmpErrorResponderModel.ErrorResponderCountersModel'
   handler = 'IcmpErrorResponderHandler.showIcmpErrExtCounters'

class ClearIcmpErrExtCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear icmp error extensions counters'

   data = {
      'clear': Clear.clearKwNode,
      'icmp': matcherIcmp,
      'error': CliCommand.Node( matcher=matcherIcmpError,
                                guard=icmpErrExtPlatformGuard ),
      'extensions': matcherIcmpErrorExtensions,
      'counters': matcherIcmpErrorExtensionsCounters,
   }
   handler = 'IcmpErrorResponderHandler.clearIcmpErrExtCounters'

if toggleIcmpExtendedErrorResponderEnabled():
   BasicCli.addShowCommandClass( ShowIcmpErrExtCountersCmd )
   BasicCli.EnableMode.addCommandClass( ClearIcmpErrExtCountersCmd )

def Plugin( entityManager ):
   global statusDir, aclCliConfig, allVrfStatusLocal
   statusDir = LazyMount.mount( entityManager,
                                'icmp/errorResponder/status',
                                'IcmpResponder::ErrorResponder::Status', 'r' )
   aclCliConfig = LazyMount.mount( entityManager,
                                   'acl/config/cli',
                                   'Acl::Input::Config', 'r' )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( 'ip/vrf/status/local' ),
                                        'Ip::AllVrfStatusLocal', 'r' )
