# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import Tac
from CliToken.Ip import ipMatcherForConfigIf
from CliMode.IgdUpnp import IgdMode, IgdVrfMode
from CliPlugin import IntfCli
from CliPlugin.IgdUpnpCliLib import (
      getIgdConfig,
      getIgdVrfConfig,
      upnpIgdSupportedGuard )
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.UpnpConfigCli import (
      UpnpConfigMode,
      UpnpDependentBase,
      UpnpCli,
      nodeUpnp )
from CliPlugin.NatCli import (
      nodeNat,
      matcherProfileName,
      nodePool,
      matcherPoolName )
from CliPlugin.AclCli import ipAclNameMatcher

nodeIgd = CliCommand.guardedKeyword( 'igd',
      helpdesc='UPNP IGD service parameters',
      guard=upnpIgdSupportedGuard )

igdProtoMatcher = CliMatcher.EnumMatcher( {
   "http": "Use HTTP for IGD",
   "https": "Use HTTPS for IGD"
   } )

class IgdConfigCleaner( UpnpDependentBase ):
   def clearConfig( self ):
      IgdCmd.clearConfig_()

UpnpCli.registerDependentClass( IgdConfigCleaner, priority=10 )

class IgdConfigMode( IgdMode, BasicCli.ConfigModeBase ):
   name = "IGD configuration"

   def __init__( self, parent, session ):
      IgdMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class IgdVrfConfigMode( IgdVrfMode, BasicCli.ConfigModeBase ):
   name = "IGD VRF configuration"

   def __init__( self, parent, session, vrfName ):
      igdVrfConfig = getIgdVrfConfig( vrfName )
      igdVrfConfig.ipv4State = igdVrfConfig.defaultIpv4State
      IgdVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# Modelet for interface configuration
class IgdUpnpIntfModelet( CliParser.Modelet ):
   pass

IntfCli.IntfConfigMode.addModelet( IgdUpnpIntfModelet )

class IgdUpnpIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      igdConfig = getIgdConfig()
      del igdConfig.interface[ self.intf_.name ]

IntfCli.Intf.registerDependentClass( IgdUpnpIntf, priority=32 )

# [no|default] ip upnp igd
class IgdUpnpIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip upnp igd'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfigIf,
      'upnp': nodeUpnp,
      'igd': 'Enable IGD service',
   }

   @staticmethod
   def handler( mode, args ):
      igdConfig = getIgdConfig()
      igdConfig.interface.add( mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igdConfig = getIgdConfig()
      del igdConfig.interface[ mode.intf.name ]

IgdUpnpIntfModelet.addCommandClass( IgdUpnpIntfCmd )

# [ no ] igd
class IgdCmd( CliCommand.CliCommandClass ):
   syntax = '''igd'''
   noOrDefaultSyntax = syntax
   data = {
      'igd': nodeIgd,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( IgdConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      IgdCmd.clearConfig_()

   @staticmethod
   def clearConfig_():
      igdConfig = getIgdConfig()
      for vrfName in igdConfig.igdVrfConfig:
         IgdVrfCmd.clearConfig_( vrfName )
      igdConfig.protocolAndPort = Tac.Value( 'IgdUpnp::IgdProtocolAndPort' )

UpnpConfigMode.addCommandClass( IgdCmd )

# [no] server protocol PROTO port PORT
class IgdProtocolAndPortCmd( CliCommand.CliCommandClass ):
   syntax = '''server protocol PROTO port PORT'''
   noOrDefaultSyntax = syntax
   data = {
      'server': 'IGD server settings',
      'protocol': 'HTTP protocol mode',
      'PROTO': igdProtoMatcher,
      'port': 'TCP port number for IGD',
      'PORT': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='TCP port number 1-65535' )
   }

   @staticmethod
   def handler( mode, args ):
      igdConfig = getIgdConfig()
      protoAndPort = Tac.Value( 'IgdUpnp::IgdProtocolAndPort' )
      protoAndPort.protocol = args[ 'PROTO' ]
      protoAndPort.port = args.get( 'PORT', 0 )
      igdConfig.protocolAndPort = protoAndPort

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igdConfig = getIgdConfig()
      igdConfig.protocolAndPort = Tac.Value( 'IgdUpnp::IgdProtocolAndPort' )

IgdConfigMode.addCommandClass( IgdProtocolAndPortCmd )

# [no] vrf VRF
class IgdVrfCmd( CliCommand.CliCommandClass ):
   syntax = '''VRF'''
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfExprFactory( helpdesc='Configure the VRF name', inclDefaultVrf=True )
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      childMode = mode.childMode( IgdVrfConfigMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args.get( 'VRF' )
      IgdVrfCmd.clearConfig_( vrfName )

   @staticmethod
   def clearConfig_( vrfName ):
      igdConfig = getIgdConfig()
      del igdConfig.igdVrfConfig[ vrfName ]

IgdConfigMode.addCommandClass( IgdVrfCmd )

# [no] disabled
class IgdVrfDisabledCmd( CliCommand.CliCommandClass ):
   syntax = '''disabled'''
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable IGD service',
   }

   @staticmethod
   def handler( mode, args ):
      igdVrfConfig = getIgdVrfConfig( mode.vrfName )
      igdVrfConfig.ipv4State = "disabled"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      igdVrfConfig = getIgdVrfConfig( mode.vrfName )
      igdVrfConfig.ipv4State = igdVrfConfig.defaultIpv4State

IgdVrfConfigMode.addCommandClass( IgdVrfDisabledCmd )

# [no] nat service-profile PROFILE access-list ACL pool POOL
class IgdVrfNatProfileCmd( CliCommand.CliCommandClass ):
   syntax = '''nat service-profile PROFILE access-list ACL pool POOL'''
   noOrDefaultSyntax = '''nat service-profile'''
   data = {
      'nat': nodeNat,
      'service-profile': 'NAT profile name',
      'PROFILE': matcherProfileName,
      'access-list': 'ACL used in the full-cone rule',
      'ACL': ipAclNameMatcher,
      'pool': nodePool,
      'POOL': matcherPoolName
   }

   @staticmethod
   def handler( mode, args ):
      igdVrfConfig = getIgdVrfConfig( mode.vrfName )
      profile = args.get( 'PROFILE', "" )
      acl = args.get( 'ACL', "" )
      pool = args.get( 'POOL', "" )
      natCfg = Tac.Value( 'IgdUpnp::IgdVrfNatInfo',
            profile, acl, pool )
      igdVrfConfig.natInfo = natCfg

   # Since args.get above would have nil values in the handler
   # method above, we can just reuse handler for deleting the config as well.
   noOrDefaultHandler = handler

IgdVrfConfigMode.addCommandClass( IgdVrfNatProfileCmd )
