#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

import Arnet
from ArnetModel import IpAddrAndMask
from Toggles import IgmpHostProxyToggleLib
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IraIpIntfCli as IraIpIntfCli
import CliPlugin.McastCommonCli as McastCommonCli
import CliPlugin.IgmpHostProxyModel as IgmpHostProxyModel
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.RouterIgmp import RouterIgmpSharedModelet
from CliPlugin import IgmpGlobalConfigMode
from CliPlugin.RouterMulticastCliLib import RouterMulticastDefaultIpv4Mode
from CliPlugin import MrouteCli
import CliToken.Clear
import CliToken.Ip
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
from McastCommonCliLib import mcastRoutingSupported
import ShowCommand
import Tac

#globals
igmpHostProxyStatus = None
igmpHostProxyConfig = None
_entityManager = None
globalVar = CliGlobal.CliGlobal( igmpHostProxyVrfsConfig=None )

IpAddress = IpAddrMatcher.Arnet.IpAddress
ReportIntervalType = Tac.Type(
      'IgmpHostProxy::UnsolicitedReportInterval' )
IgmpVersionType = Tac.Type(
      'IgmpHostProxy::IgmpVersion' )

def isMulticast( addr ):
   addrStr = addr if isinstance( addr, str ) else str( addr )
   return IpAddrMatcher.validateMulticastIpAddr( addrStr ) is None

def isUnicast( addr ):
   addrStr = addr if isinstance( addr, str ) else str( addr )
   return IpAddrMatcher.isReserved( addrStr ) or \
       IpAddrMatcher.validateMulticastIpAddr( addrStr )

matcherHostProxy = CliMatcher.KeywordMatcher( 'host-proxy',
      helpdesc='IGMP host-proxy information' )
nodeIgmp = CliCommand.guardedKeyword( 'igmp',
      helpdesc='IGMP related status and configuration',
      guard=McastCommonCli.mcastRoutingSupportedGuard )

modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

def ipIgmpHostProxySupportedGuard( mode=None, token=None ):
   if mcastRoutingSupported( _entityManager.root(), routingHardwareStatus=None ):
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# The IgmpHostProxyIntf class is used to remove the IgmpHostProxy IntfConfig
# object when an interface is deleted. The Intf class will create a new instance
# of IgmpHostProxyIntf and call destroy when the interface is deleted
# -------------------------------------------------------------------------------
class IgmpHostProxyIntf( IntfCli.IntfDependentBase ):
#----------------------------------------------------------------------------
# Destroys the IntfConfig object for this interface if it exists.
#----------------------------------------------------------------------------
   def setDefault( self ):
      name = self.intf_.name
      del igmpHostProxyConfig.intfConfig[ name ]

def _getOrCreateIgmpHostProxyIntfConfig( mode, printError=True ):
   intfConfig = igmpHostProxyConfig.intfConfig.get( mode.intf.name )
   if not intfConfig:
      intfConfig = igmpHostProxyConfig.newIntfConfig( mode.intf.name )
   return intfConfig

def getVrfConfig( vrfName ):
   vrfConfig = globalVar.igmpHostProxyVrfsConfig.\
            vrfConfig.get( vrfName )
   return vrfConfig

def deleteIgmpHostProxyIntfConfig(mode, printError=True):
   intfName = mode.intf.name
   ic = igmpHostProxyConfig.intfConfig.get( intfName )
   if not ic:
      return
   del igmpHostProxyConfig.intfConfig[ intfName ]

def deleteIhpIntfConfigIfDefault( intfName ):
   if intfName not in igmpHostProxyConfig.intfConfig:
      return
   intfConfig = igmpHostProxyConfig.intfConfig.get( intfName )
   if intfConfig.defaultHostProxy or \
         intfConfig.unsolicitedReportIntvl != \
            ReportIntervalType() or \
         intfConfig.igmpVersion != IgmpVersionType() or \
         intfConfig.aclList or \
         intfConfig.groupSourceList or \
         intfConfig.deletedGroupSourceList:
      return

   del igmpHostProxyConfig.intfConfig[ intfName ]

def deleteRouterIgmpVrfCb( mode, vrfName ):
   del globalVar.igmpHostProxyVrfsConfig.vrfConfig[ vrfName ]

def deleteIgmpHostProxyAllVrfsConfig( mode, args ):
   # Cleaning ihp vrf config in case of  no router igmp
   globalVar.igmpHostProxyVrfsConfig.vrfConfig.clear()

def addRouterIgmpVrfCb( vrfName ):
   globalVar.igmpHostProxyVrfsConfig.newVrfConfig( vrfName )

def deleteRouterMulticastCb( vrfName ):
   if vrfName == DEFAULT_VRF:
      globalVar.igmpHostProxyVrfsConfig.aclRuleOrderSequenceNum = False

#--------------------------------------------------------------------------------
# show ip igmp host-proxy ( debug | ( interface [ INTF ] [ detail ] ) )
#--------------------------------------------------------------------------------
def applyMask( ipAddr, mask, base=True ):
   """Returns a specific IP Address resulting from applying the
   32-bit mask to the given IP Address ipAddr.
   If base is true, returns the all-zeros address of the mask,
   otherwise returns the all ones address of the mask."""
   if base:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value & mask ) )
   else:
      return IpAddress( 0xFFFFFFFF & ( ipAddr.value | ~mask ) )

def createAclInfo( action=None, srcAddr=None, destination=None, errorMsg=None ):
   aclRuleInfo = IgmpHostProxyModel.AclInfo()
   aclRuleInfo.action = action
   aclRuleInfo.source = srcAddr
   aclRuleInfo.group = destination
   if errorMsg is not None:
      aclRuleInfo.errorMsg = errorMsg
   return aclRuleInfo

# examines the fitness of an acl IpRuleConfig as a SourceGroup range.
def checkSourceGroupRule( rule ):
   # Ip addr/prefix len
   destination = rule.filter.destination
   destAddr = IpAddress( destination.address )
   destmask = 0xFFFFFFFF & destination.mask
   destmin = applyMask( destAddr, destmask )
   destmax = applyMask( destAddr, destmask, base=False )
   source = rule.filter.source
   srcAddr =  IpAddress( source.address )
   srcmask =  0xFFFFFFFF & source.mask

   if not( isMulticast( destmin ) and isMulticast( destmax )  ) :
      ruleInfo = createAclInfo( rule.action, source.stringValue,
            destination.stringValue,
            "Group address range contains invalid multicast address." )
      return ruleInfo
   if srcmask == 0:
      if srcAddr.value != 0:
         ruleInfo = createAclInfo( rule.action, source.stringValue,
               destination.stringValue,
               "Source address is an invalid unicast address." )
         return ruleInfo
      else:
         ruleInfo = createAclInfo( rule.action, source.stringValue,
               destination.stringValue, None )
         return ruleInfo
   elif srcmask == 0xFFFFFFFF:
      if not isUnicast( srcAddr ):
         ruleInfo = createAclInfo( rule.action, source.stringValue,
               destination.stringValue,
               "Source address is an invalid unicast address." )
         return ruleInfo

      else:
         ruleInfo = createAclInfo( rule.action, source.stringValue,
               destination.stringValue, None )
         return ruleInfo
   else:
      ruleInfo = createAclInfo( rule.action, source.stringValue,
            destination.stringValue,
         "Source address must be a single host or *, not a range." )
      return ruleInfo

def igmpHostProxyFromConfig( intfConfig ):
   intf = IgmpHostProxyModel.IgmpHostProxyInterface()
   intf.unsolicitedReportIntvl = intfConfig.unsolicitedReportIntvl.val
   intf.igmpHostProxyVersion = intfConfig.igmpVersion.val
   intf.deviceName = None
   for grpAddr, gsInfo in intfConfig.groupSourceList.items():
      gsDetail = IgmpHostProxyModel.GroupDetail()
      for incSrc in gsInfo.includeSourceList:
         gsDetail.includeSourceList.append( incSrc )
      for excSrc in gsInfo.excludeSourceList:
         gsDetail.excludeSourceList.append(excSrc )
      intf.groupSource[grpAddr] = gsDetail

   for aclName in intfConfig.aclList:
      ihpAclInfo = IgmpHostProxyModel.IgmpHostProxyAclInfo()
      acl = AclCli.getAclConfig('ip', cliConf=True ).get( aclName )
      if not acl: # pylint: disable=no-else-continue
         ruleInfo = createAclInfo( None, None, None,
               "ACL is added, but not configured" )
         ihpAclInfo.aclInfo[ 0 ] = ruleInfo
         intf.igmpHostProxyAclInfo[ aclName ] = ihpAclInfo
         continue
      else:
         for seq, uid in acl.currCfg.ruleBySequence.items():
            rule = acl.currCfg.ipRuleById[ uid ]
            res = checkSourceGroupRule( rule )
            ihpAclInfo.aclInfo[ seq ] = res
         intf.igmpHostProxyAclInfo[ aclName ] = ihpAclInfo
   return intf

def doShowIpIgmpHostProxyDebug( mode, intf=None ):
   igmpHostProxyInterfaces = IgmpHostProxyModel.IgmpHostProxyInterfaces()
   if intf:
      intfId = intf.name
      config = igmpHostProxyConfig.intfConfig.get( intfId )
      status = igmpHostProxyStatus.intfStatus.get( intfId )
      if not status and not config:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "%s : No IGMP host-proxy configuration found "
               "on this interface" % intf.name )
      else:
         igmpHostProxyInterfaceModel = igmpHostProxyFromConfig(
               igmpHostProxyConfig.intfConfig[ intfId ] )
         igmpHostProxyInterfaces.IgmpHostProxyInterfaces = \
               { intfId : igmpHostProxyInterfaceModel }
   else:
      if not igmpHostProxyConfig.intfConfig:
         mode.addWarning( "IGMP host-proxy not configured on any interface." )
      for intfId, intfConfig in igmpHostProxyConfig.intfConfig.items():
         igmpHostProxyInterfaceModel = igmpHostProxyFromConfig( intfConfig )
         igmpHostProxyInterfaces.IgmpHostProxyInterfaces[ intfId ] = \
               igmpHostProxyInterfaceModel

   return igmpHostProxyInterfaces

def igmpHostProxyFromStatus( intfStatus, intfId, detail ):
   intf = IgmpHostProxyModel.IgmpHostProxyInterface()
   intf.unsolicitedReportIntvl = intfStatus.unsolicitedReportIntvl.val
   intf.deviceName = intfStatus.deviceName
   intf.igmpHostProxyVersion = intfStatus.igmpVersion.val
   if intfStatus.iifAware:
      intf.mrouteMatchCriteria = 'iif'
   else:
      intf.mrouteMatchCriteria = 'all'
   if IgmpHostProxyToggleLib.toggleIgmpHostProxyAclEnabled():
      if intfStatus.aclRuleOrderSequenceNum:
         intf.aclRulesPriorityOrder = 'sequenceNumber'
      else:
         intf.aclRulesPriorityOrder = 'denyPrecedence'

   for grpAddr, gsInfo in intfStatus.groupSourceList.items():
      gsDetail = IgmpHostProxyModel.GroupDetail()
      incSrcs = gsInfo.includeSourceList
      excSrcs = gsInfo.excludeSourceList

      # Group has been denied and should not be sent a join
      if "0.0.0.0" in incSrcs:
         continue

      for incSrc in incSrcs:
         gsDetail.includeSourceList.append( incSrc )
      for excSrc in excSrcs:
         gsDetail.excludeSourceList.append(excSrc )
      intf.groupSource[grpAddr] = gsDetail
   if detail:
      intfDetails = IgmpHostProxyModel.IgmpHostProxyInterfaceDetails()
      intfDetails.v1QueryReceived = intfStatus.v1QueryReceived
      intfDetails.v2GeneralQueryReceived = intfStatus.v2GeneralQueryReceived
      intfDetails.v2GroupQueryReceived = intfStatus.v2GroupQueryReceived
      intfDetails.v3GeneralQueryReceived = intfStatus.v3GeneralQueryReceived
      intfDetails.v3GroupQueryReceived = intfStatus.v3GroupQueryReceived
      intfDetails.v3GroupSourceQueryReceived = intfStatus.v3GroupSourceQueryReceived
      intfDetails.v1ReportsSent = intfStatus.v1ReportsSent
      intfDetails.v2ReportsSent = intfStatus.v2ReportsSent
      intfDetails.v3ReportsSent = intfStatus.v3ReportsSent
      intf.details = intfDetails
   return intf

def doShowIpIgmpHostProxyInterface( mode, intf=None, detail=False ):
   igmpHostProxyInterfaces = IgmpHostProxyModel.IgmpHostProxyInterfaces()
   if intf:
      intfId = intf.name
      status = igmpHostProxyStatus.intfStatus.get( intfId )
      if not status:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "%s : No IGMP host-proxy configuration found "
               "on this interface" % intf.name )
         return igmpHostProxyInterfaces
      else:
         igmpHostProxyInterfaceModel = igmpHostProxyFromStatus(
               igmpHostProxyStatus.intfStatus[ intfId ],
               intfId, detail )
         igmpHostProxyInterfaces.IgmpHostProxyInterfaces = \
               { intfId : igmpHostProxyInterfaceModel }
   else:
      if not igmpHostProxyStatus.intfStatus:
         mode.addWarning( "IGMP host-proxy not configured on any interface." )
      for intfId, ihpIntfStatus in igmpHostProxyStatus.intfStatus.items():
         igmpHostProxyInterfaceModel = igmpHostProxyFromStatus(
               ihpIntfStatus, intfId, detail )
         igmpHostProxyInterfaces.IgmpHostProxyInterfaces[ intfId ] = \
               igmpHostProxyInterfaceModel

   return igmpHostProxyInterfaces

class IpIgmpHostProxyDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp host-proxy ( debug | ( interface [ INTF ] [ detail ] ) )'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : nodeIgmp,
      'host-proxy' : matcherHostProxy,
      'debug' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'debug',
            helpdesc='IGMP host-proxy debug information' ),
         hidden=True ),
      'interface' : 'Show IgmpHostProxy interface information',
      'INTF' : IntfCli.Intf.matcher,
      'detail' : 'Display information in detail',
   }
   cliModel = IgmpHostProxyModel.IgmpHostProxyInterfaces

   @staticmethod
   def handler( mode, args ):
      if 'interface' in args:
         return doShowIpIgmpHostProxyInterface( mode, args.get( 'INTF' ),
               'detail' in args )
      else:
         return doShowIpIgmpHostProxyDebug( mode )

BasicCli.addShowCommandClass( IpIgmpHostProxyDebugCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp host-proxy version VERSION
#--------------------------------------------------------------------------------
class IgmpHostProxyInterfaceVersionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp host-proxy version VERSION'
   noOrDefaultSyntax = 'ip igmp host-proxy version ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'host-proxy' : matcherHostProxy,
      'version' : 'IGMP version on IGMP host-proxy interface',
      'VERSION' : CliMatcher.IntegerMatcher( IgmpVersionType().min,
         IgmpVersionType().max, helpdesc='Version' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIgmpHostProxyIntfConfig( mode )
      assert intfConfig
      intfConfig.igmpVersion = IgmpVersionType( args[ 'VERSION' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfName = mode.intf.name
      intfConfig = igmpHostProxyConfig.intfConfig.get( intfName )
      if intfConfig:
         intfConfig.igmpVersion = IgmpVersionType()
         deleteIhpIntfConfigIfDefault( intfName )

modelet.addCommandClass( IgmpHostProxyInterfaceVersionCmd )

#-------------------------------------------------------------------------------
# [ no | default ] host-proxy match mroute ( all | iif )
#-------------------------------------------------------------------------------
class IgmpHostProxyMatchMrouteCmd( CliCommand.CliCommandClass ):
   syntax = 'host-proxy match mroute ( all | iif )'
   noOrDefaultSyntax = 'host-proxy match mroute ...'
   data = {
         'host-proxy' : matcherHostProxy,
         'match': 'Conditional proxying of IGMP joins',
         'mroute': 'Match multicast route attributes',
         'all': 'All multicast routes( default )',
         'iif': 'Incoming interface for multicast route',
   }
   @staticmethod
   def handler( mode, args ):
      # The callback hooks registered with
      # IgmpGlobalConfigMode.addRouterIgmpVrfModeHook
      # ensure that we have a vrfConfig entry,
      # hence directly fetch the entry and use it
      vrfConfig = getVrfConfig( mode.vrfName )
      if 'iif' in args:
         vrfConfig.iifAware = True
      else:
         vrfConfig.iifAware = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfConfig = getVrfConfig( mode.vrfName )
      vrfConfig.iifAware = False

RouterIgmpSharedModelet.addCommandClass( IgmpHostProxyMatchMrouteCmd )

class IgmpHostProxyAclRulePriorityOrderCmd( CliCommand.CliCommandClass ):
   syntax = 'host-proxy access-list rules priority order sequence-number'
   noOrDefaultSyntax = syntax
   data = {
         'host-proxy': matcherHostProxy,
         'access-list': 'Non-standard Access List',
         'rules': 'ACL rules',
         'priority': 'IGMP host-proxy ACL rules priority',
         'order': 'IGMP host-proxy ACL rules priority order',
         'sequence-number': 'IGMP host-proxy ACL rules priority order per'
          ' sequence number',
   }

   @staticmethod
   def handler( mode, args ):
      globalVar.igmpHostProxyVrfsConfig.aclRuleOrderSequenceNum = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      globalVar.igmpHostProxyVrfsConfig.aclRuleOrderSequenceNum = False

if IgmpHostProxyToggleLib.toggleIgmpHostProxyAclEnabled():
   RouterMulticastDefaultIpv4Mode.addCommandClass(
         IgmpHostProxyAclRulePriorityOrderCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp host-proxy
#             [ ( GRP_ADDR [ { ( include INCLUDE_SRC ) | 
#                              ( exclude EXCLUDE_SRC ) } ] ) |
#               ( access-list ACL_NAME ) ]
#--------------------------------------------------------------------------------
#enables the proxy service. router sends IGMP join reports
#for all (*,G) entries in the mroute status table and (S,G)
#entries whose IIF matches our interface
#also react to any (*,G) entries that get added in the future
#group or groupACL - reports generated only for the specific groups

def setIgmpHostProxy ( mode, args ):
   aclName = args.get( 'ACL_NAME' )
   group = args.get( 'GRP_ADDR' )
   includeSource = args.get( 'INCLUDE_SRC', [ None ] )[ 0 ]
   excludeSource = args.get( 'EXCLUDE_SRC', [ None ] )[ 0 ]
   includeSourceSpecified = 'INCLUDE_SRC' in args
   excludeSourceSpecified = 'EXCLUDE_SRC' in args

   #No options
   if not aclName and not group:
      ihpIntfConfig = _getOrCreateIgmpHostProxyIntfConfig( mode )
      assert ihpIntfConfig
      ihpIntfConfig.defaultHostProxy = True

      ihpIntfConfig.deletedGroupSourceList.clear()
      ihpIntfConfig.groupSourceList.clear()
      ihpIntfConfig.aclList.clear()
      return

   if aclName:
      acl = AclCli.getAclConfig( 'ip', cliConf=True ).get( aclName )
      if not acl:
         print( 'ACL not configured, but added it anyway' )
      elif acl.standard:
         # pylint: disable-next=consider-using-f-string
         mode.addError( 'Please configure Non-standard ACL %s' % aclName )
         return
      elif not acl.standard:
         # pylint: disable-next=consider-using-f-string
         print( 'Non-standard ACL %s added' % aclName )
      ihpIntfConfig = _getOrCreateIgmpHostProxyIntfConfig( mode )
      assert ihpIntfConfig
      ihpIntfConfig.aclList[ aclName ] = True
      return

   if includeSourceSpecified and excludeSourceSpecified:
      if includeSource == excludeSource:
         print( "Include and exclude source are same!" )
         return

   if not isMulticast( group ):
      print( "Group address is not a valid multicast address" )
      return
   if includeSourceSpecified:
      if not isUnicast( includeSource ):
         print( "Include source address is not a valid unicast address" )
         return
      if includeSource == '0.0.0.0':
         print( "Include source address can't be zero address" )
         print( "use 'ip igmp host-proxy [group]' to configure *,G" )
         return
   if excludeSourceSpecified:
      if not isUnicast( excludeSource ):
         print( "Exclude source address is not a valid unicast address" )
         return
      if excludeSource == '0.0.0.0':
         print( "Exclude source address can't be zero address" )
         print( "use 'ip igmp host-proxy [group]' to configure *,G" )
         return

   starSource = Arnet.IpAddr( '0.0.0.0' )
   ihpIntfConfig = _getOrCreateIgmpHostProxyIntfConfig( mode )
   assert ihpIntfConfig

   if group not in ihpIntfConfig.groupSourceList:
      groupSourceLists = ihpIntfConfig.groupSourceList.newMember( group )
   else:
      groupSourceLists = ihpIntfConfig.groupSourceList[group]
   if includeSourceSpecified:
      if starSource in groupSourceLists.excludeSourceList:
         del groupSourceLists.excludeSourceList[ starSource ]
      if includeSource in groupSourceLists.excludeSourceList:
         del groupSourceLists.excludeSourceList[ includeSource ]
      groupSourceLists.includeSourceList[includeSource] = True

      if group in ihpIntfConfig.deletedGroupSourceList:
         if includeSource in ihpIntfConfig.deletedGroupSourceList[ group
               ].includeSourceList:
            del ihpIntfConfig.deletedGroupSourceList[ group
                  ].includeSourceList[ includeSource ]
   if excludeSourceSpecified:
      if starSource in groupSourceLists.excludeSourceList:
         del groupSourceLists.excludeSourceList[ starSource ]
      if excludeSource in groupSourceLists.includeSourceList:
         del groupSourceLists.includeSourceList[ excludeSource ]
      groupSourceLists.excludeSourceList[excludeSource] = True

      if group in ihpIntfConfig.deletedGroupSourceList:
         if excludeSource in ihpIntfConfig.deletedGroupSourceList[ group
            ].excludeSourceList:
            del ihpIntfConfig.deletedGroupSourceList[ group
               ].excludeSourceList[ excludeSource ]

   if not includeSourceSpecified and not excludeSourceSpecified:
      groupSourceLists.includeSourceList.clear()
      groupSourceLists.excludeSourceList.clear()
      groupSourceLists.excludeSourceList[ starSource ] = True

      if group in ihpIntfConfig.deletedGroupSourceList:
         del ihpIntfConfig.deletedGroupSourceList[ group ]

   if group in ihpIntfConfig.deletedGroupSourceList:
      if not ihpIntfConfig.deletedGroupSourceList[ group ].includeSourceList and \
            not ihpIntfConfig.deletedGroupSourceList[ group ].excludeSourceList:
         del ihpIntfConfig.deletedGroupSourceList[ group ]

def noIgmpHostProxy ( mode, args ):
   aclName = args.get( 'ACL_NAME' )
   group = args.get( 'GRP_ADDR' )
   includeSource = args.get( 'INCLUDE_SRC', [ None ] )[ 0 ]
   excludeSource = args.get( 'EXCLUDE_SRC', [ None ] )[ 0 ]
   includeSourceSpecified = 'INCLUDE_SRC' in args
   excludeSourceSpecified = 'EXCLUDE_SRC' in args
   intfName = mode.intf.name

   #No options
   if not aclName and not group:
      if intfName in igmpHostProxyConfig.intfConfig:
         del igmpHostProxyConfig.intfConfig[ intfName ]
      return

   #Acl
   if aclName:
      ihpIntfConfig = igmpHostProxyConfig.intfConfig.get( intfName )
      if not ihpIntfConfig:
         return
      del ihpIntfConfig.aclList[ aclName ]
      deleteIhpIntfConfigIfDefault( intfName )
      return

   starSource = Arnet.IpAddr( '0.0.0.0' )
   if not isMulticast( group ):
      print( "Group address is not a valid multicast address" )
      return
   if includeSourceSpecified:
      if not isUnicast( includeSource ):
         print( "Include source address is not a valid unicast address" )
         return
      if includeSource == '0.0.0.0':
         print( "Include source address can't be zero address" )
         print( "use 'no ip igmp host-proxy [group]' to remove *,G" )
         return
   if excludeSourceSpecified:
      if not isUnicast( excludeSource ):
         print( "Exclude source address is not a valid unicast address" )
         return
      if excludeSource == '0.0.0.0':
         print( "Exclude source address can't be zero address" )
         print( "use 'no ip igmp host-proxy [group]' to remove *,G" )
         return

   if intfName not in igmpHostProxyConfig.intfConfig:
      return
   ihpIntfConfig = igmpHostProxyConfig.intfConfig.get( intfName )

   if CliCommand.isDefaultCmd( args ):
      defaultIgmpHostProxy( ihpIntfConfig, group, includeSource, excludeSource )
      return

   if group in ihpIntfConfig.groupSourceList:
      gs = ihpIntfConfig.groupSourceList[ group ]
      if not includeSourceSpecified and not excludeSourceSpecified:
         if group not in ihpIntfConfig.deletedGroupSourceList:
            ihpIntfConfig.deletedGroupSourceList.newMember( group )

         for iSrc in gs.includeSourceList:
            ihpIntfConfig.deletedGroupSourceList[ group
                  ].includeSourceList[ iSrc ] = True
         for eSrc in gs.excludeSourceList:
            ihpIntfConfig.deletedGroupSourceList[ group
                  ].excludeSourceList[ eSrc ] = True
         ihpIntfConfig.deletedGroupSourceList[ group
               ].excludeSourceList[ starSource ] = True

         del ihpIntfConfig.groupSourceList[ group ]
         return

      if group in ihpIntfConfig.deletedGroupSourceList:
         deletedGroup = ihpIntfConfig.deletedGroupSourceList[ group ]
      else:
         deletedGroup = ihpIntfConfig.deletedGroupSourceList.newMember( group )

      if includeSourceSpecified:
         if includeSource in gs.includeSourceList:
            del gs.includeSourceList[includeSource]

         deletedGroup.includeSourceList[ includeSource ] = True
         if starSource in deletedGroup.excludeSourceList:
            del deletedGroup.excludeSourceList[ starSource ]
         if includeSource in deletedGroup.excludeSourceList:
            del deletedGroup.excludeSourceList[ includeSource ]

      if excludeSourceSpecified:
         if excludeSource in gs.excludeSourceList:
            del gs.excludeSourceList[excludeSource]

         deletedGroup.excludeSourceList[ excludeSource ] = True
         if starSource in deletedGroup.includeSourceList:
            del deletedGroup.includeSourceList[ starSource ]
         if excludeSource in deletedGroup.includeSourceList:
            del deletedGroup.includeSourceList[ excludeSource ]

      if not ihpIntfConfig.groupSourceList[ group ].includeSourceList and \
            not ihpIntfConfig.groupSourceList[ group ].excludeSourceList:
         del ihpIntfConfig.groupSourceList[ group ]
         deleteIhpIntfConfigIfDefault( intfName )
   else:
      if group not in ihpIntfConfig.deletedGroupSourceList:
         dgs = ihpIntfConfig.deletedGroupSourceList.newMember( group )
      else:
         dgs = ihpIntfConfig.deletedGroupSourceList[ group ]
      if includeSourceSpecified:
         if starSource in dgs.excludeSourceList:
            del dgs.excludeSourceList[ starSource ]
         if includeSource in dgs.excludeSourceList:
            del dgs.excludeSourceList[ includeSource ]
         dgs.includeSourceList[ includeSource ] = True
      if excludeSourceSpecified:
         if starSource in dgs.excludeSourceList:
            del dgs.excludeSourceList[ starSource ]
         if excludeSource in dgs.includeSourceList:
            del dgs.includeSourceList[ excludeSource ]
         dgs.excludeSourceList[excludeSource] = True
      if not includeSourceSpecified and not excludeSourceSpecified:
         dgs.includeSourceList.clear()
         dgs.excludeSourceList.clear()
         dgs.excludeSourceList[ starSource ] = True

def defaultIgmpHostProxy( ihpIntfConfig, group, includeSource=None,
                          excludeSource=None ):
   if not ihpIntfConfig or not group:
      return

   if not includeSource and not excludeSource:
      if group in ihpIntfConfig.groupSourceList:
         del ihpIntfConfig.groupSourceList[ group ]
      if group in ihpIntfConfig.deletedGroupSourceList:
         del ihpIntfConfig.deletedGroupSourceList[ group ]
      deleteIhpIntfConfigIfDefault( ihpIntfConfig.intfId )
      return

   if includeSource:
      if group in ihpIntfConfig.groupSourceList:
         if includeSource in ihpIntfConfig.groupSourceList[ group
               ].includeSourceList:
            del ihpIntfConfig.groupSourceList[ group
                  ].includeSourceList[ includeSource ]
      if group in ihpIntfConfig.deletedGroupSourceList:
         if includeSource in ihpIntfConfig.deletedGroupSourceList[ group
               ].includeSourceList:
            del ihpIntfConfig.deletedGroupSourceList[ group ].includeSourceList[
                  includeSource ]

   if excludeSource:
      if group in ihpIntfConfig.groupSourceList:
         if excludeSource in ihpIntfConfig.groupSourceList[ group
               ].excludeSourceList:
            del ihpIntfConfig.groupSourceList[ group
                  ].excludeSourceList[ excludeSource ]
      if group in ihpIntfConfig.deletedGroupSourceList:
         if excludeSource in ihpIntfConfig.deletedGroupSourceList[ group
               ].excludeSourceList:
            del ihpIntfConfig.deletedGroupSourceList[ group
                  ].excludeSourceList[
                  excludeSource ]

   if group in ihpIntfConfig.groupSourceList:
      if not ihpIntfConfig.groupSourceList[ group ].includeSourceList and \
            not ihpIntfConfig.groupSourceList[ group ].excludeSourceList:
         del ihpIntfConfig.groupSourceList[ group ]

   if group in ihpIntfConfig.deletedGroupSourceList:
      if not ihpIntfConfig.deletedGroupSourceList[ group ].includeSourceList and \
            not ihpIntfConfig.deletedGroupSourceList[ group ].excludeSourceList:
         del ihpIntfConfig.deletedGroupSourceList[ group ]

   deleteIhpIntfConfigIfDefault( ihpIntfConfig.intfId )

class IpIgmpHostProxyCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip igmp host-proxy '
                     '[ ( GRP_ADDR '
                           '[ { ( include INCLUDE_SRC ) | '
                               '( exclude EXCLUDE_SRC ) } ] ) | '
                       '( access-list ACL_NAME ) ]' )
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'host-proxy' : matcherHostProxy,
      'GRP_ADDR' : IpAddrMatcher.IpAddrMatcher( helpdesc='Group Address' ),
      'include' : CliCommand.singleKeyword( 'include', helpdesc='Include source' ),
      'INCLUDE_SRC' : IpAddrMatcher.IpAddrMatcher( helpdesc='Source address' ),
      'exclude' : CliCommand.singleKeyword( 'exclude', helpdesc='Exclude source' ),
      'EXCLUDE_SRC' : IpAddrMatcher.IpAddrMatcher( helpdesc='Source address' ),
      'access-list' : 'Non-standard Access List name',
      'ACL_NAME' : AclCli.ipAclNameMatcher
   }

   handler = setIgmpHostProxy
   noOrDefaultHandler = noIgmpHostProxy

modelet.addCommandClass( IpIgmpHostProxyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip igmp host-proxy report-interval INTERVAL
#--------------------------------------------------------------------------------
class UnsolicitedReportIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip igmp host-proxy report-interval INTERVAL'
   noOrDefaultSyntax = 'ip igmp host-proxy report-interval ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'igmp' : nodeIgmp,
      'host-proxy' : matcherHostProxy,
      'report-interval' : 'Time interval between unsolicited reports',
      'INTERVAL' : CliMatcher.IntegerMatcher( ReportIntervalType().min,
         ReportIntervalType().max,
         helpdesc='Set the time interval between unsolicited reports' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIgmpHostProxyIntfConfig( mode )
      if intfConfig:
         val = ReportIntervalType( float( args[ 'INTERVAL' ] ) )
         intfConfig.unsolicitedReportIntvl = val

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      #default unsolicited-report-interval is 1 second
      #no version sets the unsolicited-report-interval to default value
      intfName = mode.intf.name
      intfConfig =  igmpHostProxyConfig.intfConfig.get( intfName )
      if intfConfig:
         intfConfig.unsolicitedReportIntvl = ReportIntervalType()
         deleteIhpIntfConfigIfDefault( intfName )

modelet.addCommandClass( UnsolicitedReportIntervalCmd )

#--------------------------------------------------------------------------------
# show ip igmp host-proxy config-sanity
#--------------------------------------------------------------------------------
def checkOverlap( action, groupAddr, srcAddr, ihpConfigs, overlappingGroups ):
   configStr = action + '_' + groupAddr + '_' + srcAddr
   ihpConfigs.add( configStr )
   oppAction = None
   if action == 'permit':
      oppAction =  'deny'
   elif action == 'deny':
      oppAction = 'permit'
   elif action == 'no_permit':
      oppAction = 'permit'
   elif action == 'no_deny':
      oppAction = 'deny'
   oppConfigStr = oppAction + '_' + groupAddr + '_' + srcAddr
   if oppConfigStr in ihpConfigs:
      overlappingGroups.add( groupAddr )

class IpIgmpHostProxyConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip igmp host-proxy config-sanity'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'igmp' : nodeIgmp,
      'host-proxy' : matcherHostProxy,
      'config-sanity' : 'Show hints for potential IGMP Host-Proxy problems',
   }
   cliModel = IgmpHostProxyModel.IgmpHostProxyConfigSanity

   @staticmethod
   def handler( mode, args ):
      model = {}

      for intfId, intfConfig in igmpHostProxyConfig.intfConfig.items():
         ihpConfigs = set()
         overlappingGroups = set()
         sourceFilteredGroups = set()
         # Empty if IGMP host-proxy is not configured as version 1 or 2
         igmpHostProxyInterfaceModel = igmpHostProxyFromConfig( intfConfig )

         model[ intfId ] = IgmpHostProxyModel.IHPPotentialMisconfigs()
         if igmpHostProxyInterfaceModel.igmpHostProxyVersion in [ 1, 2 ]:
            model[ intfId ].version = \
                  str( igmpHostProxyInterfaceModel.igmpHostProxyVersion )

         for( aclName, aclDetails ) in \
               sorted( igmpHostProxyInterfaceModel.igmpHostProxyAclInfo.items() ):
            for aclInfoDetail in aclDetails.aclInfo.values():
               if aclInfoDetail.errorMsg == \
                     "ACL is added, but not configured":
                  # if the acl is empty
                  continue

               if aclInfoDetail.action == 'deny' \
                     and aclInfoDetail.source == '0.0.0.0/0' \
                     and aclInfoDetail.group == '0.0.0.0/0':
                  model[ intfId ].denyIpMrouteAcls.append( aclName )
                  continue

               if igmpHostProxyInterfaceModel.igmpHostProxyVersion in [ 1, 2 ] \
                     and not aclInfoDetail.source == '0.0.0.0/0':
                  sourceFilteredGroups.add( aclInfoDetail.group )

               checkOverlap( aclInfoDetail.action, aclInfoDetail.group,
                     aclInfoDetail.source, ihpConfigs, overlappingGroups )

         for( groupAddr, groupDetail ) in \
               igmpHostProxyInterfaceModel.groupSource.items():
            if not groupDetail.includeSourceList and \
                  groupDetail.excludeSourceList == [ '0.0.0.0' ]:
               checkOverlap( 'permit', groupAddr + '/32', '0.0.0.0/0',
                     ihpConfigs, overlappingGroups )
            else:
               if igmpHostProxyInterfaceModel.igmpHostProxyVersion in [ 1, 2 ]:
                  sourceFilteredGroups.add( groupAddr + '/32'  )
               for incSrc in groupDetail.includeSourceList:
                  checkOverlap( 'permit', groupAddr + '/32',
                        incSrc + '/32', ihpConfigs, overlappingGroups )
               for excSrc in groupDetail.excludeSourceList:
                  checkOverlap( 'deny', groupAddr + '/32',
                        excSrc + '/32', ihpConfigs, overlappingGroups )

         #Go through the deleted group source list ('no' configs)
         for( groupAddr, groupDetail ) in \
               ( igmpHostProxyConfig.intfConfig[
                     intfId ].deletedGroupSourceList ).items():
            if not groupDetail.includeSourceList and \
                  list( groupDetail.excludeSourceList ) == [ '0.0.0.0' ]:
               checkOverlap( 'deny', groupAddr + '/32',
                     '0.0.0.0/0', ihpConfigs, overlappingGroups )
            else:
               if igmpHostProxyInterfaceModel.igmpHostProxyVersion in [ 1, 2 ]:
                  sourceFilteredGroups.add( groupAddr + '/32'  )
               for incSrc in groupDetail.includeSourceList:
                  checkOverlap( 'no_permit', groupAddr + '/32',
                        incSrc + '/32', ihpConfigs, overlappingGroups )
               for excSrc in groupDetail.excludeSourceList:
                  checkOverlap( 'no_deny', groupAddr + '/32',
                        excSrc + '/32', ihpConfigs, overlappingGroups )

         for overlappingGroup in sorted( overlappingGroups ):
            ( groupIp, groupMask ) = overlappingGroup.split( '/' )
            model[ intfId ].overlappingGroups.append( \
                  IpAddrAndMask( ip=groupIp, mask=int( groupMask ) ) )
         for sourceFilteredGroup in sorted( sourceFilteredGroups ):
            ( groupIp, groupMask ) = sourceFilteredGroup.split( '/' )
            model[ intfId ].sourceFilteredGroups.append( \
                  IpAddrAndMask( ip=groupIp, mask=int( groupMask ) ) )

      return IgmpHostProxyModel.IgmpHostProxyConfigSanity( intfMisconfigs=model )

BasicCli.addShowCommandClass( IpIgmpHostProxyConfigSanityCmd )

IgmpGlobalConfigMode.deleteRouterIgmpModeHook.addExtension(
      deleteIgmpHostProxyAllVrfsConfig )
IgmpGlobalConfigMode.deleteRouterIgmpVrfModeHook.\
      addExtension( deleteRouterIgmpVrfCb )
IgmpGlobalConfigMode.addRouterIgmpVrfModeHook.\
      addExtension( addRouterIgmpVrfCb )
MrouteCli.routerMcastVrfDeletionHook.\
      addExtension( deleteRouterMulticastCb )
#----------------------------------------------------------------------------
def Plugin (entityManager ):
   global igmpHostProxyConfig, igmpHostProxyStatus
   global _entityManager

   _entityManager = entityManager
   igmpHostProxyConfig = ConfigMount.mount( entityManager,
      'igmphostproxy/config',
      'IgmpHostProxy::Config', 'w' )
   tacType = 'IgmpHostProxy::AllVrfConfig'
   globalVar.igmpHostProxyVrfsConfig = ConfigMount.mount( entityManager,
         Tac.Type( tacType ).mountPath, tacType, 'w' )
   igmpHostProxyStatus = LazyMount.mount( entityManager,
      'igmphostproxy/status',
      'IgmpHostProxy::Status', 'r' )
   IntfCli.Intf.registerDependentClass(IgmpHostProxyIntf, priority=10 )

