#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=too-many-nested-blocks

import Tac
from CliModel import Model, List, Dict, Str, Float, Submodel
from CliModel import Int, Enum
from ArnetModel import Ip4Address, IpAddrAndMask 
from IntfModels import Interface
from Toggles import IgmpHostProxyToggleLib

groupAddressWidth = 17
intfWidth = 25
includeSrcWidth = 20
excludeSrcWidth = 20
actionWidth = 15
srcWidth = 12
grpWidth = 18
errorMsgWidth = 71
delimiter = ';'

class IgmpHostProxyInterfaceDetails( Model ):
   '''Render the IgmpHostProxyInterface model with counter information'''
   v1QueryReceived = Int( help='IGMP V1 Queries recieved' )
   v2GeneralQueryReceived = Int( help='IGMP V2 General Queries received' )
   v2GroupQueryReceived = Int( help='IGMP V2 Group Queries received' )
   v3GeneralQueryReceived = Int( help='IGMP V3 General Queries received' )
   v3GroupQueryReceived = Int( help='IGMP V3 Group Queries received' )
   v3GroupSourceQueryReceived = Int( help='IGMP V3 Group-Source specific received' )
   v1ReportsSent = Int( help='IGMP V1 reports sent' )
   v2ReportsSent = Int( help='IGMP V2 reports sent' )
   v3ReportsSent = Int( help='IGMP V3 reports sent' )

class AclInfo( Model ):
   '''ACL info'''
   action = Str( help = "Action on the ACL rule( permit/deny )", optional=True )
   source = Str( help = "IpAddress of the source in the rule", optional=True )
   group = Str( help = "IpAddress of the group in the rule", optional=True )
   errorMsg = Str( help="Message describing the error in the rule, if any.",
         optional=True )

class IgmpHostProxyAclInfo( Model ):
   '''ACL info'''
   aclInfo = Dict( keyType=int, valueType=AclInfo,
         help="Mapping between sequence number and information about the rule" )

class GroupDetail( Model ):
   '''IGMP host-proxy model for group detail '''
   includeSourceList = List( valueType=Ip4Address,
         help='List of include sources' )
   excludeSourceList = List( valueType=Ip4Address,
         help='List of exclude sources' )

class IgmpHostProxyInterface( Model ):
   '''CAPI model for IGMP host-proxy interface'''
   unsolicitedReportIntvl = Float( 
         help='Unsolicited-report interval on this interface in seconds' )
   igmpHostProxyVersion = Int( 
         help='IGMP host-proxy version on this interface')
   deviceName = Str( help='Device name', optional=True )
   mrouteMatchCriteria = Enum( help='host-proxy IGMP join criteria',
                        values=( 'all', 'iif' ),
                        default='all' )
   if IgmpHostProxyToggleLib.toggleIgmpHostProxyAclEnabled():
      aclRulesPriorityOrder = Enum( help='ACL rules priority order',
                             values=( 'denyPrecedence', 'sequenceNumber' ),
                             default='denyPrecedence' )
   groupSource = Dict( keyType=Ip4Address, valueType=GroupDetail,
         help='Mapping between group address and sources in \
               include/exclude mode' )
   igmpHostProxyAclInfo = Dict( keyType=str, valueType=IgmpHostProxyAclInfo, 
         help="Mapping between ACL name and information about ACL rules" )
   details = Submodel( help="Detailed interface information", 
         valueType=IgmpHostProxyInterfaceDetails, optional=True )
   
class IgmpHostProxyInterfaces( Model ):
   '''CAPI Model for show ip igmp host-proxy interface <> '''
   IgmpHostProxyInterfaces = Dict( keyType=Interface, valueType=
         IgmpHostProxyInterface, help='Map of IGMP host-proxy interfaces' )

   def render( self ):
      for( intfId, igmpHostProxyIntf ) in sorted( \
            self.IgmpHostProxyInterfaces.items() ):
         print( 'IGMP host-proxy configured on: %s ' % intfId )
         print( 'IGMP host-proxy version: %s'
                % igmpHostProxyIntf.igmpHostProxyVersion )
         print( 'Unsolicited-report interval: %s'
                % igmpHostProxyIntf.unsolicitedReportIntvl )
         if igmpHostProxyIntf.deviceName:
            print( 'Device name: %s' % igmpHostProxyIntf.deviceName )
         if igmpHostProxyIntf.mrouteMatchCriteria == 'all':
            print( 'Mroute match criteria: ', 'All routes' )
         else:
            print( 'Mroute match criteria: ', 'Incoming interface' )
         if IgmpHostProxyToggleLib.toggleIgmpHostProxyAclEnabled():
            if igmpHostProxyIntf.aclRulesPriorityOrder == 'denyPrecedence':
               print( 'ACL rules priority order: ', 'Deny precedence' )
            else:
               print( 'ACL rules priority order:', 'Sequence number' )
         if igmpHostProxyIntf.groupSource:
            print( '\nInterface'.ljust( intfWidth, ' ' ) + 
                  'Group Address'.ljust( groupAddressWidth, ' ' ) + 
                  'IncludeSrc'.ljust( includeSrcWidth, ' ' ) +
                  'ExcludeSrc'.ljust( excludeSrcWidth, ' ' ) )
         for group, gs in igmpHostProxyIntf.groupSource.items():
            for incSrc in gs.includeSourceList:
               print( ( '%s' % intfId ).ljust( 
                  intfWidth ) + ( '%s' % group ).ljust( groupAddressWidth ) +
                  ( '%s' % incSrc ).ljust( includeSrcWidth ) )
            for excSrc in gs.excludeSourceList:
               if excSrc != '0.0.0.0' :
                  print( ( '%s' % intfId ).ljust( 
                     intfWidth ) + ( '%s' % group ).ljust( groupAddressWidth ) +
                     ''.ljust( includeSrcWidth ) + 
                     ( '%s' % excSrc ).ljust( excludeSrcWidth ) )
               else:
                  print( ( '%s' % intfId ).ljust( 
                     intfWidth ) + ( '%s' % group ).ljust( groupAddressWidth ) )
         if igmpHostProxyIntf.igmpHostProxyAclInfo:
            print( "Configuration through Access-list:" )
            print( '\nAccess-List'.ljust( intfWidth, ' ' ) + 
                  'Action'.ljust( actionWidth, ' ' ) +
                  'Source'.ljust( actionWidth, ' ' ) +
                  'Group Address'.ljust( groupAddressWidth, ' ' ) + 
                  'Errors' )

            for aclName, aclDetails in \
                  igmpHostProxyIntf.igmpHostProxyAclInfo.items():
               for aclInfoDetail in aclDetails.aclInfo.values():
                  if aclInfoDetail.action is not None:
                     if aclInfoDetail.errorMsg:
                        print( ( '%s' % aclName ).ljust( intfWidth ) +
                              ( '%s' % aclInfoDetail.action ).ljust( srcWidth ) + 
                              ( '%s' % aclInfoDetail.source ).ljust( actionWidth ) +
                              ( '%s' % aclInfoDetail.group ).ljust( grpWidth ) + 
                              ( '%s' % aclInfoDetail.errorMsg ) )
                     else:
                        print( ( '%s' % aclName ).ljust( intfWidth ) +
                              ( '%s' % aclInfoDetail.action ).ljust( srcWidth ) + 
                              ( '%s' % aclInfoDetail.source ).ljust( actionWidth ) +
                              ( '%s' % aclInfoDetail.group ).ljust( grpWidth ) )
                  else:
                     print( ( '%s' % aclName ).ljust( errorMsgWidth ) +
                           ( '%s' % aclInfoDetail.errorMsg ) )
         if igmpHostProxyIntf.details:
            print( '\nIGMP host-proxy statistics:\n' )
            print( 'IGMP v1 Queries received: %s '
                   % igmpHostProxyIntf.details.v1QueryReceived )
            print( 'IGMP v2 General-Queries received: %s '
                   % igmpHostProxyIntf.details.v2GeneralQueryReceived )
            print( 'IGMP v2 Group-Queries received: %s '
                   % igmpHostProxyIntf.details.v2GroupQueryReceived )
            print( 'IGMP v3 General-Queries received: %s '
                   % igmpHostProxyIntf.details.v3GeneralQueryReceived )
            print( 'IGMP v3 Group-Queries recevied: %s '
                   % igmpHostProxyIntf.details.v3GroupQueryReceived )
            print( 'IGMP v3 Group-Source Queries received: %s '
                   % igmpHostProxyIntf.details.v3GroupSourceQueryReceived )
            print( 'IGMP v1 Reports sent: %s '
                   % igmpHostProxyIntf.details.v1ReportsSent )
            print( 'IGMP v2 Reports sent: %s '
                   % igmpHostProxyIntf.details.v2ReportsSent )
            print( 'IGMP v3 Reports sent: %s '
                   % igmpHostProxyIntf.details.v3ReportsSent )

class IHPPotentialMisconfigs( Model ):
   '''IGMP Host Proxy Potential misconfigurations on an
   interface for show igmp host-proxy config-sanity'''
   denyIpMrouteAcls = List( help='Access-lists with "deny ip any any" '\
                                 'rule', valueType=str, optional=True)
   overlappingGroups = List( help='Groups with overlapping permit and deny '\
                                  'configurations', valueType=IpAddrAndMask, 
                                  optional=True )
   sourceFilteredGroups = List( help='Groups with include or exclude sources ', 
                                valueType=IpAddrAndMask, optional=True )
   version = Enum( help='IGMP Host-Proxy configured as IGMP version ( 1 or 2 )', 
                   values=( '1', '2' ), optional=True )

class IgmpHostProxyConfigSanity( Model ):
   '''CAPI Model for show igmp host-proxy config-sanity'''
   intfMisconfigs = Dict( keyType=Interface, valueType=IHPPotentialMisconfigs,
                          help='Mapping between interface and the IHP '\
                               'potential misconfigs on it' )
   def render( self ):
      print( 'Below are hints of potential IGMP Host-Proxy misconfigurations' )
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.intfMisconfigs ):
         print( 'IGMP Host-Proxy is not configured on any interface' )
         return
      for( intfId, hints ) in sorted( self.intfMisconfigs.items() ):
         print( 'IGMP host-proxy configured on interface %s:' % intfId )
         if( hints.denyIpMrouteAcls or hints.overlappingGroups 
               or hints.sourceFilteredGroups ):
            if hints.denyIpMrouteAcls:
               print( 'Access-lists having "deny ip any any" rule:' )
               for acl in hints.denyIpMrouteAcls:
                  print( '%s' % acl )
            if hints.overlappingGroups:
               print( 'Groups with overlapping permit and deny configurations:' )
               for group in hints.overlappingGroups:
                  print( '%s' % group.formatStr() )
            if hints.sourceFilteredGroups:
               print( 'Groups with source filters configured with IGMP '
                      'Host-Proxy set to version %s :' % hints.version )
               for group in hints.sourceFilteredGroups:
                  print( '%s' % group.formatStr() )
         else:
            print( 'No IGMP Host-Proxy misconfiguration hints '
                   'found on this interface' )
         print()
