#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel

class DestinationInfluxdb( CliModel.Model ):
   vrfName = CliModel.Str( help='VRF telegraf will use to transmit data' )
   url = CliModel.Str( help='URL of the InfluxDB destination' )
   databaseName = CliModel.Str( help='Database name of the InfluxDB destination' )
   retentionPolicyName = CliModel.Str(
      help='Retention policy name of the InfluxDB destination' )

   def fetchSysdbState( self, dest ):
      self.vrfName = dest.vrfName
      self.url = dest.url
      self.databaseName = dest.databaseName
      self.retentionPolicyName = dest.retentionPolicyName

class SourceSocket( CliModel.Model ):
   url = CliModel.Str( help='URL of the socket' )
   connectionLimit = CliModel.Int( help='Maximum number of connections allowed' )

   def fetchSysdbState( self, socket ):
      self.url = socket.url
      self.connectionLimit = socket.connectionLimit

class InfluxTelemetry( CliModel.Model ):
   sourceGroups = CliModel.Dict(
      help='A mapping between a source group name and its state',
      keyType=str, valueType=str )
   destinationInfluxdb = CliModel.Dict(
      help='A mapping between an InfluxDB destination name and an '
      'InfluxDB destination context',
      keyType=str, valueType=DestinationInfluxdb )
   sourceSocket = CliModel.Dict(
      help='A mapping between a socket name and a socket context',
      keyType=str, valueType=SourceSocket )
   globalTags = CliModel.Dict(
      help='A mapping between a global tag key and a global tag value',
      keyType=str, valueType=str )
   telegrafRunning = CliModel.Bool( help='Telegraf running status' )

   def render( self ):
      print( 'Telegraf status: {}'.format(
         'running' if self.telegrafRunning else 'stopped'
      ) )
      print( '' )

      print( 'Source Groups' )
      for group, enabled in sorted( self.sourceGroups.items() ):
         print( f'Name: {group} Status: {enabled}' )
      print( '' )

      print( 'Destinations' )
      for name, dest in sorted( self.destinationInfluxdb.items() ):
         print( '' )
         print( f'Name: {name}' )
         print( 'Destination type: InfluxDB' )
         print( f'VRF: {dest.vrfName} URL: {dest.url}' )
         print( f'Database name: {dest.databaseName}' )
         print( f'Retention policy: {dest.retentionPolicyName}' )
      print( '' )

      print( 'Sockets' )
      for name, socket in sorted( self.sourceSocket.items() ):
         print( '' )
         print( f'Name: {name}' )
         print( 'Source type: socket' )
         print( f'URL: {socket.url}' )
         print( 'Connection limit: {}'.format(
            socket.connectionLimit
            if socket.connectionLimit else 'unlimited' ) )
      print( '' )

      print( 'Global Tags' )
      for key, value in sorted( self.globalTags.items() ):
         print( f'{key}: {value}' )
      print( '' )
