#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=raise-missing-from

import random, sys
import os
import re
import Tac
import BasicCliUtil
import Fru
from CliPlugin import FruCli
from CliPlugin import ReloadCli
import EosVersion
import LazyMount
import FileReplicationCmds, Url, SimpleConfigFile, Cell
import FileUrl

#------------------------------------------------------
# The "install" command, in "enable" mode
#
# The full syntax of this command is:
#
#   install image_src [image_dest] [reload] [now]
#------------------------------------------------------

def getSupportedOptimizationsFromSwi( swiPath ):
   """ Looks for supported optimization in a given swi.

   This function acts exactly like the one in SwimHelperLib, but it does not use
   'memory hogs' like zipfile.

   Args:
      swiPath - str - Path to swi image
   Returns:
      List of supported optimizations
   """
   optimizations = []
   zippedFiles = Tac.run( [ 'zipinfo', '-1', swiPath ], stdout=Tac.CAPTURE )
   zippedFiles = set( zippedFiles.split( '\n' ) )
   if EosVersion.swimHdrFile not in zippedFiles:
      return optimizations

   swimSqshMapData = Tac.run( [ 'unzip', '-p', swiPath, EosVersion.swimHdrFile ],
                              stdout=Tac.CAPTURE )
   for l in swimSqshMapData.splitlines():
      optimization, sqshList = l.split( '=', 1 )
      sqshes = sqshList.split( ':' )
      if set( sqshes ).issubset( zippedFiles ):
         optimizations.append( optimization )

   return optimizations

class RevertChangesException( Exception ):
   def __init__( self, error='', aborted=False ):
      self.error = error
      self.aborted = aborted

class CommandFailedException( Exception ):
   def __init__( self, error='', aborted=False ):
      self.error = error
      self.aborted = aborted

class StandbyTransferException( Exception ):
   def __init__( self, error, isMessage=False, standbyMissing=False ):
      self.error = error
      self.isMessage = isMessage
      self.standbyMissing = standbyMissing

entityManager_ = None
isModularSystem_ = False
redundancyStatus_ = None

def printAndFlush( msg ):
   print( msg, end=' ' )
   sys.stdout.flush()

class Installer:
   # NOTE: Tac.run(...) uses stderr=Tac.CAPTURE because without it
   # Tac.SystemCommandError.output is an empty string in case the
   # command fails and throws the exception

   def __init__( self ):
      self.currId_ = Fru.slotId()
      self.bootConfFile_ = None
      self.swiFile_ = None
      self.bootExtFile_ = None
      self.tmpBootConfFile_ = None
      self.tmpSwiFile_ = None
      self.isModularSystem_ = False
      self.mode_ = None
      self.now_ = False
      self.skipDownload_ = False
      self.extensions_ = set()

   def _promptUser( self, question, defaultChoice ):
      if self.now_:
         return defaultChoice

      return BasicCliUtil.confirm( self.mode_, question,
                                   answerForReturn=defaultChoice,
                                   nonInteractiveAnswer=defaultChoice )

   def removeFileForSure( self, faile ):
      if os.path.isfile( faile ):
         with BasicCliUtil.RootPrivilege( 0 ):
            os.unlink( faile )

   def _deleteTempFile( self, filename, onActive=True ):
      assert 'tmp' in filename

      if onActive:
         self.removeFileForSure( filename )
      else:
         cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                               target=filename )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )

   def _revertChanges( self ):
      assert self.tmpSwiFile_ and self.tmpBootConfFile_

      printAndFlush( 'Trying to revert changes...' )
      try:
         if not self.skipDownload_:
            self.removeFileForSure( self.tmpSwiFile_ )
         self.removeFileForSure( self.tmpBootConfFile_ )
         if self.isModularSystem_:
            cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                                  target=self.tmpSwiFile_ )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
            cmd = FileReplicationCmds.deleteFile( self.currId_, useKey=True,
                                                  target=self.tmpBootConfFile_ )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
         print( 'done.' )
      except Tac.SystemCommandError as e:
         print( '' )
         # should never get here, but this is an attempt to die gracefully
         # incase something unexpected happens
         self.mode_.addError( 'Unexpected error: %s' % e )

   def _syncFileStandby( self, path ):
      cmd = FileReplicationCmds.syncFile( self.currId_, path, useKey=True )
      try:
         Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
      except Tac.SystemCommandError as e:
         if 'command not found' not in e.output:
            # The standby might be running an older version,
            # ignore this error
            raise
         print( "syncFile: command not found, continue..." )

   def _commitChanges( self ):
      try:
         if self.isModularSystem_:
            printAndFlush( 'Committing changes on standby supervisor...' )
            if self.tmpSwiFile_ != self.swiFile_:
               cmd = FileReplicationCmds.rename( self.currId_, self.swiFile_, 
                                                 self.tmpSwiFile_, useKey=True )
               Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
               self._syncFileStandby( self.swiFile_ )

            cmd = FileReplicationCmds.rename( self.currId_, self.bootConfFile_, 
                                              self.tmpBootConfFile_, useKey=True )
            Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
            self._syncFileStandby( self.bootConfFile_ )
            print( 'done.' )
         s = ' on this supervisor' if self.isModularSystem_ else ''
         printAndFlush( 'Committing changes%s...' % s )
         if not self.skipDownload_:
            Tac.run( [ 'mv', self.tmpSwiFile_, self.swiFile_ ], asRoot=True )
         Tac.run( [ 'SyncFile', self.swiFile_ ], asRoot=True )
         Tac.run( [ 'mv', self.tmpBootConfFile_, self.bootConfFile_ ],
                  asRoot=True )
         Tac.run( [ 'SyncFile', self.bootConfFile_ ], asRoot=True )
         print( 'done.' )
      except Tac.SystemCommandError as e:
         print( '' )
         # should never get here, but this is an attempt to wind up changes
         # and die gracefully in case something unexpected happens
         raise RevertChangesException( error='Unexpected error: %s' % e )

   def _transferToStandby( self, srcAddr, destAddr ):
      idx = destAddr.rfind( '/' )
      # if transfering file to a specific directory, create it first
      if idx > 0:
         cmd = FileReplicationCmds.createDir( self.currId_, destAddr[:idx], 
                                              useKey=True )
         Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
      cmd = FileReplicationCmds.copyFile( self.currId_, destAddr, srcAddr, 
                                          useKey=True )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )  
      self._syncFileStandby( destAddr )

   def _initializeFilenames( self, srcUrl, destUrl ):
      try:
         delim = '/' if '/' in srcUrl.url else ':'
         f = srcUrl.url.split( delim )[ -1 ]
         if not f.endswith( '.swi' ):
            question = 'Extension does not match \'.swi\'. Continue? [Y/n]'
            if not self._promptUser( question, True ):
               raise CommandFailedException( aborted=True )
         # if destUrl not provided AND the source url is on flash:/, make sure that
         # the file exists, and then skip download
         if destUrl is None:
            if srcUrl.fs.scheme in [ 'flash:', 'file:' ]:
               srcUrl.open()
               if srcUrl.fs.scheme == 'flash:':
                  self.skipDownload_ = True
                  destUrl = srcUrl
            else:
               context = Url.Context( *Url.urlArgsFromMode( self.mode_ ) )
               destUrl = Url.parseUrl( 'flash:/' + f, context )
         elif destUrl.isdir():
            # Destination is a directory, so add the file name taken
            # from src url.
            # Note: isdir() is also True for scheme only urls (e.g. "flash:",
            #       "file:")
            context = Url.Context( *Url.urlArgsFromMode( self.mode_ ) )
            destUrl = Url.parseUrl( destUrl.url + '/' + f, context )
      except ValueError as e:
         raise CommandFailedException( error='Error: %s' % e )
      except OSError as e:
         errmsg = e.strerror
         raise CommandFailedException( error='Error: %s' % errmsg )

      bootConfig = FileUrl.localBootConfig( *Url.urlArgsFromMode( self.mode_ ) )
      self.bootConfFile_ = bootConfig.localFilename()
      self.swiFile_ = destUrl.localFilename()

      context = Url.Context( *Url.urlArgsFromMode( self.mode_ ) )
      bootExtUrl = Url.parseUrl( 'flash:/' + 'boot-extensions', context )
      self.bootExtFile_ = bootExtUrl.localFilename()

      # the operations are performed in two steps: 
      # 1) perform all actions on temporary files
      # 2) override original files with temporary files
      # this makes recovery easier in case it is necessary
      rSeq = '_tmp%04x' % random.randrange( 0, 65536 )
      self.tmpSwiFile_ = self.swiFile_ + rSeq
      self.tmpBootConfFile_ = self.bootConfFile_ + rSeq

      return destUrl

   def updateExtensionArgs( self ):
      if not os.path.exists( self.bootExtFile_ ):
         return

      with open( self.bootExtFile_ ) as f:
         ext = []
         for line in f:
            if s := line.split( maxsplit=1 ):
               ext.append( s[ 0 ] )
         self.extensions_.update( ext )

   # swadapt is in root of the zip image (so it can be used in Aboot): move it to 
   # where PATH will find it. TODO: package the link into the image so no asRoot...
   def maybeInstallSwadapt( self ):
      if os.path.exists( "/mnt/flash/disable_swi_adapt" ):
         return None
      if os.path.exists( "/usr/bin/swadapt" ):
         return "swadapt"
      if not os.path.exists( "/export/swi/swadapt" ):
         # vEOS-lab/cEOS - swadapt doesn't apply to these images
         # and won't be installed
         return None
      Tac.run( [ "ln", "-s", "/export/swi/swadapt", "/usr/bin/swadapt" ],
               asRoot=True)
      return "swadapt"

   # acquire SWI from source path
   def _downloadFromSource( self, srcUrl ):
      try:
         s = ' to this supervisor' if self.isModularSystem_ else ''
         printAndFlush( 'Downloading image%s...' % s )
         filterCmd = self.maybeInstallSwadapt()
         if filterCmd == "swadapt":
            self.updateExtensionArgs()
            if self.extensions_:
               extStr = ",".join( self.extensions_ )
               filterCmd = f"swadapt extensions {extStr}"

         srcUrl.writeLocalFile(srcUrl, self.tmpSwiFile_, filterCmd=filterCmd )
         print( 'done.' )

      except OSError as e:
         errmsg = e.strerror
         # cleanup: operators would not grok "swadapt: write error: No such..."
         r = re.search( "([a-zA-Z]+: )(write error: )(.*)", errmsg )
         if r:
            errmsg = r.groups()[2]
         print( '' )
         raise CommandFailedException( 'Error: %s' % errmsg )

   def _validateSwi( self ):
      # Doing a lazy import to only load zipfile when this command gets
      # executed. This way, even if there are multiple instances of Cli
      # running, zipfile will not act as a memory hog for each of them
      import zipfile # pylint: disable=import-outside-toplevel
      try:
         f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_
         if not zipfile.is_zipfile( f ):
            raise zipfile.BadZipfile()
         z = zipfile.ZipFile( f ) # pylint: disable=consider-using-with
         if 'version' not in z.namelist():
            raise zipfile.BadZipfile()
      except zipfile.BadZipfile:
         error = 'Error: Software image seems to be corrupted'
         raise RevertChangesException( error=error )

   def _verifyImageCompatibility( self, imgSrcUrl ):
      # Test image used in InstallCliNetworkUrlTest.py not compat check resistant
      if os.environ.get( "SKIP_IMAGE_COMPAT_CHECK" ):
         return
      # Check if the image being installed is compatible (ReloadPolicy checks
      # pass, swEpoch >= hwEpoch and if 2Gb image, it supports the hardware)
      # with the hardware
      # pylint: disable-next=import-outside-toplevel
      import CliPlugin.ReloadImageEpochCheckCli as epochHook
      # pylint: disable-next=import-outside-toplevel
      import CliPlugin.ReloadPolicyCheckCli as rpHook

      f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_

      # ReloadPolicy check includes SWI signature check and factors in whether
      # or not SecureBoot is enabled in that check
      rpComp, rpResult = rpHook.checkReloadPolicy( self.mode_, f )
      for rpWarning in rpResult.warnings:
         self.mode_.addWarning( rpWarning )
      if not rpComp:
         rpError = '\n'.join( rpResult.errors )
         raise RevertChangesException( error=rpError )

      epochComp, epochErr = epochHook.checkImageEpochAllowed( self.mode_, f,
                                                              imgSrcUrl )
      if not epochComp :
         raise RevertChangesException( error=epochErr )

   # modify boot-config with new SWI filename
   def _modifyBootConfig( self, destUrl ):
      try:
         printAndFlush( 'Preparing new boot-config...' )
         cf = SimpleConfigFile.SimpleConfigFileDict( self.bootConfFile_, True )
         tmpCf = SimpleConfigFile.SimpleConfigFileDict( self.tmpBootConfFile_, True )
         for key in cf:
            tmpCf[key] = cf[key]
         tmpCf['SWI'] = str( destUrl )
         print( 'done.' )
      except ( OSError, SimpleConfigFile.ParseError ) as e:
         errmsg = e.strerror
         print( '' )
         # if the boot-config file could not be modified, revert changes, as the
         # file would now just end up consuming space without being used
         raise RevertChangesException( error='Error: %s' % errmsg )


   def _transferChangesToStandby( self ):
      if redundancyStatus_.peerMode != 'standby': # pylint: disable=no-else-raise
         e = 'Standby supervisor is disabled and will not be upgraded.'
         raise StandbyTransferException( error=e, isMessage=True, 
                                         standbyMissing=True )
      else:
         try:
            printAndFlush( 
               'Copying new software image to standby supervisor...' )
            f = self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_
            self._transferToStandby( f, self.tmpSwiFile_ )
            print( 'done.' )
            printAndFlush( 'Copying new boot-config to standby supervisor...' )
            self._transferToStandby( self.tmpBootConfFile_, self.tmpBootConfFile_ )
            print( 'done.' )
         except Tac.SystemCommandError as e:
            print( '' )
            errMsg = e.output
            if 'No space left on device' in e.output:
               errMsg = 'Error: No space left on standby supervisor'
            elif 'Network is unreachable' in e.output:
               errMsg = 'Error: Unable to communicate with standby supervisor'
            raise StandbyTransferException( error=errMsg )

   def run( self, mode, srcUrl, doReload, now, destUrl, extensions, allArg ):
      self.mode_ = mode
      self.now_ = now
      self.skipDownload_ = False
      self.isModularSystem_ = isModularSystem_
      self.extensions_ = set( [ 'all' ] ) if allArg else set( extensions )

      try:
         if self.isModularSystem_:
            # pylint: disable-next=consider-using-in
            if( redundancyStatus_.mode != 'active' and 
                redundancyStatus_.mode != 'switchover' ):
               e = 'Error: This command is only available on the active supervisor'
               raise CommandFailedException( e )
            self.isModularSystem_ = ( redundancyStatus_.peerState != 'notInserted' )

         destUrl = self._initializeFilenames( srcUrl, destUrl )

         if not self.skipDownload_:
            self._downloadFromSource( srcUrl )

         self._validateSwi()
         # Check if image was adapted. If image has only one optimization, image
         # most likely was adapted. There is a possibility that `install source`
         # was called on an already optimized swi. In such case we would still
         # report that image was adapted
         optims = getSupportedOptimizationsFromSwi( 
            self.swiFile_ if self.skipDownload_ else self.tmpSwiFile_ )
         if len( optims ) == 1:
            print( "Image optimized to: " + optims[0] )

         self._verifyImageCompatibility( srcUrl )
         self._modifyBootConfig( destUrl )

         if self.isModularSystem_:
            try:
               self._transferChangesToStandby()
            except StandbyTransferException as e:
               if e.isMessage:
                  self.mode_.addWarning( e.error )
                  defaultChoice = True
               else:
                  self.mode_.addError( e.error )
                  # We should abort
                  defaultChoice = False
               if not e.standbyMissing:
                  self._deleteTempFile( self.tmpSwiFile_, onActive=False )
                  self._deleteTempFile( self.tmpBootConfFile_, onActive=False )
               self.isModularSystem_ = False
               question = 'Commit changes on this supervisor? [%s]' % (
                  'Y/n' if defaultChoice else 'y/N' )
               if not self._promptUser( question, defaultChoice ):
                  raise RevertChangesException( aborted=True )

         self._commitChanges()

         # (if needed) reload both supes
         if doReload:
            print( 'Reloading...' )
            sys.stdout.flush()
            ReloadCli.doReload( mode, supe='all', power=True, now=self.now_ )

      except RevertChangesException as e:
         if not e.aborted:
            self.mode_.addError( e.error )
         self._revertChanges()
         return None if e.aborted else False
      except CommandFailedException as e:
         if not e.aborted:
            self.mode_.addError( e.error )
         return None if e.aborted else False
      return True

def doInstall( mode, args ):
   srcUrl = args[ 'SOURCE' ]
   destUrl = args.get( 'DESTINATION' )
   doReload = args.get( 'reload' )
   now = args.get( 'now' )
   extensions = args.get( 'EXTENSION', [] )
   allArg = args.get( 'all' )

   installer = Installer()
   success = installer.run( mode, srcUrl, doReload, now, destUrl,
                            extensions, allArg )
   if success is None:
      print( 'Installation aborted.' )
   elif success:
      print( 'Installation succeeded.' )
   else:
      print( 'Installation failed.' )

def Plugin( entityManager ):
   global entityManager_, redundancyStatus_
   entityManager_ = entityManager
   redundancyStatus_ = LazyMount.mount( entityManager,
      Cell.path( 'redundancy/status' ), 'Redundancy::RedundancyStatus', 'r' )

def _isModular():
   global isModularSystem_
   isModularSystem_ = True

FruCli.registerModularSystemCallback( _isModular )
