# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import CliParser

from CliPlugin import ( EbraEthIntfCli, L2ProtocolShowCli, IntfCli, EthIntfCli )
from CliPlugin.IntfStatusMatcher import IntfStatusMatcher, DefaultIntfNumberMatcher
from CliPlugin.InternalRecircIntfModel import InternalRecircInterfaceStatus
import Intf.IntfRange as IntfRange

import LazyMount
import ConfigMount
from TypeFuture import TacLazyType

from IntfRangePlugin.InternalRecircIntf import InternalRecircAutoIntfType

traceHandle = Tracing.Handle( 'InternalRecircIntfCli' )
t0 = traceHandle.trace0

internalRecircIntfConfigDir = None
internalRecircIntfStatusDir = None
hwCapabilities = None

InternalRecircIntfId = TacLazyType( "Arnet::InternalRecircIntfId" )

# If Fru has not started yet or entity mib root has not yet fully been created,
# each element in limits is the virtual limit for each part of an InternalRecirc
# interface
limits = [ ( 0, 18 ), ( 0, 8 ), ( 1, 3 ) ] # 0 for module part due to BUG678118
defaultMatcher = DefaultIntfNumberMatcher( limits, minParts=2, maxParts=3 )

def internalRecircIntfStatus( mode ):
   return hwCapabilities.availablePort

def internalRecircIntfGuard( mode, token ):
   if hwCapabilities.internalRecircIntfSupported:
      return None
   return CliParser.guardNotThisPlatform

class InternalRecircIntf( EthIntfCli.EthIntf ):
   # ----------------------------------------------------------------------------
   # Creates a new InternalRecircIntf instance of the specified name.
   # ----------------------------------------------------------------------------
   def __init__( self, name, mode ):
      t0( "InternalRecircIntf::__init__", name )
      EthIntfCli.EthIntf.__init__( self, name, mode )
      self.ethIntfConfigDir = internalRecircIntfConfigDir
      self.ethIntfStatusDir = internalRecircIntfStatusDir
      self.ethIntfStatuses = internalRecircIntfStatusDir.intfStatus

   def setDefault( self ):
      self.setTimestampMode( 'timestampModeDisabled' )
      EthIntfCli.EthIntf.setDefault( self )

   # ----------------------------------------------------------------------------
   # Creates the Interface::InternalRecircIntfConfig object for this interface if
   # it does not already exist.
   # ----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      t0( "InternalRecircIntf::createPhysical: ", self.name )
      self.ethIntfConfigDir.intfConfig.newMember( self.name )

   # ----------------------------------------------------------------------------
   # Deletes the Interface::InternalRecircIntfConfig object for this interface
   # ----------------------------------------------------------------------------
   def destroyPhysical( self ):
      if self.ethIntfConfigDir.intfConfig.get( self.name ):
         del self.ethIntfConfigDir.intfConfig[ self.name ]

   # ----------------------------------------------------------------------------
   # EthIntf is a subclass of PhysicalIntf, but we are actually a virtual
   # interface and we're allowed to be deleted.
   # ----------------------------------------------------------------------------
   def noInterface( self ):
      self.destroy()

   # ----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   # ----------------------------------------------------------------------------
   # TODO: These function needs investigation. See BUG679463
   def addrStr( self ):
      return "address is %s" % self.addr()

   def countersSupported( self ):
      return False

   def counter( self ):
      return None

   def hardware( self ):
      return "internalRecirc"

   def bandwidth( self ):
      return 0

   def getIntfDuplexStr( self ):
      return "duplexFull"

   def xcvrTypeStr( self ):
      return "N/A"

   def autonegActive( self ):
      return False

   def flowcontrolRxPause( self ):
      return None

   def flowcontrolTxPause( self ):
      return None

   def getIntfStatusModel( self ):
      return InternalRecircInterfaceStatus( name=self.status().intfId )

   def showLoopbackMode( self, intfStatusModel ):
      pass

   def showLinkTypeSpecific( self, intfStatusModel ):
      pass

   # This port is eligible for "switchport" commands
   def switchportEligible( self ):
      return True

   def routingSupported( self ):
      return False

   # ----------------------------------------------------------------------------
   # The rule for matching recirculation interface names. When this pattern
   # matches, it returns an instance of the InternalRecircIntf class.
   # ----------------------------------------------------------------------------
   matcher = IntfStatusMatcher( 'InternalRecirc',
                  internalRecircIntfStatus,
                  defaultMatcher,
                  value=lambda mode, intf:
                     InternalRecircIntf( intf, mode ), # pylint: disable-msg=E0602
                  guard=internalRecircIntfGuard )

   # ----------------------------------------------------------------------------
   # Returns an unsorted list of InternalRecircIntf objects representing all the
   # existing Interface::EthPhyIntfStatus objects for recirc interfaces.
   # ----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in internalRecircIntfStatusDir.intfStatus:
         if not name.startswith( "InternalRecirc" ):
            continue
         intf = InternalRecircIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

# -------------------------------------------------------------------------------
# Register the InternalRecircIntf class as a type of physical interface.
# -------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( InternalRecircIntf,
                                  InternalRecircAutoIntfType )
IntfRange.registerIntfTypeGuard( InternalRecircAutoIntfType,
                                 internalRecircIntfGuard )

EbraEthIntfCli.switchPortMatcher |= InternalRecircIntf.matcher

# -------------------------------------------------------------------------------
# Register the InternalRecircIntf class as able to participate in interface ranges.
# -------------------------------------------------------------------------------
EbraEthIntfCli.registerSwitchPortIntfType( InternalRecircAutoIntfType )
L2ProtocolShowCli.l2ProtocolIntfTypes.append( InternalRecircAutoIntfType )

# -------------------------------------------------------------------------------
# Enable InternalRecircIntf commands to be added to "config-if" mode.
# -------------------------------------------------------------------------------
class InternalRecircIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, InternalRecircIntf )

IntfCli.IntfConfigMode.addModelet( InternalRecircIntfConfigModelet )

def Plugin( entityManager ):
   global internalRecircIntfConfigDir, internalRecircIntfStatusDir
   global hwCapabilities

   internalRecircIntfConfigDir = ConfigMount.mount( entityManager,
                                   "interface/config/recirc",
                                   "Interface::InternalRecircIntfConfigDir", "w" )
   internalRecircIntfStatusDir = LazyMount.mount( entityManager,
                                    "interface/status/recirc",
                                    "Interface::InternalRecircIntfStatusDir", "r" )
   hwCapabilities = LazyMount.mount( entityManager,
                                     "hardware/internalport/hwcapabilities",
                                     "Arnet::InternalRecircIntfHwCapabilities", "r" )
