# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time

import Ark
import Arnet
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   Model,
)
from IntfModels import Interface
from TableOutput import (
   Format,
   Headings,
   TableFormatter,
)
import Tac

class InterfaceMacData( Model ):
   adminEnabled = Bool( help="Administratively enabled" )

   phyState = Enum( values=( "unknown",
                             "init",
                             "detectingXcvr",
                             "autonegotiating",
                             "linkDown",
                             "linkUp",
                             "errDisabled" ),
                    help="Phy State", optional=True )
   phyStateChangeCount = Int( help="Total number of PHY state changes",
                              optional=True )
   phyStateLastChangeTime = Float( help="Time of last PHY state change",
                                   optional=True )

   intfState = Enum( values=( "linkUnknown",
                              "linkDown",
                              "linkUp",
                              "inactive" ),
                     help="Interface State" )
   intfStateChangeCount = Int( help="Total number of interface state changes" )
   intfStateLastChangeTime = Float( help="Time of last interface state change" )

   macRxLocalFault = Bool( help="A current local Rx fault has been detected" )
   macRxLocalFaultChangeCount = Int( help="Total number of local MAC Rx faults" )
   macRxLastLocalFaultChangeTime = Float( help="Time of last local MAC Rx fault" )

   macRxRemoteFault = Bool( help="A current remote Rx fault has been detected" )
   macRxRemoteFaultChangeCount = Int( help="Total number of remote MAC Rx faults" )
   macRxLastRemoteFaultChangeTime = Float( help="Time of last remote MAC Rx fault" )

   additionalStatus = Dict( keyType=str,
                            valueType=str,
                            help="A mapping of platform specific fields to "\
                                                                     "their values",
                            optional=True )


class InterfacesMac( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfaceMacData,
                      help='A mapping of interfaces to their MAC information' )
   _detail = Bool( help="Use detail render format", optional=True)

   def detailIs( self, detail ):
      self._detail = detail

   def detail( self ):
      return self._detail

   def _renderSummary( self ):
      print( '''Key:
      L  = Rx Local Fault Reported
      R  = Rx Remote Fault Reported
      Last Change: Time when most recent fault status changed
      ''' )

      table = TableFormatter()
      headers = Headings( [ 'Interface',
                            'Config State',
                            'Oper State',
                            'PHY State',
                            'MAC Fault',
                            'Last Change' ] )
      headers.doApplyHeaders( table )

      table.formatColumns(
         Format( align='bottom', justify='left' ),
         Format( align='bottom', justify='left' ),
         Format( align='bottom', justify='left' ),
         Format( align='bottom', justify='left' ),
         Format( align='bottom', justify='left' ),
         Format( align='bottom', justify='right' ) )

      # Keep the table compact by removing padding columns on each side
      for fmt in table.columnFormats_:
         fmt.padLimitIs( True )

      for intf in Arnet.sortIntf( self.interfaces ):
         macData = self.interfaces[ intf ]

         # Local and/or Remote fault reception could have been reported.
         # But we will make sure that the reported last change is
         # whatever the latest among the 4, including PHY state change
         # and Interface change.

         lastChangeTime = 0.0
         fault = '-'

         # If the PHY state is anything other than link up/down, what's
         # the point of looking at MAC status anyway?  (Or, if we're on
         # virtual hardware, with no phyState, we look at the mac status.)
         if macData.phyState in ( 'linkDown', 'linkUp' ) or \
               macData.phyState is None:
            fault = ''
            lastChangeTime = max( macData.intfStateLastChangeTime,
                                  macData.macRxLastLocalFaultChangeTime,
                                  macData.macRxLastRemoteFaultChangeTime,
                                  lastChangeTime )
            if macData.phyStateLastChangeTime is not None:
               lastChangeTime = max( macData.phyStateLastChangeTime,
                                     lastChangeTime )

            if macData.macRxLocalFault:
               fault += 'L'
            if macData.macRxRemoteFault:
               fault += 'R'

         table.newRow( intf,
                       'Up' if macData.adminEnabled else 'Down',
                       macData.intfState,
                       # Use interface state if phyState no present
                       macData.phyState if macData.phyState else macData.intfState,
                       fault,
                       Ark.utcTimeRelativeToNowStr( lastChangeTime ) )

      print( table.output() )

   def _renderDetail( self ):

      print( 'Current System Time: '+ str( time.ctime( Tac.utcNow() ) ) )

      fmt = '  %-27.27s %-16.16s %8.8s %22.22s'
      fmt2 = '  %-27.27s %s'

      for intf in Arnet.sortIntf( self.interfaces ):
         print( '' )
         print( intf )

         macData = self.interfaces[ intf ]

         print( fmt % ( '', 'Current State', 'Changes', 'Last Change' ) )

         # Use interface state if phyState no present
         phyState = macData.phyState
         phyStateChangeCount = macData.phyStateChangeCount
         phyStateLastChangeTime = macData.phyStateLastChangeTime
         if not phyState:
            phyState = macData.intfState
            phyStateChangeCount = macData.intfStateChangeCount
            phyStateLastChangeTime = macData.intfStateLastChangeTime
         print( fmt % ( 'PHY State',
                       phyState,
                       phyStateChangeCount,
                       Ark.utcTimeRelativeToNowStr( phyStateLastChangeTime ) ) )

         print( fmt % ( 'Interface State',
                       macData.intfState,
                       macData.intfStateChangeCount,
                       Ark.utcTimeRelativeToNowStr(
                          macData.intfStateLastChangeTime ) ) )

         print( fmt % ( 'MAC Rx Local Fault',
                       macData.macRxLocalFault,
                       macData.macRxLocalFaultChangeCount,
                       Ark.utcTimeRelativeToNowStr(
                          macData.macRxLastLocalFaultChangeTime ) ) )

         print( fmt % ( 'MAC Rx Remote Fault',
                       macData.macRxRemoteFault,
                       macData.macRxRemoteFaultChangeCount,
                       Ark.utcTimeRelativeToNowStr(
                          macData.macRxLastRemoteFaultChangeTime ) ) )

         for item in sorted( macData.additionalStatus.items() ):
            print( fmt2 % item )

   def render( self ):
      if not self.interfaces:
         return

      if not self._detail:
         self._renderSummary()
      else:
         self._renderDetail()

