#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re

import Arnet
import Tac
from ArnetModel import MacAddress
from CliMode.Intf import IntfMode
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import Float
from CliModel import List
from CliModel import Submodel
from CliModel import Bool
import IntfModels
from TableOutput import Format, createTable
import CliExtensions
import Ethernet

_hardwareTypeMap = { "portChannel": "Port-Channel",
                     "ethernet": "Ethernet",
                     "vlan": "Vlan",
                     "loopback": "Loopback",
                     "tunnel": "Tunnel",
                     "test": "Test",
                     "vxlan": "Vxlan",
                     "subinterface": "Subinterface",
                     "fabric": "Fabric",
                     "fpga": "FPGA",
                     "cpu": "Cpu",
                     "application": "Application FPGA connector",
                     "metamux": "Metamux",
                     "metawatch": "Metawatch",
                     "switch": "Switch",
                     "internalRecirc": "Internal Recirculation",
                     "trafficGen": "Traffic Generation",
                     "multiaccess": "MultiAccess",
                     "dps": "DPS",
                     "hati": "HATI",
                     "fabricChannel": "Fabric-Channel" }

prettyIntfStatusMap = { 'uninitDown': 'uninit down',
                        'adminDown': 'admin down',
                        'up': 'up',
                        'down': 'down',
                        'adminIsolated': 'isolated',
                        'autoIsolated': 'isolated',
                        'notApplicable': 'N/A' }

CONNECTED_STATUSES = [ "connected", "signal" ]

def formatInterfaceStatus( model ):
   """Helper function to render interface status.

   Arg:
     - model: An instance of a model that has at least two attributes:
       an `interfaceStatus' and a `lineProtocolStatus', such as in the
       IntfStatus model.
   """
   if model.interfaceStatus in CONNECTED_STATUSES:
      intfStatus = "up"
   elif model.interfaceStatus == "disabled":
      intfStatus = "administratively down"
   elif model.interfaceStatus == 'adminIsolated':
      intfStatus = "administratively isolated"
   elif model.interfaceStatus == 'autoIsolated':
      intfStatus = "isolated"
   else:
      intfStatus = "down"
   return ( "{}, line protocol is {} ({})".format( intfStatus,
                                                   model.lineProtocolStatus.lower(),
                                                   formatIntfStatus(
                                                      model.interfaceStatus ) ) )

def fmtShortIntfStatus( interfaceStatus ):
   """
   Helper function to render a short interface status.

   used in formatShortIntfStatus to render a model interfaceStatus, but also can be
   imported by tests to map a model's interfaceStatus to the rendered output.
   """

   if interfaceStatus == "uninitialized":
      return "uninit down"

   # pylint: disable-next=consider-using-in
   if interfaceStatus == "disabled" or interfaceStatus == "admin":
      return "admin down"

   if interfaceStatus in CONNECTED_STATUSES:
      return "up"

   return "down"

def formatShortIntfStatus( model ):
   """
   Helper function to render a short interface status for CliModel.

   Arg:
     - model: An instance of a model that has `interfaceStatus' attribute
       such as in the IntfStatus model.
   """
   return fmtShortIntfStatus( model.interfaceStatus )

def formatIntfStatus( intfStatus ):
   if intfStatus == "noSignal":
      return "no signal"
   if intfStatus in ( "adminIsolated", "autoIsolated" ):
      return "isolated"
   if intfStatus == "no-resource":
      return "no resource"
   return intfStatus

def formatMac( macAddr ):
   return Ethernet.convertMacAddrToDisplay( macAddr.stringValue )

def intfOperStatusToEnum( st ):
   # Converts "intfOperFooBar" to "fooBar", for example.
   return re.sub( "intfOper([A-Z])", lambda m: m.group( 1 ).lower(), st )
intfOperStatuses = [ intfOperStatusToEnum( s ) for s in \
                        Tac.Type( 'Interface::IntfOperStatus' ).attributes ]
# For some interfaces line protocol is not really applicable
# BUG591027 - OperStatus values come from IF-MIB, so we need to investigate whether
# we can add notApplicable to the enum list without issues.
intfOperStatuses += [ 'notApplicable' ]

prettyLineProtoStatusMap = { x : x.lower() for x in intfOperStatuses }
prettyLineProtoStatusMap[ 'notApplicable' ] = 'N/A'

#--------------------------------------------------------------------------------
# EAPI Models
#--------------------------------------------------------------------------------   
class InterfaceAddressIp4Base( Model ):
   # All IPv4 address models that can be returned inherit from this
   lossOfConnectivityReason = Enum( values=( 'cfm', ),
                                   help="IPv4 layer 3 interface connectivity status",
                                   optional=True )

class InterfaceAddressIp6Base( Model ):
   # All IPv6 address models that can be returned inherit from this
   lossOfConnectivityReason = Enum( values=( 'cfm', ),
                                   help="IPv6 layer 3 interface connectivity status",
                                   optional=True )

class InterfaceStatus( Model ):
   # This is also present as the intfStatuses key - but it
   # useful to repeat it here, so that IntfStatus is self-
   # contained.
   name = IntfModels.Interface( help="Name of the interface" )
   forwardingModel = Enum( values=( "linkQualification",
                                    "layer1Patch", "layer1PatchAndLldp",
                                    "recirculation",
                                    "quietDataLink", "unauthorized",
                                    "dataLink", "bridged", "routed" ),
                            help="LinkQualification -- the interface forwards "
                            "packets for link qualification purposes. "
                            "Protocols do not run on this interface, "
                            "Layer1Patch -- the interface runs in Layer 1 mode "
                            "to patch receive and/or transmit paths of various "
                            "ports in the system. Protocols do not run on this "
                            "interface, "
                            "Layer1PatchAndLldp -- the interface runs in Layer 1 "
                            "mode to patch receive and/or transmit paths of "
                            "various ports in the system. LLDP is the only protocol "
                            "supported on this interface, "
                            "Recirculation -- the interface recirculates "
                            "packets. Protocols do not run on this port, "
                            "QuietDataLink -- the interface forwards packets "
                            "but does not participate in bridging or routing and "
                            "protocols do not run on this interface, " 
                            "Unauthorized -- the interface does not participate "
                            "in bridging or routing and only EAPOL runs on this "
                            "interface, "
                            "DataLink -- the interface forwards packets "
                            "given to it but is not participating in bridging or "
                            "routing operations, "
                            "Routed -- the interface can route packets, "
                            "Bridged -- the interface can bridge packets but not "
                            "route them" )
   lineProtocolStatus = Enum( help="Line protocol status",
                              values=intfOperStatuses )
   interfaceStatus = Enum( values=( 'disabled', 'connected', 'notconnect', 'unknown',
                                    'admin', 'errdisabled', 'uninitialized',
                                    'inactive', 'maint-down', 'notapplicable',
                                    'signal', 'noSignal', 'initializing',
                                    'adminIsolated', 'autoIsolated', 'no-resource' ),
                           help="Connection status of the interface" )
   maintenanceEnterTime = Float( help="Time at which interface went under "
                               "maintenance", optional=True )
   hardware = Enum( values=_hardwareTypeMap, help="Hardware Description" )

   # Should not be a list but defined as such for historical reasons.
   # Original intent for the list was to contain multiple instances
   # of multiple address classes. But JSON does not return class types
   # for list elements which makes it difficult for applications to
   # process said list. Adding a class type enum to the base class
   # is not backwards compatible. Hence the list has always been in
   # practice a single instance of an IPv4 address submodel.
   interfaceAddress = List( valueType=InterfaceAddressIp4Base,
                            help="IPv4 addresses for this interface" )

   interfaceAddressIp6 = Submodel( valueType=InterfaceAddressIp6Base,
                                   help="IPv6 addresses for this interface",
                                   optional=True )
   
   # Loopback interfaces don't have MAC addresses.
   physicalAddress = MacAddress( help="Physical Layer Address", optional=True )
   burnedInAddress = MacAddress( help="Burned-in Address", optional=True )
   description = Str( help="Configured interface description" )
   bandwidth = Int( help="Bandwidth (bit)" )
   mtu = Int( help="MTU (bytes)" )
   l3MtuConfigured = Bool( help="Per-interface MTU is set under interface " \
                                "configuration mode", default=False )
   l2Mru = Int( help="L2 MRU (bytes)", default=0 )
   lastStatusChangeTimestamp = Float( help="Timestamp of last status change", 
                                      optional=True )
   
   def renderHeader( self ):
      print( f"{self.name.stringValue} is {formatInterfaceStatus( self )}" )
      
   def renderHardware( self ):
      output = "  Hardware is %s" % _hardwareTypeMap[ self.hardware ]
      if self.physicalAddress is not None:
         output += ", address is %s" % formatMac( self.physicalAddress )
      if self.burnedInAddress is not None:
         output += " (bia %s)" %  formatMac( self.burnedInAddress )
      print( output )
      
      if self.description:
         print( "  Description: %s" % self.description )
                
   def renderMtuMruAndBw( self ):
      tokens = []
      # Process MTU
      if self.forwardingModel == "routed":
         mtuToken = "IP MTU %d bytes" % self.mtu
         mtuToken += " (default)"  if not self.l3MtuConfigured else ""
         tokens.append( mtuToken )
      elif self.mtu is not None:
         tokens.append( "Ethernet MTU %d bytes" % self.mtu )
      else:
         assert not self.bandwidth, repr( self.bandwidth )

      # Process L2 MRU
      if self.l2Mru:
         tokens.append( "Ethernet MRU %d bytes" % self.l2Mru )

      # Process Bandwidth
      if self.bandwidth:
         tokens.append( "BW %d kbit" % ( self.bandwidth // 1000 ) )

      if tokens:
         output = "  "
         output += ", ".join( tokens )
         print( output )

   def timeString( self, elapsed, output, units ):
      assert isinstance( elapsed, int )
      if units:
         _ , unit = units[ -1 ]
         assert ( unit == 1 ), "Last unit must have 1 as a unit value"
      if ( elapsed > 0 ) and units:
         localName, localUnits = units[0] 
         value = elapsed % localUnits if len( units ) > 1 else elapsed
         if( value > 0 ): # pylint: disable=superfluous-parens
            output = ( "%d %s"%( value, localName + ( "s" if value > 1 else "" ) )
                        + ( ( ", " + output ) if output else "" ) )
         return( self.timeString( elapsed//localUnits, 
                 output, units[1:] ) )
      else: 
         return( output )  # pylint: disable=superfluous-parens
      
   def renderUptime( self ):
      if self.lastStatusChangeTimestamp is None:
         return
      
      timeDelta = int( Tac.utcNow() - self.lastStatusChangeTimestamp )
      uptimeStatus = "  Up " if self.lineProtocolStatus == "up" else "  Down "
      uptimeString = self.timeString( timeDelta, "",
                                      [ ( "second", 60 ), ( "minute", 60 ),
                                        ( "hour", 24 ), ( "day", 365 ),
                                        ( "year", 1 ) ] )
      print( uptimeStatus + uptimeString )
      
   def renderMaintEnterTime( self ):
      if self.maintenanceEnterTime is None:
         return

      timeDelta = int( Tac.utcNow() - self.maintenanceEnterTime )
      maintenanceEnterTimeString = self.timeString( timeDelta, "",
                                                    [ ( "second", 60 ),
                                                      ( "minute", 60 ),
                                                      ( "hour", 24 ),
                                                      ( "day", 365 ),
                                                      ( "year", 1 ) ] )
      print( "  Under maintenance for %s" % maintenanceEnterTimeString )

   def renderInterfaceAddress( self ):
      ip4LocReason = None
      ip6LocReason = None
      for address in self.interfaceAddress:
         address.render()
         ip4LocReason = address.lossOfConnectivityReason
      if self.interfaceAddressIp6:
         self.interfaceAddressIp6.render()
         ip6LocReason = self.interfaceAddressIp6.lossOfConnectivityReason
      if ip4LocReason or ip6LocReason:
         locReason = None
         if ip4LocReason and ip6LocReason:
            proto = 'IPv4, IPv6'
            locReason = ip4LocReason
         elif ip4LocReason:
            proto = 'IPv4'
            locReason = ip4LocReason
         elif ip6LocReason:
            proto = 'IPv6'
            locReason = ip6LocReason
         print( f"  {locReason.upper()} connectivity status for {proto} is DOWN" )

   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderInterfaceAddress()
      self.renderMtuMruAndBw()
      self.renderUptime()
      self.renderMaintEnterTime()

# Container for all interfaces
class InterfaceStatuses( Model ):
   interfaces = Dict( keyType=IntfModels.Interface, valueType=InterfaceStatus,
                      help="Interface Status" )

   def render( self ):
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].render()

class InterfaceConfigSanity( InterfaceStatus ):
   
   # Default behavior; override Intf.getConfigSanityModel to provide an
   # InterfaceStatus model
   def render( self ):
      print( 'Configuration sanity checking for interface %s is not '
             'supported.' % str( self.name ) )

class InterfaceCountersBase( Model ):
   _name = IntfModels.Interface( help="Name of the interface" )
   
   _l3Counters = Bool( help="True if counters are for Layer 3 interface",
                        default=False )
   _tunnelIntfCounters = Bool( help="True if counters are for static "
                                    "Tunnel interface", default=False )
   _vxlanCounters = Bool( help="True if counters are for aggregate VXLAN interfaces",
                          default=False )

   # some interfaces ( SubIntfs for example ) might not support
   # Ingress/Egress/BUM counters
   _ingressCounters = Bool( help="True if interface supports ingress counters",
                            default=True )
   _egressCounters = Bool( help="True if interface supports egress counters",
                           default=True )

   def l3Counters( self ):
      return self._l3Counters

   def tunnelIntfCounters( self ):
      return self._tunnelIntfCounters

   def vxlanCounters( self ):
      return self._vxlanCounters

   def renderHeader( self, direction='in' ):
      pass

   def renderIncoming( self ):
      pass

   def renderOutgoing( self ):
      pass 

   def counterSupported( self, direction ):
      return self._ingressCounters if direction == 'in' else self._egressCounters

# Interface Counters
class InterfaceCounters( InterfaceCountersBase ):

   inOctets = Int( help="Input octets" )
   inUcastPkts = Int( help="Input unicast packets" )
   inMulticastPkts = Int( help="Input multicast packets" )
   inBroadcastPkts = Int( help="Input broadcast packets  " )
   inDiscards = Int( help="Input packets with errors  ", optional=True )
   inTotalPkts = Int( help="Input packets", optional=True )

   outOctets = Int( help="Output octets" )
   outUcastPkts = Int( help="Output unicast packets" )
   outMulticastPkts = Int( help="Output multicast packets" )
   outBroadcastPkts = Int( help="Output broadcast packets" )
   outDiscards = Int( help="Output packets with errors  ", optional=True )
   outTotalPkts = Int( help="Output packets", optional=True )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

   # some interfaces ( L3 interfaces on T2 for example ) might only
   # support unicast and multicast counters, but not broadcast
   _umCounters = Bool( 
      help="True if interface supports Unicast and Multicast counters",
      default=False )
   _bumCounters = Bool( help="True if interface supports BUM counters",
                        default=True )

   def printRow( self, port, totalOctets, ucastPkts, mcastPkts=None, 
                 bcastPkts=None ):
      def checkValue( value ):
         return value if value is not None else 'n/a'
      
      if self._bumCounters:
         counterDisplayFormat = "%-20s %18s %15s %15s %15s"
         print( counterDisplayFormat % ( port, checkValue( totalOctets ),
                                         checkValue( ucastPkts ),
                                         checkValue( mcastPkts ),
                                         checkValue( bcastPkts ) ) )
      elif self._umCounters:
         counterDisplayFormat = "%-20s %18s %15s %15s"
         print( counterDisplayFormat % ( port, checkValue( totalOctets ),
                                         checkValue( ucastPkts ),
                                         checkValue( mcastPkts ) ) )
      else:
         counterDisplayFormat = "%-20s %18s %15s"
         print( counterDisplayFormat % ( port, checkValue( totalOctets ),
                                         checkValue( ucastPkts ) ) )
   
   def renderHeader( self, direction='in' ):
      if self._l3Counters:
         heading = "L3 Interface"
      elif self._tunnelIntfCounters:
         heading = "Tunnel"
      elif self._vxlanCounters:
         heading = "VXLAN Interface"
      else:
         heading = "Port"
      if direction == 'in':
         if not self._ingressCounters:
            return
         if self._bumCounters:
            self.printRow( heading, "InOctets", "InUcastPkts", "InMcastPkts",
                           "InBcastPkts" )
         elif self._umCounters:
            self.printRow( heading, "InOctets", "InUcastPkts", "InMcastPkts" )
         else:
            self.printRow( heading, "InOctets", "InPkts" )
      elif direction == 'out':
         if not self._egressCounters:
            return
         if self._bumCounters:
            self.printRow( heading, "OutOctets", "OutUcastPkts", "OutMcastPkts",
                           "OutBcastPkts" )
         elif self._umCounters:
            self.printRow( heading, "OutOctets", "OutUcastPkts", "OutMcastPkts" )
         else:
            self.printRow( heading, "OutOctets", "OutPkts" )
      else:
         assert False, "Unhandled direction"

   def renderIncoming( self ):
      if not self._ingressCounters:
         return
      if self._bumCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets, 
                        self.inUcastPkts, self.inMulticastPkts,
                        self.inBroadcastPkts )
      elif self._umCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets, 
                        self.inUcastPkts, self.inMulticastPkts )
      else:
         self.printRow( IntfMode.getShortname( self._name ), self.inOctets,
                        self.inUcastPkts )

   def renderOutgoing( self ):
      if not self._egressCounters:
         return
      if self._bumCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts, self.outMulticastPkts,
                        self.outBroadcastPkts )
      elif self._umCounters:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts, self.outMulticastPkts )
      else:
         self.printRow( IntfMode.getShortname( self._name ), self.outOctets,
                        self.outUcastPkts )

class VirtualInterfaceCountersBase( InterfaceCountersBase ):
   _vxlanVniCounters = Bool( help="True if counters are per-VNI on VXLAN interfaces",
                             default=False )

   def vxlanVniCounters( self ):
      return self._vxlanVniCounters

# Container base class
class InterfacesCountersBase( Model ):
   _outputType = Enum( values=( 'all', 'incoming', 'outgoing' ),
                       default='all', help="Which counters to show" )

   def getInterfaces( self ):
      raise NotImplementedError

   def getInterfaceCounters( self, intf ):
      raise NotImplementedError

   def l3Counters( self, intf ):
      return False

   def tunnelIntfCounters( self, intf ):
      return False

   def vxlanCounters( self, intf ):
      return False

   def partition( self, intfs ):
      # partition intfs into list of intfs that
      # have counters rendered separately
      intfList = []
      subIntfList = []
      l3IntfList = []
      tunnelIntfList = []
      vxlanIntfList = []
      for intf in intfs:
         if self.l3Counters( intf ):
            # subIntf will be accounted here if they support only l3 counters
            l3IntfList.append( intf )
         elif '.' in intf: 
            subIntfList.append( intf )
         elif self.tunnelIntfCounters( intf ):
            tunnelIntfList.append( intf )
         elif self.vxlanCounters( intf ):
            vxlanIntfList.append( intf )
         else:
            intfList.append( intf )
      return [ intfList, subIntfList, l3IntfList, tunnelIntfList, vxlanIntfList ]

   def renderWithHeader( self, intfList, direction='in' ):
      prevValueType = None
      for key in intfList:
         intfCounter = self.getInterfaceCounters( key )
         if not intfCounter.counterSupported( direction ):
            continue
         valueType = intfCounter.__class__
         if valueType != prevValueType:
            prevValueType = valueType
            intfCounter.renderHeader( direction )
         if direction == 'in':
            intfCounter.renderIncoming()
         else:
            intfCounter.renderOutgoing()

   def render( self ):
      intfs = self.getInterfaces()
      if not intfs:
         return

      sortedIntfs = Arnet.sortIntf( intfs )
      partitionedIntfs = self.partition( sortedIntfs )
      for intfList in partitionedIntfs:
         if intfList and partitionedIntfs.index( intfList ) != 0:
            # print empty line for separation only if intfList is not empty
            print()

         if self._outputType != 'outgoing':
            self.renderWithHeader( intfList )

         if intfList and self._outputType == 'all':
            print()

         if self._outputType != 'incoming':
            self.renderWithHeader( intfList, direction='out' )

# Container class
class InterfacesCounters( InterfacesCountersBase ):
   interfaces = Dict( keyType=IntfModels.Interface, valueType=InterfaceCountersBase,
                      help="Interface Counters" )

   def getInterfaces( self ):
      return self.interfaces

   def getInterfaceCounters( self, intf ):
      return self.interfaces[ intf ]

   def l3Counters( self, intf ):
      return self.interfaces[ intf ].l3Counters()

   def tunnelIntfCounters( self, intf ):
      return self.interfaces[ intf ].tunnelIntfCounters()

   def vxlanCounters( self, intf ):
      return self.interfaces[ intf ].vxlanCounters()

# Base class for virtual interface counters, i.e. for interfaces that require more
# than one identifier (e.g. for VXLAN, we need VTI and VNI)
class VirtualInterfacesCountersBase( InterfacesCountersBase ):
   def vxlanVniCounters( self, intf ):
      return False

   def partition( self, intfs ):
      raise NotImplementedError

class VxlanInterfacesVniCounters( VirtualInterfacesCountersBase ):
   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=VirtualInterfaceCountersBase,
                      help="Interface Counters" )

   def getInterfaces( self ):
      return self.interfaces

   def getInterfaceCounters( self, intf ):
      return self.interfaces[ intf ]

   def vxlanVniCounters( self, intf ):
      return self.interfaces[ intf ].vxlanVniCounters()

   def partition( self, intfs ):
      vxlanIntfList = []
      for intf in intfs:
         if self.vxlanVniCounters( intf ):
            vxlanIntfList.append( intf )
      return [ vxlanIntfList ]

class InterfaceCountersRateBase( Model ):
   _name = IntfModels.Interface( help="Name of the interface" )

   def intfName( self ):
      return self._name

   def usesTableOutput( self ):
      return False

# Interface rate counters
class InterfaceCountersRate( InterfaceCountersRateBase ):
   description = Str( help="Port description" )
   interval = Int( help="Interval in seconds" )

   inBpsRate = Float( help="Input bps rate" )
   inPktsRate = Float( help="Input packets rate percentage", optional=True )
   inPpsRate = Float( help="Input pps rate" )

   outBpsRate = Float( help="Output bps rate" )
   outPktsRate = Float( help="Output packets rate percentage", optional=True )
   outPpsRate = Float( help="Output Kpps rate" )

   lastUpdateTimestamp = Float( help="Time of last update", optional=True )

   def usesTableOutput( self ):
      return True

   @staticmethod
   def createTable( scaleKbpsAndPps=False ):
      # Port
      portMinLen = 9
      portMaxLen = 13

      # Name
      nameMinLen = 9
      nameMaxLen = 20

      # Intvl
      intvlMinLen = 5

      # In rate pct, Out rate Pct
      ratePctMinLen = 6

      # In Mbps, Out Mbps
      mbpsMinLen = 8

      # In Kpps, Out Kpps
      kppsMinLen = 8

      if scaleKbpsAndPps:
         interfaceCountersColumnList = [
            "Port", "Name", "Intvl",
            "In Kbps", "%", "In Pps",
            "Out Kbps", "%", "Out Pps", ]
      else:
         interfaceCountersColumnList = [
            "Port", "Name", "Intvl",
            "In Mbps", "%", "In Kpps",
            "Out Mbps", "%", "Out Kpps", ]
      table = createTable( interfaceCountersColumnList )

      # Port
      fPort = Format( justify="left", minWidth=portMinLen,
                      maxWidth=portMaxLen, isHeading=True )
      fPort.noPadLeftIs( True )
      fPort.padLimitIs( True )

      # Name
      fName = Format( justify="left", minWidth=nameMinLen,
                      maxWidth=nameMaxLen )
      fName.noPadLeftIs( True )
      fName.padLimitIs( True )

      # Intvl
      fIntvl = Format( justify="left", minWidth=intvlMinLen )
      fIntvl.noPadLeftIs( True )
      fIntvl.padLimitIs( True )

      # In rate pct, Out rate Pct
      fRatePct = Format( justify="right", dotAlign=True, minWidth=ratePctMinLen )
      fRatePct.noPadLeftIs( True )
      fRatePct.padLimitIs( True )

      # In Mbps, Out Mbps
      fMbps = Format( justify="right", dotAlign=True, minWidth=mbpsMinLen )
      fMbps.noPadLeftIs( True )
      fMbps.padLimitIs( True )

      # In Kpps
      fKpps = Format( justify="right", minWidth=kppsMinLen )
      fKpps.noPadLeftIs( True )
      fKpps.padLimitIs( True )

      # Out Kpps - The last column
      # For the last column we need to add no trailing space.
      # Hence we create a new format.
      fOutKpps = Format( justify="right", minWidth=kppsMinLen )
      fOutKpps.noPadLeftIs( True )
      fOutKpps.padLimitIs( True )
      fOutKpps.noTrailingSpaceIs( True )

      table.formatColumns( fPort, fName, fIntvl, fMbps, fRatePct, fKpps,
                           fMbps, fRatePct, fOutKpps )
      table.formatRows( Format( isHeading=True, border=False ) )

      return table

   def insertRow( self, table, scaleKbpsAndPps=False ):
      # Setting interval in the format mm:ss
      minutes = self.interval // 60
      seconds = self.interval % 60
      intervalStr = f'{minutes:2d}:{seconds:02d}'

      # Calculating and setting In Mbps and Out Mbps rate.
      if scaleKbpsAndPps:
         inBitRate = self.inBpsRate / 1000
         outBitRate = self.outBpsRate / 1000
      else:
         inBitRate = self.inBpsRate / 1000 / 1000
         outBitRate = self.outBpsRate / 1000 / 1000
      inBitRateStr = f'{inBitRate:.1f}'
      outBitRateStr = f'{outBitRate:.1f}'

      # Calculating and setting In Kpps and Out Kpps rate.
      if scaleKbpsAndPps:
         inPacketRate = self.inPpsRate
         outPacketRate = self.outPpsRate
      else:
         inPacketRate = self.inPpsRate // 1000
         outPacketRate = self.outPpsRate // 1000
      inPacketRateStr = f'{inPacketRate:.0f}'
      outPacketRateStr = f'{outPacketRate:.0f}'

      # Calculating and setting In and Out rate %
      inRatePct = self.inPktsRate
      inRatePctStr = '-' if inRatePct is None else f'{inRatePct:.1f}%'

      outRatePct = self.outPktsRate
      outRatePctStr = '-' if outRatePct is None else f'{outRatePct:.1f}%'

      # Creating a new row.
      table.newRow( IntfMode.getShortname( self.intfName() ),
                    self.description, intervalStr,
                    inBitRateStr, inRatePctStr, inPacketRateStr,
                    outBitRateStr, outRatePctStr, outPacketRateStr )

class InterfaceCountersRates( Model ):

   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=InterfaceCountersRateBase,
                      help="Mapping between an interface rate counters and model" )
   _scaleKbpsAndPps = Bool( default=False,
                            help="Scale counter rates to Kbps and Pps" )

   def render( self ):
      if not self.interfaces:
         return

      prevValueType = None
      table = None
      for key in Arnet.sortIntf( self.interfaces ):
         valueType = self.interfaces[ key ].__class__
         if valueType != prevValueType:
            if table:
               print( table.output() )
               table = None

            if self.interfaces[ key ].usesTableOutput():
               table = self.interfaces[ key ].createTable(
                                             scaleKbpsAndPps=self._scaleKbpsAndPps )
            else:
               self.interfaces[ key ].renderHeader()

            prevValueType = valueType

         if self.interfaces[ key ].usesTableOutput():
            self.interfaces[ key ].insertRow( table,
                                              scaleKbpsAndPps=self._scaleKbpsAndPps )
         else:
            self.interfaces[ key ].renderRates()

      if table:
         print( table.output(), end='' )

# Interface Counters Discards
class InterfaceCounterDiscards( Model ):
   _name = IntfModels.Interface( help="Name of the interface" )
   inDiscards = Int( help="Input packets discarded" )
   outDiscards = Int( help="Output packets discarded", optional=True )

# Container class
class InterfacesCountersDiscards( Model ):
   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=InterfaceCounterDiscards,
                      help="Interface Discard Counters" )

   inDiscardsTotal = Int( help="Total input packets discarded", default=0 )
   outDiscardsTotal = Int( help="Total output packets discarded", optional=True )

   def render( self ):
      if len( self.interfaces ) == 0:
         return
      
      # Set up the column formatting objects.
      fl = Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = Format( justify="right" )

      headings = ( "Port", "InDiscards", "OutDiscards" )

      table = createTable( headings )
      table.formatColumns( fl, fr, fr )

      for key in Arnet.sortIntf( self.interfaces ):
         if self.interfaces[ key ].outDiscards is not None:
            outDiscards = self.interfaces[ key ].outDiscards
         else:
            outDiscards = "N/A"
         table.newRow( IntfMode.getShortname( key ),
                       self.interfaces[ key ].inDiscards,
                       outDiscards )

      # If discard statistics are shown for two or more interfaces, then
      # also display a row showing summary statistics for all interfaces.
      if len( self.interfaces ) >= 2:
         row = [ "Totals", self.inDiscardsTotal ]

         # Only include the "outDiscards" sum if there is at least one
         # interface that supports it. Otherwise, show "N/A".
         if self.outDiscardsTotal is not None:
            row.append( self.outDiscardsTotal )
         else:
            row.append( "N/A" )

         sep = "---------"
         table.newRow( sep, sep, sep )
         table.newRow( *row )

      print( table.output() )
      
# Interface Traffic-class Counters
class InterfaceTrafficClassCounters( Model ):
   inPkts = Int( help="Input packets" )
   inOctets = Int( help="Input octets" )
   inUcastPkts = Int( help="Input unicast packets", optional=True )
   inMcastPkts = Int( help="Input multicast packets", optional=True )
   inBcastPkts = Int( help="Input broadcast packets", optional=True )
   inUcastOctets = Int( help="Input unicast octets", optional=True )
   inMcastOctets = Int( help="Input multicast octets", optional=True )
   inDroppedUcastPkts = Int( help="Dropped unicast packets", optional=True )
   inDroppedMcastPkts = Int( help="Dropped multicast packets", optional=True )
   inDroppedUcastOctets = Int( help="Dropped unicast octets", optional=True )
   inDroppedMcastOctets = Int( help="Dropped multicast octets", optional=True )

# Container class for traffic classes
class InterfaceTrafficClassesCounters( Model ):
   _name = IntfModels.Interface( help="Name of the interface" )
   _bumCounters = Bool( help="True if interface supports BUM counters",
                        default=True )
   _umCounters = Bool( help="True if interface supports unicast and "
                            "multicast counters", default=False )
   _egressCounters = Bool( help="True if interface supports egress counters",
                           default=False )
   trafficClasses = Dict( keyType=int, valueType=InterfaceTrafficClassCounters,
         help="A mapping between a traffic class (in range 0 to 7)"
              " and its counters" )

   def formatter( self ):
      if self._bumCounters:
         return '%-26s %7s %18s %18s %18s %18s'
      elif self._umCounters:
         return '%-26s %7s %14s %18s %18s %18s %18s'
      else:
         return '%-26s %7s %18s %18s'

   def headerFormatter( self ):
      totalWidth = 125 if self._umCounters else 72
      return self.formatter() + '\n%-26s %-7s\n' + ( '-' * totalWidth )

   def renderHeader( self, direction='in' ):
      if direction == 'in':
         print( 'Traffic-Class Counters' )
         if self._bumCounters:
            print( self.headerFormatter() % (
                      'Ingress', 'Traffic', 'InOctets', 'InUcastPkts', 'InMcastPkts',
                      'InBcastPkts', 'Port', 'Class' ) )
         elif self._umCounters:
            print( self.headerFormatter() % (
                      'Ingress', 'Traffic', 'DestType', 'Pkts', 'Octets', 'DropPkts',
                      'DropOctets', 'Port', 'Class' ) )
         else:
            print( self.headerFormatter() % (
                      'Ingress', 'Traffic', 'Pkts', 'Octets', 'Port', 'Class' ) )
      elif self._egressCounters:
         # Implement when we support egress counters
         pass

   def renderIncoming( self ):
      for tc, tcCounter in sorted( self.trafficClasses.items() ):
         tcStr = 'TC%d' % tc
         if self._bumCounters:
            print( self.formatter() % (
                      self._name.shortName, tcStr, tcCounter.inOctets,
                      tcCounter.inUcastPkts, tcCounter.inMcastPkts,
                      tcCounter.inBcastPkts ) )
         elif self._umCounters:
            print( self.formatter() % (
                      self._name.shortName, tcStr, 'UC',
                      tcCounter.inUcastPkts, tcCounter.inUcastOctets,
                      tcCounter.inDroppedUcastPkts,
                      tcCounter.inDroppedUcastOctets ) )
            print( self.formatter() % (
                      self._name.shortName, tcStr, 'MC',
                      tcCounter.inMcastPkts, tcCounter.inMcastOctets,
                      tcCounter.inDroppedMcastPkts,
                      tcCounter.inDroppedMcastOctets ) )
         else:
            print( self.formatter() % (
                      self._name.shortName, tcStr,
                      tcCounter.inPkts, tcCounter.inOctets ) )

   def renderOutgoing( self ):
      # Implement when we support egress counters
      pass

   def counterSupported( self, direction ):
      return True if direction == 'in' else self._egressCounters

# Container class
class InterfacesTrafficClassesCounters( InterfacesCountersBase ):
   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=InterfaceTrafficClassesCounters,
                      help="A mapping between an interface and its traffic class "
                           "counters" )

   def getInterfaces( self ):
      return self.interfaces

   def getInterfaceCounters( self, intf ):
      return self.interfaces[ intf ]

class InterfacesDescriptions( Model ):
   class InterfaceDescription( Model ):
      description = InterfaceStatus.description
      lineProtocolStatus = InterfaceStatus.lineProtocolStatus
      interfaceStatus = Enum( values=prettyIntfStatusMap,
                              help='Connection status of the interface' )
      
   interfaceDescriptions = Dict( keyType=IntfModels.Interface,
                                 valueType=InterfaceDescription,
                                 help='Mapping between an interface and its '
                                      'description' )
   
   def render( self ):
      if not self.interfaceDescriptions:
         return 
      
      fmt = '%-30s %-14s %-18s %s'
      print( fmt % ( 'Interface', 'Status', 'Protocol', 'Description' ) )
      for intf in Arnet.sortIntf( self.interfaceDescriptions ):
         interfaceDescription = self.interfaceDescriptions[ intf ]
         print( fmt % ( IntfMode.getShortname( intf ),
                        prettyIntfStatusMap[ interfaceDescription.interfaceStatus ],
                        prettyLineProtoStatusMap[
                           interfaceDescription.lineProtocolStatus ],
                        interfaceDescription.description ) )

# Error counters for interfaces   
class ErrorCounters( Model ):
   fcsErrors = Int( help="FCS errors" )
   alignmentErrors = Int( help="Alignment errors" )
   symbolErrors = Int( help="Symbol Errors" )
   inErrors = Int( help="Input errors" )
   frameTooShorts = Int( help="Frame too short errors" )
   frameTooLongs = Int( help="Frame too long errors" )
   outErrors = Int( help="Output errors" )

#Class that holds error counters for given interfaces
class InterfacesErrorCounters( Model ):

   interfaceErrorCounters = Dict( keyType=IntfModels.Interface,
                                  valueType=ErrorCounters,
                                  help="Mapping between an interface and error "
                                       "counters" )

   def render( self ):
      if not self.interfaceErrorCounters:
         return
      fmt = '%-10s %11s %8s %8s %8s %8s %8s %8s'
      print( fmt % ( 'Port', 'FCS', 'Align', 'Symbol', 'Rx',
                     'Runts', 'Giants', 'Tx' ) )
      for intf in Arnet.sortIntf( self.interfaceErrorCounters ):
         x = self.interfaceErrorCounters[ intf ]
         print( fmt % ( IntfMode.getShortname( intf ), x.fcsErrors,
                        x.alignmentErrors, x.symbolErrors, x.inErrors,
                        x.frameTooShorts, x.frameTooLongs, x.outErrors ) )

# Half-duplex error counters for interfaces
class HalfDuplexErrorCounters( Model ):
   singleCollisionFrames = Int( help="Single collision frames" )
   multipleCollisionFrames = Int( help="Multiple collision frames" )
   deferredTransmissions  = Int( help="Deferred transmissions" )
   lateCollisions = Int( help="Late collisions" )
   excessiveCollisions = Int( help="Excessive collisions" )

#Class that holds half-duplex error counters for given interfaces
class InterfacesHalfDuplexErrorCounters( Model ):

   intfHalfDuplexErrorCounters = Dict( keyType=IntfModels.Interface,
                                       valueType=HalfDuplexErrorCounters,
                                       help="Mapping between an interface and "
                                            "half-duplex error counters" )

   def render( self ):
      if not self.intfHalfDuplexErrorCounters:
         return
      # Set up the column formatting objects.
      fl = Format( justify="left" )
      fr = Format( justify="right" )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr.padLimitIs( True )

      headings = ( "\nInterface", "Single   \nCollision", "Multiple \nCollision",
                   "Late     \nCollision", "Excessive\nCollision",
                   "Deferred    \nTransmission", )

      table = createTable( headings )
      table.formatColumns( fl, fr, fr, fr, fr )
      for intf in Arnet.sortIntf( self.intfHalfDuplexErrorCounters ):
         counter = self.intfHalfDuplexErrorCounters[ intf ]
         table.newRow( IntfMode.getShortname( intf ),
                       counter.singleCollisionFrames,
                       counter.multipleCollisionFrames,
                       counter.lateCollisions,
                       counter.excessiveCollisions,
                       counter.deferredTransmissions )
      print( table.output() )

# CPU TX counters for interfaces
class InterfacesCounterCpuTx( Model ):
   cpuTxPkts = Int( help="CPU TX Packets" )

#Class that holds CPU TX counters for given interfaces
class InterfacesCountersCpuTx( Model ):

   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=InterfacesCounterCpuTx,
                      help="Mapping between an interface and CPU Tx counters" )

   def render( self ):
      if not self.interfaces:
         return

      # Set up the column formatting objects.
      fl = Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = Format( justify="right" )

      headings = ( "Interface", "CPU TX packets" )

      table = createTable( headings )
      table.formatColumns( fl, fr )
      for intf in Arnet.sortIntf( self.interfaces ):
         intfCounters = self.interfaces[ intf ]
         table.newRow( IntfMode.getShortname( intf ),
                       intfCounters.cpuTxPkts )
      print( table.output() )

# Acl drop counters for interfaces
class InterfacesCounterAclDrop( Model ):
   _name = IntfModels.Interface( help="Name of the interface" )
   aclDrops = Int( help="Packets denied by an ACL" )

#Class that holds acl drop counters for given interfaces
class InterfacesCountersAclDrop( Model ):

   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=InterfacesCounterAclDrop,
                      help="Mapping between an interface and ACL drop counters" )

   def render( self ):
      if len( self.interfaces ) == 0:
         return

      # Set up the column formatting objects.
      fl = Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = Format( justify="right" )

      headings = ( "Port", "InAclDrops" )

      table = createTable( headings )
      table.formatColumns( fl, fr )
      for intf in Arnet.sortIntf( self.interfaces ):
         x = self.interfaces[ intf ]
         table.newRow( IntfMode.getShortname( intf ),
                       x.aclDrops )
      print( table.output() )

interfaceInfoHook = CliExtensions.CliHook()

class DpCounters( Model ):
   inPkts = Int( help="Ingress packets", optional=True )
   inOctets = Int( help="Ingress octets (bytes)", optional=True )
   outPkts = Int( help="Egress packets", optional=True )
   outOctets = Int( help="Egress octets (bytes)", optional=True )

class IntfDropPrecedenceCounters( Model ):
   dpCounters = Dict( keyType=int, valueType=DpCounters,
                 help="Mapping between DP ID and its counter data" )

class DropPrecedenceCounters( Model ):
   interfaces = Dict( keyType=IntfModels.Interface,
                      valueType=IntfDropPrecedenceCounters,
                      help=( "Mapping between interface and all corresponding "
                             "drop-precedence counters" ) )

   def renderHeaders( self, *headers ):
      if not self.interfaces:
         return
      anyCounter = next( iter( self.interfaces.values() ) )

      fl = Format( justify="left" )
      fl.noPadLeftIs( True )
      fr = Format( justify="center", minWidth=5 )
      fr.padLimitIs( True )

      dropPrecedences = sorted( anyCounter.dpCounters )
      table = createTable( [ "Interface", "DP" ] + list( headers ) )
      table.formatColumns( fl, *( [ fr ] * ( 1 + len( headers ) ) ) )

      for intf in Arnet.sortIntf( self.interfaces ):
         for dp in dropPrecedences:
            def checkVal( countObj, header ):
               return getattr( countObj, header, "N/A" )
            dpCounters = self.interfaces[ intf ].dpCounters
            row = [ checkVal( dpCounters[ dp ], header ) for header in headers ]
            table.newRow( intf, dp, *row )
      print( table.output() )

   def render( self ):
      self.renderHeaders( "inPkts", "inOctets" )
      self.renderHeaders( "outPkts", "outOctets" )
