# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliUtil
import Cell
from CliMode.IntfOctal import (
   InterfaceDefaultsSlotTypeMode,
   InterfaceDefaultsModuleMode
)
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import EthIntfLib
import Tac

# -------------------------------------------------------------------------------
# (config-interface-defaults)# slot-type SLOT_TYPE
# (config-<SLOT_TYPE>)#
# -------------------------------------------------------------------------------
class InterfaceDefaultsSlotTypeConfigMode( InterfaceDefaultsSlotTypeMode,
                                           BasicCli.ConfigModeBase ):
   name = 'Interface Defaults Octal Configuration'

   def __init__( self, parent, session, slotType ):
      InterfaceDefaultsSlotTypeMode.__init__( self, slotType )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# Expression to handle the slot-types which can be input to the command
# This allows it to be a bit more extensible
#
# (config-interface-defaults)# slot-type ?
#   osfp-400     Osfp ports with maximum speed of 400G
#   qsfp-dd-400  Qsfp-dd ports with maximum speed of 400G
#   ...
# --------------------------------------------------------------------------------

class HwXcvrSlotConfigCmd( CliCommand.CliCommandClass ):
   # 800g configuration is only supported on silverstrandP
   # BUG951363, we should be able to remove this once we deprecate the command
   slotTypeKeys_ = [
      k for k in EthIntfLib.slotTypeFormat[ 'toCamelCase' ]
      if '800' not in k or 'silverstrand' in Cell.product().lower()
   ]
   syntax = 'slot-type SLOT_TYPE'
   noOrDefaultSyntax = syntax
   data = {
      'slot-type': 'Configure default values for all ports of a given slot type',
      'SLOT_TYPE': CliMatcher.EnumMatcher( {
         # pylint: disable-next=consider-using-f-string
         k: '%s ports with maximum speed of %sG'
             % tuple( k.capitalize().rsplit( '-', 1 ) )
         for k in slotTypeKeys_ } ),
   }

   @staticmethod
   def handler( mode, args ):
      slotConfigDir = EthIntfCli.ethPhyIntfDefaultConfigDir.hwXcvrSlotDefaultConfig
      slotType = EthIntfLib.slotTypeFormat[ 'toCamelCase' ][ args[ 'SLOT_TYPE' ] ]
      slotConfigDir.newMember( slotType )
      slotConfigDir[ slotType ].globalL1DefaultConfig = ( '', )

      newSubMode = mode.childMode( InterfaceDefaultsSlotTypeConfigMode,
                                   slotType=args[ 'SLOT_TYPE' ] )
      mode.session_.gotoChildMode( newSubMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      slotConfigDir = EthIntfCli.ethPhyIntfDefaultConfigDir.hwXcvrSlotDefaultConfig
      if slotType := args.get( 'SLOT_TYPE' ):
         slotType = EthIntfLib.slotTypeFormat[ 'toCamelCase' ][ slotType ]
         del slotConfigDir[ slotType ]
      else:
         slotConfigDir.clear()

IntfCli.InterfaceDefaultsConfigMode.addCommandClass( HwXcvrSlotConfigCmd )

# -------------------------------------------------------------------------------
# (config-<SLOT_TYPE>)# module MODULE
# (config-<SLOT_TYPE>-module-<MODULE>)#
# -------------------------------------------------------------------------------
class InterfaceDefaultsModuleConfigMode( InterfaceDefaultsModuleMode,
                                         BasicCli.ConfigModeBase ):
   name = 'Interface Defaults Octal Configuration'

   def __init__( self, parent, session, slotType, module ):
      InterfaceDefaultsModuleMode.__init__( self, ( slotType, module ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# (config-<SLOT_TYPE>)# module ?
#   Linecard  Linecard numbers
#   <3-6>  Linecard numbers
#
# (config-<SLOT_TYPE>)# module Linecard ?
#   $      list end
#   <3-6>  Linecard numbers
# --------------------------------------------------------------------------------

# This is to support the following:
# 1. module 6
# 2. module Linecard 6
# 3. module linecard 6
# 4. module Linecard6
# 5. module linecard6
class LinecardExpressionFactory( FruCli.SlotExpressionFactory ):
   moduleTypes_ = ( 'Linecard', )
   slotMatcher_ = FruCli.SlotMatcher( overriddenDefaultTags=[ 'Linecard', ] )

class ModuleConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'module MOD'
   noOrDefaultSyntax = syntax
   data = {
      'module': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'module',
            'Configure default values to be applied on '
            'all ports of a given slot type on the given '
            'module' ),
         guard=FruCli.modularSystemGuard ),
      'MOD': LinecardExpressionFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      # pylint: disable-msg=protected-access
      slotType = EthIntfLib.slotTypeFormat[ 'toCamelCase' ][ mode.param_ ]
      slotConfigDir = EthIntfCli.ethPhyIntfDefaultConfigDir.hwXcvrSlotDefaultConfig
      moduleConfigDir = slotConfigDir[ slotType ].moduleL1DefaultConfig

      moduleSlotDesc = args[ 'MOD' ]
      # pylint: disable-next=consider-using-f-string
      module = '%s%s' % ( moduleSlotDesc.tag, moduleSlotDesc.label )
      moduleConfigDir.newMember( module )

      newSubMode = mode.childMode( InterfaceDefaultsModuleConfigMode,
                                   module=module,
                                   slotType=mode.param_ )
      mode.session_.gotoChildMode( newSubMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # pylint: disable-msg=protected-access
      slotType = EthIntfLib.slotTypeFormat[ 'toCamelCase' ][ mode.param_ ]
      slotConfigDir = EthIntfCli.ethPhyIntfDefaultConfigDir.hwXcvrSlotDefaultConfig
      moduleConfigDir = slotConfigDir[ slotType ].moduleL1DefaultConfig

      moduleSlotDesc = args[ 'MOD' ]
      # pylint: disable-next=consider-using-f-string
      module = '%s%s' % ( moduleSlotDesc.tag, moduleSlotDesc.label )
      del moduleConfigDir[ module ]

InterfaceDefaultsSlotTypeConfigMode.addCommandClass( ModuleConfigCmd )

# -------------------------------------------------------------------------------
# (config-<SLOT_TYPE>)# speed SPEED_TYPE
# or
# (config-<SLOT_TYPE>-<MODULE>)# speed SPEED_TYPE
# -------------------------------------------------------------------------------
# We should handle this in IPM in the future(BUG637392).
def getCliPromptIntfs( mode, slotType, module ):
   cliPromptIntfs = list() # pylint: disable=use-list-literal
   intfs = EthIntfCli.getAllIntfs( mode, None, module, EthIntfCli.EthPhyIntf )
   for intf in intfs:
      currentAutonegStatus = intf.status().autonegStatus.mode
      currentForcedLinkMode = intf.config().linkModeLocal
      # We are only interested in interfaces of the provided slotType, also
      # they have to be at default speed (no forced speed, no autoneg).
      if( intf.intfXcvrStatus().hwXcvrSlotType == slotType and
           ( currentAutonegStatus == 'anegModeUnknown' and
             currentForcedLinkMode == 'linkModeUnknown' ) ):
         cliPromptIntfs.append( intf.name )
   return cliPromptIntfs

# Grabs a list of all interfaces with the specified slot type and module,
# and warns the user that those interfaces will be affected by the command.
def warnSpeedDefaults( mode, slotType, module ):
   warning = 'The following interfaces ' \
             'will be affected by this configuration\n'
   prompt = 'Do you wish to proceed with this command? [y/N]'

   cliPromptIntfs = getCliPromptIntfs( mode, slotType, module )
   if cliPromptIntfs:
      for intf in cliPromptIntfs:
         warning += '\n  ' + intf
      mode.addWarning( warning )
      if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
         mode.addMessage( 'Command aborted by user' )
         return False
   return True

# Handle the logic for the speed command
def speedDefault( mode, args ):
   # The slot-type and module are stored within the current mode
   param = mode.param_ # pylint: disable-msg=protected-access
   if isinstance( param, tuple ):
      slotType = param[ 0 ]
      module = param[ 1 ]
   else:
      slotType = param
      module = None
   noOrDefaultCmd = CliCommand.isNoOrDefaultCmd( args )

   # 'toCamelCase' converts cli-formatted slot-type to camelCase
   # 'toDashCase' does the opposite
   slotType = EthIntfLib.slotTypeFormat[ 'toCamelCase' ][ slotType ]

   if noOrDefaultCmd:
      linkMode = Tac.Type( 'Interface::EthLinkMode' ).linkModeUnknown
   else:
      linkMode = EthIntfCli.speedLinkMode[ args[ 'SPEED_TYPE' ] ]

   slotConfigDir = EthIntfCli.ethPhyIntfDefaultConfigDir.hwXcvrSlotDefaultConfig
   slotConfig = slotConfigDir[ slotType ]

   if warnSpeedDefaults( mode, slotType, module ):
      if not module:
         slotConfig.globalL1DefaultConfig.linkMode = linkMode
      else: 
         moduleConfigDir = slotConfig.moduleL1DefaultConfig
         moduleConfigDir[ module ].linkMode = linkMode

# ---------------------------------------------------------------------
# (config-<SLOT_TYPE>)# speed ?
# or
# (config-<SLOT_TYPE>-<MODULE>)# speed ?
#   100g-2  100Gbps 2 lane
#   100g-1  100Gbps 1 lane
# ---------------------------------------------------------------------
class DefaultSpeedExpression( CliCommand.CliExpression ):
   expression = 'SPEED_NODE'
   # List of supported speeds in the format 'speed-lanes'.
   # Ex: 100g-1
   supportedSpeedsStr_ = list() # pylint: disable=use-list-literal
   for ethLinkMode_ in Tac.Type( 'Xcvr::HwXcvrSlotCliSpeeds' ).attributes:
      speed_, lanes_, _ =  EthIntfLib.linkModeToSpeedLanesDuplex[ ethLinkMode_ ]
      # pylint: disable-msg=protected-access
      supportedSpeedsStr_.append( EthIntfCli.EthPhyIntf._speedLanesStr(
                                  speed_, lanes_, True, 'Format2' ).lower() )
   helpdesc_ = 'Default to %s full duplex operation over %s lane'
   helplist_ = []
   # Concise way of getting speed & lanecount from speedStr for the whole list
   for speed_, lanes_ in [ i_.rsplit( '-', 1 ) for i_ in supportedSpeedsStr_ ]: 
      if int( lanes_ ) > 1:
         helplist_.append( helpdesc_ % ( speed_, lanes_ ) + 's' )
      else:
         helplist_.append( helpdesc_ % ( speed_, lanes_ ) )

   data = {
         'SPEED_NODE': CliCommand.Node(
            matcher=CliMatcher.EnumMatcher( dict( zip( supportedSpeedsStr_,
                                                       helplist_ ) ) ),
            alias='SPEED_TYPE' ),
   }

class DefaultSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'speed SPEED_TYPE'
   noOrDefaultSyntax = 'speed ...'
   data = {
         'speed': 'Set the default speed for given interfaces',
         'SPEED_TYPE': DefaultSpeedExpression
   }

   handler = speedDefault
   noOrDefaultHandler = speedDefault

InterfaceDefaultsSlotTypeConfigMode.addCommandClass( DefaultSpeedCmd )
InterfaceDefaultsModuleConfigMode.addCommandClass( DefaultSpeedCmd )
