# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import threading

import Cell
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.XcvrAllStatusDir as XcvrAllStatusDir
import LazyMount
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "IntfPolicyManagerCli" )
t0 = __defaultTraceHandle__.trace0

defaultIpmCache = threading.local()

IntfSwState = TacLazyType( "Interface::IntfSwState::IntfSwState" )
IntfHwState = TacLazyType( "Interface::IntfHwState::IntfHwState" )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )

ResourceType = TacLazyType( "Interface::ResourceType" )
LogicalPortAllocationPriority = TacLazyType(
   "AlePhy::LogicalPortAllocationPriority::LogicalPortAllocationPriority" )
XcvrAdapterManagementType = TacLazyType(
   "AlePhy::XcvrAdapterManagementType::XcvrAdapterManagementType" )

allIntfConfigDir = None
allIntfStatusDir = None
defaultCapsRootDir = None
xcvrAllStatusProxy = None
resourceManagerSliceConfig = None
sysdbResourceManagerSliceStatus = None
sissBaseDir = None
l1CardPowerGenerationRootDir = None

def getIpm( resourceConsumerDir, sysdbResourceManagerStatus=None,
            intfStatusDir=None, ipmCache=defaultIpmCache ):
   """
   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir

   Returns
   -------
   An instance of AlePhy::IntfPolicyManager that computes states for all the
   interfaces in resourceConsumerDir.
   """
   if not resourceConsumerDir:
      return None

   rcdToIpm = getattr( ipmCache, 'rcdToIpm', None )
   if not rcdToIpm:
      ipmCache.rcdToIpm = {}

   ipm = ipmCache.rcdToIpm.get( resourceConsumerDir )

   if not ipm:
      ipm = IntfPolicyManager( resourceConsumerDir, sysdbResourceManagerStatus,
                               intfStatusDir )
      ipmCache.rcdToIpm[ resourceConsumerDir ] = ipm

   return ipm

def getRmcFromRcd( resourceConsumerDir ):
   """Retrieve Interface::ResourceManagerConfig object in Sysdb for the same subslice
   as the one in the given resourceConsumerDir.

   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir

   Returns
   -------
   Interface::ResourceManagerConfig in Sysdb or None
   """
   for sliceDir in resourceManagerSliceConfig.values():
      if sliceDir.name == resourceConsumerDir.parent.name:
         for subSliceDir in sliceDir.values():
            if subSliceDir.name == resourceConsumerDir.name:
               return subSliceDir
   return None

def getSysdbRmsFromRcd( resourceConsumerDir ):
   """Retrieve Interface::ResourceManagerStatus object in Sysdb for the same subslice
   as the one in the given resourceConsumerDir.

   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir

   Returns
   -------
   Interface::ResourceManagerStatus in Sysdb or None
   """
   for sliceDir in sysdbResourceManagerSliceStatus.values():
      if sliceDir.name == resourceConsumerDir.parent.name:
         for subSliceDir in sliceDir.values():
            if subSliceDir.name == resourceConsumerDir.name:
               return subSliceDir
   return None

def getRcdForIntf( intf, resourceConsumerSliceDir ):
   """
   Parameters
   ----------
   intf : CliPlugin.EthIntfCli.EthPhyIntf
   resourceConsumerSliceDir : Tac::Dir
      Dir at /ar/Sysdb/interface/resources/consumers/slice

   Returns
   -------
   AlePhy::ResourceConsumerDir that contains intf.
   """
   for sliceDir in resourceConsumerSliceDir.values():
      for subSliceDir in sliceDir.values():
         if intf.name in subSliceDir.serdesResourceConsumer:
            return subSliceDir
   t0( "Failed to find RCD:", intf.name )
   return None

def getRcdToIntfListMap( intfList, resourceConsumerSliceDir ):
   """
   Parameters
   ----------
   intfList : [ list of CliPlugin.EthIntfCli.EthPhyIntf ]
   resourceConsumerSliceDir : Tac::Dir
      Dir at /ar/Sysdb/interface/resources/consumers/slice

   Returns
   -------
   dict { AlePhy::ResourceConsumerDir : [ list of CliPlugin.EthIntfCli.EthPhyIntf ] }
   """
   rcdToIntfListMap = defaultdict( list )

   for intf in intfList:
      rcd = getRcdForIntf( intf, resourceConsumerSliceDir )
      if not rcd:
         continue
      rcdToIntfListMap[ rcd ].append( intf )

   return rcdToIntfListMap

def getSubdomainFromIntf( intfId, baseSissDir ):
   """
   Parameters
   ----------
   intf : Arnet::IntfId
   baseSissDir : Tac::Dir
      Dir at /ar/Sysdb/interface/intfSlot/status/[frontPanel|internal]/slot

   Returns
   -------
   Tac::String
   """
   for sliceSissDir in baseSissDir.values():
      for subdomainDir in sliceSissDir.values():
         if subdomainDir.fetchSlot( intfId ):
            return subdomainDir.name
   t0( "Failed to find SubdomainIntfSlotStatusDir:", intfId )
   return None
   
def getSissDirFromRcd( resourceConsumerDir, resourceManagerConfig ):
   """
   Parameters
   ----------
   resourceConsumerDir : AlePhy::ResourceConsumerDir
   resourceManagerConfig : Interface::ResourceManagerConfig

   Returns
   -------
   Interface::SubdomainIntfSlotStatusDir::PtrConst aggregated for a particular
   subdomain.
   """
   if not resourceManagerConfig.useSubdomains:
      # The forwarding agent initializing resourceManagerConfig doesn't use
      # subdomains.
      return None

   intfId = next( iter( resourceConsumerDir.serdesResourceConsumer ) )
   sliceId = resourceConsumerDir.parent.name
   # TODO: Replace with an API to fetch correct dir when given the IntfId
   if intfId.startswith( "Ethernet" ):
      # Look in the front panel dir
      dirName = "frontPanel"
   else:
      dirName = "internal"

   sissDir = sissBaseDir[ dirName ][ "slot" ]
   subdomainName = getSubdomainFromIntf( intfId, sissDir )
   if not subdomainName:
      # Subdomains are not enabled for the system, return None.
      # TODO: When subdomains are enabled for all systems, we should change this into
      # an assertion since all interfaces should be tagged with a subdomain.
      return None
   allSissDir = Tac.newInstance( "Interface::SubdomainIntfSlotStatusPtrConstDir",
                                 subdomainName )
   if sliceId.startswith( "Switchcard" ):
      # In order to maintain consistency with the forwarding agent users of IPM, we
      # need to aggregate across all slices if we're running on a switchcard.
      sliceId = Tac.Type( "Interface::SubdomainAggregationConstants" ).allSlice

   # Lazy mounted paths must be force mounted prior to passing them to any TACC code
   # to avoid deadlocks.
   LazyMount.force( l1CardPowerGenerationRootDir )

   # Aggregate if the target dirs exist
   for target in [ "internal", "frontPanel" ]:
      sissDir = sissBaseDir[ target ][ "slot" ]
      if sissDir:
         Tac.newInstance( "Interface::SubdomainBaseIntfSlotStatusAggregator",
                          sissDir,
                          l1CardPowerGenerationRootDir,
                          sliceId,
                          subdomainName,
                          allSissDir )
   return allSissDir

class IntfPolicyManager:
   """
   This class is an interface for IntfPolicyManager. It takes the necessary
   directories and processes them as IPM would, so that configuration changes
   can be tested before applying.

   Expected usage is get an instance of the class with getIpm, synchronize its input
   directories with synchronize, modify the directories to apply the changes to be
   tested, call processIntfs, and determine what has changed by looking at
   intfStateDir and other directories.

   getHwActiveChangeIntfs and getDisabledIntfs are provided for convenience for speed
   change.
   """
   def __init__( self, resourceConsumerDir, sysdbResourceManagerStatus=None,
                 intfStatusDir=None ):
      self.resourceManagerConfig = None
      self.sysdbResourceManagerStatus = sysdbResourceManagerStatus
      self.resourceManagerStatus = None
      self.resourceConsumerDir = resourceConsumerDir
      self.intfStateDir = None
      self.mockEthIntfModeDir = None
      self.mockAllIntfConfigDir = None
      self.mockIntfConfigDir = None
      self.internalEthIntfModeDir = None
      self.intfStatusDir = intfStatusDir
      self.xcvrStatusDir = None
      self.dfltCapsRootDir = None
      self.subdomainIntfSlotStatusDir = None
      self.mockSpeedGroupStatusDir = None

      self.createDirectories()
      if ( self.subdomainIntfSlotStatusDir and
           self.subdomainIntfSlotStatusDir.subdomainIntfSlotStatus ):
         # Xcvr status dir is not needed
         xcvrStatusDir = None
         sissDir = self.subdomainIntfSlotStatusDir
      else:
         # Subdomain dir is not needed
         xcvrStatusDir = self.xcvrStatusDir
         sissDir = None
      self.ipm = Tac.newInstance( "AlePhy::IntfPolicyManager",
                                  self.resourceManagerConfig,
                                  self.resourceManagerStatus,
                                  self.mockEthIntfModeDir,
                                  self.intfStatusDir,
                                  xcvrStatusDir,
                                  sissDir )
      self.addResourceManagers()

   def createDirectories( self ):
      # Fetch the resource manager config or create an empty one. An empty resource
      # manager config has no effect.
      self.resourceManagerConfig = getRmcFromRcd( self.resourceConsumerDir )
      if not self.resourceManagerConfig:
         self.resourceManagerConfig = Tac.newInstance(
            "Interface::ResourceManagerConfig", "rmc" )

      if self.sysdbResourceManagerStatus is None:
         self.sysdbResourceManagerStatus = getSysdbRmsFromRcd(
            self.resourceConsumerDir )
      self.resourceManagerStatus = Tac.newInstance(
         "Interface::ResourceManagerStatus", "rms" )
      self.resourceManagerStatus.ethIntfModeDir = ( "leimd", )
      self.resourceManagerStatus.intfStateDir = ( "risd", )

      self.intfStateDir = self.resourceManagerStatus.intfStateDir

      self.mockAllIntfConfigDir = Tac.newInstance(
         "Interface::AllEthPhyIntfConfigDir", "strataepicd" )

      self.mockIntfConfigDir = Tac.newInstance( "Interface::EthPhyIntfConfigDir",
                                                "sandepicd" )

      self.mockEthIntfModeDir = Tac.newInstance( "Interface::EthIntfModeDir",
                                                 "eimd" )

      self.internalEthIntfModeDir = self.resourceManagerStatus.ethIntfModeDir

      # We need to force the mount, otherwise mounting does not occur and
      # newInstance hangs as that is the first access to the proxy.
      if self.intfStatusDir is None:
         LazyMount.force( allIntfStatusDir )
         self.intfStatusDir = allIntfStatusDir
      if self.dfltCapsRootDir is None:
         LazyMount.force( defaultCapsRootDir )
         self.dfltCapsRootDir = defaultCapsRootDir

      self.xcvrStatusDir = xcvrAllStatusProxy.force()

      # Aggregated siss dir
      self.subdomainIntfSlotStatusDir = getSissDirFromRcd(
            self.resourceConsumerDir, self.resourceManagerConfig )

      self.mockSpeedGroupStatusDir = Tac.newInstance(
            "Interface::SpeedGroupStatusDir", "sgsd" )

   def _updateLogicalPortResourceManager( self ):
      if not self.ipm:
         return

      expManagement = self.resourceConsumerDir.logicalPortAllocationPriority
      if ResourceType.logicalPort not in self.ipm.resourceManager:
         curManagement = LogicalPortAllocationPriority.none
      else:
         manager = self.ipm.resourceManager[ ResourceType.logicalPort ]
         curManagement = manager.priority

      if expManagement == curManagement:
         return

      if expManagement == LogicalPortAllocationPriority.none:
         del self.ipm.resourceManager[ ResourceType.logicalPort ]
      elif expManagement == LogicalPortAllocationPriority.intfIdBlackhawkGen3:
         self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::BlackhawkGen3RemapResourceManager",
                            self.mockEthIntfModeDir,
                            self.resourceConsumerDir,
                            self.ipm.reactorSmRegistry,
                            self.mockAllIntfConfigDir ) )
      elif expManagement == LogicalPortAllocationPriority.intfId:
         self.ipm.resourceManager.addMember(
               Tac.newInstance( "AlePhy::LogicalPortResourceManager",
                                self.mockEthIntfModeDir,
                                self.resourceConsumerDir,
                                self.ipm.reactorSmRegistry ) )
      elif expManagement == LogicalPortAllocationPriority.intfIdAndIntfEnabled:
         self.ipm.resourceManager.addMember(
               Tac.newInstance( "AlePhy::LogicalPortReleaseOnShutResourceManager",
                                self.mockEthIntfModeDir,
                                self.resourceConsumerDir,
                                self.ipm.reactorSmRegistry,
                                self.mockAllIntfConfigDir ) )
      else:
         assert False, "Unknown logial port allocation priority type."

   def _updateXcvrAdapterResourceManager( self ):
      if not self.ipm:
         return

      expManagement = self.resourceConsumerDir.xcvrAdapterManagementType
      if ResourceType.xcvrAdapter not in self.ipm.resourceManager:
         curManagement = None
      else:
         manager = self.ipm.resourceManager[ ResourceType.xcvrAdapter ]
         curManagement = manager.managementType()

      if expManagement == curManagement:
         return

      if expManagement == XcvrAdapterManagementType.legacyManagement:
         self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::LegacyXcvrAdapterResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )
      elif expManagement == XcvrAdapterManagementType.defaultManagement:
         if ( self.subdomainIntfSlotStatusDir and
              self.subdomainIntfSlotStatusDir.subdomainIntfSlotStatus ):
            xarmType = "AlePhy::SubdomainXcvrAdapterResourceManager"
         else:
            xarmType = "AlePhy::XcvrAdapterResourceManager"
         self.ipm.resourceManager.addMember(
            Tac.newInstance( xarmType,
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )
      elif expManagement == XcvrAdapterManagementType.hitfulSpeedChangeManagement:
         self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::HitfulSpeedChangeXcvrAdapterResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )
      else:
         # pylint: disable-next=consider-using-f-string
         assert False, "Unknown xcvr adapter resource management type: {}".format(
                          expManagement )


   def addResourceManagers( self ):
      """
      This method unconditionally adds all resource managers that could be used. It
      is assumed that they have no side effects if their resources do not exist.

      Strata and Sand use different resource managers so this needs to reflect the
      correct resource managers for the product

      Common:
         SerdesResourceManager
         XcvrAdapterResourceManager
         PllResourceManager
      Strata resource managers:
         BlackhawkGen3RemapResourceManager
         LogicalPortResourceManager
         LogicalPortReleaseOnShutResourceManager
         IntfCapsResourceManager
         IntfRestrictedResourceManager
      Sand resource managers:
         IntfPermissionsResourceManager
         IntfShutResourceManager
      """

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::SerdesResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )

      self._updateXcvrAdapterResourceManager()

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::PllResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry,
                             self.mockSpeedGroupStatusDir,
                             None, None ) )

      self._updateLogicalPortResourceManager()

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfCapsResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfRestrictedResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry ) )

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfPermissionsResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry,
                             self.mockIntfConfigDir,
                             None ) )

      self.ipm.resourceManager.addMember(
            Tac.newInstance( "AlePhy::IntfDefaultCapsResourceManager",
                             self.mockEthIntfModeDir,
                             self.resourceConsumerDir,
                             self.ipm.reactorSmRegistry,
                             None,
                             self.dfltCapsRootDir ) )

      if self.resourceConsumerDir.portGroupData is not None:
         self.ipm.resourceManager.addMember(
               Tac.newInstance( "AlePhy::PortGroupResourceManager",
                                self.mockEthIntfModeDir,
                                self.resourceConsumerDir,
                                self.ipm.reactorSmRegistry ) )

      # TODO BUG571752 Add BlackhawkAutonegRestrictionManager support to CLI IPM

      # Empty the reactorSmRegistry. We don't need or want reactions.
      self.ipm.reactorSmRegistry.reactorSm.clear()

   def synchronize( self ):
      """
      Synchronizes our mocked IPM input directories to match the systems.
      """
      self.resetEthIntfModeDir()
      self.resetIntfConfigDir()
      self.resetSpeedGroupStatusDir()
      self._updateXcvrAdapterResourceManager()
      self._updateLogicalPortResourceManager()
      # Always reset intfStateDir at last as it depends on other inputs.
      self.resetAndUpdateIntfStateDir()

   def processIntfs( self ):
      # allow IPM to run
      self.ipm.initialized = True

      t0( "processing intf state" )
      self.ipm.processIntfState()

      # stop allowing IPM to run
      self.ipm.initialized = False

   def enableInterface( self, intfName ):
      # Set it for the strata object
      if intfName in self.mockAllIntfConfigDir.intfConfig:
         self.mockAllIntfConfigDir.intfConfig[ intfName ].adminEnabled = True

      # Set it for the sand object
      if intfName in self.mockIntfConfigDir.intfConfig:
         self.mockIntfConfigDir.intfConfig[ intfName ].adminEnabled = True

   def resetEthIntfModeDir( self ):
      # Resets and re-syncs mode with the current SysdbIntfMode
      self.mockEthIntfModeDir.ethIntfMode.clear()
      self.ipm.prevEthIntfModeDir.ethIntfMode.clear()
      for ethIntfModeSliceDir in EthIntfCli.ethIntfModeSliceDir.values():
         for ethIntfModeDir in ethIntfModeSliceDir.values():
            for mode in ethIntfModeDir.ethIntfMode.values():
               self.mockEthIntfModeDir.ethIntfMode.addMember( mode )
               self.ipm.prevEthIntfModeDir.ethIntfMode.addMember( mode )

   def resetIntfConfigDir( self ):
      # bug483510 it is assumed that a system will have only one of
      # mockAllIntfConfigDir or mockIntfConfigDir. Currently this works
      # because no resource manager using these objects is supported on both
      # strata and sand so we only ever pass in the correct one to the
      # correct RM. In order to pass in both we will need to find a better
      # way to handle, populating these since one will be assumed to be empty.
      self.mockAllIntfConfigDir.intfConfig.clear()
      self.mockIntfConfigDir.intfConfig.clear()
      for intfId, intfConfig in allIntfConfigDir.intfConfig.items():
         epic = Tac.newInstance( "Interface::EthPhyIntfConfig", intfId )
         epic.adminEnabled = intfConfig.adminEnabled
         epic.adminEnabledStateLocal = intfConfig.adminEnabledStateLocal
         self.mockAllIntfConfigDir.intfConfig.addMember( epic )

         self.mockIntfConfigDir.newIntfConfig( intfId )
         self.mockIntfConfigDir.intfConfig[ intfId ].adminEnabled = \
               intfConfig.adminEnabled
         self.mockIntfConfigDir.intfConfig[ intfId ].adminEnabledStateLocal = \
               intfConfig.adminEnabledStateLocal

   def resetAndUpdateIntfStateDir( self ):
      self.intfStateDir.intfState.clear()
      self.ipm.localIntfStateDir.intfState.clear()
      for intf in self.resourceConsumerDir.serdesResourceConsumer:
         # Skip interfaces that are not ready.
         sysDbResourceMgrSts = self.sysdbResourceManagerStatus
         if sysDbResourceMgrSts and \
            intf not in sysDbResourceMgrSts.intfStateDir.intfState:
            continue
         if intf not in self.mockEthIntfModeDir.ethIntfMode:
            continue
         # BUG483510 - we currently assume these two config collections are equal.
         assert set( self.mockAllIntfConfigDir.intfConfig ) == \
               set( self.mockIntfConfigDir.intfConfig )
         if intf not in self.mockAllIntfConfigDir.intfConfig:
            continue
         if intf not in self.intfStatusDir.intfStatus:
            continue
         if Cell.cellType() != "generic":
            module = EthIntfId.module( intf )
            port = EthIntfId.port( intf )
            # pylint: disable-next=consider-using-f-string
            xcvrName = ( "Ethernet%d/%d" % ( module, port )
                         # pylint: disable-next=consider-using-f-string
                         if module else "Ethernet%d" % port )
         else:
            delimPos = intf.rfind( '/' )
            xcvrName = intf if delimPos < 0 else intf[ :delimPos ]
         sissDir = self.subdomainIntfSlotStatusDir
         if ( ( sissDir and sissDir.subdomainIntfSlotStatus
                and xcvrName not in sissDir.subdomainIntfSlotStatus )
              or ( self.xcvrStatusDir
                   and xcvrName not in self.xcvrStatusDir.xcvrStatus ) ):
            continue

         # Synchronizing local intfStates with SysDb
         rmsIntfState = self.intfStateDir.intfState.newMember( intf )
         localIntfState = self.ipm.localIntfStateDir.intfState.newMember( intf )

         if sysDbResourceMgrSts:
            rmsIntfState.state = \
               sysDbResourceMgrSts.intfStateDir.intfState.get( intf ).state
            localIntfState.state = rmsIntfState.state
            rmsIntfState.inactiveReason.update(
               sysDbResourceMgrSts.intfStateDir.intfState.get( intf ).inactiveReason
               )
            localIntfState.inactiveReason.update( rmsIntfState.inactiveReason )
            rmsIntfState.modeGeneration = \
               sysDbResourceMgrSts.intfStateDir.intfState.get( intf ).modeGeneration


   def resetSpeedGroupStatusDir( self ):
      sgsd = self.resourceConsumerDir.speedGroupStatusDir
      mockSgsd = self.mockSpeedGroupStatusDir
      sysdbResourceMgrSts = self.sysdbResourceManagerStatus
      localResourceMgrSts = self.resourceManagerStatus

      mockSgsd.group.clear()
      mockSgsd.intfSpeedGroup.clear()
      localResourceMgrSts.speedGroupConfig.clear()

      if not sgsd:
         return

      for k, v in sgsd.intfSpeedGroup.items():
         mockSgsd.intfSpeedGroup[ k ] = v
      for groupName, groupStatus in sgsd.group.items():
         mockStatus = mockSgsd.group.newMember( groupName )

         for k, v in groupStatus.memberIntf.items():
            mockStatus.memberIntf[ k ] = v
         mockStatus.masterIntf = groupStatus.masterIntf
         mockStatus.settingCount = groupStatus.settingCount
         for k, v in groupStatus.supportedModes.items():
            mockStatus.supportedModes[ k ] = v
         for k, v in groupStatus.settingPriority.items():
            mockStatus.settingPriority[ k ] = v
         mockStatus.capabilitiesGeneration = groupStatus.capabilitiesGeneration

         for k, v in groupStatus.setting.items():
            mockStatus.setting[ k ] = v
         mockStatus.settingGeneration = groupStatus.settingGeneration
         mockStatus.speedGroupAutoSupported = groupStatus.speedGroupAutoSupported

      if not sysdbResourceMgrSts:
         return

      sysdbIpmSpeedGroupConfig = sysdbResourceMgrSts.speedGroupConfig

      # Syncing localRMS with sysDb's RMS speedGroupConfig.
      for groupName, groupStatus in sysdbIpmSpeedGroupConfig.items():
         localRmsConfig = localResourceMgrSts.speedGroupConfig.newMember(
            groupName )
         localRmsConfig.setting.update( groupStatus.setting )
         localRmsConfig.settingGeneration = groupStatus.settingGeneration

   def getHwActiveChangeIntfs( self ):
      """
      Returns the set of interfaces whose activeness would change if the current
      state takes affect.
      """
      affected = set()
      for intfId, newState in self.intfStateDir.intfState.items():
         epis = self.intfStatusDir.intfStatus.get( intfId )
         # Interface is currently active,
         if epis.hwActive():
            # speed change would make it inactive.
            if newState.state.hwState == IntfHwState.inactive:
               affected.add( intfId )
         # Interface is currently inactive,
         else:
            # speed change would make it active.
            if newState.state.hwState == IntfHwState.active:
               affected.add( intfId )

      return affected

   def getDisabledIntfs( self, reason=ResourceType.logicalPort ):
      """
      Returns the set of interfaces which will become inactive due to the given
      reason if the current state takes affect.
      """
      # interfaces which have become inactive due to logical port
      disabled = set()
      for intfId, newState in self.intfStateDir.intfState.items():
         epis = self.intfStatusDir.intfStatus.get( intfId )
         if not epis.hwActive():
            continue
         # Interface is currently active,
         # speed change would make it inactive.
         if ( newState.state.swState == IntfSwState.inactive
              and reason in newState.inactiveReason
              and len( newState.inactiveReason ) == 1 ):
            disabled.add( intfId )

      return disabled

def Plugin( em ):
   global allIntfConfigDir
   global allIntfStatusDir
   global defaultCapsRootDir
   global xcvrAllStatusProxy
   global resourceManagerSliceConfig
   global sysdbResourceManagerSliceStatus
   global sissBaseDir
   global l1CardPowerGenerationRootDir

   allIntfConfigDir = LazyMount.mount( em, "interface/config/eth/phy/all",
                                       "Interface::AllEthPhyIntfConfigDir", "r" )
   allIntfStatusDir = LazyMount.mount( em, "interface/status/eth/phy/all",
                                       "Interface::AllEthPhyIntfStatusDir", "r" )
   defaultCapsRootDir = LazyMount.mount(
      em, "interface/archer/status/eth/phy/capabilities/default/slice", "Tac::Dir",
      "r" )
   xcvrAllStatusProxy = XcvrAllStatusDir.xcvrAllStatusDir( em )
   resourceManagerSliceConfig = LazyMount.mount(
         em, "interface/resources/config/slice", "Tac::Dir", "ri" )
   sysdbResourceManagerSliceStatus = LazyMount.mount(
         em, "interface/resources/status/slice", "Tac::Dir", "ri" )
   sissBaseDir = LazyMount.mount(
         em, "interface/intfSlot/status", "Tac::Dir", "ri" )
   l1CardPowerGenerationRootDir = LazyMount.mount(
         em, "interface/archer/config/eth/phy/generation/slice", "Tac::Dir", "ri" )
