# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin.EthIntfCli import EthIntf, L1Intf
from CliPlugin.IntfCli import ShowIntfCommand, Intf
from CliPlugin.SubIntfCli import SubIntf

# --------------------------------------------------------------------------------
# Base class for all "show interfaces status EXTRA_SYNTAX"
# -------------------------------------------------------------------------------
showIntfStatusKw = CliMatcher.KeywordMatcher( 'status',
                                      helpdesc='Details on the state of interfaces' )
filterData = { t: CliCommand.singleKeyword( t, f'Include {t} interfaces' )
               for t in ( 'connected', 'notconnect', 'disabled', 'inactive',
                          'signal' ) }
filterData[ 'sub-interfaces' ] = CliCommand.singleKeyword( 'sub-interfaces',
                                                           'Include sub interfaces' )
filterData[ 'no-signal' ] = CliCommand.singleKeyword(
   'no-signal', 'Include no signal interfaces' )
filterData[ 'no-resource' ] = CliCommand.singleKeyword(
   'no-resource', 'Include no resource interfaces' )
statusData = dict( status=showIntfStatusKw )
statusData.update( filterData )

class ShowIntfStatusCommand( ShowIntfCommand ):
   '''Defines a base class for all "show interfaces status" commands.

   Basic Syntax:
      show interface status [ { status1 | status2 | ... } ] EXTRA_SYNTAX

   The subclass may customize this syntax using the following controls:
      extraStatusFilters: When set, extra status filters will be added in addition
      to the existing filters defined in filterData above
   '''
   statusSyntax = 'show interfaces status [ {statusFilter} ] {extraSyntax}'
   statusData_ = statusData
   moduleAtEnd = True
   allowAllSyntax = True

   extraStatusFilters = {}
   ALLOWED_FIELDS = tuple( list( ShowIntfCommand.ALLOWED_FIELDS ) +
                           [ 'extraStatusFilters' ] )

   @classmethod
   def _generateSyntaxAndData( cls ):
      cls._assert( cls.syntax.startswith( 'show interfaces status' ),
                   'must define syntax with \'show interfaces status ...\'' )

      statusFilter = filterData
      statusFilter.update( cls.extraStatusFilters )
      filters = ' | '.join( statusFilter )
      filterStr = f'{{ {filters} }}'

      extraSyntax = cls.syntax.removeprefix( 'show interfaces status' ).strip()
      cls.syntax = cls.statusSyntax.format( statusFilter=filterStr,
                                            extraSyntax=extraSyntax )

      cls.statusData_.update( cls.extraStatusFilters )
      for k in cls.statusData_:
         cls._assert( k not in cls.data,
                      f'{k!r} should not be defined in data' )

      cls.data.update( cls.statusData_ )
      super()._generateSyntaxAndData()

   @staticmethod
   def getIntfs( mode, args ):
      intf = args.get( 'INTF' )
      mod = args.get( 'MOD' )
      filters = set( args ) & set( filterData )
      includeSubIntfs = 'sub-interfaces' in filters
      filters.discard( 'sub-interfaces' )

      if 'no-signal' in filters:
         filters.remove( 'no-signal' )
         filters.add( 'noSignal' )

      # We need to set exposeInactive to True for Intf.getAll calls, when
      # trying to show only inactive interfaces, otherwise inactive
      # interfaces will be hidden.
      kwargs = {
            'exposeInactive': 'inactive' in filters,
            'exposeInternal': 'all' in args,
      }

      intfType = SubIntf
      if intf:
         if '.' not in str( intf ):
            if includeSubIntfs:
               mode.addError( 'Invalid option: sub-interfaces' )
               return []
            del kwargs[ 'exposeInternal' ]
            intfType = ( EthIntf, L1Intf )
      else:
         if not includeSubIntfs:
            intfType = ( SubIntf, EthIntf, L1Intf )

      intfs = Intf.getAll( mode, intf, mod, intfType=intfType, **kwargs )

      # BUG91241: need to call lookup
      intfs = ( i for i in intfs if i.lookup() )
      if filters:
         intfs = ( i for i in intfs if i.linkStatus() in filters )

      return intfs
