# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module implements the following command
 - "show interfaces status detail"
 - "show interfaces [<name>] status { <status1> | <status2> ... } detail"
'''

import re

import BasicCli
from CliPlugin.EthIntfCli import L1Intf
from CliPlugin.ErrdisableCli import statusErrdisableKw
from CliPlugin.IntfStatusCli import ShowIntfStatusCommand
from CliPlugin.IntfStatusDetailCliModel import (
   StatusDetail,
   InterfaceStatusDetailEntry,
   ShowStatusDetailModel )
from Intf.IntfRange import intfListToCanonical
from Plugins import SkipPluginModule
from Toggles.EbraToggleLib import toggleStatusDetailCommandEnabled
from TypeFuture import TacLazyType

if not toggleStatusDetailCommandEnabled():
   raise SkipPluginModule( 'Show interfaces status detail command not enabled' )

IntfSwState = TacLazyType( 'Interface::IntfSwState::IntfSwState' )
L1PolicyVersion = TacLazyType( 'L1::PolicyVersion' )
ResourceType = TacLazyType( 'Interface::ResourceType' )
StatusDetailConstants = TacLazyType( 'L1::IntfStatusDetailConstants' )

# Caution: the value in this dict will affect CLI output
#
# TODO yhua: will need to improve how we define these constants, this is only
# a temporary solution because IPM is the only publisher for status details for now
resourceTypeToReason = {
   ResourceType.logicalPort: StatusDetailConstants.logicalPortReason(),
   ResourceType.pll: StatusDetailConstants.pllReason(),
   ResourceType.xcvrAdapter: StatusDetailConstants.xcvrAdapterReason(),
   ResourceType.intfCaps: StatusDetailConstants.capsReason(),
   # BUG957450, serdes will only be serdes capabilities until we support gearbox
   ResourceType.serdes: StatusDetailConstants.capsReason(),
   ResourceType.fec: StatusDetailConstants.fecReason(),
   ResourceType.intfLane: StatusDetailConstants.subsumedReason()
}
intfStateToStatus = {
   IntfSwState.active: 'connected',
   IntfSwState.errdisabled: 'errdisabled',
   IntfSwState.inactive: 'inactive'
}

def parseLogicalPortDetail( detail ):
   '''The logical port status detail we received from IPM will look like this:
      'Pool 1 ([ Ethernet1/1, Ethernet1/2, Ethernet1/3, Ethernet1/4 ])
      of 2 logical ports has been depleted'
      We would like to further parse the interface list to more customer-readable
      interface ranges such as 'Et1/1-4', since there can be over 20 interfaces
      in a logical port pool and we don't want to print all of them individually.
      If we cannot properly parse the interface list, we will return the original
      detail.
   '''
   listMatcher = re.compile( r'\[(?P<interfaces>.+)\]' )
   matchingList = listMatcher.search( detail )
   if not matchingList:
      return detail

   intfList = matchingList.group( 1 ).strip().split( ', ' )

   intfRange = intfListToCanonical(
       intfList, noHoleRange=True, verifyIntfHoles=True )
   if not intfRange:
      return detail
   intfRangeStr = ','.join( intfRange )

   return detail.replace( f'[{matchingList.group( 1 )}]', intfRangeStr )

def consolidateStateDetails( stateDetails ):
   '''Consolidate interface state details.
   For now the only case we want to consolidate is when there is an unsupported
   speed, both caps and pll will report error, but it is only useful to display
   caps as the reason of errdisabling
   '''
   capsReason = StatusDetailConstants.capsReason()
   if capsDetail := stateDetails.get( capsReason ):
      return { capsReason: capsDetail }

   return stateDetails


def getIntfStatusDetailFromIpm( intf ):
   '''Note: This is V1 of status detail command, we simply read from IPM because it
   contains the first group of interface statuses that we would like to publish
   details for, later we will support getting details from other entities'''

   if not isinstance( intf, L1Intf ):
      return []

   intfState = None
   intfStateDetail = None
   resourceMgrStatus = intf.resourceManagerStatus()
   if not resourceMgrStatus:
      return []

   intfState = resourceMgrStatus.intfStateDir.intfState.get( intf.name )
   intfStateDetail = resourceMgrStatus.intfStateDir.intfStateDetail.get(
      intf.name )

   if not intfState or not intfStateDetail:
      return []

   if intfState.state.swState not in intfStateToStatus:
      return []

   # We don't want to display because interface status is not being driven by IPM
   if intfStateToStatus[ intfState.state.swState ] != intf.linkStatus():
      return []

   stateDetails = {}
   for resource, detail in intfStateDetail.stateDetail.items():
      # Filter away any unintended reason being published
      if resource not in resourceTypeToReason:
         continue

      # Further parse logical port interfaces to more readable interface ranges
      if resource == ResourceType.logicalPort:
         detail = parseLogicalPortDetail( detail )
      stateDetails[ resourceTypeToReason[ resource ] ] = detail

   stateDetails = consolidateStateDetails( stateDetails )
   return [
      StatusDetail( reason=reason or None,
                    detail=detail or None )
      for reason, detail in stateDetails.items()
   ]

def showStatusDetailHandler( mode, args ):
   showStatusDetails = ShowStatusDetailModel()

   intfs = ShowIntfStatusCommand.getIntfs( mode, args )
   for intf in intfs:
      try:
         intfStatus = intf.linkStatus()
      except AttributeError:
         # Some interfaces may have been deleted during the process,
         # description() and linkStatus() reads from Sysdb, and AttributeError
         # will be thrown for non-existing paths
         continue

      # We currently only support status detail for interfaces managed under
      # L1 Policy V3
      if ( not isinstance( intf, L1Intf )
           or intf.l1PolicyVersion() < L1PolicyVersion( 3, 0 ) ):
         continue

      showStatusDetails.interfaces[ intf.name ] = \
         InterfaceStatusDetailEntry(
            description=intf.description() or None,
            status=intfStatus,
            details=getIntfStatusDetailFromIpm( intf )
         )

   return showStatusDetails

class ShowIntfStatusDetail( ShowIntfStatusCommand ):
   syntax = 'show interfaces status detail'
   extraStatusFilters = { 'errdisabled': statusErrdisableKw }
   data = {
      'detail': 'Detailed information explaining interface statuses'
   }
   cliModel = ShowStatusDetailModel
   handler = showStatusDetailHandler

BasicCli.addShowCommandClass( ShowIntfStatusDetail )
