# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import (
   Dict,
   List,
   Model,
   Str,
)
from CliPlugin import IntfModel
from CliPlugin.IntfCli import Intf
from IntfModels import Interface
from TableOutput import createTable, Format

class StatusDetail( Model ):
   reason = Str( help='Reason for the interface status', optional=True )
   detail = Str( help='Detail for the interface status', optional=True )

class InterfaceStatusDetailEntry( Model ):
   description = Str( help='Interface description', optional=True )
   status = IntfModel.InterfaceStatus.interfaceStatus # Enum
   details = List( valueType=StatusDetail,
                         help='List of interface status details' )

class ShowStatusDetailModel( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=InterfaceStatusDetailEntry,
                      help='Map interfaces to status details' )

   def render( self ):
      if not self.interfaces:
         return

      t = createTable( ( 'Interface', 'Name', 'Status', 'Reason', 'Detail' ) )
      left = Format( justify='left' )
      left20 = Format( justify='left', maxWidth=20, wrap=True )
      left100 = Format( justify='left', maxWidth=100, wrap=True )

      t.formatColumns( left, left20, left, left, left100 )

      for fmt in t.columnFormats_:
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )

      for intf in Arnet.sortIntf( self.interfaces ):
         statusDetailEntry = self.interfaces[ intf ]
         statusDetailList = statusDetailEntry.details
         if not statusDetailList:
            statusDetailList = [ StatusDetail( reason='n/a', detail='n/a' ) ]

         for statusDetail in statusDetailList:
            t.newRow( Intf.getShortname( intf ),
                      statusDetailEntry.description or '',
                      statusDetailEntry.status,
                      statusDetail.reason or 'n/a',
                      statusDetail.detail or 'n/a' )

      print( t.output() )
