# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import CliMatcher
import CliParser
from CliPlugin.VirtualIntfRule import _VirtualIntfMatcher, VirtualIntfMatcherBase
from TypeFuture import TacLazyType

# The regex can match several cases:
# 1. An empty string: ''
# 2. A path: '12' or '12/34' ( no limit on how many parts in the path )
# 3. A path followed by a slash: '12/' or '12/34/' ( also no limit )
# This regex is used to verify that the path string typed in Cli is valid
# It checks the digit is at the start of the string to prevent a leading slash from
# being at the start of the string
numberSyntaxRe = re.compile( r'^\d*(\/$|(\/\d+))*$' )
# This regex is used to extract the path from the intfs in intfStatus
numberSearchRe = re.compile( r'\d*(\/$|(\/\d+))*$' )

SubIntfRangeDetails = TacLazyType( "Interface::SubIntfIdConstants" )

class IntfStatusNumberMatcher:

   def __init__( self, basename, allIntfStatusFunc, defaultMatcher ):
      self.basename = basename
      self.allIntfStatusFunc = allIntfStatusFunc
      self.defaultMatcher = defaultMatcher

   def _entityMibRoot( self, mode ):
      return mode.entityManager.lookup( 'hardware/entmib' ).root

   def _intfStatus( self, mode ):
      return self.allIntfStatusFunc( mode )

   def _intfExists( self, mode, token ):
      try:
         return self.basename + token in self._intfStatus( mode )
      except IndexError: # Invalid IntfId
         return False

   def match( self, mode, context, token ):
      if self._entityMibRoot( mode ) is None:
         return self.defaultMatcher.match( mode, context, token )
      elif self._intfExists( mode, token ):
         return CliMatcher.MatchResult( token, token )
      return CliMatcher.noMatch

   def completions( self, mode, context, token ):
      if self._entityMibRoot( mode ) is None:
         return self.defaultMatcher.completions( mode, context, token )
      ct = []
      # Check if token's syntax is valid
      if numberSyntaxRe.match( token ):
         # If token ends with a slash, the last element of the tokens list
         # will be an empty string. parts is a tuple of numbers in string form
         parts = tuple( token.split( '/' ) )
         # prevParts represents all the previous parts in token. It is used to
         # determine which part values are valid for the next part given the values
         # of the previous parts.
         prevParts = parts[ : -1 ]
         # allParts is a list of tuples of numbers in string form. It represents
         # all of the parts of all the intfs in the intf status
         allParts = [ tuple( numberSearchRe.search( intf ).group( 0 ).split( '/' ) )
                      for intf in self._intfStatus( mode ) ]
         # relevantParts is a modified subset of allParts. We filter each tuple in
         # allParts on the condition that the tuple's values start with the same
         # elements as parts. Since every element in relevantParts would start with
         # the same elements under this filter, we remove the previous parts.
         relevantParts = [ p[ len( parts ) - 1 : ] for p in allParts
                           if tuple( p[ : len( prevParts ) ] ) == prevParts ]
         relevantParts = [ p for p in relevantParts if p ]
         # If relevantParts is not empty, then the current token's parts matches
         # at least one intf in the intf status
         if relevantParts:
            # We need to find the min and max value for the next part
            intParts = [ int( p[ 0 ] ) for p in relevantParts ]
            minPart, maxPart = min( intParts ), max( intParts )
            # pylint: disable-next=consider-using-f-string
            ct.append( CliParser.Completion( '<%d-%d>' % ( minPart, maxPart ), '',
                                             literal=False ) )
            # If token does not end with a slash and there is at least one intf in
            # the intf status that has more parts that matches prevParts and the
            # current part then a slash can be added
            if token and parts[ -1 ]:
               if any( p[ 0 ] == parts[ -1 ] and len( p ) > 1 for p in
                       relevantParts ):
                  ct.append( CliParser.Completion( '/', '', literal=False ) )
      return ct

class IntfStatusMatcher( VirtualIntfMatcherBase ):

   def __init__( self, basename, allIntfStatusFunc, defaultNumberMatcher,
                 helpdesc=None, alternates=None, subIntf=False, subIntfGuard=None,
                 **kwargs ):
      self.basename = basename
      intfNumberMatcher = IntfStatusNumberMatcher( basename, allIntfStatusFunc,
                                                   defaultNumberMatcher )
      intfNameMatcher = _VirtualIntfMatcher( basename, intfNumberMatcher,
                                             alternates=alternates,
                                             subIntf=subIntf,
                                             subIntfGuard=subIntfGuard )
      VirtualIntfMatcherBase.__init__( self, basename, intfNameMatcher,
                                       intfNumberMatcher, helpdesc=helpdesc,
                                       subIntf=subIntf,
                                       **kwargs )

class DefaultIntfNumberMatcher:

   def __init__( self, limits, minParts=1, maxParts=3, subIntf=False ):
      self.limits = limits
      self.minParts = minParts
      self.maxParts = maxParts
      self.subIntf = subIntf

   def match( self, mode, context, token ):
      portToken = token
      if self.subIntf:
         subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
         subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax
         if '.' not in token:
            return CliMatcher.noMatch
         tokens = token.split( '.' )
         if len( tokens ) != 2:
            return CliMatcher.noMatch
         ( portToken, subToken ) = tokens
         try:
            # This can fail due to the subToken containing a comma when trying to
            # select multiple subIntfs
            subIntf = int( subToken )
         except ValueError:
            return CliMatcher.noMatch
         if not subLowerBound <= subIntf <= subUpperBound:
            return CliMatcher.noMatch
      elif '.' in token:
         return CliMatcher.noMatch
      tokens = portToken.split( '/' )
      if not self.minParts <= len( tokens ) <= self.maxParts:
         return CliMatcher.noMatch
      parts = []
      for t in tokens:
         try:
            parts.append( int( t ) )
         except ValueError:
            return CliMatcher.noMatch
      for i, part in enumerate( parts ):
         lowerLimit, upperLimit = self.limits[ i ]
         if not lowerLimit <= part <= upperLimit:
            return CliMatcher.noMatch
      return CliMatcher.MatchResult( token, token )

   def completions( self, mode, context, token ):
      subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
      subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax
      ct = []
      ( portToken, subToken ) = token.split( '.' ) if '.' in token else \
         ( token, None )
      if '.' in token:
         if not self.subIntf:
            # If the token contains '.', let the subIntf supported matcher deal with
            # it.
            return []
         parts = portToken.split( '/' )
         if len( parts ) < self.minParts:
            # There is no need to check > self.maxParts as the normal matcher will
            # reject the '/' after the max part.
            return []
         if subToken and ( not subLowerBound <= int( subToken ) <= subUpperBound ):
            return []
         # pylint: disable-next=consider-using-f-string
         helpName = '<%d-%d>' % ( subLowerBound, subUpperBound )
         helpDesc = 'sub interface number'
         ct.append( CliParser.Completion( helpName, helpDesc, literal=False ) )
         return ct
      token = portToken
      # Check if token's syntax is valid
      if numberSyntaxRe.match( token ):
         # If token ends with a slash, the last element of the tokens list
         # will be an empty string. parts is a tuple of numbers in string form
         parts = token.split( '/' )
         if len( parts ) > self.maxParts:
            return []
         # prevParts represents all the previous parts in token. It is used to
         # determine which part values are valid for the next part given the values
         # of the previous parts.
         prevParts = parts[ : -1 ]
         for i, part in enumerate( prevParts ):
            minPart, maxPart = self.limits[ i ]
            if not minPart <= int( part ) <= maxPart:
               return []
         if parts:
            lastPartIndex = len( parts ) - 1
            lastPart = parts[ lastPartIndex ]
         else:
            lastPartIndex = 0
            lastPart = ''
         minPart, maxPart = self.limits[ lastPartIndex ]
         # pylint: disable-next=consider-using-f-string
         helpName = '<%d-%d>' % ( minPart, maxPart )
         helpDesc = 'interface number'
         ct.append( CliParser.Completion( helpName, helpDesc, literal=False ) )
         if ( lastPart and lastPartIndex < self.maxParts - 1 and
              ( minPart <= int( lastPart ) <= maxPart ) ):
            helpDesc = 'slash to specify interface number'
            ct.append( CliParser.Completion( '/', helpDesc, literal=False ) )
         if ( self.subIntf and lastPart and
              ( self.minParts <= ( lastPartIndex + 1 ) <= self.maxParts ) ):
            ct.append( CliParser.Completion( '.', 'subinterface', literal=False ) )
      return ct

class DefaultIntfMatcher( VirtualIntfMatcherBase ):
   def __init__( self, basename, defaultNumberMatcher,
                 helpdesc=None, alternates=None, subIntf=False, subIntfGuard=None,
                 **kwargs ):
      self.basename = basename
      intfNumberMatcher = defaultNumberMatcher
      intfNameMatcher = _VirtualIntfMatcher( basename, intfNumberMatcher,
                                             alternates=alternates,
                                             subIntf=subIntf,
                                             subIntfGuard=subIntfGuard )
      VirtualIntfMatcherBase.__init__( self, basename, intfNameMatcher,
                                       intfNumberMatcher, helpdesc=helpdesc,
                                       subIntf=subIntf, **kwargs )
