#!/usr/bin/env python3
# Copyright (c) 2009-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import sys
import threading
import CliPlugin.TechSupportCli
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import ArpModel
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin import IraRouteCommon
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
from CliPlugin.SubIntfCli import subMatcher
from CliPlugin.VirtualIntfRule import IntfMatcher
import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import CliParser
from CliPrint import CliPrint
import CliToken.Clear
import CliToken.Ip
import Ethernet
import HostnameCli
import LazyMount
import ShowCommand
import SharedMem
import SmashLazyMount
import Smash
import Tac

cprinter = CliPrint().lib

allIntfStatusDir = None
ipStatus = None
ipConfig = None
ip6Status = None
arpSmash = None
vrfIdMap = None
allVrfStatusGlobal = None
noArpTableForVrfMsg = "ARP table for VRF %s does not exist."
bridgingStatus = None
cliHelper = None
routingHardwareStatusCommon = None
arpHwStatus = None
arpCliStatus = None

em = None

routing = IraRouteCommon.routing( IraRouteCommon.Ip4() )
routing6 = IraRouteCommon.routing( IraRouteCommon.Ip6() )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

arpAfterShowKw = CliMatcher.KeywordMatcher( 'arp', helpdesc='ARP entries' )
arpAfterShowIpKw = CliMatcher.KeywordMatcher( 'arp',
      helpdesc='Address Resolution Protocol table' )
neighborAfterShowKw = CliMatcher.KeywordMatcher( 'neighbors',
                                                 helpdesc='IPv6 neighbor table' )

threadLock = threading.RLock()
def getCliHelper():
   global cliHelper
   # cliHelper added to EosImage/test/cliGlobalsExcusedList.dat
   with threadLock:
      if not cliHelper:
         assert IntfCli.intfIdDisplayContextHelper
         cliHelper = Tac.newInstance( "IpEth::ArpCliHelper", bridgingStatus,
                                      IntfCli.intfIdDisplayContextHelper )
   return cliHelper

#-------------------------------------------------------------------------------
# The "show ip arp [resolve] [IP] [host HOST] [mac-address MAC]
#                  [interface INTF] [summary [ total ]]" command.
#-------------------------------------------------------------------------------
def _doShowArp( fd, outputFormat, macAddr, ipAddr, hostName, ifName, summary,
                resolve, summaryType, vrfs, portIntfId, isStandby, domain,
                printVrfName ):

   helperFlags = Tac.Type( "IpEth::ArpHelperFlags" )()
   if macAddr:
      if macAddr != 'Incomplete':
         macAddr = Ethernet.convertMacAddrToDisplay( macAddr )
      helperFlags.macAddr = macAddr
      helperFlags.macAddrValid = True
   else:
      helperFlags.macAddrValid = False
   if ipAddr:
      helperFlags.ipAddr = ipAddr
      helperFlags.ipAddrValid = True
   else:
      helperFlags.ipAddrValid = False
   if hostName:
      helperFlags.hostName = hostName
   if ifName:
      helperFlags.intfId = Tac.Type( "Arnet::IntfId" )( ifName.name )
   if summary:
      helperFlags.summary = True
   if resolve or hostName:
      helperFlags.resolve = True
      helperFlags.domain = domain
   if summaryType:
      helperFlags.summaryType = summaryType
   for vrf in vrfs:
      helperFlags.vrfs[ vrf ] = True
   helperFlags.defaultVrf = DEFAULT_VRF
   if portIntfId:
      helperFlags.portIntfId = portIntfId
   helperFlags.standby = isStandby
   helperFlags.printVrfName = printVrfName

   # call to ipStatus to ensure that
   # they are instantiated before passing into ArpHelper
   ipStatus.urpfConfiguredOnAnyIntf # pylint: disable=W0104

   helper = Tac.newInstance( "IpEth::ArpHelper", arpSmash, vrfIdMap, arpCliStatus,
                             allIntfStatusDir.force(),
                             ipStatus, getCliHelper(), helperFlags )

   if vrfs == [ DEFAULT_VRF ]:
      euid = -1
   else:
      # need root privilege
      euid = 0
   with BasicCliUtil.RootPrivilege( euid=euid ):
      helper.doShowArp( fd, outputFormat.value )

def showArp( mode, options, vrfName=None, resolve=None,
             summary=False, summaryType=None ):
   assert vrfName != ''
   # After this call vrfName can not be None, it would have
   # asserted if it were.  Therefore there are only 3 cases
   # to be concerned with all, default, or other
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
   fd = sys.stdout.fileno()

   macAddr = None
   ipAddr = None
   hostName = None
   ifName = None
   portIntfId = None
   domain = None
   isStandby = mode.session_.entityManager_.redundancyStatus().mode == 'standby'

   for filterType, value in options:
      if filterType == 'mac':
         macAddr = value
      elif filterType == 'ip':
         ipAddr = value
      elif filterType == 'host':
         hostName = value
      elif filterType == 'intf':
         # The interface filter can be the ip or port interface name
         # If there is NOT an ip status, then its a port name
         intfId = Tac.Type( "Arnet::IntfId" )( str( value ) )
         ipIntfStatus = ipStatus.ipIntfStatus.get( intfId )
         if ipIntfStatus:
            ifName = value
         else:
            portIntfId = intfId

   if resolve or hostName:
      host = Tac.run( ['hostname'], stdout=Tac.CAPTURE ).strip()
      pos = host.find('.')
      if pos > 0:
         domain = host[pos:]
         domainLen = len( domain )
      else:
         domain = ''

   if hostName:
      if domain and hostName.endswith( domain ):
         hostName = hostName[:-domainLen]
      else:
         hostName = hostName[:15]

   if vrfName == 'all':
      vrfs = list( allVrfStatusGlobal.vrf ) + [ 'default' ]
      _doShowArp( fd, outputFormat, macAddr, ipAddr, hostName, ifName, summary,
                  resolve, summaryType, vrfs, portIntfId, isStandby, domain,
                  printVrfName=True )
      # return the Class itself, not an instance
      return ArpModel.VrfIpV4Neighbors
   elif vrfName != DEFAULT_VRF:
      if vrfName not in allVrfStatusGlobal.vrf:
         mode.addError( noArpTableForVrfMsg % vrfName )
         return None

   _doShowArp( fd, outputFormat, macAddr, ipAddr, hostName, ifName, summary,
               resolve, summaryType, [ vrfName ], portIntfId, isStandby, domain,
               printVrfName=False )
   return ArpModel.IpV4Neighbors

def ipV6Vrf( mode, token ):
   if not routingHardwareStatusCommon.vrfCapability.\
      ipv6EnabledDefault:
      return CliParser.guardNotThisPlatform
   return None

class ShowArpExpr( CliCommand.CliExpression ):
   # 'show arp' and 'show ip arp'
   expression = "arp | ( ip arpAfterIp )"
   data = {
      'arp' : arpAfterShowKw,
      'ip' : CliToken.Ip.ipMatcherForShow,
      'arpAfterIp': arpAfterShowIpKw,
   }

class FilterExpr( CliCommand.CliExpression ):
   expression = "{ IP | ( host HOST ) | ( mac-address MAC ) | " \
                "( interface INTF ) }"
   data = {
      'IP' : CliCommand.Node(
         IpAddrMatcher.IpAddrMatcher( helpdesc="IP address filter" ),
         maxMatches=1 ),
      'host' : CliCommand.singleKeyword( 'host', helpdesc='Hostname filter' ),
      'HOST' : HostnameCli.HostnameMatcher( helpname="WORD",
                                            helpdesc="Hostname" ),
      'mac-address' : CliCommand.singleKeyword( 'mac-address',
                                                helpdesc='Mac address filter' ),
      'MAC' : CliMatcher.PatternMatcher( Ethernet.macAddrPattern,
                                         helpname='H.H.H',
                                         helpdesc='Ethernet address' ),
      'interface' : CliCommand.singleKeyword( 'interface',
                                              helpdesc='Interface selector' ),
      'INTF' : IntfCli.Intf.matcher
   }
   @staticmethod
   def adapter( mode, args, argsList ):
      options = []
      for n in ( 'IP', 'HOST', 'MAC', 'INTF' ):
         if n in args:
            val = args[ n ]
            if n != 'IP':
               val = val[ 0 ]
            options.append( ( n.lower(), val ) )
      if options:
         args[ 'OPTIONS' ] = options

vrfExprFactory = VrfExprFactory(
      helpdesc='Show ARP entries in a VRF',
      inclAllVrf=True )

resolveKw = CliMatcher.KeywordMatcher( 'resolve', helpdesc='Resolve host names' )

summaryKw = CliMatcher.KeywordMatcher(
   'summary',
   helpdesc='Display a summary of the ARP entries' )

summaryTotalKw = CliMatcher.KeywordMatcher(
   'total',
   helpdesc='Display a count of the total ARP entries' )

class ShowArpBase( ShowCommand.ShowCliCommandClass ):
   data = {
      'arp' : ShowArpExpr,
      'VRF' : vrfExprFactory,
      'OPTIONS' : FilterExpr,
      }
   cliModel = ArpModel.IpV4Neighbors

   @staticmethod
   def handler( mode, args ):
      options = args.get( 'OPTIONS', [] )
      vrfName = args.get( 'VRF' )
      resolve = 'resolve' in args
      summary = 'summary' in args
      summaryType = args.get( 'total' )
      return showArp( mode, options, vrfName=vrfName, resolve=resolve,
                      summary=summary, summaryType=summaryType )

class ShowArp( ShowArpBase ):
   syntax = "show arp [ VRF ] [ resolve ] [ OPTIONS ]"
   data = ShowArpBase.data.copy()
   data[ 'resolve' ] = resolveKw

class ShowArpSummary( ShowArpBase ):
   syntax = "show arp [ VRF ] [ OPTIONS ] summary [ total ]"
   data = ShowArpBase.data.copy()
   data.update( { 'summary' : summaryKw,
                  'total' : summaryTotalKw } )

class ShowArpResolveSummary( ShowArpBase ):
   # The "show arp resolve summary" command is hidden and there is no longer
   # testing for this command as it is now deprecated. See BUG171017.
   syntax = "show [ ip ] arp [ VRF ] resolve [ OPTIONS ] summary total"
   data = ShowArpBase.data.copy()
   # keep summary hidden for resolve command
   data.update( { 'resolve' : resolveKw,
                  'summary' : CliCommand.Node( summaryKw, hidden=True ),
                  'total' : summaryTotalKw } )

subIntfMatcher = IntfMatcher()
subIntfMatcher |= subMatcher

BasicCli.addShowCommandClass( ShowArp )
BasicCli.addShowCommandClass( ShowArpSummary )
BasicCli.addShowCommandClass( ShowArpResolveSummary )

#----------------------------------------------------------------------------------
# show ipv6 neighbors [ interface-type interface-number | ipv6-address ]
# [ summary [ total ] ]
#----------------------------------------------------------------------------------
def _doShowIpv6Neighbor( fd, outputFormat, revision, ipv6IntfName, ipv6Addr,
                         summaryType, vrfs, summary, printVrfName ):
   helperFlags = Tac.Type( "IpEth::Ipv6NeighborHelperFlags" )()
   if ipv6IntfName:
      # The interface filter can be the ip or port interface name
      # If there is NOT an ip status, then its a port name
      intfId = Tac.Type( "Arnet::IntfId" )( ipv6IntfName.name )
      ip6IntfStatus = ip6Status.intf.get( intfId )
      if ip6IntfStatus:
         helperFlags.intfId = intfId
      else:
         helperFlags.portIntfId = intfId

   if ipv6Addr:
      helperFlags.ipv6Addr = ipv6Addr
   if summaryType:
      helperFlags.summaryType = summaryType
   for vrf in vrfs:
      helperFlags.vrfs[ vrf ] = True
   helperFlags.defaultVrf = DEFAULT_VRF
   helperFlags.summary = summary
   helperFlags.printVrfName = printVrfName

   # call to arpSmash/ip6Status to ensure that
   # it's instantiated before passing into Ipv6NeighborHelper
   ip6Status.name # pylint: disable=W0104

   helper = Tac.newInstance( "IpEth::Ipv6NeighborHelper", arpSmash, vrfIdMap,
                             allIntfStatusDir.force(), ip6Status, getCliHelper(),
                             helperFlags )

   if vrfs == [ DEFAULT_VRF ]:
      euid = -1
   else:
      # need root privilege
      euid = 0
   with BasicCliUtil.RootPrivilege( euid=euid ):
      helper.doShowIpv6Neighbor( fd, outputFormat.value, revision )

def cmdShowIpv6Neighbor( mode, vrfName=None, ipv6IntfName=None, ipv6Addr=None,
                         summary=False, summaryType=None ):
   assert vrfName != ''
   # After this call vrfName can not be None, it would have
   # asserted if it were.  Therefore there are only 3 cases
   # to be concerned with all, default, or other
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
   fd = sys.stdout.fileno()
   revision = mode.session_.requestedModelRevision()

   if vrfName == 'all':
      vrfs = list( allVrfStatusGlobal.vrf ) + [ 'default' ]
      _doShowIpv6Neighbor( fd, outputFormat, revision, ipv6IntfName, ipv6Addr,
                           summaryType, vrfs, summary, printVrfName=True )
      return ArpModel.VrfIpV6Neighbors
   elif vrfName != DEFAULT_VRF:
      if vrfName not in allVrfStatusGlobal.vrf:
         mode.addError( noArpTableForVrfMsg % vrfName )
         return None

   _doShowIpv6Neighbor( fd, outputFormat, revision, ipv6IntfName, ipv6Addr,
                        summaryType, [ vrfName ], summary, printVrfName=False )
   return ArpModel.IpV6Neighbors

class ShowNeighbor( ShowCommand.ShowCliCommandClass ):
   syntax = "show ipv6 neighbor [ VRF ] [ INTF ] [ IP6_ADDR ] [ summary [ total ] ]"
   data = {
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'neighbor' : neighborAfterShowKw,
      'VRF' : vrfExprFactory,
      'INTF' : IntfCli.Intf.matcher,
      'IP6_ADDR' : Ip6AddrMatcher.Ip6AddrMatcher( "IPv6 address of neighbor" ),
      'summary' : summaryKw,
      'total' : summaryTotalKw
      }
   cliModel = ArpModel.IpV6Neighbors

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      intf = args.get( 'INTF' )
      ipv6Addr = args.get( 'IP6_ADDR' )
      summary = 'summary' in args
      summaryType = args.get( 'total' )
      return cmdShowIpv6Neighbor( mode, vrfName=vrfName, ipv6IntfName=intf,
                                  ipv6Addr=ipv6Addr, summary=summary,
                                  summaryType=summaryType )

BasicCli.addShowCommandClass( ShowNeighbor )

# Add commands to 'show tech-support summary' output
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-05-13 08:12:57',
   summaryCmds=[ 'show arp vrf all summary',
                 'show ipv6 neighbor vrf all summary' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )
   routing6.plugin( entityManager )

   global em
   global allIntfStatusDir
   global ipStatus
   global ip6Status
   global ipConfig
   global routingHardwareStatusCommon
   global arpSmash
   global arpHwStatus
   global arpCliStatus
   global vrfIdMap
   global allVrfStatusGlobal

   em = entityManager
   arpSmash = SmashLazyMount.mount( entityManager, "arp/status",
                                    "Arp::Table::Status",
                                    SmashLazyMount.mountInfo( 'reader' ) )
   vrfIdMap = SmashLazyMount.mount( entityManager, "vrf/vrfIdMapStatus",
                                    "Vrf::VrfIdMap::Status",
                                    SmashLazyMount.mountInfo( 'reader' ),
                                    autoUnmount=True )

   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager,
      "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )

   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   arpHwStatus = LazyMount.mount( entityManager, "arp/hardware/status",
                                  "Arp::Hardware::Status", "r" )
   arpCliStatus = LazyMount.mount( entityManager, "arp/clistatus",
                                   "Arp::CliStatus", "r" )

   global bridgingStatus
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   bridgingStatus = shmemEm.doMount( "bridging/status", "Smash::Bridging::Status",
                                     Smash.mountInfo( 'reader' ) )

   allVrfStatusGlobal = LazyMount.mount( entityManager, 'ip/vrf/status/global',
                                         'Ip::AllVrfStatusGlobal', 'r' )
