# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliParser
from CliPlugin import IntfCli
from CliPlugin import VlanCli
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import ip6AddrMatcher
from CliPlugin.IpAddrMatcher import ipAddrMatcher
from CliPlugin.MacAddr import macAddrMatcher
import Toggles.IpLockingToggleLib as ILTL
from BasicCliModes import EnableMode
from CliMode.IpLocking import (
      AddressLockingMode,
      IpLockingEthIntfAddressLockingMode,
      IpLockingLagIntfAddressLockingMode,
      IpLockingVlanAddressLockingMode
)
from CliToken.IpLocking import (
      ipLockingAddressMatcherForConfig,
      ipLockingLockingMatcherForConfig,
      ipLockingAddressMatcherForConfigIf,
      ipLockingLockingMatcherForConfigIf,
      ipLockingDisabledMatcherForConfigIf,
      ipLockingAddressMatcherForClear,
      ipLockingLockingMatcherForClear,
      ipLockingCountersMatcherForClear,
      ipLockingDhcpMatcherForConfig,
      ipLockingServerMatcherForConfig,
      ipLockingIpv4MatcherForConfig,
      ipLockingIpv6MatcherForConfig,
      ipLockingProtocolMatcherForConfig,
      ipLockingAristaQueryMatcherForConfig,
      ipLockingleaseMatcherForConfig,
      ipLockingQueryMatcherForConfig,
      ipLockingRetryMatcherForConfig,
      ipLockingIntervalMatcherForConfig,
      ipLockingRetryIntervalValueMatcherForConfig,
      ipLockingTimeoutMatcherForConfig,
      ipLockingTimeoutValueMatcherForConfig,
      matcherLockedAddr,
      matcherLockedAddrDisable,
      matcherLockedAddrEnforcement
      )
from CliToken.Clear import clearKwNode
import ConfigMount
from IpLockingLib import (
      intfIsEthAndSupportsIpLocking,
      intfIsLag
      )
import LazyMount

# globals
cliConfig = None
hwConfig = None

def ipLockingConfigured():
   return cliConfig.configured

def enforcementDisabledGuard( mode, token ):
   if hwConfig.v4EnforcementDisabledSupported:
      return None
   return CliParser.guardNotThisPlatform

def addressLockingHwV6Guard( mode, token ):
   if token == 'ipv6':
      return hwV6SupportedGuard( mode, token )
   return None

def hwV6SupportedGuard( mode, token ):
   if hwConfig.v6featureSupported:
      return None
   return CliParser.guardNotThisPlatform

def arIpV6SupportedGuard( mode, token ):
   if hwV6SupportedGuard( mode, token ):
      return hwV6SupportedGuard( mode, token )
   if ILTL.toggleIpLockingArIpEnabled():
      return None
   return CliParser.guardNotThisEosVersion

class IpLockingEthIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return intfIsEthAndSupportsIpLocking( mode.intf.name )

class IpLockingLagIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return intfIsLag( mode.intf.name )

#-------------------------------------------------------------------------------
# The "[no|default] address locking" command, in config mode.
#-------------------------------------------------------------------------------
class AddressLockingModeEnter( CliCommand.CliCommandClass ):
   syntax = 'address locking'
   noOrDefaultSyntax = syntax

   data = {
      'address': ipLockingAddressMatcherForConfig,
      'locking': ipLockingLockingMatcherForConfig,
   }

   handler = "IpLockingCliHandler.doAddressLockingModeEnter"

   # Reset all non-interface attributes to default
   noOrDefaultHandler = "IpLockingCliHandler.noAddressLockingModeEnter"

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax

   data = {
      'disabled': 'Disable IP locking on configured ports',
   }

   handler = "IpLockingCliHandler.setIpLockingDisabled"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingDisabled"

#-------------------------------------------------------------------------------
# The "[no|default] local-interface <intf>" command, in "address locking"
# config mode.
#-------------------------------------------------------------------------------
class IpLockingLocalInterface( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface [ INTF ]'

   data = {
      'local-interface': 'Configuring local interface',
      'INTF': Intf.matcherWithIpSupport,
   }

   handler = "IpLockingCliHandler.setIpLockingLocalInterface"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingLocalInterface"

#-------------------------------------------------------------------------------
# The "[no|default] dhcp server ipv4 ADDR" command, in "address locking"
# config mode.
#-------------------------------------------------------------------------------
class IpLockingDhcpServer( CliCommand.CliCommandClass ):
   syntax = 'dhcp server ipv4 V4ADDR'
   noOrDefaultSyntax = 'dhcp server ipv4 [ V4ADDR ] ...'

   data = {
      'dhcp': ipLockingDhcpMatcherForConfig,
      'server': ipLockingServerMatcherForConfig,
      'ipv4': ipLockingIpv4MatcherForConfig,
      'V4ADDR': ipAddrMatcher,
   }

   handler = "IpLockingCliHandler.setIpLockingDhcpServer"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingDhcpServer"

#-------------------------------------------------------------------------------
# The "[no|default] dhcp server ipv4 ADDR [protocol arista-query]" command, in
# "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingServer( CliCommand.CliCommandClass ):
   syntax = 'dhcp server ipv4 V4ADDR [ protocol arista-query ]'
   noOrDefaultSyntax = 'dhcp server ipv4 [ V4ADDR ] [ protocol arista-query ]'

   data = {
      'dhcp': ipLockingDhcpMatcherForConfig,
      'server': ipLockingServerMatcherForConfig,
      'ipv4': ipLockingIpv4MatcherForConfig,
      'V4ADDR': ipAddrMatcher,
      'protocol': ipLockingProtocolMatcherForConfig,
      'arista-query': ipLockingAristaQueryMatcherForConfig,
   }

   handler = "IpLockingCliHandler.setIpLockingServer"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingServer"

class IpLockingV6Server( CliCommand.CliCommandClass ):
   syntax = 'dhcp server ipv6 V6ADDR protocol arista-query'
   noOrDefaultSyntax = 'dhcp server ipv6 [ V6ADDR ] [ protocol arista-query ]'

   data = {
      'dhcp': ipLockingDhcpMatcherForConfig,
      'server': ipLockingServerMatcherForConfig,
      'ipv6': ipLockingIpv6MatcherForConfig,
      'V6ADDR': ip6AddrMatcher,
      'protocol': ipLockingProtocolMatcherForConfig,
      'arista-query': ipLockingAristaQueryMatcherForConfig,
   }

   handler = "IpLockingCliHandler.setIpLockingV6Server"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingV6Server"

#-------------------------------------------------------------------------------
# The "[no|default] lease <ipv4-or-ipv6-addr>  mac <m>" command,
# in "address locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingStaticLease( CliCommand.CliCommandClass ):
   syntax = 'lease ( V4ADDR | V6ADDR ) mac MAC_ADDR'
   noOrDefaultSyntax = 'lease ( ( V4ADDR ) | ( V6ADDR ) )...'

   data = {
      'lease': ipLockingleaseMatcherForConfig,
      'V4ADDR': ipAddrMatcher,
      'V6ADDR': CliCommand.Node( ip6AddrMatcher, guard=arIpV6SupportedGuard ),
      'mac': 'Configure mac for static lease',
      'MAC_ADDR': macAddrMatcher,
   }

   handler = "IpLockingCliHandler.setIpLockingStaticLease"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingStaticLease"

def nodeIpv4( guard=None ):
   return CliCommand.guardedKeyword( 'ipv4', helpdesc='Configuration for IPv4',
                                     guard=guard )

def nodeIpv6( guard=None ):
   return CliCommand.guardedKeyword( 'ipv6', helpdesc='Configuration for IPv6',
                                     guard=guard )

class IpLockingIntfVlanLockedAddressEnforcementDisabled(
      CliCommand.CliCommandClass ):
   syntax = 'locked-address ipv4 enforcement disabled'
   noOrDefaultSyntax = syntax

   data = {
         'locked-address': matcherLockedAddr,
         'ipv4': nodeIpv4( guard=enforcementDisabledGuard ),
         'enforcement': matcherLockedAddrEnforcement,
         'disabled': matcherLockedAddrDisable
         }

   handler = ( "IpLockingCliHandler."
               "setIpLockingIntfVlanLockedAddressEnforcementDisabled" )

   noOrDefaultHandler = ( "IpLockingCliHandler."
                          "noIpLockingIntfVlanLockedAddressEnforcementDisabled" )

class IpLockingGlobalLockedAddressEnforcementDisabled(
      CliCommand.CliCommandClass ):
   syntax = 'locked-address ( ipv4 | ipv6 ) enforcement disabled'
   noOrDefaultSyntax = syntax

   data = {
         'locked-address': matcherLockedAddr,
         'ipv4': nodeIpv4( guard=enforcementDisabledGuard ),
         'ipv6': nodeIpv6( guard=hwV6SupportedGuard ),
         'enforcement': matcherLockedAddrEnforcement,
         'disabled': matcherLockedAddrDisable
         }

   handler = "IpLockingCliHandler.setIpLockingGlobalLockedAddressEnforcementDisabled"

   noOrDefaultHandler = ( "IpLockingCliHandler."
                          "noIpLockingGlobalLockedAddressEnforcementDisabled" )

#-------------------------------------------------------------------------------
# The "[no|default] locked-address expiration mac disabled" command in "address
# locking" config mode.
#-------------------------------------------------------------------------------
class IpLockingLockedAddressExpirationMacDisabled( CliCommand.CliCommandClass ):
   syntax = 'locked-address expiration mac disabled'
   noOrDefaultSyntax = syntax

   data = {
      'locked-address': 'Configuration options for locked addresses',
      'expiration': 'Configure expiration mode for locked addresses',
      'mac': 'Configure deauthorizing locked addresses upon MAC aging out',
      'disabled': 'Disable deauthorizing locked addresses upon MAC aging out',
   }

   handler = "IpLockingCliHandler.setIpLockingLockedAddressExpirationMacDisabled"

   noOrDefaultHandler = ( "IpLockingCliHandler."
                          "noIpLockingLockedAddressExpirationMacDisabled" )

afData = {
   'ipv4': 'Enable address locking for IPv4',
   'ipv6': 'Enable address locking for IPv6'
}

def intfConfigured( intfConfig ):
   return ( intfConfig and
            ( intfConfig.ipv4 or intfConfig.ipv6 or
              intfConfig.ipv4Disabled or intfConfig.ipv6Disabled or
              intfConfig.ipv4LockedAddressEnforcementDisabled or
              intfConfig.ipv4DeniedAddr ) )

def vlanConfigured( vlanConfig ):
   return ( vlanConfig and
            ( vlanConfig.ipv4 or vlanConfig.ipv6 or
              vlanConfig.ipv4LockedAddressEnforcementDisabled ) )

# ---------------------------------------------------------------------------------
# The "[no|default] address locking" command, in "config-if-Po*" mode to enter the
# LAG interface address locking sub mode
# ---------------------------------------------------------------------------------
class EnterLagInterfaceAddressLockingMode( CliCommand.CliCommandClass ):
   syntax = 'address locking'
   noOrDefaultSyntax = syntax

   data = {
      'address': 'Interface address locking config commands',
      'locking': 'Address locking feature'
   }

   handler = "IpLockingCliHandler.doEnterLagInterfaceAddressLockingMode"

   noOrDefaultHandler = "IpLockingCliHandler.noEnterLagInterfaceAddressLockingMode"

# ---------------------------------------------------------------------------
# The "[no|default] address-family ipv4 disabled" command, in
# "config-if-Po*-address-locking" mode. This command is only available on
# intfIsEthAndSupportsIpLocking interface
# -----------------------------------------------------
class IpLockingLagInterfaceAddressFamilyDisable( CliCommand.CliCommandClass ):
   syntax = 'address-family ipv4 disabled'
   noOrDefaultSyntax = 'address-family [ ipv4 ] disabled'

   data = {
      'address-family': "Address family configuration options",
      'ipv4': nodeIpv4(),
      'disabled': ipLockingDisabledMatcherForConfigIf
   }

   handler = "IpLockingCliHandler.setIpLockingLagInterfaceAddressFamilyDisabled"

   noOrDefaultHandler = \
         "IpLockingCliHandler.noIpLockingLagInterfaceAddressFamilyDisabled"

# ---------------------------------------------------------------------------------
# The "[no|default] address-family ( ipv4 | ipv6 ) [ disabled ]" command, in
# "config-if-Et*-address-locking" mode. This command is only available on
# intfIsEthAndSupportsIpLocking interface
# -----------------------------------------------------
class IpLockingInterfaceAddressFamily( CliCommand.CliCommandClass ):
   syntax = 'address-family ( ipv4 | ipv6 ) [ disabled ]'
   noOrDefaultSyntax = 'address-family [ ( ipv4 | ipv6 ) ] [ disabled ]'

   data = {
      'address-family': "Address family configuration options",
      'ipv4': nodeIpv4(),
      'ipv6': nodeIpv6( guard=addressLockingHwV6Guard ),
      'disabled': ipLockingDisabledMatcherForConfigIf
   }

   handler = "IpLockingCliHandler.setIpLockingInterfaceAddressFamily"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingInterfaceAddressFamily"

# ---------------------------------------------------------------------------------
# The "[no|default] address locking ( ipv4 | ipv6 )" command, in "config-if" mode.
# This command is only available on intfIsEthAndSupportsIpLocking interface
# ---------------------------------------------------------------------------------
class IpLockingInterfaceEnable( CliCommand.CliCommandClass ):
   syntax = 'address locking [ ADDRESS_FAMILY ]'
   noOrDefaultSyntax = 'address locking [ ADDRESS_FAMILY ]'

   data = {
      'address': ipLockingAddressMatcherForConfigIf,
      'locking': ipLockingLockingMatcherForConfigIf,
      'ADDRESS_FAMILY': CliCommand.SetEnumMatcher( afData,
                                                   guard=addressLockingHwV6Guard )
   }

   handler = "IpLockingCliHandler.setIpLockingInterfaceEnable"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingInterfaceEnable"

# ---------------------------------------------------------------------------------
# The "[no|default] deny { IP_ADDR }" command, in
# "config-if-address-locking" mode. This command is only available on
# intfIsEthAndSupportsIpLocking interface
# ---------------------------------------------------------------------------------
class IpLockingIntfDenyAddress( CliCommand.CliCommandClass ):
   syntax = 'deny IP_ADDR'
   noOrDefaultSyntax = syntax

   data = {
      'deny': 'Deny specific IP address',
      'IP_ADDR': ipAddrMatcher
   }

   handler = "IpLockingCliHandler.setIpLockingIntfDenyAddress"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingIntfDenyAddress"

# ---------------------------------------------------------------------------------
# The "[no|default] address locking" command, in "config-vl" mode to enter the
# VLAN address locking sub mode
# ---------------------------------------------------------------------------------
class EnterVlanAddressLockingMode( CliCommand.CliCommandClass ):
   syntax = 'address locking'
   noOrDefaultSyntax = syntax

   data = {
      'address': 'VLAN address locking config commands',
      'locking': 'Address locking feature'
   }

   handler = "IpLockingCliHandler.doEnterVlanAddressLockingMode"

   noOrDefaultHandler = "IpLockingCliHandler.noEnterVlanAddressLockingMode"

# ---------------------------------------------------------------------------------
# The "[no|default] address-family ( ipv4 | ipv6 )" command, in
# address locking sub mode within the "config-vl" mode.
# ---------------------------------------------------------------------------------
class IpLockingVlanAddressFamily( CliCommand.CliCommandClass ):
   syntax = 'address-family ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'address-family [ ( ipv4 | ipv6 ) ]'

   data = {
      'address-family': "Address family configuration options",
      'ipv4': nodeIpv4(),
      'ipv6': nodeIpv6( guard=addressLockingHwV6Guard )
   }

   handler = "IpLockingCliHandler.setIpLockingVlanAddressFamily"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingVlanAddressFamily"

#-------------------------------------------------------------------------------
# The "clear address locking lease ( ipv4 V4ADDR | ipv6 V6ADDR | interface INTF | \
#      all )" command, in enable mode.
#-------------------------------------------------------------------------------
class IpLockingClearLease( CliCommand.CliCommandClass ):
   syntax = 'clear address locking lease ( (ipv4 V4ADDR) | (ipv6 V6ADDR) | \
         (interface INTF) | all )'
   data = {
      'clear': clearKwNode,
      'address': ipLockingAddressMatcherForClear,
      'locking': ipLockingLockingMatcherForClear,
      'lease': 'Lease information',
      'ipv4': 'IPv4 address of the lease to be cleared',
      'V4ADDR': ipAddrMatcher,
      'ipv6': CliCommand.guardedKeyword( 'ipv6',
                                         'IPv6 address of the lease to be cleared',
                                         hwV6SupportedGuard ),
      'V6ADDR': ip6AddrMatcher,
      'interface': 'The interface of the leases to be cleared',
      'INTF': EthPhyIntf.ethMatcher,
      'all': 'The entire lease table',
   }

   handler = "IpLockingCliHandler.setIpLockingClearLease"

# --------------------------------------------------------------------------------
# The "clear address locking counters" command, in enable mode.
#---------------------------------------------------------------------------------
class IpLockingClearCounters( CliCommand.CliCommandClass ):
   syntax = 'clear address locking counters'
   data = {
      'clear': clearKwNode,
      'address': ipLockingAddressMatcherForClear,
      'locking': ipLockingLockingMatcherForClear,
      'counters': ipLockingCountersMatcherForClear,
   }

   handler = "IpLockingCliHandler.setIpLockingClearCounters"

# -------------------------------------------------------------------------
# The "[no|default] lease query retry interval N timeout M" command in
# "address locking" config mode.
# -------------------------------------------------------------------------
class IpLockingLeaseQueryTimeout( CliCommand.CliCommandClass ):
   syntax = 'lease query retry interval RETRY_INTERVAL timeout TIMEOUT'
   noOrDefaultSyntax = 'lease query retry ...'

   data = {
      'lease': ipLockingleaseMatcherForConfig,
      'query': ipLockingQueryMatcherForConfig,
      'retry': ipLockingRetryMatcherForConfig,
      'interval': ipLockingIntervalMatcherForConfig,
      'RETRY_INTERVAL': ipLockingRetryIntervalValueMatcherForConfig,
      'timeout': ipLockingTimeoutMatcherForConfig,
      'TIMEOUT': ipLockingTimeoutValueMatcherForConfig,
   }

   handler = "IpLockingCliHandler.setIpLockingLeaseQueryTimeout"

   noOrDefaultHandler = "IpLockingCliHandler.noIpLockingLeaseQueryTimeout"

BasicCli.GlobalConfigMode.addCommandClass( AddressLockingModeEnter )
AddressLockingMode.addCommandClass( IpLockingDisabled )
AddressLockingMode.addCommandClass( IpLockingLocalInterface )
if ILTL.toggleIpLockingArIpEnabled():
   AddressLockingMode.addCommandClass( IpLockingV6Server )
   AddressLockingMode.addCommandClass( IpLockingServer )
else:
   AddressLockingMode.addCommandClass( IpLockingDhcpServer )
AddressLockingMode.addCommandClass( IpLockingStaticLease )
AddressLockingMode.addCommandClass(
      IpLockingLockedAddressExpirationMacDisabled )
AddressLockingMode.addCommandClass(
      IpLockingGlobalLockedAddressEnforcementDisabled )
AddressLockingMode.addCommandClass( IpLockingLeaseQueryTimeout )
EnableMode.addCommandClass( IpLockingClearLease )
EnableMode.addCommandClass( IpLockingClearCounters )
VlanCli.VlanConfigMode.addCommandClass( EnterVlanAddressLockingMode )
IpLockingVlanAddressLockingMode.addCommandClass( IpLockingVlanAddressFamily )
IpLockingVlanAddressLockingMode.addCommandClass(
   IpLockingIntfVlanLockedAddressEnforcementDisabled )
IpLockingEthIntfAddressLockingMode.addCommandClass(
   IpLockingIntfVlanLockedAddressEnforcementDisabled )
IpLockingEthIntfConfigModelet.addCommandClass( IpLockingInterfaceEnable )
IpLockingEthIntfAddressLockingMode.addCommandClass(
   IpLockingIntfDenyAddress )
IpLockingEthIntfAddressLockingMode.addCommandClass(
   IpLockingInterfaceAddressFamily )
IpLockingLagIntfConfigModelet.addCommandClass(
   EnterLagInterfaceAddressLockingMode )
IpLockingLagIntfAddressLockingMode.addCommandClass(
   IpLockingLagInterfaceAddressFamilyDisable )

#-------------------------------------------------------------------------------
# Associate IpLockingEthIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( IpLockingEthIntfConfigModelet )

#-------------------------------------------------------------------------------
# Associate IpLockingLagIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( IpLockingLagIntfConfigModelet )

#-------------------------------------------------------------------------------
# Have the Cli agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig
   global hwConfig

   cliConfig = ConfigMount.mount( entityManager, "iplocking/cliConfig",
                                  "IpLocking::CliConfig", "w" )
   hwConfig = LazyMount.mount( entityManager, "iplocking/hardware/config",
                               "IpLocking::Hardware::Config", "r" )
