#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=use-dict-literal

import Tac, Tracing, QuickTrace
import SharedMem, Smash, SmashLazyMount, LazyMount, Cell
import SharkLazyMount
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from GenericReactor import GenericReactor
from Toggles.TacSharkToggleLib import toggleSharkPassiveReaderEnabled
from Toggles.IpRibToggleLib import toggleNoRouteConfigTrieUsageEnabled

sharkMode = 'reader' if toggleSharkPassiveReaderEnabled() else 'shadow'

ipRibProtocols = set( Tac.Type( "Routing::Rib::RoutingProtocol" ).attributes )
ipRibProtocols.remove( 'staticRouteCacheConfig' )
ipRibProtocols.remove( 'routeInputRouteCacheConfig' )
ipRibProtocols.remove( 'routingProtocols' )

ipRibProtocolString = Tac.newInstance( "Routing::Rib::RoutingProtocolString" )

ipRibTransport = Tac.Value( "Routing::Rib::Transport" )
tacTypeViaSetStatus = Tac.Type( 'Routing::Rib::ViaSetStatus' )

t0 = Tracing.trace0

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt8 = QuickTrace.trace8

class NhDependencyMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.nhDepGraph = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getNhDependencyGraph( self ):
      if not self.nhDepGraph:
         readerInfo = SmashLazyMount.mountInfo( 'reader' )
         mountPath = "routing/rib/nextHopDependencyGraph"
         self.nhDepGraph = self.shmemEm.doMount( mountPath,
                           "Routing::Rib::Ip::GlobalNextHopDependencyGraph",
                           readerInfo )
      return self.nhDepGraph

def isRouteConfigFlattened( proto, af, rib="rib" ):
   return Tac.Type( "Routing::RouteConfigLib" )().isAllVrfRouteConfig( proto,
                                                     af, rib )

def getCommonIgpResultStatusReaderHelper( shmemEm ):
   staticConfigIrs = Tac.Type( "Routing::IgpRedistribute::IgpResultStatus" )\
                     .createIgpResultStatusReader( 'staticConfig', shmemEm,
                                                   'reader' )
   igpResultStatusReaderHelper = Tac.newInstance(
      "Routing::IgpRedistribute::IgpResultStatusReaderHelperBase", staticConfigIrs )
   return igpResultStatusReaderHelper

class IpRibVrfMounter:

   def __init__( self ):

      self.entityManager = None
      self.shmemEm = None
      self.allVrfConfig = None
      self.vrfNameStatus = None
      self.ribMounters = []
      self.vrfReactor = None
      self.vrfIdStatus = None

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getVrfNames( self, mode ):

      if self.entityManager is None:
         return []

      return list( VrfCli.getVrfNames() )

   def getAllVrfNames( self, mode ):

      vrflist = self.getVrfNames( mode )
      vrflist.append( DEFAULT_VRF )
      return vrflist

   def iterVrfNames( self, mode, vrfArg ):
      '''Returns a generator of vrfNames
      If vrfArg is ALL_VRF_NAME, then returns
         [ DEFAULT_VRF ] + sorted( self.getVrfNames( mode ) )
      Else returns
         [ vrfArg ]
      '''
      if vrfArg == ALL_VRF_NAME:
         yield DEFAULT_VRF
         yield from sorted( self.getVrfNames( mode ) )
      else:
         yield vrfArg

   def getVrfId( self, vrfName ):

      if self.vrfNameStatus is None:
         self.vrfNameStatus = LazyMount.mount( self.entityManager,
                                               Cell.path( 'vrf/vrfNameStatus' ),
                                               'Vrf::VrfIdMap::NameToIdMapWrapper',
                                               'r' )
      if self.vrfNameStatus.nameToIdMap is None:
         return None

      if self.vrfReactor is None:
         self.vrfReactor = GenericReactor( self.vrfNameStatus.nameToIdMap,
                                           [ 'vrfNameToId' ], self.handleVrf )
      return self.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )

   def getVrfIdStatus( self ):

      if self.vrfIdStatus is None:
         self.vrfIdStatus = SmashLazyMount.mount(
            self.entityManager,
            "vrf/vrfIdMapStatus",
            "Vrf::VrfIdMap::Status",
            SmashLazyMount.mountInfo( 'reader' ),
            autoUnmount=True )
      return self.vrfIdStatus

   def handleVrf( self, reactor, vrfName ):
      for ribMounter in self.ribMounters:
         ribMounter.handleVrf( vrfName )

class IpRibOrrPositionMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.vrfReactor = None
      self.positionReactor = None
      self.positionRibMounters = []
      self.orrPositionRoot = None
      self.defaultVrfId = 0

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getPosition( self, position ):
      if self.orrPositionRoot is None:
         self.orrPositionRoot = LazyMount.mount(
               self.entityManager,
               Cell.path( 'routing/bgp/orrPositionRoot' ),
               'Routing::BgpOrrPositionsRoot', 'ri' )

      orrPositionsByVrf = \
            self.orrPositionRoot.orrPositionsByVrf.get(
                  self.defaultVrfId )
      if orrPositionsByVrf is None:
         return None

      if self.vrfReactor is None:
         self.vrfReactor = GenericReactor( self.orrPositionRoot,
                                           [ 'orrPositionsByVrf' ],
                                           self.handleVrf )

      self.positionReactor = GenericReactor( orrPositionsByVrf,
                                             [ 'orrPositions' ],
                                             self.handlePosition )

      orrPosition = orrPositionsByVrf.orrPositions.get( position )
      if orrPosition:
         return orrPosition.name
      else:
         return None

   def handlePosition( self, reactor, position ):
      for ribMounter in self.positionRibMounters:
         ribMounter.handlePosition( position )

   def handleVrf( self, reactor, vrfId ):
      for ribMounter in self.positionRibMounters:
         ribMounter.handleVrf( vrfId )

class IpRibOrrPositionCliMounter:
   '''This class mounts IpRib Orr Routeconfig/ViaConfig entries in Smash.
   It mounts entities the first time they're needed and caches them'''

   def __init__( self, orrPositionMounter, rib="rib" ):

      assert rib == "rib"

      self.entityManager = None
      self.shmemEm = None
      self.readerInfo = SmashLazyMount.mountInfo( 'reader' )
      self.keyShadowInfo = SmashLazyMount.mountInfo( 'keyshadow' )

      self.orrPositionMounter = orrPositionMounter
      # So the OrrPosition mounter can notify us about Position/Vrf
      # additions/deletions
      self.orrPositionMounter.positionRibMounters.append( self )
      self.rib = rib
      # Only Default Vrf is supported for BGP ORR positions
      self.defaultVrfId = 0

      self.routeConfigs = {}
      if not toggleNoRouteConfigTrieUsageEnabled():
         self.routeConfigTries = {}
         self.routeConfigTrieSms = {}

      self.viaConfigs = {}
      self.viaConfigByProtoByVrf = {}
      self.viaConfigSplitterSmColl = {}
      self.vms = {}

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def handlePosition( self, position ):
      qt8( "Handling Position ", qv( position ) )
      orrPositionRoot = self.orrPositionMounter.orrPositionRoot
      orrPositionsByVrf = \
                   orrPositionRoot.orrPositionsByVrf.get(
                         self.defaultVrfId )

      if position in orrPositionsByVrf.orrPositions:
         return

      qt8( "Clearing Position ", qv( position ) )

      self.routeConfigs.pop( position, None )
      if not toggleNoRouteConfigTrieUsageEnabled():
         self.routeConfigTries.pop( position, None )
         self.routeConfigTrieSms.pop( position, None )

      self.vms.pop( position, None )
      self.viaConfigs.pop( position, None )
      self.viaConfigSplitterSmColl.pop( position, None )
      self.viaConfigByProtoByVrf.pop( position, None )

   def handleVrf( self, vrfId ):
      # Only Default-VRF is supported for BGP ORR positions
      assert vrfId == self.defaultVrfId

      # If this is a VRF addtion to BgpOrrPositionsByVrf, just return.
      # Mounting and caching of data will be done on demand
      orrPositionRoot = self.orrPositionMounter.orrPositionRoot
      orrPositionsByVrf = \
            orrPositionRoot.orrPositionsByVrf.get( vrfId )
      if orrPositionsByVrf:
         return

      # If there is a VRF deletion from BgpOrrPositionsByVrf, clear entries for
      # all the positions.
      qt8( "Handling VRF ", qv( vrfId ) )
      self.routeConfigs = {}
      if not toggleNoRouteConfigTrieUsageEnabled():
         self.routeConfigTries = {}
         self.routeConfigTrieSms = {}

      self.vms = {}
      self.viaConfigs = {}
      self.viaConfigSplitterSmColl = {}
      self.viaConfigByProtoByVrf = {}

   def getRouteConfig( self, af, vrfName, position ):

      # Only Default-VRF is supported for BGP ORR positions
      assert vrfName == DEFAULT_VRF

      vrfId = self.defaultVrfId

      rc = self.routeConfigs.get( position, dict() )
      if not rc:
         self.routeConfigs[ position ] = rc

      if not toggleNoRouteConfigTrieUsageEnabled():
         rcTrie = self.routeConfigTries.get( position, dict() )
         if not rcTrie:
            self.routeConfigTries[ position ] = rcTrie

         rcTrieSm = self.routeConfigTrieSms.get( position, dict() )
         if not rcTrieSm:
            self.routeConfigTrieSms[ position ] = rcTrieSm

      if af not in rc or ( not toggleNoRouteConfigTrieUsageEnabled()
                          and ( af not in rcTrie or af not in rcTrieSm ) ):
         # For each AF, we store a dict by VRF Name
         rc[ af ] = dict()
         if not toggleNoRouteConfigTrieUsageEnabled():
            rcTrie[ af ] = dict()
            rcTrieSm[ af ] = dict()

         vrfConfig = Tac.newInstance( "Routing::Rib::RouteConfigByProtocol",
                                      vrfName, vrfId )
         if not toggleNoRouteConfigTrieUsageEnabled():
            vrfConfigTrie = Tac.newInstance(
                  "Routing::Rib::RouteConfigTrieByProtocol",
                  vrfName )
            vrfConfigTrieSm = Tac.newInstance(
                  "Routing::Rib::RouteConfigTrieSmByProtocol",
                  vrfName )
         proto = 'isis'
         mountPath = 'routing/' + self.rib + '/routeConfig/' + af + "/" + \
               str( vrfId ) + "/" + position + "/" + proto

         protoConfig = self.shmemEm.doMount( mountPath,
                                             'Routing::Rib::RouteConfig',
                                             self.keyShadowInfo )
         vrfConfig.protocolConfig.addMember( protoConfig )
         vrfConfig.isRouteConfigFlattened.remove( proto )
         if not toggleNoRouteConfigTrieUsageEnabled():
            protoConfigTrie = vrfConfigTrie.routeConfigTrie.newMember( proto )
            protoConfigTrie.trie = ( proto, )
            protoConfigTrie.trie.af = af
            vrfConfigTrieSm.routeConfigTrieSm.newMember( protoConfig,
                                                         protoConfigTrie,
                                                         vrfId )
         qt8( "Vrf ", qv( vrfName ), " Routes: ",
               qv( len( protoConfig.route ) ),
               " Trie Routes: " + str( len( protoConfigTrie.trie.trieNode ) )
               if not toggleNoRouteConfigTrieUsageEnabled() else "",
               " Protocol: ", qv( proto ), " VRF ID: ", qv( vrfId ), " Position: ",
               qv( position ) )
         rc[ af ][ vrfName ] = vrfConfig
         if not toggleNoRouteConfigTrieUsageEnabled():
            rcTrie[ af ][ vrfName ] = vrfConfigTrie
            rcTrieSm[ af ][ vrfName ] = vrfConfigTrieSm
      else:
         qt8( "Returning existing route config for VRF ", qv( vrfName ),
               " VRF ID: ", qv( vrfId ), " Position: ", qv( position ) )

      return ( rc[ af ][ vrfName ], rcTrie[ af ][ vrfName ]
               if not toggleNoRouteConfigTrieUsageEnabled() else None )

   def getIpViaConfig( self, vrfName, transport, position ):

      # Only Default-VRF is supported for BGP ORR positions
      assert vrfName == DEFAULT_VRF
      # pylint: disable-next=consider-using-in
      assert ( transport == 'ipv4' or transport == 'ipv6' )

      vrfId = self.defaultVrfId

      vc = self.viaConfigs.get( position, dict() )
      if not vc:
         self.viaConfigs[ position ] = vc

      vcSplitterSm = self.viaConfigSplitterSmColl.get( position, dict() )
      if not vcSplitterSm:
         self.viaConfigSplitterSmColl[ position ] = vcSplitterSm

      if transport not in vc or transport not in vcSplitterSm:
         vc[ transport ] = dict()
         vcSplitterSm[ transport ] = dict()
         # The Via Config doesn't exist, instantiate a new one
         vcbpType = 'Routing::Rib::Ip::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Ip::ViaConfig'

         vrfConfig = Tac.newInstance( vcbpType, vrfId )
         proto = 'bgp'
         mp = 'routing/' + self.rib + '/viaConfig/' + transport + '/' \
               + position + '/' + proto

         viaConfig = self.shmemEm.doMount( mp, vcType, self.keyShadowInfo )
         viaConfig.rp = proto
         vrfIdArg = Tac.Value( "Vrf::VrfIdMap::VrfId" )
         t = getattr( ipRibTransport, transport )
         transportVal = Tac.Value( 'Routing::Rib::Transport', t )
         viaKeyConfigAllVrfKey = Tac.Value(
                                 "Routing::Rib::Ip::ViaKeyConfigAllVrfKey",
                                 proto, transportVal, vrfIdArg, True )
         viaKeyConfigAllVrf = Tac.newInstance(
                              "Routing::Rib::Ip::ViaKeyConfigAllVrf",
                              viaKeyConfigAllVrfKey )
         viaConfigSplitterSm = Tac.newInstance(
                               "Routing::Rib::ViaConfigSplitterSm",
                               mp, viaConfig, viaKeyConfigAllVrf )

         qt8( "Instantiated viaKeyConfigAllVrf and SplitterSm for MountPath ",
               qv( mp ), " and Position ", qv( position ) )
         vcSplitterSm[ mp ] = viaConfigSplitterSm
         # This creates a viaKeyConfig for the VRF is not present already.
         viaKeyConfig = viaKeyConfigAllVrf.viaKeyConfigIs( vrfId, viaConfig )
         # The VRF is valid and hence mark it as not tentative
         viaKeyConfig.tentative = False
         vrfConfig.protocolConfig.addMember( viaKeyConfig )
         vc[ transport ][ vrfName ] = vrfConfig
      else:
         qt8( "Returning existing via config for VRF ", qv( vrfName ),
               " VRF ID: ", qv( vrfId ), " Position: ", qv( position ) )

      return vc[ transport ][ vrfName ]

   def getAllViaConfigPerTransport( self, transport, position ):
      vrfId = self.defaultVrfId
      vrfName = DEFAULT_VRF

      vcByProtoByVrf = self.viaConfigByProtoByVrf.get( position, dict() )
      if not vcByProtoByVrf:
         self.viaConfigByProtoByVrf[ position ] = vcByProtoByVrf

      if transport not in vcByProtoByVrf:
         vcByProtoByVrf[ transport ] = \
               Tac.newInstance( "Routing::Rib::Ip::ViaConfigByProtocolByVrf" )

      viaConfigByProtoByVrf = vcByProtoByVrf[ transport ]

      viaConfig = self.getIpViaConfig( vrfName, transport, position )
      viaConfigByProtoByVrf.protocolConfig[ vrfId ] = viaConfig

      return viaConfigByProtoByVrf

   def getViaMetricStatus( self, position ):
      vms = self.vms.get( position, dict() )
      if not vms:
         vms = self.vms[ position ] = SharkLazyMount.mount(
               self.entityManager,
               Cell.path(
                  'routing/' + self.rib + '/' + position + '/viaMetricStatus' ),
               'Routing::Rib::ViaMetricStatusByVrf',
               SharkLazyMount.mountInfo( sharkMode ),
               True )

      return SharkLazyMount.force( vms )

   def getIgpResultStatusReaderHelper( self ):
      return getCommonIgpResultStatusReaderHelper( self.shmemEm )

class TunnelRibMounter:
   def __init__( self ):
      self.entityManager = None
      self._tunnelRib = None
      self._tunnelRibNameIdMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def tunnelRib( self ):
      if self._tunnelRib is None:
         self._tunnelRib = SmashLazyMount.mount(
            self.entityManager, 'tunnel/tunnelRibs/status/0',
            'Tunnel::TunnelTable::TunnelRib',
            SmashLazyMount.mountInfo( 'reader' ) )
      return self._tunnelRib

   @property
   def tunnelRibNameIdMap( self ):
      if self._tunnelRibNameIdMap is None:
         self._tunnelRibNameIdMap = LazyMount.mount(
            self.entityManager, 'tunnel/tunnelRibs/tunnelRibNameIdMap',
            'Tunnel::TunnelTable::TunnelRibNameIdMap', 'r' )
         LazyMount.force( self._tunnelRibNameIdMap )
      return self._tunnelRibNameIdMap

class FibRouteStatusMounter:
   def __init__( self, vrfMounter ):
      self.entityManager = None
      self.shmemEm = None
      self.fib4RouteStatus = {}
      self.fib6RouteStatus = {}
      self.vrfMounter = vrfMounter

      self.vrfMounter.ribMounters.append( self )

   def handleVrf( self, vrfName ):
      qt8( "Handling VRF ", qv( vrfName ) )
      if vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         return

      def releaseRouteStatus( self, routeStatus ):
         if vrfName in routeStatus and vrfName != DEFAULT_VRF:
            del routeStatus[ vrfName ]

      releaseRouteStatus( self, self.fib4RouteStatus )
      releaseRouteStatus( self, self.fib6RouteStatus )

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getFib4RouteStatus( self, vrfName ):
      if vrfName in self.fib4RouteStatus:
         return self.fib4RouteStatus[ vrfName ]

      mountPath = None
      if vrfName == DEFAULT_VRF:
         mountPath = 'routing/status'
         self.fib4RouteStatus[ vrfName ] = self.shmemEm.doMount(
            mountPath, 'Smash::Fib::RouteStatus',
            Smash.mountInfo( 'reader' ) )
      else:
         # pylint: disable-next=consider-using-f-string
         mountPath = 'routing/vrf/status/%s' % vrfName
         self.fib4RouteStatus[ vrfName ] = SmashLazyMount.force(
            SmashLazyMount.mount(
               self.shmemEm, mountPath, 'Smash::Fib::RouteStatus',
               SmashLazyMount.mountInfo( 'reader' ), autoUnmount=True ) )
      return self.fib4RouteStatus[ vrfName ]

   def getFib6RouteStatus( self, vrfName ):
      if vrfName in self.fib6RouteStatus:
         return self.fib6RouteStatus[ vrfName ]

      mountPath = None
      if vrfName == DEFAULT_VRF:
         mountPath = 'routing6/status'
         self.fib6RouteStatus[ vrfName ] = self.shmemEm.doMount(
            mountPath, 'Smash::Fib6::RouteStatus',
            Smash.mountInfo( 'reader' ) )
      else:
         # pylint: disable-next=consider-using-f-string
         mountPath = 'routing6/vrf/status/%s' % vrfName
         self.fib6RouteStatus[ vrfName ] = SmashLazyMount.force(
            SmashLazyMount.mount(
               self.shmemEm, mountPath, 'Smash::Fib6::RouteStatus',
               SmashLazyMount.mountInfo( 'reader' ), autoUnmount=True ) )
      return self.fib6RouteStatus[ vrfName ]

class IpRibCliMounter:
   '''This class mounts IpRib config and status entities in sysdb/smash.
   It mounts entities the first time they're needed and caches them'''

   def __init__( self, vrfMounter, rib="rib" ):

      assert rib in [ "rib", "mrib" ]

      self.entityManager = None
      self.shmemEm = None
      self.readerInfo = SmashLazyMount.mountInfo( 'reader' )
      self.keyShadowInfo = SmashLazyMount.mountInfo( 'keyshadow' )

      self.vrfMounter = vrfMounter
      # So the VRF mounter can notify us about VRF additions/deletions
      self.vrfMounter.ribMounters.append( self )
      self.rib = rib

      self.routeConfigs = {}
      self.cachedRouteConfig = {}
      if not toggleNoRouteConfigTrieUsageEnabled():
         self.routeConfigTries = {}
         self.routeConfigTrieSms = {}
      self.viaConfigByProtoByVrf = {}
      self.winningRouteStatuses = {}
      self.loopingRouteStatus = None
      self.tunnelFib = None
      self.nhResPolicyStatus = None
      self.ribConfig = None
      self.ribConfigBgp = None
      self.crLeakStatuses = {}

      # For each AF, we store a dict by VRF Name
      for af in [ 'ipv4', 'ipv6' ]:
         self.routeConfigs[ af ] = {}
         if not toggleNoRouteConfigTrieUsageEnabled():
            self.routeConfigTries[ af ] = {}
            self.routeConfigTrieSms[ af ] = {}
         self.winningRouteStatuses[ af ] = {}

      self.viaSetConfigByProto = \
         Tac.newInstance( "Routing::Rib::ViaSetConfigByProtocol" )

      vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey" )
      self.viaSetConfig = Tac.newInstance( "Routing::Rib::ViaSetConfig", vsck )

      self.viaSetStatusByProto = \
         Tac.newInstance( "Routing::Rib::ViaSetStatusByProtocol" )

      self.viaConfig = {}
      self.transports = [ 'ipv4', 'ipv6', 'evpn', 'mpls' ]
      for transport in self.transports:
         # For each transport, we store a dict by VRF Name
         self.viaConfig[ transport ] = {}

         self.viaConfigByProtoByVrf[ transport ] = \
            Tac.newInstance( "Routing::Rib::Ip::ViaConfigByProtocolByVrf" )
      # We maintain a dict of ViaConfigSplitterSm, keyed by the
      # ViaConfig mount path
      self.viaConfigSplitterSmColl = {}

      self.viaStatus = None
      self.viaMetricStatus = None
      self.vrfNameToViaSetConfigId = {}
      self.nhgEntryStatus = None

   def setEntityManager( self, em ):

      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def handleVrf( self, vrfName ):
      qt8( "Handling VRF ", qv( vrfName ) )
      if vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         return

      qt8( "Clearing VRF Name", qv( vrfName ) )

      # Vrf has been deleted. Delete all the objects we cached for this VRF,
      # they will be recreated the next time a CLI involving that VRF is issued.
      for af in [ 'ipv4', 'ipv6' ]:
         if vrfName in self.routeConfigs[ af ]:
            del self.routeConfigs[ af ][ vrfName ]
         if not toggleNoRouteConfigTrieUsageEnabled():
            if vrfName in self.routeConfigTries[ af ]:
               del self.routeConfigTries[ af ][ vrfName ]

            if vrfName in self.routeConfigTrieSms[ af ]:
               del self.routeConfigTrieSms[ af ][ vrfName ]

         if vrfName in self.winningRouteStatuses[ af ]:
            del self.winningRouteStatuses[ af ][ vrfName ]

      vrfId = None
      for transport in self.transports:
         if vrfName in self.viaConfig[ transport ]:
            if not vrfId:
               vrfId = self.viaConfig[ transport ][ vrfName ].vrfId
            del self.viaConfig[ transport ][ vrfName ]

            if vrfId in self.viaConfigByProtoByVrf[ transport ].protocolConfig:
               del self.viaConfigByProtoByVrf[ transport ].protocolConfig[ vrfId ]

      # Cleanup the viaKeyConfig's for this VRF
      if vrfId is not None:
         for splitterSm in list( self.viaConfigSplitterSmColl.values() ):
            del splitterSm.viaKeyConfigAllVrf.viaKeyConfigPerVrf[ vrfId ]

         ViaConfigLib = Tac.Type( 'Routing::ViaConfigLib' )
         # Cleanup the splitterSm for the unflattened mount paths
         for transport in self.transports:
            for proto in ipRibProtocols:
               if not ViaConfigLib.isAllVrfViaConfig( proto, transport, self.rib ):
                  mp = f'routing/{self.rib}/viaConfig/{transport}/{vrfId}/{proto}'
                  if mp in self.viaConfigSplitterSmColl:
                     del self.viaConfigSplitterSmColl[ mp ]

      vsci = self.vrfNameToViaSetConfigId.get( vrfName )
      if vsci is None:
         return

      for proto in ipRibProtocols:
         if not self.viaSetConfig.perVrf( proto ):
            continue

         vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey", proto, vsci )
         if vsck in self.viaSetConfigByProto.protocolConfig:
            del self.viaSetConfigByProto.protocolConfig[ vsck ]

         del self.viaSetStatusByProto.protocolStatus[ proto ]

   def getRouteConfig( self, af, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None, None

      routeConfig = self.routeConfigs[ af ]
      vrfConfig = routeConfig.get( vrfName )

      if not toggleNoRouteConfigTrieUsageEnabled():
         routeConfigTrie = self.routeConfigTries[ af ]
         vrfConfigTrie = routeConfigTrie.get( vrfName )

      if vrfConfig is None :
         vrfConfig = Tac.newInstance( "Routing::Rib::RouteConfigByProtocol",
                                       vrfName, vrfId )
         if not toggleNoRouteConfigTrieUsageEnabled():
            vrfConfigTrie = Tac.newInstance(
               "Routing::Rib::RouteConfigTrieByProtocol",
               vrfName )
            vrfConfigTrieSm = Tac.newInstance(
               "Routing::Rib::RouteConfigTrieSmByProtocol",
               vrfName )

         for proto in ipRibProtocols:

            isRCFlattened = isRouteConfigFlattened( proto, af, self.rib )
            if isRCFlattened:
               mountPath = 'routing/' + self.rib + '/routeConfig/' + af + '/' + \
                           proto
               protoConfig = self.cachedRouteConfig.get( mountPath )
               if protoConfig is None:
                  protoConfig = self.shmemEm.doMount( mountPath,
                                     'Routing::Rib::RouteConfig',
                                     self.keyShadowInfo )
                  self.cachedRouteConfig[ mountPath ] = protoConfig
            else:
               mountPath = 'routing/' + self.rib + '/routeConfig/' + af + "/" + \
                           str( vrfId ) + "/" + proto
               protoConfig = self.shmemEm.doMount( mountPath,
                                                   'Routing::Rib::RouteConfig',
                                                   self.keyShadowInfo )
            vrfConfig.protocolConfig.addMember( protoConfig )
            if isRCFlattened:
               vrfConfig.isRouteConfigFlattened.add( proto )
            else:
               vrfConfig.isRouteConfigFlattened.remove( proto )
            if not toggleNoRouteConfigTrieUsageEnabled():
               protoConfigTrie = vrfConfigTrie.routeConfigTrie.newMember( proto )
               protoConfigTrie.trie = ( proto, )
               protoConfigTrie.trie.af = af
               vrfConfigTrieSm.routeConfigTrieSm.newMember( protoConfig,
                                                            protoConfigTrie,
                                                            vrfId )
            qt8( "Vrf ", qv( vrfName ), " Routes: ", 
                  qv( len( protoConfig.route ) ),
                  " Trie Routes: " + str( len( protoConfigTrie.trie.trieNode ) )
                  if not toggleNoRouteConfigTrieUsageEnabled() else "",
                  " Protocol: ", qv( proto ), " VRF ID: ", qv( vrfId ) )
         self.routeConfigs[ af ][ vrfName ] = vrfConfig
         if not toggleNoRouteConfigTrieUsageEnabled():
            self.routeConfigTries[ af ][ vrfName ] = vrfConfigTrie
            self.routeConfigTrieSms[ af ][ vrfName ] = vrfConfigTrieSm
      else:
         qt8( "Returning existing route config for VRF ", qv( vrfName ),
                             " VRF ID: ", qv( vrfId ) )

      return ( vrfConfig, vrfConfigTrie
               if not toggleNoRouteConfigTrieUsageEnabled() else None )

   def getAllViaSetConfig( self ):
      '''Calls getViaSetConfig for all vrfs in order to mount ViaSetConfigs for all
      vrfs. This is required for the vrf leaking scenario where a route is in one
      vrf, and the ViaSetConfig is in another. Returns a ViaSetConfigByProtocol 
      instance, which is a flattened entity of ViaConfigByProtocol accross all vrfs.
      '''
      for vrfName in self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId:
         # We do not care about the return value, we simply need this
         # to mount the collection for all the vrfs.
         _ = self.getViaSetConfig( vrfName )

      return self.viaSetConfigByProto

   def getViaSetConfig( self, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      for proto in ipRibProtocols:

         viaSetConfigId = None
         if self.viaSetConfig.perVrf( proto ):
            viaSetConfigId = vrfId
            self.vrfNameToViaSetConfigId[ vrfName ] = viaSetConfigId
         else:
            viaSetConfigId = Tac.Value( "Routing::Rib::ViaSetKey" ).allVrfId

         vsck = Tac.newInstance( "Routing::Rib::ViaSetConfigKey", proto,
                                 viaSetConfigId )
         if vsck in self.viaSetConfigByProto.protocolConfig:
            continue

         viaSetConfig = self.viaSetConfigByProto.protocolConfig.newMember( vsck )
         mountPath = self.viaSetConfig.mountPath( self.rib, proto, vrfName )
         qt8( "Mount Path ", qv( mountPath ) )
         smashColl = self.shmemEm.doMount( mountPath,
                                           'Routing::Rib::Smash::ViaSetCollection',
                                           self.readerInfo )
         viaSetConfig.configViaSetColl = smashColl

      return self.viaSetConfigByProto

   def getAllViaSetStatus( self ):
      for proto in ipRibProtocols:
         if proto in self.viaSetStatusByProto.protocolStatus:
            continue
         viaSetStatus = self.viaSetStatusByProto.protocolStatus.newMember( proto )
         mountPath = tacTypeViaSetStatus.mountPath( self.rib, proto )
         smashStatus = self.shmemEm.doMount( mountPath,
                                             'Routing::Rib::Smash::ViaSetStatus',
                                             self.readerInfo )
         viaSetStatus.smashStatus = smashStatus
      return self.viaSetStatusByProto

   def getAllViaConfigPerTransport( self, transport ):
      viaConfigByProtoByVrf = self.viaConfigByProtoByVrf[ transport ]

      for vrfName, vrfId in \
            self.vrfMounter.vrfNameStatus.nameToIdMap.vrfNameToId.items():
         viaConfig = self.getIpViaConfig( vrfName, transport )
         viaConfigByProtoByVrf.protocolConfig[ vrfId ] = viaConfig

      return viaConfigByProtoByVrf

   def getViaConfigByViaType( self, vrfName, transport, viaType ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      transportConfig = self.viaConfig.get( transport )
      vrfConfig = transportConfig.get( vrfName )
      if vrfConfig is not None:
         qt8( "Returning existing via config for VRF ", qv( vrfName ) )
         return vrfConfig

      # The Via Config doesn't exist, instantiate a new one

      if viaType == 'ip':
         vcbpType = 'Routing::Rib::Ip::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Ip::ViaConfig'
      elif viaType == 'evpn':
         vcbpType = 'Routing::Rib::Evpn::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Evpn::ViaConfig'
      elif viaType == 'mpls':
         vcbpType = 'Routing::Rib::Mpls::ViaConfigByProtocol'
         vcType = 'Routing::Rib::Mpls::ViaConfig'
      else:
         return None

      vrfConfig = Tac.newInstance( vcbpType, vrfId )

      viaConfigLib = Tac.Type( 'Routing::ViaConfigLib' )

      for proto in ipRibProtocols:
         if viaConfigLib.isAllVrfViaConfig( str( proto ), transport, self.rib ):
            mp = f'routing/{self.rib}/viaConfig/{transport}/{proto}'
            vrfIdArg = Tac.Value( "Vrf::VrfIdMap::VrfId" )
         else:
            mp = f'routing/{self.rib}/viaConfig/{transport}/{vrfId}/{proto}'
            vrfIdArg = vrfId

         viaConfigProxy = SmashLazyMount.mount( self.shmemEm, mp, vcType,
               self.keyShadowInfo, autoUnmount=True )
         viaConfig = SmashLazyMount.force( viaConfigProxy )
         viaConfig.rp = proto

         viaConfigSplitterSm = self.viaConfigSplitterSmColl.get( mp )
         if not viaConfigSplitterSm:
            if viaType == 'ip':
               # pylint: disable-next=consider-using-in
               if not ( transport == 'ipv4' or transport == 'ipv6' ):
                  return None

               t = getattr( ipRibTransport, transport )
               transportVal = Tac.Value( 'Routing::Rib::Transport', t )
               viaKeyConfigAllVrfKey = Tac.Value(
                                       "Routing::Rib::Ip::ViaKeyConfigAllVrfKey",
                                       proto, transportVal, vrfIdArg,
                                       self.rib == "rib" )
               viaKeyConfigAllVrf = Tac.newInstance(
                                       "Routing::Rib::Ip::ViaKeyConfigAllVrf",
                                       viaKeyConfigAllVrfKey )
               viaConfigSplitterSm = Tac.newInstance(
                                       "Routing::Rib::ViaConfigSplitterSm",
                                       mp, viaConfig, viaKeyConfigAllVrf )
            elif viaType == 'evpn':
               viaKeyConfigAllVrf = Tac.newInstance(
                                    "Routing::Rib::Evpn::ViaKeyConfigAllVrf", mp )
               viaConfigSplitterSm = Tac.newInstance(
                                     "Routing::Rib::EvpnViaConfigSplitterSm",
                                     mp, viaConfig, viaKeyConfigAllVrf )
            elif viaType == 'mpls':
               viaKeyConfigAllVrf = Tac.newInstance(
                                    "Routing::Rib::Mpls::ViaKeyConfigAllVrf", mp )
               viaConfigSplitterSm = Tac.newInstance(
                                     "Routing::Rib::MplsViaConfigSplitterSm",
                                     mp, viaConfig, viaKeyConfigAllVrf )

            qt8( "Instantiated viaKeyConfigAllVrf and SplitterSm for ",
                 qv( mp ) )
            self.viaConfigSplitterSmColl[ mp ] = viaConfigSplitterSm
         else:
            viaKeyConfigAllVrf = viaConfigSplitterSm.viaKeyConfigAllVrf
         # This creates a viaKeyConfig for the VRF is not present already.
         viaKeyConfig = viaKeyConfigAllVrf.viaKeyConfigIs( vrfId, viaConfig )
         # The VRF is valid and hence mark it as not tentative
         viaKeyConfig.tentative = False
         vrfConfig.protocolConfig.addMember( viaKeyConfig )

      self.viaConfig[ transport ][ vrfName ] = vrfConfig
      return vrfConfig

   def getIpViaConfig( self, vrfName, transport ):
      return self.getViaConfigByViaType( vrfName, transport, 'ip' )

   def getEvpnViaConfig( self, vrfName ):
      return self.getViaConfigByViaType( vrfName, 'evpn', 'evpn' )

   def getMplsViaConfig( self, vrfName ):
      return self.getViaConfigByViaType( vrfName, 'mpls', 'mpls' )

   def getWinningRouteStatus( self, af, vrfName ):

      routeStatus = self.winningRouteStatuses[ af ]

      vrfStatus = routeStatus.get( vrfName )
      if vrfStatus is not None:
         return vrfStatus

      vrfStatus = Tac.newInstance( "Routing::Rib::WinningRouteStatusByProtocol",
                                   vrfName )

      for proto in ipRibProtocols:
         mountPath = 'routing/' + self.rib + '/winningRoute/' + af + "/" + \
                     proto
         protoStatus = self.shmemEm.doMount( mountPath,
                                      'Routing::Rib::UnifiedWinningRouteStatus',
                                       self.readerInfo )
         vrfStatus.protocolStatus[ proto ] = protoStatus

      self.winningRouteStatuses[ af ][ vrfName ] = vrfStatus

      return vrfStatus

   def getLoopingRouteStatus( self, af, vrfName ):

      vrfId = self.vrfMounter.getVrfId( vrfName )
      if vrfId is None:
         return None

      if self.loopingRouteStatus is None:
         path = Cell.path( "routing/" + self.rib + "/allVrfLoopingRouteStatus" )

         self.loopingRouteStatus = \
               LazyMount.mount( self.entityManager, path,
                                'Routing::Rib::AllVrfLoopingRouteStatus', 'r' )
         LazyMount.force( self.loopingRouteStatus )

      if af == "ipv4":
         afStatus = self.loopingRouteStatus.ipv4VrfLoopingRouteStatus
      else:
         afStatus = self.loopingRouteStatus.ipv6VrfLoopingRouteStatus

      return afStatus.get( vrfId )

   def getViaStatus( self ):

      if self.viaStatus is None:
         path = Cell.path( 'routing/' + self.rib + '/viaStatus' )
         self.viaStatus = LazyMount.mount( self.entityManager, path,
                                           'Routing::Rib::ViaStatusByVrf', 'r' )
         LazyMount.force( self.viaStatus )

      return self.viaStatus

   def getNhgEntryStatus( self ):
      if not self.nhgEntryStatus:
         mountPath = 'routing/nexthopgroup/entrystatus'
         self.nhgEntryStatus = self.shmemEm.doMount( mountPath,
                               "NexthopGroup::EntryStatus",
                               self.readerInfo )
      return self.nhgEntryStatus

   def getTunnelFib( self ):

      if self.tunnelFib is None:
         self.tunnelFib = self.shmemEm.doMount(
            'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
            Smash.mountInfo( 'reader' ) )
      return self.tunnelFib

   def getNhResStatus( self, vrfName ):
      if self.nhResPolicyStatus is None:
         path = Cell.path( 'routing/' + self.rib + '/nhResPolicyStatus' )
         self.nhResPolicyStatus = \
            LazyMount.mount( self.entityManager, path,
                             'Routing::Rib::NextHopResPolicyStatusByVrf', 'r' )
         LazyMount.force( self.nhResPolicyStatus )
      return self.nhResPolicyStatus.vrfStatus.get( vrfName )

   def getRibConfig( self ):
      if self.ribConfig is None:
         self.ribConfig = LazyMount.mount( self.entityManager,
                                           Cell.path( 'routing/rib/config' ),
                                           'Tac::Dir', 'ri' )
         LazyMount.force( self.ribConfig )
      return self.ribConfig

   def getRibConfigBgp( self, vrfId ):
      if self.ribConfig is None:
         self.ribConfig = self.getRibConfig()

      if self.ribConfigBgp is None:
         self.ribConfigBgp = self.ribConfig.get( 'bgp' )

      if self.ribConfigBgp:
         return self.ribConfigBgp.vrfConfig.get( vrfId )
      
      return None

   def getCrLeakStatus( self, af ):
      '''Mount connected route leak status if needed and then sort it'''

      status = self.crLeakStatuses.get( af )
      if status is None:
         mountPath = "routing/" if af == "ipv4" else "routing6/"
         mountPath += "connectedRouteLeakStatus"

         status = self.shmemEm.doMount( mountPath,
                                        "Routing::ConnectedRouteLeakStatus",
                                        self.readerInfo )
         self.crLeakStatuses[ af ] = status

      return status

   def getIgpResultStatusReaderHelper( self ):
      return getCommonIgpResultStatusReaderHelper( self.shmemEm )

class IpRibShmemCliMounter:
   '''This class mounts IpRib config and status entities from shared-memory.
      After a period of not being used, the underlying entity in the
      SharkLazyMount will get unmounted to release resources. It will
      seamlessly get re-mounted under-the-hood on subsequent access.'''
   def __init__( self, em, rib ):
      self._vms = SharkLazyMount.mount(
         em,
         Cell.path( 'routing/' + rib + '/viaMetricStatus' ),
         'Routing::Rib::ViaMetricStatusByVrf',
         SharkLazyMount.mountInfo( sharkMode ),
         True )

   def getViaMetricStatus( self, force=False ):
      if force:
         # Force returns the actual underlying entity instead of a
         # proxy. This is necessary if the entity is to be passed to
         # the c++ layer via Tac.newInstance or similar. The downside
         # is that the autoUnmount cannot occur as long as the raw
         # entity is referenced somewhere
         return SharkLazyMount.force( self._vms )
      return self._vms

class AllVrfIgpReadyMounter:
   def __init__( self ):
      self.entityManager = None
      self._allVrfIgpReady = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def allVrfIgpReady( self ):
      if self._allVrfIgpReady is None:
         self._allVrfIgpReady = \
            LazyMount.mount( self.entityManager,
                             Cell.path( "routing/rib/ribReady/status/igpReady" ),
                             "Routing::Rib::AllVrfIgpReady", "r" )
      return self._allVrfIgpReady

class AllVrfRibReadyMounter:
   def __init__( self ):
      self.entityManager = None
      self._allVrfRibReady = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def allVrfRibReady( self ):
      if self._allVrfRibReady is None:
         self._allVrfRibReady = \
            LazyMount.mount( self.entityManager,
                             Cell.path( "routing/rib/ribReady/status/all" ),
                             "Routing::Rib::AllVrfRibReady", "r" )
      return self._allVrfRibReady

class FibReadyDirMounter:
   def __init__( self ):
      self.entityManager = None
      self._fibReadyDir = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def fibReadyDir( self ):
      if self._fibReadyDir is None:
         self._fibReadyDir = LazyMount.mount( self.entityManager,
                                              Cell.path( "routing/fibReady" ),
                                              "Tac::Dir", "ri" )
      return self._fibReadyDir

class ForwardingStatusMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.fec4Status = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getFec4Status( self ):
      if self.fec4Status is None:
         readerInfo = SmashLazyMount.mountInfo( 'reader' )
         mountPath = "forwarding/unifiedStatus"
         self.fec4Status = self.shmemEm.doMount( mountPath,
                           "Smash::Fib::ForwardingStatus",
                           readerInfo )
      return self.fec4Status

class FecOverrideTableMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.fecOverrideTable = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getFecOverrideTable( self ):
      if self.fecOverrideTable is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = "te/segmentrouting/fecoverride/config"
         self.fecOverrideTable = self.shmemEm.doMount( mountPath,
                                    "Qos::Smash::FecOverrideConfig",
                                    readerInfo )
      return self.fecOverrideTable

class SrTeColorDscpMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._srTeCbfColorDscpMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def srTeCbfColorDscpMap( self ):
      if self._srTeCbfColorDscpMap is None:
         self._srTeCbfColorDscpMap = LazyMount.mount( self.entityManager,
               'te/segmentrouting/cbf/color-dscp/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._srTeCbfColorDscpMap

class TeColorDscpMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._teCbfColorDscpMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def teCbfColorDscpMap( self ):
      if self._teCbfColorDscpMap is None:
         self._teCbfColorDscpMap = LazyMount.mount( self.entityManager,
               'te/cbf/color-dscp/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._teCbfColorDscpMap

class TeColorTcMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._teCbfColorTcMap = None

   def setEntityManager( self, em ):
      self.entityManager = em
      
   @property
   def teCbfColorTcMap( self ):
      if self._teCbfColorTcMap is None:
         self._teCbfColorTcMap = LazyMount.mount( self.entityManager,
               'te/cbf/color-tc/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._teCbfColorTcMap

class LfibVskOverrideTableMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.lfibVskOverrideTable = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getLfibVskOverrideTable( self ):
      if self.lfibVskOverrideTable is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = "te/cbf/vskoverride/mpls/config"
         self.lfibVskOverrideTable = self.shmemEm.doMount( mountPath,
                                             "Mpls::Override::LfibVskOverrideConfig",
                                             readerInfo )
      return self.lfibVskOverrideTable

# Utility functions required for implementing and testing 'show rib ready' command
def toUtc( timestamp ):
   """Convert a timestamp got from Tac::now() to UTC"""
   utcTime = timestamp + Tac.utcNow() - Tac.now() if timestamp else timestamp
   return utcTime
