# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from CliModel import ( Bool, Enum, List, Model, Str )

typeName = "Tunnel::TunnelTable::TunnelRibNameIdMap"
systemColoredTunnelRibName = Tac.Type( typeName ).systemColoredTunnelRibName

class ResolutionRib( Model ):
   name = Str( help="Name of the resolution RIB." )
   ribType = Enum( [ 'tunnel', 'ip' ], help="Type of the resolution RIB." )
   colored = Bool( optional=True, help="The tunnel RIB is colored." )

   @staticmethod
   def fromTac( tacValue ):
      if tacValue.resolutionRibSource == 'undefined':
         return None

      if tacValue.resolutionRibSource in ( 'systemUnicastRib',
                                           'systemConnected',
                                           'systemUnicastRibSrv6' ):
         # Omit 'colored' as it's optional
         params = { 'name' : tacValue.stringValue(), 'ribType' : 'ip' }
      elif tacValue.resolutionRibSource == 'tunnelRib':
         name = tacValue.tunnelRibName
         params = {
               'name' : name,
               'ribType' : 'tunnel',
               'colored' : name == systemColoredTunnelRibName
               }

      return ResolutionRib( **params )

   @staticmethod
   def stringValueIs( stringValue ):
      """
      Convert a string representation of a ResolutionRib into the CAPI type, e.g.
      'system-unicast-rib' or 'tunnel-rib colored foo'.
      """
      name = stringValue.split()[ -1 ]
      ribType = 'tunnel' if 'tunnel-rib' in stringValue else 'ip'
      params = {
            'name' : name,
            'ribType' : ribType,
            }
      if ribType == 'tunnel':
         params[ 'colored' ] = 'colored' in stringValue
      return ResolutionRib( **params )

   @staticmethod
   def tacResolutionRibConfig( resolutionRibSource ):
      return Tac.Value( "Routing::Rib::ResolutionRibConfig", resolutionRibSource )

class ResolutionRibProfileConfig( Model ):
   resolutionMethods = List( valueType=ResolutionRib,
                             help="List of resolution methods" )
   systemDefault = Bool( optional=True,
                         help="System default profile is set for next-hop" )

   def toTac( self ):
      """
      Converts an instance of ResolutionRibProfileConfig into an instance of
      Tac.Type( Rib::ResolutionRibProfileConfig ).
      """
      dictRepr = self.toDict()
      resolutionMethods = dictRepr[ 'resolutionMethods' ]
      systemDefault = dictRepr.get( 'systemDefault', False )
      return ResolutionRibProfileConfig.fromJson( resolutionMethods, systemDefault )

   @staticmethod
   def fromTac( tacValue ):
      """
      Converts an instance of Tac.Type( Rib::ResolutionRibProfileConfig ) into an
      instance of ResolutionRibProfileConfig
      """
      if not isinstance( tacValue,
                         Tac.Type( 'Routing::Rib::ResolutionRibProfileConfig' ) ):
         raise ValueError

      resolutionMethods = [ _f for _f in (
         ResolutionRib.fromTac( v ) for v in tacValue.resolutionMethod.values() )
         if _f ]
      # If it's a 'system-default' profile, it has no ribType or any real
      # ResolutionRibConfig instances at all, so hardcode the required CAPI model,
      #
      #    { 'systemDefault' : 'true' }
      if tacValue.useDefault:
         assert not resolutionMethods
         return ResolutionRibProfileConfig( systemDefault=True,
                                            resolutionMethods=[] )

      return ResolutionRibProfileConfig( resolutionMethods=resolutionMethods )

   @staticmethod
   def tacResolutionRibProfileConfig( resolutionMethods=None, useDefault=False ):
      """
      @param resolutionMethods - list of tuples of the form ( ResolutionMethodIndex,
      ResolutionRibSource, String )

      Usage:

      resRibProfile = self.tacResolutionRibProfile( [
         ( 'tunnelRib', 'foo' ), # CTRIB foo
         ( 'tunnelRib', 'bar' ), # CTRIB bar
         ( 'systemUnicastRib', '' ), # system unicast rib
      ] )

      Raises a ValueError if a tunnel name is nonempty and the resolution source is
      not 'tunnelRib'.
      """
      resRibProfileConfig = Tac.Value( "Routing::Rib::ResolutionRibProfileConfig",
                                       useDefault=useDefault )
      if not resolutionMethods:
         resolutionMethods = []
      for idx, ( src, tunName ) in enumerate( resolutionMethods ):
         if src != 'tunnelRib' and tunName != '':
            raise ValueError
         config = \
               ResolutionRib.tacResolutionRibConfig( resolutionRibSource=src )
         config.tunnelRibName = tunName
         resRibProfileConfig.resolutionMethod[ idx ] = config
      return resRibProfileConfig

   @staticmethod
   def fromJson( resolutionMethods, systemDefault=False ):
      """
      Returns a Rib::ResolutionRibProfileConfig from a list of ResolutionRib JSON
      objects represented as a Python dictionary, e.g.

      {
         "resolutionMethods": [
            {
               "ribType": "tunnel",
               "colored": true,
               "name": "system-colored-tunnel-rib"
            },
            {
               "ribType": "tunnel",
               "colored": false,
               "name": "system-tunnel-rib"
            },
            {
               "ribType": "ip",
               "name": "system-unicast-rib"
            }
         ]
      }
      """
      methods = []
      for resolutionMethod in resolutionMethods:
         src = None
         tunnelRibName = ''
         if resolutionMethod[ 'ribType' ] == 'tunnel':
            src = 'tunnelRib'
            tunnelRibName = resolutionMethod[ 'name' ]
         elif resolutionMethod[ 'name' ] == 'system-unicast-rib':
            src = 'systemUnicastRib'
         elif resolutionMethod[ 'name' ] == 'system-connected':
            src = 'systemConnected'
         elif resolutionMethod[ 'name' ] == 'system-unicast-rib-srv6':
            src = 'systemUnicastRibSrv6'
         # The CliModelChecker.py generates bogus name / ribType combinations (e.g.
         # { 'ribType': 'ip', name: 'test string' }. It is not possible to specify
         # the contstraints (e.g. 'ribType' == 'ip' => 'name' =~
         # 'system-{unicast-rib,connected}') using the CAPI model types, so be
         # generous, and _silently_ ignore such entries.
         if src in Tac.Type( 'Routing::Rib::ResolutionRibSource' ).attributes:
            methods.append( ( src, tunnelRibName ) )
      return ResolutionRibProfileConfig.tacResolutionRibProfileConfig(
            methods, useDefault=systemDefault )

class ResolutionRibProfile( Model ):
   resolutionMethods = List( valueType=ResolutionRib,
                             help="List of resolution methods" )
   error = Str( help="Invalid resolution profile reason", optional=True )

   @staticmethod
   def stringValueIs( stringValue ):
      if stringValue == "No profile set for this next-hop":
         return ResolutionRibProfile( error=stringValue, resolutionMethods=[] )
      methods = [ ResolutionRib.stringValueIs( val )
                  for val in stringValue.split( ',' ) ] if stringValue else []
      return ResolutionRibProfile( resolutionMethods=methods )

class ViaFlagDetails( Model ):
   requireMplsResolution = Bool( optional=True, 
         help='Via may recursively resolve over mpls '
              'tunnel or RIB/connected routes' )
   arpResolved = Bool( optional=True, 
         help="ARP/ND resolution information is available for this via" )
   mustResolveViaConnected = Bool( optional=True, 
         help="Via must recursively resolve only over connected routes" )
   suppressResolution = Bool( optional= True, 
         help="Resolution status information is not published to protocols" )
   routeCacheConnectedVia = Bool( optional=True, 
         help="This via is for a route cache connected route" )
   martianVia = Bool( optional=True, 
         help="This via is for a martian route" )
   bfdMonitoredVia = Bool( optional=True, 
         help="The via is monitored by BFD" )
   bgpPeerNexthop = Bool( optional=True,
         help="The via is for BGP peer reachability" )
