#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpRibCliLib as IpRibCliLib
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpRibCliShow as IpRibCliShow
from CliPlugin.RoutingBgpCli import orrPositionNameFn
from CliPlugin.IpAddrMatcher import PREFIX_OVERLAP_AUTOZERO, ipAddrWithMaskExpr
from CliPlugin.IpGenAddrMatcher import IpGenAddrOrPrefixExprFactory
from CliPlugin.TechSupportCli import techSupportKwMatcher
from CliPlugin.VrfCli import VrfExprFactory, vrfKwForShow, ALL_VRF_NAME
import IpRibModel
import ShowCommand
import CliCommand
from CliCommand import Node
# BUG282923 - deprecate ip (make it hidden) and use ipv4 instead
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv4 import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from CliToken.Agent import agentKwForShow
from ConnectedRouteLeakModel import LeakedConnectedRoutesByVrf

from Toggles import IpRibToggleLib
from Toggles.RoutingLibToggleLib import toggleShowRibRouteDetailEnabled,\
                                        toggleOspfShamLinkEnabled,\
                                        toggleOspf3ShamLinkEnabled
from Toggles.SrTePolicyLibToggleLib import toggleDynamicSrTeEnabled
from Toggles.IpRibLibToggleLib import toggleIsisConditionalAdvertiseEnabled

matcherBrief = CliMatcher.KeywordMatcher( 'brief',
      helpdesc='Show a single combined summary of the RIB for all VRFs' )
matcherConnected = CliMatcher.KeywordMatcher( 'connected',
      helpdesc='Display connected route leaking state' )
matcherInternal = CliMatcher.KeywordMatcher( 'internal',
      helpdesc='Dump internal state' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Show viaSetIDs and viaIDs' )
matcherLeak = CliMatcher.KeywordMatcher( 'leak',
      helpdesc='Show VRF leaking state' )
matcherLoop = CliMatcher.KeywordMatcher( 'loop',
      helpdesc='Recursive route resolution loops' )
matcherLoopRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routes that are part of recursive route resolution loops' )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Multicast Routing Information Base commands' )
matcherNextHop = CliMatcher.KeywordMatcher( 'next-hop',
      helpdesc='Next-hop vias with their corresponding resolved/unresolved vias' )
matcherPrefix = IpGenAddrOrPrefixExprFactory( ipOverlap=PREFIX_OVERLAP_AUTOZERO,
                                              ip6Overlap=PREFIX_OVERLAP_AUTOZERO )
matcherResolution = CliMatcher.KeywordMatcher( 'resolution',
      helpdesc='Recursive route resolution policy' )
matcherRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routing Information Base routes' )
matcherResolutionRoute = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Routes that are used for recursive resolution' )
matcherRib = CliMatcher.KeywordMatcher( 'rib',
      helpdesc='Routing Information Base commands' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Routing Information Base summary' )
matcherVrf = VrfExprFactory( helpdesc='Show routes in a VRF' )
matcherOrr = CliMatcher.KeywordMatcher( 'orr-position',
      helpdesc='Show routes for a ORR-Position' )
matcherOrrPosition = CliMatcher.DynamicNameMatcher( orrPositionNameFn,
      helpdesc='Position name', pattern=r'[a-zA-Z0-9][-a-zA-Z0-9_.:]*' )
matcherEvpn = CliMatcher.KeywordMatcher( 'evpn',
      helpdesc='Details related to EVPN' )


protocols = list( IpRibCliLib.ipRibProtocols ) + [ 'host' ]
matcherProtocolOrHost = CliMatcher.EnumMatcher(
   # pylint: disable-next=consider-using-f-string
   { IpRibCliLib.ipRibProtocolString.internalToExternal( proto ) : 'Show only %s '
      'routes' % IpRibCliLib.ipRibProtocolString.internalToExternal( proto )
      for proto in protocols } )

evpnProtocols = [ 'bgp', 'static' ]
matcherEvpnProtocol = CliMatcher.EnumMatcher(
    #pylint: disable-next=consider-using-f-string
    { IpRibCliLib.ipRibProtocolString.internalToExternal( proto ) : 'Show only %s '
      'routes' % IpRibCliLib.ipRibProtocolString.internalToExternal( proto )
      for proto in evpnProtocols } )

matcherVrfDefault = VrfExprFactory( helpdesc='Show routes in a VRF',
                                    inclDefaultVrf=True )
prefixMatcher = ipAddrWithMaskExpr( 'IP address', 'Subnet\'s mask value',
      'IPv4 address prefix', overlap=PREFIX_OVERLAP_AUTOZERO )
matcherOspfShow = CliMatcher.KeywordMatcher( 'ospf', helpdesc='OSPF information' )
matcherOspfv3AfterShow = CliMatcher.KeywordMatcher( 'ospfv3',
                                                    helpdesc='OSPFv3 information' )
matcherAfIpv6AfterShow = CliMatcher.KeywordMatcher( 'ipv6',
                                                     helpdesc='IPv6 related' )
vrfExprFactory = VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True,
      inclAllVrf=True )

def showIpOspfShamLinkRoute( mode, args ):
   # if no vrf is specified show all vrfs output, this is aligned
   # with 'sh ip ospf sham-link' command
   args[ 'VRF' ] = args.get( 'VRF', ALL_VRF_NAME )
   # below args are needed for integration with show ip route
   if 'ip' in args:
      args[ 'ip' ] = 'ipv4'
      args[ 'PROTOCOL' ] = 'ospf'
   else:
      args[ 'PROTOCOL' ] = 'ospf3'
   args[ 'PREFIX' ] = args.get( 'PREFIX', None )
   args[ 'TAG' ] = None
   args[ 'shamlink' ] = True
   return IpRibCliShow.showIpRoute( mode, args )
# --------------------------------------------------------------------------------
# show ip ospf sham-link [vrf] route
# --------------------------------------------------------------------------------

class ShowIpOspfShamLinkRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf sham-link [ VRF ] route [ PREFIX ]'
   data = { 'ip' : ipMatcherForShow,
            'ospf' : matcherOspfShow,
            'sham-link' : 'Status for sham link',
            'VRF' : vrfExprFactory,
            'route' : 'Routes with nexthop as sham link',
            'PREFIX' : prefixMatcher,
          }
   handler = showIpOspfShamLinkRoute
   cliModel = IpRibModel.RibRoutesByVrf

if toggleOspfShamLinkEnabled():
   BasicCli.addShowCommandClass( ShowIpOspfShamLinkRouteCmd )

# --------------------------------------------------------------------------------
# show ospfv3 ipv6 sham-link [vrf] route
# --------------------------------------------------------------------------------

class ShowIpOspf3ShamLinkRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 ipv6 sham-link [ VRF ] route [ PREFIX ]'
   data = {
            'ospfv3' : matcherOspfv3AfterShow,
            'ipv6' : matcherAfIpv6AfterShow,
            'sham-link' : 'Status for sham link',
            'VRF' : vrfExprFactory,
            'route' : 'Routes with nexthop as sham link',
            'PREFIX' : matcherPrefix
          }
   handler = showIpOspfShamLinkRoute
   cliModel = IpRibModel.RibRoutesByVrf

if toggleOspf3ShamLinkEnabled():
   BasicCli.addShowCommandClass( ShowIpOspf3ShamLinkRouteCmd )

#--------------------------------------------------------------------------------
# show rib next-hop ( ip | ipv6 ) dependency
#--------------------------------------------------------------------------------
class ShowNhDependencyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib next-hop ( ip | ipv6 ) dependency'
   data = { 'rib' : matcherRib,
            'next-hop' : matcherNextHop,
            'ip' : ipMatcherForShow,
            'ipv6' : ipv6MatcherForShow,
            'dependency' : 'Recursive dependencies of next-hops',
   }

   handler = IpRibCliShow.showNhDependency
   cliModel = IpRibModel.RecursiveNhDependencyByVrf

BasicCli.addShowCommandClass( ShowNhDependencyCmd )

# Using separate handler method instead of directly referencing
# showIpRoute in order to be able to stub showIpRoute in breadth test.
def ribRouteIpCmdHandler( mode, args ):
   return IpRibCliShow.showIpRoute( mode, args )

#--------------------------------------------------------------------------------
# show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#    [ fib policy excluded ] [ orr-position POSITION ] [ debug ]
#--------------------------------------------------------------------------------
class RibRouteIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
               [ fib policy excluded ] [ orr-position POSITION ]'''
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PREFIX' : matcherPrefix,
      'PROTOCOL' : matcherProtocolOrHost,
      'fib' : 'Match FIB criteria',
      'policy' : 'Match FIB policy',
      'excluded' : 'Show only routes excluded from FIB',
      'orr-position' : matcherOrr,
      'POSITION' : matcherOrrPosition,
   }

   if toggleShowRibRouteDetailEnabled():
      syntax += ' [ debug ]'
      data[ 'debug' ] = matcherDebug

   handler = ribRouteIpCmdHandler
   cliModel = IpRibModel.RibRoutesByVrf

BasicCli.addShowCommandClass( RibRouteIpCmd )

#--------------------------------------------------------------------------------
# show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#    [ fib policy excluded ] tag ( TAG )
#--------------------------------------------------------------------------------
tagValueMatcher = CliMatcher.IntegerMatcher( 0, 4294967295, helpdesc="Tag value to "
                                             "filter the routes with" )

class RibRouteIpCmdWithTagFiltering( ShowCommand.ShowCliCommandClass ):
   syntax = '''show rib route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
               [ fib policy excluded ] tag ( TAG )'''
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PREFIX' : matcherPrefix,
      'PROTOCOL' : matcherProtocolOrHost,
      'fib' : 'Match FIB criteria',
      'policy' : 'Match FIB policy',
      'excluded' : 'Show only routes excluded from FIB',
      'tag' : 'Filter the routes by tag',
      'TAG' : tagValueMatcher,
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByVrf

BasicCli.addShowCommandClass( RibRouteIpCmdWithTagFiltering )

#--------------------------------------------------------------------------------
# show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ] [ PROTOCOL ]
#--------------------------------------------------------------------------------
class RibRouteIpMulticastCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show rib multicast route ( ip | ipv6 ) [ VRF ] [ PREFIX ]
               [ PROTOCOL ]'''
   data = {
      'rib' : matcherRib,
      'multicast' : matcherMulticast,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PREFIX' : matcherPrefix,
      'PROTOCOL' : matcherProtocolOrHost,
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByVrf

BasicCli.addShowCommandClass( RibRouteIpMulticastCmd )

#--------------------------------------------------------------------------------
# show rib route ( ip | ipv6 ) internal [ VRF ]
#--------------------------------------------------------------------------------
class RibRouteIpInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route ( ip | ipv6 ) internal [ VRF ]'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'internal' : Node( matcher=matcherInternal, hidden=True ),
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showIpRoute
   cliModel = IpRibModel.RibRoutesByVrf

BasicCli.addShowCommandClass( RibRouteIpInternalCmd )

#--------------------------------------------------------------------------------
# show rib route summary [ ip | ipv6 ] [ VRF ]
#--------------------------------------------------------------------------------
class ShowRibRouteSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route summary [ ip | ipv6 ] [ VRF ]'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'summary' : matcherSummary,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showRibRouteSummary
   cliModel = IpRibModel.RibRouteSummary

BasicCli.addShowCommandClass( ShowRibRouteSummary )

#--------------------------------------------------------------------------------
# show rib route summary [ ip | ipv6 ] brief
#--------------------------------------------------------------------------------
class ShowRibRouteSummaryBrief( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib route summary [ ip | ipv6 ] brief'
   data = {
      'rib' : matcherRib,
      'route' : matcherRoute,
      'summary' : matcherSummary,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'brief' : matcherBrief,
   }

   handler = IpRibCliShow.showRibRouteSummaryBrief
   cliModel = IpRibModel.RibRouteSummaryForVrf

BasicCli.addShowCommandClass( ShowRibRouteSummaryBrief )

#--------------------------------------------------------------------------------
# show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]
#--------------------------------------------------------------------------------
class RibLoopIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib [ multicast ] loop ( ip | ipv6 ) [ VRF ] route [ PREFIX ]'
   data = {
      'rib' : matcherRib,
      'loop' : matcherLoop,
      'multicast' : matcherMulticast,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'route' : matcherLoopRoute,
      'PREFIX' : matcherPrefix,
   }

   handler = IpRibCliShow.showIpRouteLooped
   cliModel = IpRibModel.LoopedRoutes

BasicCli.addShowCommandClass( RibLoopIpRouteCmd )

#--------------------------------------------------------------------------------
# show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]
#--------------------------------------------------------------------------------
class RibNextHopResolutionRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib next-hop resolution route ( ipv4 | ipv6 ) [ VRF ]'
   data = {
      'rib' : matcherRib,
      'next-hop' : matcherNextHop,
      'resolution' : matcherResolution,
      'route' : matcherResolutionRoute,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
   }

   handler = IpRibCliShow.showResolutionRoutes
   cliModel = IpRibModel.ResolutionRoutes

BasicCli.addShowCommandClass( RibNextHopResolutionRouteCmd )

#--------------------------------------------------------------------------------
# show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]
#    [ orr-position POSITION ]
#--------------------------------------------------------------------------------
class RibNextHopIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib [ multicast ] next-hop ( ip | ipv6 ) [ VRF ] [ PROTOCOL ]'
   syntax += ' [ orr-position POSITION ]'
   syntax += ' [ detail ]'

   data = {
      'rib' : matcherRib,
      'multicast' : matcherMulticast,
      'next-hop' : matcherNextHop,
      'ip' : ipMatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'VRF' : matcherVrf,
      'PROTOCOL' : matcherProtocolOrHost,
      'orr-position' : matcherOrr,
      'POSITION' : matcherOrrPosition,
      'detail' : 'detail',
   }

   handler = IpRibCliShow.showIpRibNextHop
   cliModel = IpRibModel.RibNextHopsByProtocol

BasicCli.addShowCommandClass( RibNextHopIpCmd )

class RibNextHopEvpnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib next-hop evpn [ VRF ] [ PROTOCOL ]'
   data = {
      'rib' : matcherRib,
      'next-hop' : matcherNextHop,
      'evpn' : matcherEvpn,
      'VRF' : matcherVrf,
      'PROTOCOL' : matcherEvpnProtocol,
   }

   handler = IpRibCliShow.showIpRibNextHop
   cliModel = IpRibModel.RibNextHopsByProtocol

if IpRibToggleLib.toggleShowIpRibEvpnNextHopEnabled():
   BasicCli.addShowCommandClass( RibNextHopEvpnCmd )


#--------------------------------------------------------------------------------
# show vrf leak connected ( ipv4 | ipv6 )
#--------------------------------------------------------------------------------
class VrfLeakConnectedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vrf leak connected ( ipv4 | ipv6 )'
   data = {
      'vrf' : vrfKwForShow,
      'leak' : matcherLeak,
      'connected' : matcherConnected,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
   }

   handler = IpRibCliShow.showVrfLeakConnected
   cliModel = LeakedConnectedRoutesByVrf

BasicCli.addShowCommandClass( VrfLeakConnectedCmd )

# -----------------------------------------------------------------------------------
# show tech-support iprib [ { graceful-restart | optimal-route-reflection } ]
# show tech-support iprib [ VRF ] [ PROTOCOL ] CATEGORIES [ verbose ] [ limit NUM ]
# -----------------------------------------------------------------------------------
class ShowTechIpRibSubCmd( ShowCommand.ShowCliCommandClass ):

   if IpRibToggleLib.toggleShowTechIpRibUpdateEnabled():
      syntax = '''show tech-support iprib [ VRF ] [ PROTOCOL ] \
            CATEGORIES [ verbose NUM ]'''
      data = {
         'tech-support' : techSupportKwMatcher,
         'iprib' : 'Show aggregated status details for IpRib',
         'CATEGORIES' : CliCommand.SetEnumMatcher( {
            'graceful-restart' : 'Graceful restart information for IpRib agent',
            'optimal-route-reflection' : 'Show BGP ORR related information',
            'via-config' : 'Show Via Configs',
            'via-set-config' : 'Show dump of ViaSetConfigByProtocol',
            'via-set-status' : 'Show dump of ViaSetStatusByProtocol',
            'via-status' : 'Show Via Status related information',
            'resolved-via' : 'Show Resolved Via information',
         } ),
         'VRF' : matcherVrfDefault,
         'PROTOCOL' : matcherProtocolOrHost,
         'verbose' : 'set custom limit on entries printed for collections',
         'NUM' : CliMatcher.IntegerMatcher( 1, 10000000,
                            helpdesc='max entries printed for collection' ),
      }
   else:
      syntax = '''show tech-support iprib 
                  ( graceful-restart | optimal-route-reflection )'''
      data = {
            'tech-support' : techSupportKwMatcher,
            'iprib' : 'Show aggregated status details for IpRib',
            'graceful-restart' : 'Graceful restart information for IpRib agent',
            'optimal-route-reflection' : 'Show BGP ORR related information',
      }

   privileged = True
   handler = IpRibCliShow.showTechIpRib

BasicCli.addShowCommandClass( ShowTechIpRibSubCmd )

# --------------------------------------------------------
# show agent IpRib debug ( dynamic-endpoints | unicast-rib )
# [ verbose NUM ]
# --------------------------------------------------------
class ShowAgentDebugFeatureCmd( ShowCommand.ShowCliCommandClass ):
   # TODO: Update /src/Umbrella/test/allowedShowCmds.dat accordingly
   # when removing these toggles.
   if toggleDynamicSrTeEnabled() and toggleIsisConditionalAdvertiseEnabled():
      syntax = '''show agent IpRib debug ( dynamic-endpoints | unicast-rib )
                  [ verbose NUM ]'''
   elif toggleDynamicSrTeEnabled():
      syntax = 'show agent IpRib debug dynamic-endpoints [ verbose NUM ]'
   else:
      syntax = 'show agent IpRib debug unicast-rib [ verbose NUM ]'
   data = {
      'agent' : agentKwForShow,
      'IpRib' : 'Show aggregated status details for IpRib',
      'debug' : "Agent's debugging information",
      'dynamic-endpoints' : 'Show Dynamic Endpoints related information',
      'unicast-rib' : 'Show Custom Unicast RIB related information',
      'verbose' : 'set custom limit on entries printed for collections',
      'NUM' : CliMatcher.IntegerMatcher( 1, 10000000,
                        helpdesc='max entries printed for collection' ),
   }
   privileged = True
   handler = IpRibCliShow.showAgentDebugIpRib

if toggleDynamicSrTeEnabled() or toggleIsisConditionalAdvertiseEnabled():
   BasicCli.addShowCommandClass( ShowAgentDebugFeatureCmd )

#--------------------------------------------------------------------------------
# show rib ready [ VRF ]
#--------------------------------------------------------------------------------
class ShowRibReadyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show rib ready [ VRF ]'
   data = {
      'rib' : matcherRib,
      'ready' : 'Show protocols\' routes ready state',
      'VRF' : VrfExprFactory( helpdesc='Show routes in a VRF', inclAllVrf=True,
                              inclDefaultVrf=True )
   }

   handler = IpRibCliShow.showRibReady
   cliModel = IpRibCliShow.ribReadyVrfModel

BasicCli.addShowCommandClass( ShowRibReadyCmd )
