# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ShowCommand
import CliCommand
from CliMatcher import KeywordMatcher
from CliToken.Clear import clearKwNode
from CliToken.Flow import flowMatcherForClear, flowMatcherForShow
import Tac
import LauncherLib
import LazyMount
from CliPlugin import IpfixModel
from EcoLib import getJSONData, deleteData, ianaModel

agentConfigCliDir = None
daemonConfigDir = None

flowMatcher = flowMatcherForShow
telemetryMatcher = KeywordMatcher( 'telemetry',
                                   helpdesc='Flow telemetry information' )
ipfixMatcher = KeywordMatcher( 'ipfix',
                               helpdesc='Flow telemetry IPFIX information' )
sflowMatcher = KeywordMatcher( 'sflow',
                               helpdesc='Flow telemetry sFlow information' )
postcardMatcher = KeywordMatcher( 'postcard',
                               helpdesc='Flow telemetry postcard information' )

def getData( path ):
   return getJSONData( agentConfigCliDir, daemonConfigDir, path )

def clearData( path ):
   deleteData( agentConfigCliDir, daemonConfigDir, path )

def handleCounter( protocol ):
   allCounter = IpfixModel.IpfixAllCounter()
   j = getData( '/eco/%s/counter' % protocol )
   if j:
      allCounter.exporters = []
      allCounter.receivedMessages = j[ 'ReceivedMessages' ]
      allCounter.sentMessages = j[ 'SentMessages' ]
      if 'Exporters' not in j:
         return allCounter

      for exporter in j[ 'Exporters' ]:
         c = exporter[ 'Counters' ]
         exporterCounter = IpfixModel.IpfixExporterCounter()
         exporterCounter.receivedMessages = c[ 'ReceivedMessages' ]
         exporterCounter.receivedTrs = c[ 'ReceivedTRs' ]
         exporterCounter.receivedOtrs = c[ 'ReceivedOTRs' ]
         exporterCounter.receivedDrs = c[ 'ReceivedDRs' ]
         exporterCounter.receivedOdrs = c[ 'ReceivedODRs' ]
         exporterCounter.invalidMsgHdrErrors = c[ 'InvalidMsgHdrErrors' ]
         exporterCounter.invalidSetHdrErrors = c[ 'InvalidSetHdrErrors' ]
         exporterCounter.invalidSetLenErrors = c[ 'InvalidSetLenErrors' ]
         exporterCounter.invalidSetIdErrors = c[ 'InvalidSetIDErrors' ]
         exporterCounter.invalidTrErrors = c[ 'InvalidTRErrors' ]
         exporterCounter.invalidOtrErrors = c[ 'InvalidOTRErrors' ]
         exporterCounter.unknownTidErrors = c[ 'UnknownTIDErrors' ]
         exporterCounter.invalidFieldErrors = c[ 'InvalidFieldErrors' ]
         exporterCounter.interpretFieldErrors = c[ 'InterpretFieldErrors' ]
         exporterCounter.unsupportedOdrErrors = c[ 'UnsupportedODRErrors' ]
         exporterCounter.invalidFlowKeyErrors = c[ 'InvalidFlowKeyErrors' ]

         e = exporter[ 'Exporter' ]
         exporterId = IpfixModel.IpfixExporterId()
         exporterId.address = e[ 'Address' ]
         exporterId.sourcePort = e[ 'SourcePort' ]
         exporterId.observationDomainId = e[ 'ObservationDomainID' ]
         exporterId.protocol = protocol
         exporterCounter.exporterId = exporterId

         allCounter.exporters.append( exporterCounter )

   return allCounter

# ------------------------------------------------------
# show flow telemetry ipfix counters
# ------------------------------------------------------
class ShowCounter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry ipfix counters'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'counters': 'Show flow telemetry IPFIX counters' }
   cliModel = IpfixModel.IpfixAllCounter

   @staticmethod
   def handler( mode, args ):
      return handleCounter( 'ipfix' )

BasicCli.addShowCommandClass( ShowCounter )

# ------------------------------------------------------
# show flow telemetry ipfix template
# ------------------------------------------------------
class ShowTemplate( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry ipfix template'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'template': 'Show flow telemetry IPFIX template' }
   cliModel = IpfixModel.IpfixAllTemplate

   @staticmethod
   def handler( mode, args ):
      allTemplate = IpfixModel.IpfixAllTemplate()
      j = getData( '/eco/ipfix/template' )
      if j and 'Exporters' in j:
         allTemplate.exporters = []

         for exporter in j[ 'Exporters' ]:
            exporterTemplate = IpfixModel.IpfixExporterTemplate()
            exporterTemplate.templateInfos = []

            t = exporter.get( 'Templates' )
            if not t:
               continue

            for template in exporter[ 'Templates' ]:
               templateInfo = IpfixModel.IpfixTemplateInfo()
               templateInfo.templateId = template[ 'TemplateID' ]
               templateInfo.fields = []
               templateInfo.scopeFields = []
               templateInfo.timeReceived = template[ 'TimeReceived' ]
               if 'ScopeFields' in template:
                  for f in template[ 'ScopeFields' ]:
                     field = IpfixModel.IpfixTemplateField()
                     field.enterpriseNumber = f[ 'EnterpriseNumber' ]
                     field.informationElementId = f[ 'InformationElementID' ]
                     field.informationElementName = f[ 'InformationElementName' ]
                     field.length = f[ 'Length' ]
                     templateInfo.scopeFields.append( field )
               if 'Fields' in template:
                  for f in template[ 'Fields' ]:
                     field = IpfixModel.IpfixTemplateField()
                     field.enterpriseNumber = f[ 'EnterpriseNumber' ]
                     field.informationElementId = f[ 'InformationElementID' ]
                     field.informationElementName = f[ 'InformationElementName' ]
                     field.length = f[ 'Length' ]
                     templateInfo.fields.append( field )
               exporterTemplate.templateInfos.append( templateInfo )

            e = exporter[ 'Exporter' ]
            exporterId = IpfixModel.IpfixExporterId()
            exporterId.address = e[ 'Address' ]
            exporterId.sourcePort = e[ 'SourcePort' ]
            exporterId.observationDomainId = e[ 'ObservationDomainID' ]
            exporterId.protocol = 'ipfix'
            exporterTemplate.exporterId = exporterId

            allTemplate.exporters.append( exporterTemplate )

      return allTemplate

BasicCli.addShowCommandClass( ShowTemplate )

# ------------------------------------------------------
# show flow telemetry ipfix option-data
# ------------------------------------------------------
class ShowOption( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry ipfix option-data'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'option-data': 'Show flow telemetry IPFIX option data' }
   cliModel = IpfixModel.IpfixAllOptionData

   @staticmethod
   def handler( mode, args ):
      allOption = IpfixModel.IpfixAllOptionData()
      j = getData( '/eco/ipfix/option' )
      if j and 'Exporters' in j:
         allOption.exporters = []

         for exporter in j[ 'Exporters' ]:
            exporterOption = IpfixModel.IpfixExporterOptionData()

            interfaces = exporter.get( 'InterfaceData' )
            if interfaces:
               for intfId, value in interfaces.items():
                  exporterOption.interfaces[ intfId ] = value[ 0 ][ 'Value' ]

            vrfs = exporter.get( 'VRFData' )
            if vrfs:
               for vrfId, value in vrfs.items():
                  exporterOption.vrfs[ vrfId ] = value[ 0 ][ 'Value' ]

            templates = exporter.get( 'TemplateData' )
            if templates:
               for templateId, value in templates.items():
                  dataFields = IpfixModel.IpfixDataFields()
                  for field in value:
                     dataFields.fields[ field[ 'Name' ] ] = str( field[ 'Value' ] )
                  exporterOption.templates[ templateId ] = dataFields

            observationDomains = exporter.get( 'ObsDomainData' )
            if observationDomains:
               for obsDomainId, value in observationDomains.items():
                  dataFields = IpfixModel.IpfixDataFields()
                  for field in value:
                     dataFields.fields[ field[ 'Name' ] ] = str( field[ 'Value' ] )
                  exporterOption.observationDomains[ obsDomainId ] = dataFields

            e = exporter[ 'Exporter' ]
            exporterId = IpfixModel.IpfixExporterId()
            exporterId.address = e[ 'Address' ]
            exporterId.sourcePort = e[ 'SourcePort' ]
            exporterId.observationDomainId = e[ 'ObservationDomainID' ]
            exporterId.protocol = 'ipfix'
            exporterOption.exporterId = exporterId

            allOption.exporters.append( exporterOption )

      return allOption

BasicCli.addShowCommandClass( ShowOption )

# pylint: disable=too-many-nested-blocks
def handleFlow( protocol, showHostname ):
   allFlow = IpfixModel.IpfixAllFlow()
   url = '/eco/%s/flow' % protocol
   url = url + '?hostname' if showHostname else url
   j = getData( url )
   if j and 'Exporters' in j:
      allFlow.exporters = []

      for exporter in j[ 'Exporters' ]:
         exporterFlow = IpfixModel.IpfixExporterFlow()
         exporterFlow.v4Flows = []
         exporterFlow.v6Flows = []

         e = exporter[ 'Exporter' ]
         exporterId = IpfixModel.IpfixExporterId()
         exporterId.address = e[ 'Address' ]
         exporterId.sourcePort = e[ 'SourcePort' ]
         exporterId.observationDomainId = e[ 'ObservationDomainID' ]
         exporterId.protocol = protocol
         exporterFlow.exporterId = exporterId

         for flowRecord in exporter[ 'Flow' ]:
            flow = IpfixModel.IpfixFlow()
            for k, v in flowRecord[ 'FlowKey' ].items():
               k = ianaModel.get( k, k )
               flow.flowKey[ k ] = str( v )
            for k, v in flowRecord[ 'FlowData' ].items():
               # process INT node data, which is a list of dict
               if k == 'intNodes':
                  flow.intNodes = []
                  for node in v:
                     intNode = IpfixModel.IpfixIntNode()
                     for k1, v1 in node.items():
                        # process INT interval data, which is a list of dict
                        if k1 == 'intIntervals':
                           intNode.intIntervals = []
                           for interval in v1:
                              intInterval = IpfixModel.IpfixIntInterval()
                              for k2, v2 in interval.items():
                                 k2 = ianaModel.get( k2, k2 )
                                 intInterval.intervalData[ k2 ] = str( v2 )
                              intNode.intIntervals.append( intInterval )
                        else:
                           k1 = ianaModel.get( k1, k1 )
                           intNode.nodeData[ k1 ] = str( v1 )
                     flow.intNodes.append( intNode )
               else:
                  k = ianaModel.get( k, k )
                  flow.flowData[ k ] = str( v )
            flow.lastUpdateTime = flowRecord[ 'LastUpdateTime' ]
            if ( IpfixModel.SRCIP6 in flow.flowKey or
                 IpfixModel.DSTIP6 in flow.flowKey or
                 IpfixModel.SRCIP6 in flow.flowData or
                 IpfixModel.DSTIP6 in flow.flowData ):
               exporterFlow.v6Flows.append( flow )
            else:
               exporterFlow.v4Flows.append( flow )

         allFlow.exporters.append( exporterFlow )

   return allFlow

# ------------------------------------------------------
# show flow telemetry ipfix flow-table
# ------------------------------------------------------
class ShowFlow( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry ipfix flow-table [ hostnames ]'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'flow-table': 'Show flow telemetry IPFIX flow records',
            'hostnames': 'Show flow telemetry IPFIX flow records with hostnames' }
   cliModel = IpfixModel.IpfixAllFlow

   @staticmethod
   def handler( mode, args ):
      return handleFlow( 'ipfix', 'hostnames' in args )

BasicCli.addShowCommandClass( ShowFlow )

def handleHostname( protocol ):
   allHostname = IpfixModel.IpfixAllHostname()
   j = getData( '/eco/%s/dns' % protocol )
   if j and 'Record' in j:
      allHostname.v4Hostnames = []
      allHostname.v6Hostnames = []
      for record in j[ 'Record' ]:
         hostname = IpfixModel.IpfixHostname()
         hostname.address = record[ 'IP' ]
         hostname.hostnames = record[ 'Name' ]
         hostname.lastUpdateTime = record[ 'LastUpdateTime' ]
         if ':' in record[ 'IP' ]:
            allHostname.v6Hostnames.append( hostname )
         else:
            allHostname.v4Hostnames.append( hostname )

   return allHostname

# ------------------------------------------------------
# show flow telemetry ipfix hostname
# ------------------------------------------------------
class ShowHostname( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry ipfix hostname'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'hostname': 'Show flow telemetry IPFIX address resolution' }
   cliModel = IpfixModel.IpfixAllHostname

   @staticmethod
   def handler( mode, args ):
      return handleHostname( 'ipfix' )

BasicCli.addShowCommandClass( ShowHostname )

# ------------------------------------------------------
# clear flow telemetry ipfix counter
# ------------------------------------------------------
class ClearCounter( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry ipfix counter'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'counter': 'Clear flow telemetry IPFIX counter' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/ipfix/counter' )

BasicCli.EnableMode.addCommandClass( ClearCounter )

# ------------------------------------------------------
# clear flow telemetry ipfix flow-table
# ------------------------------------------------------
class ClearFlowTable( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry ipfix flow-table'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'flow-table': 'Clear flow telemetry IPFIX flow table' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/ipfix/flow' )

BasicCli.EnableMode.addCommandClass( ClearFlowTable )

# ------------------------------------------------------
# clear flow telemetry ipfix hostname
# ------------------------------------------------------
class ClearHostname( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry ipfix hostname'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'ipfix': ipfixMatcher,
            'hostname': 'Clear flow telemetry IPFIX address resolution' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/ipfix/dns' )

BasicCli.EnableMode.addCommandClass( ClearHostname )

# ------------------------------------------------------
# show flow telemetry sflow counters
# ------------------------------------------------------
class ShowSflowCounter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry sflow counters'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'counters': 'Show flow telemetry sFlow counters' }
   cliModel = IpfixModel.IpfixAllCounter

   @staticmethod
   def handler( mode, args ):
      return handleCounter( 'sflow' )

BasicCli.addShowCommandClass( ShowSflowCounter )

# ------------------------------------------------------
# show flow telemetry sflow flow-table
# ------------------------------------------------------
class ShowSflowFlow( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry sflow flow-table [hostnames]'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'flow-table': 'Show flow telemetry sFlow flow records',
            'hostnames': 'Show flow telemetry sFlow flow records with hostnames' }
   cliModel = IpfixModel.IpfixAllFlow

   @staticmethod
   def handler( mode, args ):
      return handleFlow( 'sflow', 'hostnames' in args )

BasicCli.addShowCommandClass( ShowSflowFlow )

# ------------------------------------------------------
# show flow telemetry sflow hostname
# ------------------------------------------------------
class ShowSflowHostname( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry sflow hostname'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'hostname': 'Show flow telemetry sFlow address resolution' }
   cliModel = IpfixModel.IpfixAllHostname

   @staticmethod
   def handler( mode, args ):
      return handleHostname( 'sflow' )

BasicCli.addShowCommandClass( ShowSflowHostname )

def handleOptionData( protocol ):
   allOption = IpfixModel.IpfixAllOptionData()
   j = getData( '/eco/%s/option' % protocol )
   if j and 'InterfaceData' in j:
      exporterOption = IpfixModel.IpfixExporterOptionData()
      for intfId, value in j[ 'InterfaceData' ].items():
         exporterOption.interfaces[ intfId ] = value

      exporterId = IpfixModel.IpfixExporterId()
      exporterId.address = '127.0.0.1'
      exporterId.sourcePort = 0
      exporterId.observationDomainId = 0
      exporterId.protocol = protocol
      exporterOption.exporterId = exporterId

      allOption.exporters = [ exporterOption ]

   return allOption

# ------------------------------------------------------
# show flow telemetry sflow option-data
# ------------------------------------------------------
class ShowSflowOption( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry sflow option-data'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'option-data': 'Show flow telemetry sFlow option data' }
   cliModel = IpfixModel.IpfixAllOptionData

   @staticmethod
   def handler( mode, args ):
      return handleOptionData( 'sflow' )

BasicCli.addShowCommandClass( ShowSflowOption )

# ------------------------------------------------------
# clear flow telemetry sflow counter
# ------------------------------------------------------
class ClearSflowCounter( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry sflow counter'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'counter': 'Clear flow telemetry sFlow counter' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/sflow/counter' )

BasicCli.EnableMode.addCommandClass( ClearSflowCounter )

# ------------------------------------------------------
# clear flow telemetry sflow flow-table
# ------------------------------------------------------
class ClearSflowTable( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry sflow flow-table'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'flow-table': 'Clear flow telemetry sFlow flow table' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/sflow/flow' )

BasicCli.EnableMode.addCommandClass( ClearSflowTable )

# ------------------------------------------------------
# clear flow telemetry sflow hostname
# ------------------------------------------------------
class ClearSflowHostname( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry sflow hostname'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'sflow': sflowMatcher,
            'hostname': 'Clear flow telemetry sFlow address resolution' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/sflow/dns' )

BasicCli.EnableMode.addCommandClass( ClearSflowHostname )

# ------------------------------------------------------
# show flow telemetry postcard counters
# ------------------------------------------------------
class ShowPostcardCounter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry postcard counters'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'counters': 'Show flow telemetry postcard counters' }
   cliModel = IpfixModel.IpfixAllCounter

   @staticmethod
   def handler( mode, args ):
      return handleCounter( 'greent' )

BasicCli.addShowCommandClass( ShowPostcardCounter )

# ------------------------------------------------------
# show flow telemetry postcard flow-table
# ------------------------------------------------------
class ShowPostcardFlow( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry postcard flow-table [hostnames]'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'flow-table': 'Show flow telemetry postcard flow records',
            'hostnames': 'Show flow telemetry postcard flow records with hostnames' }
   cliModel = IpfixModel.IpfixAllFlow

   @staticmethod
   def handler( mode, args ):
      return handleFlow( 'greent', 'hostnames' in args )

BasicCli.addShowCommandClass( ShowPostcardFlow )

# ------------------------------------------------------
# show flow telemetry postcard hostname
# ------------------------------------------------------
class ShowPostcardHostname( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry postcard hostname'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'hostname': 'Show flow telemetry postcard address resolution' }
   cliModel = IpfixModel.IpfixAllHostname

   @staticmethod
   def handler( mode, args ):
      return handleHostname( 'greent' )

BasicCli.addShowCommandClass( ShowPostcardHostname )

# ------------------------------------------------------
# show flow telemetry postcard option-data
# ------------------------------------------------------
class ShowPostcardOption( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow telemetry postcard option-data'
   data = { 'flow': flowMatcher,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'option-data': 'Show flow telemetry postcard option data' }
   cliModel = IpfixModel.IpfixAllOptionData

   @staticmethod
   def handler( mode, args ):
      return handleOptionData( 'greent' )

BasicCli.addShowCommandClass( ShowPostcardOption )

# ------------------------------------------------------
# clear flow telemetry postcard counter
# ------------------------------------------------------
class ClearPostcardCounter( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry postcard counter'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'counter': 'Clear flow telemetry postcard counter' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/greent/counter' )

BasicCli.EnableMode.addCommandClass( ClearPostcardCounter )

# ------------------------------------------------------
# clear flow telemetry postcard flow-table
# ------------------------------------------------------
class ClearPostcardFlowTable( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry postcard flow-table'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'flow-table': 'Clear flow telemetry postcard flow table' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/greent/flow' )

BasicCli.EnableMode.addCommandClass( ClearPostcardFlowTable )

# ------------------------------------------------------
# clear flow telemetry postcard hostname
# ------------------------------------------------------
class ClearPostcardHostname( CliCommand.CliCommandClass ):
   syntax = 'clear flow telemetry postcard hostname'
   data = { 'clear': clearKwNode,
            'flow': flowMatcherForClear,
            'telemetry': telemetryMatcher,
            'postcard': postcardMatcher,
            'hostname': 'Clear flow telemetry postcard address resolution' }

   @staticmethod
   def handler( mode, args ):
      clearData( '/eco/greent/dns' )

BasicCli.EnableMode.addCommandClass( ClearPostcardHostname )

def Plugin( entityManager ):
   global agentConfigCliDir, daemonConfigDir

   agentConfigCliDir = LazyMount.mount( entityManager,
                                        LauncherLib.agentConfigCliDirPath,
                                        "Launcher::AgentConfigDir",
                                        "ri" )
   daemonConfigDir = LazyMount.mount( entityManager,
                                      "daemon/agent/config",
                                      "Tac::Dir",
                                      "ri" )
