#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import Arnet
import BasicCliModes
import Toggles.TunnelToggleLib
import CliCommand
import CliMatcher
from CliToken.Test import testMatcherForExec
from CliToken.Community import CommunityConstExpr
from CliPlugin import IntfCli
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpGenAddrMatcher
from CliPlugin import IraRouteCommon
from CliPlugin import VrfCli
from CliPlugin import ConfigTagCommon
from CliPlugin.IntfCli import Intf
from CliPlugin.IraCommonModel import IpRouteSimpleVia, AllRoutes
from CliPlugin.IraCommonModel import VrfNotInstalledBfd, NotInstalledBfd, ViaList
import CliExtensions
import ConfigMount
import LazyMount
import RoutingConsts
from CliModel import UnknownEntityError
from CEosHelper import isCeosLab
from TypeFuture import TacLazyType

TunnelPayloadType = TacLazyType( "L3::Tunnel::PayloadType" )
UdpPortConfigMode = TacLazyType( "L3::Tunnel::UdpPortConfigMode" )
UdpDestPortConfigMode = TacLazyType( "L3::Tunnel::UdpDestinationPortAndConfigMode" )

udpDestPortConfig = None
allIntfStatusDir = None
ipConfig = None
ip6Config = None
vrfToIpAddrMapBuilderSm = None

# Config Tag mounts
configTagInput = None
configTagConfig = None
configTagIdState = None
configTagIpv4Config = None
configTagIpv6Config = None
configTagIpv4DynamicConfig = None
configTagIpv6DynamicConfig = None

# Maximum amount of communities in the community related Ira commands
COMMUNITY_LIMIT = 4
IPV6_MIN_MTU = 1280

@Tac.memoize
def configTagInputAllocator():
   return Tac.newInstance( "ConfigTag::ConfigTagInputAllocator" )

IpGenAddr = Tac.Type( 'Arnet::IpGenAddr' )
AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
CommandTagId = Tac.Type( "ConfigTag::ConfigTagId" )

fibKw = CliMatcher.KeywordMatcher( 'fib', helpdesc='Routing table' )
fibMatcher = fibKw

resilienceMatcher = CliMatcher.KeywordMatcher( 'resilience' , \
                                   helpdesc='Configure Resilience in ECMP routes' )
resilientEcmpCapacityMatcher = CliMatcher.KeywordMatcher( 'capacity', \
                                  helpdesc='specify capacity value' )
resilientEcmpRedundancyMatcher = CliMatcher.KeywordMatcher( 'redundancy', \
                                  helpdesc='specify redundancy value ' )
prefixLengthMatcher = CliMatcher.KeywordMatcher( 'prefix-length',
                                          helpdesc='Length of the prefix in bits' )
redundantSpecificMatcher = CliMatcher.KeywordMatcher( 'redundant-specifics',
                              helpdesc="type of route filter to use" )
filterMatcher = CliMatcher.KeywordMatcher( 'filter',
                          helpdesc="filter command" )
notInstalledMatcher = CliMatcher.KeywordMatcher( 'not-installed',
                                           helpdesc='Show not installed routes' )
bfdMatcher = CliMatcher.KeywordMatcher( 'bfd',
                                  helpdesc='Not installed routes tracked by Bfd' )
nullIntfKwMatcher = CliMatcher.KeywordMatcher( 'Null0',
                           helpdesc='Interface that drops all traffic' )

# route option config matchers
localAddrMatcherForConfig = CliMatcher.KeywordMatcher( 'local-address',
   helpdesc='Set local address for BFD session' )
localAddrMatcherDescr = 'Local interface address used for the BFD session'
metricMatcherForConfig = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Metric for this route' )
metricValueMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
      helpdesc='Value of the route metric' )
weightMatcherForConfig = CliMatcher.KeywordMatcher( 'weight',
      helpdesc='UCMP weight for the next hop' )
# weight max value needs to be limited to (2^24 -1) due to internal data casting
weightValueMatcher = CliMatcher.IntegerMatcher( 1, 16777215,
      helpdesc='Value of the UCMP weight for the next hop' )
nameMatcherForConfig = CliMatcher.KeywordMatcher( 'name',
                                                  helpdesc='Next hop name' )
nexthopNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9._-]+',
                                                helpname='WORD',
                                              helpdesc='String naming the next hop' )
preferenceRangeMatcher = CliMatcher.IntegerMatcher( 1, 255,
      helpdesc='Administrative distance for this route' )
tagMatcherForConfig = CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' )
tagNumberMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
                                              helpdesc='Tag number' )
communityMatcher = CliMatcher.KeywordMatcher( 'community',
                   helpdesc='BGP communities' )
trackMatcherForConfig = CliMatcher.KeywordMatcher( 'track',
                                                   helpdesc='Track this route' )

matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Configure tunnel settings' )

matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Configure tunnel type' )

class Null0Expr( CliCommand.CliExpression ):
   expression = 'Null0 | ( Null 0 )'
   data = { 'Null0': nullIntfKwMatcher,
            'Null': CliCommand.Node( CliMatcher.KeywordMatcher( 'Null',
                                        helpdesc='drops' ),
                                     hidden=True ),
            '0': CliCommand.Node( CliMatcher.KeywordMatcher( '0',
                                        helpdesc='drops' ),
                                     hidden=True ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'Null' in args and '0' in args:
         del args[ 'Null' ]
         del args[ '0' ]
         args[ 'Null 0' ] = 'Null 0'

leakMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'leak',
                                 helpdesc='additional output VRF for this route' ),
                                 hidden=True )
leakVrfMatcherForConfig = CliMatcher.KeywordMatcher( 'vrf',
                                 helpdesc='additional output VRF for this route' )
egressVrfMatcher = CliMatcher.KeywordMatcher( 'egress-vrf',
                                 helpdesc='egress VRF for this route' )

class LocalAddressExprFactory( CliCommand.CliExpressionFactory ):
   '''TrackRouteOptionExpr can optionally include a LocalAddressExpr.
      See the help string for TrackRouteOptionExprFactory to see how
      to pass in the kwargs for this Factory.
   '''
   def __init__( self, includeIp4=False, includeIp6=False, ip6Guard=None,
                 ip6IsHidden=False ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.includeIp4 = includeIp4
      self.includeIp6 = includeIp6
      self.ip6Guard = ip6Guard
      self.ip6IsHidden = ip6IsHidden

   def generate( self, name ):
      exprs = {
         ( True, True ): '( ADDR4 | ADDR6 )',
         ( True, False ): 'ADDR4',
         ( False, True ): 'ADDR6',
      }
      expr = 'local-address ' + exprs[ ( self.includeIp4, self.includeIp6 ) ]
      options = {
         'local-address': CliCommand.singleNode( localAddrMatcherForConfig ),
      }
      if self.includeIp4:
         options[ 'ADDR4' ] = CliCommand.singleNode(
            IpAddrMatcher.IpAddrMatcher( localAddrMatcherDescr ) )
      if self.includeIp6:
         options[ 'ADDR6' ] = CliCommand.singleNode(
            Ip6AddrMatcher.Ip6AddrMatcher( localAddrMatcherDescr ),
            guard=self.ip6Guard,
            hidden=self.ip6IsHidden )

      class LocalAddressExpr( CliCommand.CliExpression ):
         expression = expr
         data = options

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'OPTION' not in args:
               args[ 'OPTION' ] = {}
            if 'local-address' not in args:
               return
            localAddress = {}
            args[ 'OPTION' ][ 'localAddress' ] = localAddress
            if 'ADDR4' in args:
               localAddress[ 'localAddr4' ] = args.pop( 'ADDR4' )
            if 'ADDR6' in args:
               localAddress[ 'localAddr6' ] = args.pop( 'ADDR6' )

      return LocalAddressExpr

class TrackingProtocolExprFactory( CliCommand.CliExpressionFactory ):
   '''This generates the expression for the tracking protocols from the
      per-address family dictionaries for trackingProtocols and
      trackingProtocolsHidden. Currently, the only tracking protocol is 'bfd'.
   '''
   def __init__( self, trackingProtocols, trackingProtocolsHidden ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.trackingProtocols = trackingProtocols
      self.trackingProtocolsHidden = trackingProtocolsHidden

   def generate( self, name ):
      class TrackingProtocolExpr( CliCommand.CliExpression ):
         expression = 'TRACKING_PROTOCOLS | HIDDEN_TRACKING_PROTOCOLS'
         data = {
            'TRACKING_PROTOCOLS': CliCommand.singleNode(
               CliMatcher.DynamicKeywordMatcher(
                  lambda mode: self.trackingProtocols ),
               alias='TRACKING_PROTO' ),
            'HIDDEN_TRACKING_PROTOCOLS': CliCommand.singleNode(
               CliMatcher.DynamicKeywordMatcher(
                  lambda mode: self.trackingProtocolsHidden ),
               alias='TRACKING_PROTO',
               hidden=True ),
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'OPTION' not in args:
               args[ 'OPTION' ] = {}
            if 'TRACKING_PROTO' not in args:
               return
            args[ 'OPTION' ][ 'trackingProto' ] = args.pop( 'TRACKING_PROTO' )

      return TrackingProtocolExpr

class TrackRouteOptionExprFactory( CliCommand.CliExpressionFactory ):
   '''This generates a TrackRouteOptionExpr.
      @trackingProtocols is the address family dependent trackingProtocols dict
      @trackingProtocolsHidden is the address family dependent
                               trackingProtocolsHidden dict
      @includeIp4 includes an IPv4 local-address address matcher
      @includeIp6 includes an IPv6 local-address address matcher
      @ip6Guard is a guard function on the IPv6 address
      @ip6IsHidden the IPv6 local-address address matcher should be hidden from
                   completion

      'ip route' can have either an IPv4 nexthop or IPv6 nexthop. The IPv6 nexthop
      is only available on supported platforms. Hence the need for the guard. In
      addtion, the syntax is currently hidden from completion.

      'ipv6 route' can only have an IPv6 nexthop.

      The factory is intended to be used from the RouteOptionsExprFactory.
      That class has a trackKwargs constructor arg. A call site that wishes
      to include the track syntax should call it like

      'ip route'
      trackKwargs={
         'trackingProtocols': trackingProtocols,
         'trackingProtocolsHidden': trackingProtocolsHidden,
         'includeIp4': True,
         'includeIp6': True,
         'ip6Guard': v4RouteWithV6NhSupportedGuard,
         'ip6IsHidden': True,
         }

      'ipv6 route'
      trackKwargs={
         'trackingProtocols': trackingProtocols,
         'trackingProtocolsHidden': trackingProtocolsHidden,
         'includeIp4': False,
         'includeIp6': True,
         'ip6Guard': None,
         'ip6IsHidden': False,
      }
   '''
   def __init__( self, trackingProtocols, trackingProtocolsHidden, includeIp4,
                 includeIp6, ip6Guard, ip6IsHidden ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.trackingProtocols = trackingProtocols
      self.trackingProtocolsHidden = trackingProtocolsHidden
      self.includeIp4 = includeIp4
      self.includeIp6 = includeIp6
      self.ip6Guard = ip6Guard
      self.ip6IsHidden = ip6IsHidden

   def generate( self, name ):
      class TrackRouteOptionExpr( CliCommand.CliExpression ):
         expression = 'track TRACKING_PROTO_EXPR [ LOCAL_ADDRESS_EXPR ]'
         data = {
            'track': CliCommand.singleNode( trackMatcherForConfig ),
            'TRACKING_PROTO_EXPR': TrackingProtocolExprFactory(
                                    self.trackingProtocols,
                                    self.trackingProtocolsHidden ),
            'LOCAL_ADDRESS_EXPR': LocalAddressExprFactory(
                                    self.includeIp4, self.includeIp6,
                                    self.ip6Guard, self.ip6IsHidden ),
         }

      return TrackRouteOptionExpr

class MetricRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a metric'''
   expression = 'metric METRICVALUE'
   data = {
      'metric': CliCommand.singleNode( metricMatcherForConfig ),
      'METRICVALUE': CliCommand.singleNode( metricValueMatcher ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'metric' in args:
         args[ 'OPTION' ][ 'metric' ] = args.pop( 'METRICVALUE' )

class WeightRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a weight'''
   expression = 'weight WEIGHTVALUE'
   data = {
      'weight': CliCommand.singleNode( weightMatcherForConfig ),
      'WEIGHTVALUE': CliCommand.singleNode( weightValueMatcher ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'weight' in args:
         args[ 'OPTION' ][ 'weight' ] = args.pop( 'WEIGHTVALUE' )

class NexthopNameRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a nexthop name'''
   expression = 'name NEXTHOP_NAME'
   data = {
      'name': CliCommand.singleNode( nameMatcherForConfig ),
      'NEXTHOP_NAME': CliCommand.singleNode( nexthopNameMatcher ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'name' in args:
         args[ 'OPTION' ][ 'nextHopName' ] = args.pop( 'NEXTHOP_NAME' )

class TagRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a tag'''
   expression = 'tag TAGNUM'
   data = {
      'tag': CliCommand.singleNode( tagMatcherForConfig ),
      'TAGNUM': CliCommand.singleNode( tagNumberMatcher ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'tag' in args:
         args[ 'OPTION' ][ 'tag' ] = args.pop( 'TAGNUM' )

class CommunityRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a community'''
   expression = 'community {COMM_VALUES}'
   data = {
      'community': CliCommand.singleNode( communityMatcher ),
      'COMM_VALUES': CommunityConstExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'community' in args:
         args[ 'OPTION' ][ 'community' ] = args.pop( 'communityValue' )

class PreferenceRouteOptionExpr( CliCommand.CliExpression ):
   '''For types of routes that can set a preference'''
   expression = 'PREFERENCE'
   data = {
      'PREFERENCE': CliCommand.singleNode( preferenceRangeMatcher ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTION' not in args:
         args[ 'OPTION' ] = {}
      if 'PREFERENCE' in args:
         args[ 'OPTION' ][ 'preference' ] = args.pop( 'PREFERENCE' )

class RouteOptionsExprFactory( CliCommand.CliExpressionFactory ):
   '''Most variants of `ip route` and `ipv6 route` commands include a subexpression
      of route options. These route options can be supplied in any order but
      each option can be matched at most once. In other words, its an optional
      iteration rule where each subexpression of the iteration has maxMatches=1.

      A RouteOptionsExpr example for many variants is
      '[ { PREFERENCE | ( tag TAGNUM ) | ( name NEXTHOP_NAME ) } ]`
      Such a command variant accepts a preference, tag or nexthop name but there is
      only one of each that can be given and they can be supplied in any order.
      For example
      ip route PREFIX NH 111 tag 66 name baz
      ip route PREFIX NH tag 66 111 name baz
      ip route PREFIX NH 11 name baz tag 66
      ip route PREFIX NH tag 66 name baz 111
      ip route PREFIX NH name baz 111 tag 66
      ip route PREFIX NH name baz tag 66 111
      are all equivalent.

      The current RouteOptions are
      - preference ( PreferenceRouteOptionExpr )
      - tag ( TagRouteOptionExpr )
      - community ( CommunityRouteOptionExpr )
      - nexthop name ( NexthopNameRouteOptionExpr )
      - metric ( MetricRouteOptionExpr )
      - weight ( WeightRouteOptionExpr )
      - track ( TrackRouteOptionExpr )

      It's possibe, that in the future, further RouteOptions may be defined.
      These options should all fit within these constraints.

      The RouteOption subexpressions all use an adapter function to insert
      a key, value pair into the two level dictionary rooted at args[ 'OPTION' ].
      If the 'OPTION' key is not in the args dictionary no RouteOptions were
      specified in the command. If some RouteOption was specified in the command
      then it's key, value pair would be present in the args[ 'OPTION' ] dictionary.

      An example args dictionary would be
      { 'OPTION':
         {
            'preference': 111,
            'tag': 66,
            'nextHopName': 'baz',
         }
      }

   To add a new RouteOption subexpression to this factory, first define a class
   derived from CliExpression. Next, add a constructor parameter to the __init__
   function below that allows the factory's user to tell it to include or not
   that new RouteOption subexpression in the generated RouteOptionsExpr. The
   parameter should default to not including the subexpression as it is desired
   to be able to tell what options are included by looking at the call site.
   Finally, in the generate function check if the RouteOption should be included
   in the RouteOptionsExpr and, if so, add the class at some appropriately named
   key.

   Example:
   awesomenessMatcher = CliMatcher.KeywordMatcher( 'awesomeness',
      helpdesc='Route awesomeness factor' )
   awesomenessFactorMatcher = CliMatcher.IntegerMatcher( 0, 1000,
      helpdesc='Awesomeness factor' )
   class AwesomenessRouteOptionExpression( CliCommand.CliExpression ):
      syntax = 'awesomeness AWESOMENESSFACTOR'
      data = {
         'awesomeness': CliCommand.Node( awesomenessMatcher, maxMatches=1 ),
         'AWESOMENESSFACTOR': CliCommand.Node( awesomenessFactorMatcher,
                                               maxMatches=1 ),
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         if 'OPTIONS' not in args:
            args[ 'OPTIONS' ] = {}
         if 'AWESOMENESSFACTOR' in args:
            args[ 'OPTIONS' ][ 'awesomeness' ] = args.pop( 'AWESOMENESSFACTOR' )

   then to the __init__ function add the parameter `includeAwesomeness=False` so
   that, by default, its not included in the generated RouteOptionsExpr.

   Finally, in the generate function add a check like:
      if self.includeAwesomeness:
         options[ 'AWESOMENESS_EXPR' ] = AwesomenessRouteOptionExpr

   This is an example for a fairly simple subexpression syntax. A more complicated
   subexpression is possible. See TrackRouteOptionExpr, for example.
   '''
   def __init__( self,
                 includePreference=False,
                 includeTag=False,
                 includeCommunity=False,
                 includeNexthopName=False,
                 includeMetric=False,
                 includeWeight=False,
                 trackKwargs=None ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.includePreference = includePreference
      self.includeTag = includeTag
      self.includeCommunity = includeCommunity
      self.includeNexthopName = includeNexthopName
      self.includeMetric = includeMetric
      self.includeWeight = includeWeight
      # trackKwargs is a dict of the constructor args for TrackRouteOptionExprFactory
      self.trackKwargs = trackKwargs

   def generate( self, name ):
      # build the data dictionary for the returned RouteOptionsExpr
      # route options are an iterable but each option expression can only match once
      # and this factory is only and must only be used once per command variant
      # thus no need to to utilize the name parameter for the keys in the various
      # dictionaries to distinguish between multiple factory uses.
      options = {}
      if self.includePreference:
         options[ 'PREFERENCE_EXPR' ] = PreferenceRouteOptionExpr
      if self.includeTag:
         options[ 'TAG_EXPR' ] = TagRouteOptionExpr
      if self.includeNexthopName:
         options[ 'NEXTHOP_NAME_EXPR' ] = NexthopNameRouteOptionExpr
      if self.includeMetric:
         options[ 'METRIC_EXPR' ] = MetricRouteOptionExpr
      if self.includeWeight:
         options[ 'WEIGHT_EXPR' ] = WeightRouteOptionExpr
      if self.trackKwargs is not None:
         options[ 'TRACK_EXPR' ] = TrackRouteOptionExprFactory( **self.trackKwargs )

      # example expansions of expr:
      # [ { PREFERENCE_EXPR | TAG_EXPR | NEXTHOP_NAME_EXPR } ]
      # [ { PREFERENCE_EXPR | TAG_EXPR | NEXTHOP_NAME_EXPR | METRIC_EXPR } ]
      # [ { PREFERENCE_EXPR | TAG_EXPR | NEXTHOP_NAME_EXPR |
      #     METRIC_EXPR | TRACK_EXPR } ]
      expr = '[ { ' + ' | '.join( options.keys() ) + ' } ]'

      if self.includeCommunity:
         options[ 'COMMUNITY_EXPR' ] = CommunityRouteOptionExpr
         expr += ' [ COMMUNITY_EXPR ]'
         # COMMUNITY_EXPR needs to be the final expression, because a U32 community
         # can conflict with the PREFERENCE_EXPR. After a COMMUNITY_EXPR, no other
         # route option can be supplied. Example expansions of expr:
         # [ { PREFERENCE_EXPR | TAG_EXPR | NEXTHOP_NAME_EXPR } ] [COMMUNITY_EXPR]

      class RouteOptionsExpr( CliCommand.CliExpression ):
         expression = expr
         data = options

         # no adapter needed as the subexpressions deal with it

      return RouteOptionsExpr

def getConfigTagRouteKeyMount( af, dynamic ):
   if af == 'ipv4':
      if dynamic:
         return configTagIpv4DynamicConfig
      else:
         return configTagIpv4Config
   else:
      if dynamic:
         return configTagIpv6DynamicConfig
      else:
         return configTagIpv6Config

def checkAndDeleteRouteFromTag( routeKey, vrfName, rkPerTagPerVrf,
      routeConfig, expectedConfigTag=None ):
   # Check if routeKey has a tag associated with it and delete it
   # if its not the expected tag
   route = routeConfig.route.get( routeKey )
   if route:
      commandTagId = route.commandTagId
      if commandTagId:
         commandTagStr = configTagIdState.tagIdToTagStr.get( commandTagId, None )
         if commandTagStr == expectedConfigTag:
            return False
         rkPerTagForVrf = rkPerTagPerVrf.get( vrfName, None )
         if rkPerTagForVrf:
            tagsForVrf = rkPerTagForVrf.tagToRouteKeySet
            tagKeys = tagsForVrf.get( commandTagStr, None )
            if tagKeys:
               # Tag found for routeKey needs to be removed from both commandTag
               # config and routeConfig.
               tagKeys.routeKeySet.remove( routeKey )
               if not tagKeys.routeKeySet:
                  del tagsForVrf[ commandTagStr ]
               if not rkPerTagPerVrf[ vrfName ].tagToRouteKeySet:
                  del rkPerTagPerVrf[ vrfName ]
               route.commandTagId = CommandTagId.null
   return True

def manageStaticEvpnRouteConfigTag( mode, prefix, preference, vrfName, dynamic,
      routeConfig, tagStr=None, af='ipv4' ):
   prefix = Arnet.IpGenPrefix( prefix )
   if preference is None:
      preference = RoutingConsts.defaultStaticRoutePreference
   if vrfName is None:
      vrfName = VrfCli.DEFAULT_VRF
   routeKey = Tac.Value( "Routing::RouteKey",
                         prefix=prefix,
                         preference=preference )
   rkPerTagPerVrf = getConfigTagRouteKeyMount( af, dynamic ).vrfConfig
   if ( checkAndDeleteRouteFromTag( routeKey, vrfName, rkPerTagPerVrf,
      routeConfig, expectedConfigTag=tagStr ) ) and tagStr:
      # Since tagStr has been provided, associate the route with the tag
      # in both the commandTag config and routeConfig.

      # create a configTagInput Entry using ConfigTagInputAllocator
      res = configTagInputAllocator().newConfigTagInputEntry( configTagInput,
         tagStr, vrfName )
      if not res:
         mode.addErrorAndStop( configTagInputAllocator().errorMsg )
      else:
         # No command tag is associated with the routeKey at this point
         # Get tagId for tag
         tagId = configTagConfig.configTagEntry[ tagStr ].tagId
         if not tagId:
            mode.addErrorAndStop( ConfigTagCommon.commandTagErrStr( tagStr ) )
         else:
            # Store the tag in the command tag to routeKey sysdb config.
            if vrfName not in rkPerTagPerVrf:
               rkPerTagPerVrf.newMember( vrfName )
            if tagStr not in rkPerTagPerVrf[ vrfName ].tagToRouteKeySet:
               rkPerTagPerVrf[ vrfName ].tagToRouteKeySet.newMember( tagStr )
            routeKeySet = \
               rkPerTagPerVrf[ vrfName ].tagToRouteKeySet[ tagStr ]
            routeKeySet.routeKeySet.add( routeKey )

            # Store the tagId in the route config
            route = routeConfig.route.get( routeKey )
            route.commandTagId = tagId

def printWarningIfNotRoutedPort( mode, intf, configType ):
   intfStatus = allIntfStatusDir.intfStatus.get( intf )
   # Don't throw warning if operating inside config-session
   if intfStatus and intfStatus.forwardingModel != "intfForwardingModelRouted" \
      and not mode.session.inConfigSession():
      # pylint: disable-next=consider-using-f-string
      mode.addWarning( "%s configuration will be ignored while interface %s "
                       "is not a routed port." % ( configType, intf ) )

ip4 = IraRouteCommon.Ip4()
ip6 = IraRouteCommon.Ip6()
routing4 = IraRouteCommon.routing( ip4 )
routing6 = IraRouteCommon.routing( ip6 )

interfaceToken = CliMatcher.KeywordMatcher( 'interface',
                                 helpdesc='Show only via on this interface' )
nexthopToken = CliMatcher.KeywordMatcher( 'next-hop',
                                 helpdesc='Filter output by next hop' )
intfExpr = IntfCli.Intf.matcher

class NexthopCliExpression( CliCommand.CliExpression ):
   """
   SubClass for matching filters on interface or nexthop address.
   This creates all the Tacc classes needed for the Tacc filter.
   """
   expression = """{ ( interface INTERFACE ) |
                     ( nexthop IP ) }
                   [ all-vias ]"""
   data = {
      "interface": CliCommand.Node( interfaceToken,
                                    maxMatches=1 ),
      "INTERFACE": intfExpr,
      "nexthop": CliCommand.Node( nexthopToken,
                                  maxMatches=1 ),
      "IP": IpGenAddrMatcher.ipGenAddrMatcher,
      "all-vias": 'Show all the vias of a route if one those vias matches',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if "ip" in args:
         routing = routing4
      else:
         routing = routing6
      # if all-via is not given then the non matching vias will be discard
      routeFilterList = Tac.newInstance( "Ira::RouteFilterColl",
                                          "all-vias" not in args )
      intf = args.pop( 'INTERFACE', [ None ] )[ 0 ]
      if intf:
         intfStatus = routing.intfStatus( intf )
         routeFilterIntf = Tac.newInstance( "Ira::RouteFilterIntf", intfStatus )
         routeFilterList.filterColl.add( routeFilterIntf )
      ip = args.pop( 'IP', [ None ] )[ 0 ]
      if ip:
         ip = Tac.newInstance( "Ira::RouteFilterIp", ip )
         routeFilterList.filterColl.add( ip )
      args[ "nexthop-matchers" ] = routeFilterList

# isValidNextHopHook allows another package to check if the next hop
# is one of the switch's own IP addresses. Takes the next hop and
# vrfname as the two arguments and returns a boolean: True if the next
# hop is valid, False if it is not.
isValidNextHopHook4 = CliExtensions.CliHook()
isValidNextHopHook6 = CliExtensions.CliHook()

def validNextHop4( mode, hop, vrfName, allowMcast=False, egressVrfName=None ):
   if not allowMcast and IpAddrMatcher.validateMulticastIpAddr( hop ) is None or \
          hop == '255.255.255.255':
      mode.addError( "Next hop address must be unicast" )
      return False

   assert vrfName

   # Check if next hop is equal to one of the inbound vrf's IP addresses.
   invalidHop = False
   vrfToCheck = vrfName if egressVrfName is None else egressVrfName
   ipAddrMap = vrfToIpAddrMapBuilderSm.allVrfIpAddrMap \
                                      .vrfToIpAddrMap.get( vrfToCheck )
   if ipAddrMap is not None:
      invalidHop = hop in ipAddrMap.ipv4Addr

   if invalidHop or not isValidNextHopHook4.all( hop, vrfName ):
      mode.addWarning( "Invalid next hop address (it's this router)" )

   return True

def validNextHop6( mode, hop, vrfName, intfId=None, egressVrfName=None ):
   if hop.isMulticast:
      mode.addError( "Next hop address must be unicast" )
      return False
   if hop.isLinkLocal and not intfId:
      mode.addError( "Link local next hop requires an interface" )
      return False
   if not hop.isGlobalUnicast and not hop.isLinkLocal:
      mode.addError( "Next hop must be routeable" )
      return False

   assert vrfName

   # Check if next hop is equal to one of the switch's IP addresses
   invalidHop = False
   vrfToCheck = vrfName if egressVrfName is None else egressVrfName
   ipAddrMap = vrfToIpAddrMapBuilderSm.allVrfIpAddrMap \
                                      .vrfToIpAddrMap.get( vrfToCheck )
   if ipAddrMap is not None:
      invalidHop = hop in ipAddrMap.ipv6Addr

   if invalidHop or not isValidNextHopHook6.all( hop, vrfName ):
      mode.addWarning( "Invalid next hop address (it's this router)" )

   return True

def validNextHopGen( mode, hop, vrfName,
                     intfId=None, allowMcast=False, egressVrfName=None ):
   if isinstance( hop, str ) and IpGenAddr.isIpv4( hop ):
      return validNextHop4( mode, hop, vrfName, allowMcast, egressVrfName )
   elif isinstance( hop, IpGenAddr ):
      if hop.af == AddressFamily.ipv4:
         return validNextHop4( mode, hop.v4Addr, vrfName, allowMcast, egressVrfName )
      elif hop.af == AddressFamily.ipv6:
         return validNextHop6( mode, hop.v6Addr, vrfName, intfId, egressVrfName )
   assert 0, 'Invalid nexthop'

def allNotInstalledBfdTrackedStaticRoutes( routing, vrfs, addressFamily ):
   '''Generator function that returns all of the not-installed bfd route vias
      for routes in the given addressFamily.
   '''
   def allBfdTrackedStaticRoutes():
      for vrfName in vrfs:
         bfdTrackedStaticRoutesList = routing.staticBfdStatus( vrfName )
         if not bfdTrackedStaticRoutesList:
            continue
         yield vrfName, bfdTrackedStaticRoutesList

   def allTrackedRoutesByLocalAddr():
      for vrfName, bfdTrackedStaticRoutesList in allBfdTrackedStaticRoutes():
         for route in bfdTrackedStaticRoutesList.route.values():
            yield vrfName, route

   def allTrackedRoutes():
      for vrfName, route in allTrackedRoutesByLocalAddr():
         for localAddr, bfdRoute in route.bfdRoute.items():
             # If BFD is up then move to the next node as these routes are installed.
            if bfdRoute.status:
               continue
            yield vrfName, localAddr, bfdRoute

   def allDependantRoutes():
      for vrfName, localAddr, bfdRoute in allTrackedRoutes():
         for routeKey, dependantRoute in bfdRoute.dependantRoute.items():
            if routeKey.prefix.af != addressFamily:
               continue
            yield vrfName, localAddr, routeKey, dependantRoute

   for vrfName, localAddr, routeKey, dependantRoute in allDependantRoutes():
      for via in dependantRoute.trackedVia:
         yield vrfName, localAddr, routeKey, via

def viaListForRouteKey( routeKey, model ):
   '''Using routeKey's prefix and preference find the ViaList.'''
   pfx = str( routeKey.prefix )
   route = model.routes.setdefault( pfx, AllRoutes() )
   for viaGroup in route.viaGroups:
      # the model groups vias together by preference
      if viaGroup.preference == routeKey.preference:
         break
   else:
      viaGroup = ViaList()
      viaGroup.preference = routeKey.preference
      viaGroup.metric = 0
      route.viaGroups.append( viaGroup )
   return viaGroup

def showCommonNotInstalledBfd( vrfs, v4=True ):
   """
   Common function which will be invoked from IraIpCli.py and IraIp6Cli.py.
   """
   myVrfModel = VrfNotInstalledBfd()
   if v4:
      routing = routing4
      addressFamily = AddressFamily.ipv4
   else:
      routing = routing6
      addressFamily = AddressFamily.ipv6

   for vrfName, localAddr, routeKey, via in (
         allNotInstalledBfdTrackedStaticRoutes( routing, vrfs, addressFamily ) ):
      myModel = myVrfModel.vrfs.setdefault( vrfName, NotInstalledBfd() )
      viaModel = IpRouteSimpleVia()
      viaModel.nexthopAddr = via.hop
      if via.intfId:
         viaModel.interface = via.intfId
      if localAddr:
         viaModel.localAddress = localAddr

      viaList = viaListForRouteKey( routeKey, myModel )
      viaList.vias.append( viaModel )

   return myVrfModel

#---------------------------------------------------------------------------------
# Hook for show rib ready [ vrf vrfName ]
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribReadyHook = CliExtensions.CliHook()

#---------------------------------------------------------------------------------
# Hook for show rib route ip[v6]
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribRouteHook = CliExtensions.CliHook()
ribRouteModel = {}

#---------------------------------------------------------------------------------
# Hook for show rib next-hop resolution route [ipv4|ipv6] (vrf <vrfName>)
# This hook has been defined in Ira to avoid adding a dependency between gated
# and IpRib
#---------------------------------------------------------------------------------
ribResolutionRouteHook = CliExtensions.CliHook()

def initVrfToIpAddrMapBuilderSm():
   global vrfToIpAddrMapBuilderSm

   vrfToIpAddrMapBuilderSm = Tac.newInstance( 'IpLib::VrfToIpAddrMapBuilderSm',
                                              ipConfig, ip6Config )

#--------------------------------------------------------------------------------
# test routing trigger boot progression
#--------------------------------------------------------------------------------
class TestRoutingStartBootProgressionCmd( CliCommand.CliCommandClass ):
   syntax = 'test routing trigger boot progression'
   data = {
      'test': testMatcherForExec,
      'routing': 'Routing module',
      'trigger': 'Start stage progression',
      'boot': 'Boot stage class',
      'progression': 'Boot stage progression',
   }
   hidden = True
   handler = "IraCommonCliHandler.handlerTestRoutingStartBootProgressionCmd"

if isCeosLab():
   BasicCliModes.EnableMode.addCommandClass( TestRoutingStartBootProgressionCmd )

def tunnelUdpDestPortHandler( mode, args ):
   tokenToNexthopGroupTypeMap = {
      'mpls-over-udp': TunnelPayloadType.mpls,
      'ipv4-over-udp': TunnelPayloadType.ipv4,
      'ipv6-over-udp': TunnelPayloadType.ipv6
   }

   payloadType = tokenToNexthopGroupTypeMap[ args[ 'PAYLOAD_TYPE' ] ]
   destPort = args[ 'DEST_PORT' ]
   udpDestPortConfig.payloadTypeToDestPort[
      payloadType ] = UdpDestPortConfigMode( destPort,
                                             UdpPortConfigMode.tunnelConfig )

def tunnelUdpDestPortDelHandler( mode, args ):
   tokenToNexthopGroupTypeMap = {
      'mpls-over-udp': TunnelPayloadType.mpls,
      'ipv4-over-udp': TunnelPayloadType.ipv4,
      'ipv6-over-udp': TunnelPayloadType.ipv6
   }

   payloadType = tokenToNexthopGroupTypeMap[ args[ 'PAYLOAD_TYPE' ] ]
   del udpDestPortConfig.payloadTypeToDestPort[ payloadType ]

def payloadTypeMatcher( mode ):
   payloadTypes = {
      'mpls-over-udp': 'Set the tunnel payload type as MPLS-over-UDP',
      'ipv4-over-udp': 'Set the tunnel payload type as IPv4-over-UDP',
      'ipv6-over-udp': 'Set the tunnel payload type as IPv6-over-UDP',
   }
   return payloadTypes

# ---------------------------------------------------------------------------------
#  [no | default] tunnel type <payload-type> udp destination port <port>
# ---------------------------------------------------------------------------------
class TunnelTypeUdpDestinationPortCommand( CliCommand.CliCommandClass ):
   syntax = 'tunnel type PAYLOAD_TYPE udp destination port DEST_PORT'
   noOrDefaultSyntax = 'tunnel type PAYLOAD_TYPE udp destination port ...'

   data = {
      'tunnel': matcherTunnel,
      'type': matcherType,
      'PAYLOAD_TYPE': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( payloadTypeMatcher,
                                           alwaysMatchInStartupConfig=True ) ),
      'udp': 'Configure tunnel type UDP parameters',
      'destination': 'Configure destination global parameters',
      'port': 'Configure port for the tunnel',
      'DEST_PORT': CliMatcher.IntegerMatcher( 1024, 65535,
                                              helpdesc='Destination port value' ),
   }

   handler = tunnelUdpDestPortHandler
   noOrDefaultHandler = tunnelUdpDestPortDelHandler

if Toggles.TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass(
      TunnelTypeUdpDestinationPortCommand )

def Plugin( entityManager ):
   global allIntfStatusDir
   global ipConfig, ip6Config
   global configTagInput
   global configTagConfig
   global configTagIdState
   global configTagIpv4Config
   global configTagIpv4DynamicConfig
   global configTagIpv6Config
   global configTagIpv6DynamicConfig
   global udpDestPortConfig

   def doMountsComplete():
      initVrfToIpAddrMapBuilderSm()

   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   mg = entityManager.mountGroup()
   ipConfig = mg.mount( 'ip/config', 'Ip::Config', 'w' )
   ip6Config = mg.mount( 'ip6/config', 'Ip6::Config', 'w' )
   mg.close( doMountsComplete )

   configTagInput = ConfigMount.mount( entityManager, 'configTag/input/cli',
                                       'ConfigTag::ConfigTagInput',
                                       'w' )
   configTagIpv4Config = ConfigMount.mount( entityManager,
         'configTag/routing/vrf/routeListTaggedConfig',
         'Routing::VrfRouteListTaggedConfig',
         'wS' )

   # Although we just read from configTagConfig and configTagIdState in this plugin,
   # we config-mount them here to maintain uniformity across all cliPlugins.
   configTagConfig = ConfigMount.mount( entityManager, 'configTag/config',
                                        'ConfigTag::ConfigTagConfig',
                                        'w' )

   configTagIdState = LazyMount.mount( entityManager, 'configTag/configTagIdState',
                                         'ConfigTag::ConfigTagIdState',
                                         'w' )

   configTagIpv4DynamicConfig = ConfigMount.mount( entityManager,
         'configTag/routing/vrf/routeListTaggedDynamicConfig',
         'Routing::VrfRouteListTaggedConfig',
         'wS' )

   configTagIpv6Config = ConfigMount.mount( entityManager,
         'configTag/routing6/vrf/routeListTaggedConfig',
         'Routing::VrfRouteListTaggedConfig',
         'wS' )

   configTagIpv6DynamicConfig = ConfigMount.mount( entityManager,
         'configTag/routing6/vrf/routeListTaggedDynamicConfig',
         'Routing::VrfRouteListTaggedConfig',
         'wS' )

   udpDestPortConfig = ConfigMount.mount( entityManager,
         "l3/tunnel/udpDestinationPortConfig",
         "L3::Tunnel::UdpDestinationPortConfig", "wi" )

   routing4.plugin( entityManager )
   routing6.plugin( entityManager )

# As there is no exact identifier on an IPv6 address if it was generated by
# SLAAC, the latter 64 bits has to be checked against the EUI-64 of the
# relevant interface.
def isSlaacAddress( ip6Addr, intfStatus ):
   # Not all interfaces have routedAddr attribute,
   # which contains the needed MAC address
   if not hasattr( intfStatus, 'routedAddr' ):
      return False
   ethAddr = Tac.Value( "Arnet::EthAddr" )
   ethAddr.stringValue = intfStatus.routedAddr
   return ( ip6Addr.u16[ 4 ] == ethAddr.word0 ^ 0x200 ) and \
          ( ip6Addr.u16[ 5 ] == ethAddr.byte[ 2 ] << 8 | 0xff ) and \
          ( ip6Addr.u16[ 6 ] == ethAddr.byte[ 3 ] | 0xfe00 ) and \
          ( ip6Addr.u16[ 7 ] == ethAddr.word2 )

# Implements a wrapper for the lookup() method of objects derived from the
# IntfCli.Intf class. The wrapper adds an option to throw an exception if
# the lookup fails. It also is a useful function to mock during tests so
# callers of the real IntfCli.Intf class lookup() method are unaffected.
def lookupCommon( intf, useException=True ):
   assert isinstance( intf, Intf )
   if intf.lookup():
      return True
   if useException:
      # pylint: disable-next=consider-using-f-string
      raise UnknownEntityError( "Interface %s does not exist" % intf.name )
   return False

def isInvalidIpv6Mtu( mtu ):
   return 0 < mtu < IPV6_MIN_MTU
