# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP6 configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

from collections import defaultdict
import os.path
import sys
import Arnet
import Arnet.MplsLib
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Hardware
import CliToken.Icmp
import CliToken.Ipv6
import ConfigMount
import Intf.IntfRange
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import IntfCli
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpGenAddrMatcher
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin import IraCommonCli
from CliPlugin.IraCommonCli import (
   NexthopCliExpression,
   RouteOptionsExprFactory,
   preferenceRangeMatcher,
   tagMatcherForConfig,
   tagNumberMatcher,
)
from CliPlugin import IraIp6IntfCli
from CliPlugin import IraIp6RouteCliLib
from CliPlugin import IraRouteCommon
from CliPlugin.IraIpCliCommon import (
      mergedDict,
      supportedProfileType,
)
from CliPlugin.IraIpModel import VrfIpRibRoutes, V6EcmpModel
from CliPlugin.TechSupportCli import registerShowTechSupportCmd
from CliPlugin.IraIp6RouteCliLib import (
      ip6,
      routing6,
      noIpv6RouteTableForVrfMsg,
      routeMatcherForIpv6Config,
)
from CliPlugin.IraCommonModel import (
   ServiceRoutingProtoModelStatus,
   VrfNotInstalledBfd,
)
from CliPlugin.IraIp6Model import (
      Ip6Statuses,
      Ip6StatusBrief,
      Ip6StatusesBrief,
      Ip6Routes,
      Ip6RouteSummary,
      ospfTotals6,
      bgpTotals6,
      isisTotals6,
      VrfIp6Routes,
      VrfIp6RoutesHost,
      Ip6RouteSummaryForVrf,
      Ip6UrpfInterfaces,
)
from CliPlugin.VrfCli import (
      VrfExprFactory,
      vrfExists,
)
from CliPlugin.IraNexthopGroupCli import (
      nexthopGroupNamePattern,
      nexthopGroupSupportedGuard,
      getNexthopGroupNames,
)
from CliPlugin.IpRibLib import ResolutionRibsExprFactory
from CliPlugin import ConfigTagCommon
import IpLibTypes
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import LazyMount

import ShowCommand
import SharkLazyMount
import Tac
import TacSigint
from CliModel import UnknownEntityError
from CliPrint import CliPrint
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from CliToken import IpRibLibCliTokens

from Toggles.IraToggleLib import (
   toggleIntfEgressVrfRoutesEnabled,
   toggleMixedRouteSummaryOutputEnabled,
   toggleStaticRouteViaWeightEnabled )
from Toggles.RoutingLibToggleLib import (
   toggleStaticResRibProfilePerViaEnabled,
   toggleOspf3RouteTypesDisplayEnabled )

printer = CliPrint().lib

routing4 = IraRouteCommon.routing( IraRouteCommon.Ip4() )
l3Config = None
routingHardwareStatusCommon = None
routingHardwareStatus = None
routing6HardwareStatus = None
allVrfConfig = None
allVrfStatusLocal = None
allIntfStatusDir = None
allVrfRouteSummary = None
VERIFY_MAX_ROUTES = 100

addrSource = IpLibTypes.addrSource( Tac.Type( 'Arnet::AddressFamily' ).ipv6 )

configTagExpr = ConfigTagCommon.ConfigTagExpr

def mplsPushSupportedGuard( mode, token ):
   if routingHardwareStatus.mplsPushSupported:
      return None
   return CliParser.guardNotThisPlatform

labelNode = CliCommand.guardedKeyword( 'label',
      helpdesc="Push an MPLS label",
      guard=mplsPushSupportedGuard )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#--------------------------<-----------------------------------------------------
configMode = BasicCli.GlobalConfigMode

def getVrfNames( mode ):
   return allVrfConfig.vrf.members()

#-------------------------------------------------------------------------------
# The "ipv6 unicast-routing" and
#     "no ipv6 unicast-routing [keep-static-routes]" commands.
#-------------------------------------------------------------------------------
unicastRoutingMatcher = CliMatcher.KeywordMatcher(
   'unicast-routing',
   helpdesc='Enable IPv6 unicast forwarding' )

def ipV6Vrf( mode, token ):
   if routingHardwareStatusCommon.vrfCapability.\
      ipv6EnabledDefault:
      return None
   return CliParser.guardNotThisPlatform

class Ipv6UnicastRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 unicast-routing [ VRF ]'
   noOrDefaultSyntax = 'ipv6 unicast-routing ' \
                       '[ keep-static-routes | delete-static-routes ] ' \
                       '[ VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'unicast-routing': unicastRoutingMatcher,
            'keep-static-routes': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'keep-static-routes',
                  helpdesc='Disable IPv6 forwarding, ' \
                           'but do not remove the configured static routes' ),
               hidden=True ),
            'delete-static-routes': CliMatcher.KeywordMatcher(
                                    'delete-static-routes',
                                    helpdesc='Disable IPv6 forwarding, ' \
                                    'and remove the configured static routes' ),
            'VRF': VrfExprFactory( helpdesc='Enable routing in a VRF',
                                   guard=ipV6Vrf ),
          }

   handler = "IraIp6CliHandler.handlerIpv6UnicastRoutingCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerIpv6UnicastRoutingCmd"

configMode.addCommandClass( Ipv6UnicastRoutingCmd )

#-------------------------------------------------------------------------------
# The "[no] ipv6 icmp rate-limit-unreachable 0" command.
#-------------------------------------------------------------------------------
class Ipv6IcmpUnreachableRateCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 icmp rate-limit-unreachable RATE'
   noOrDefaultSyntax = 'ipv6 icmp rate-limit-unreachable ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'icmp': CliToken.Icmp.icmpMatcherForConfig,
            'rate-limit-unreachable':
                           CliToken.Icmp.rateLimitUnreachableMatcherForConfig,
            'RATE': CliMatcher.IntegerMatcher( 0, 0, helpdesc='Rate' )
          }

   handler = "IraIp6CliHandler.handlerIpv6IcmpUnreachableRateCmd"
   noOrDefaultHandler = handler

configMode.addCommandClass( Ipv6IcmpUnreachableRateCmd )

#------------------------------------------------------------------------------------
#The "(no|default) ipv6 icmp redirect"
#      commands.
#------------------------------------------------------------------------------------
redirectMatcher = CliMatcher.KeywordMatcher( 'redirect',
                                             helpdesc='Send ICMP redirects' )

class Ipv6IcmpRedirectCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 icmp redirect'
   noOrDefaultSyntax = 'ipv6 icmp redirect'

   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'icmp': CliToken.Icmp.icmpMatcherForConfig,
            'redirect': redirectMatcher }

   handler = "IraIp6CliHandler.handlerIpv6IcmpRedirectCmd"
   noHandler = "IraIp6CliHandler.noHandlerIpv6IcmpRedirectCmd"
   defaultHandler = "IraIp6CliHandler.defaultHandlerIpv6IcmpRedirectCmd"

configMode.addCommandClass( Ipv6IcmpRedirectCmd )

#------------------------------------------------------------------------------------
#The "(no|default) ipv6 nd enhanced-dad default"
#      commands.
#------------------------------------------------------------------------------------
# Determine if kernel supports enhanced DAD
def enhancedDadSupported( mode, token ):
   if os.path.isfile( '/proc/sys/net/ipv6/conf/all/enhanced_dad' ):
      return None
   else:
      return CliParser.guardNotThisPlatform

ndEnhancedDadMatcher = CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'enhanced-dad',
                           helpdesc='RFC7527 enhanced duplicate address detection' ),
                        guard=enhancedDadSupported )

class EnhancedDadCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd enhanced-dad default'
   noOrDefaultSyntax = syntax
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'nd': IraIp6IntfCli.ndMatcher,
            'enhanced-dad': ndEnhancedDadMatcher,
            'default': 'Setting for all the interfaces' }

   handler = "IraIp6CliHandler.handlerEnhancedDadCmd"
   noHandler = "IraIp6CliHandler.noHandlerEnhancedDadCmd"
   defaultHandler = handler

configMode.addCommandClass( EnhancedDadCmd )

#-------------------------------------------------------------------------------
# The "[no] ipv6 route" command.
#-------------------------------------------------------------------------------
nexthopAddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( 'Address of the nexthop router' )
genNexthopAddrMatcher =\
   IpGenAddrMatcher.IpGenAddrMatcher( "address of the next-hop router" )

intfAddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher(
      "Forwarding router's address on destination interface" )

def intfAddr6RoutesSupportedGuard( mode, token ):
   if ip6.intfAddrRoutesSupported:
      return None
   return CliParser.guardNotThisPlatform

nexthopGroupMatcherForConfig = CliCommand.Node(
                                  CliMatcher.KeywordMatcher( 'nexthop-group',
                                     helpdesc='Specify nexthop group name'),
                                  guard=nexthopGroupSupportedGuard )

nexthopGroupNameMatcher = CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
                              'Nexthop group name', pattern=nexthopGroupNamePattern )

trackingProtocols = {}
trackingProtocolsHidden = {}

def addTrackingProto( _trackingProtocols, _trackingProtocolsHidden,
                      protocol, desc, hidden=False ):
   if hidden:
      _trackingProtocolsHidden[ protocol ] = desc
   else:
      _trackingProtocols[ protocol ] = desc

# call addTrackingProto before TrackingProtocolExpr
addTrackingProto( trackingProtocols, trackingProtocolsHidden,
                  'bfd', 'Track this route using BFD' )

vrfExprFactoryForConfig = VrfExprFactory( helpdesc='Configure routes in a VRF' )

# toggle for weight option
weightEnabled = toggleStaticRouteViaWeightEnabled()

class NexthopAddrExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP'
   data = { 'NEXTHOP': nexthopAddrMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' not in args and 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )

class NexthopAddrOrIntfExpr( CliCommand.CliExpression ):
   expression = '( NEXTHOP [ label LABEL_VALUE ] ) | ( nexthop-group NHG_NAME ) ' \
                '| ( INTF [ INTF_ADDR ] ) | NULL0_EXPR'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'label': labelNode,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher,
            'nexthop-group': nexthopGroupMatcherForConfig,
            'NHG_NAME': nexthopGroupNameMatcher,
            'INTF': IntfCli.Intf.matcher,
            'INTF_ADDR': CliCommand.Node( intfAddrMatcher,
                                          guard=intfAddr6RoutesSupportedGuard ),
            'NULL0_EXPR': IraCommonCli.Null0Expr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'label' in args:
         args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
         args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )
      elif 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'nexthop-group' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ]= args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = args.pop( 'INTF_ADDR',
                                                                       None )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )

class NhMplsLabelExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( NEXTHOP label LABEL_VALUE )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'label': labelNode,
            'LABEL_VALUE': Arnet.MplsLib.labelValMatcher
          }
   data[ 'NEXTHOP' ] = genNexthopAddrMatcher
   data[ 'LABEL_VALUE' ] = Arnet.MplsLib.LabelValsWithExpNullExprFactory()

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      args[ 'NEXTHOP_INTF' ][ 'mpls' ][ 'label' ] = args.pop( 'LABEL_VALUE' )

class NexthopGroupExpr( CliCommand.CliExpression ):
   # this has an adapter, so making it a class
   expression = '( nexthop-group NHG_NAME )'
   data = { 'nexthop-group': nexthopGroupMatcherForConfig,
            'NHG_NAME': nexthopGroupNameMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'nexthopGroupName' ] = args.pop( 'NHG_NAME' )

class NexthopOrIntfNexthopExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP | NULL0_EXPR | ( INTF INTF_ADDR )'
   data = { 'NEXTHOP': nexthopAddrMatcher,
            'NULL0_EXPR': IraCommonCli.Null0Expr,
            'INTF': CliCommand.Node( IntfCli.Intf.matcher,
                                     guard=intfAddr6RoutesSupportedGuard ),
            'INTF_ADDR': intfAddrMatcher
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      if 'NEXTHOP' in args:
         args[ 'NEXTHOP_INTF' ][ 'nexthop' ] = args.pop( 'NEXTHOP' )
      elif 'Null0' in args:
         del args[ 'Null0' ]
         args[ 'NEXTHOP_INTF' ][ 'Null0' ] = 'Null0'
      elif 'Null 0' in args:
         args[ 'NEXTHOP_INTF' ][ 'Null 0' ] = args.pop( 'Null 0' )
      elif 'INTF' in args:
         args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ]= args.pop( 'INTF' )
         args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = args.pop( 'INTF_ADDR' )

defaultVrfMatcher = CliMatcher.KeywordMatcher( DEFAULT_VRF,
      helpdesc='Default virtual routing and forwarding instance' )

class LeakVrfCliExpr( CliCommand.CliExpression ):
   expression = 'leak LEAK_VRF ( CONFIG_VRFNAME | DEFAULT_VRF )'
   data = { 'leak': IraCommonCli.leakMatcherForConfig,
            'LEAK_VRF': IraCommonCli.leakVrfMatcherForConfig,
            'CONFIG_VRFNAME': CliMatcher.DynamicNameMatcher( getVrfNames,
                                       'VRF name', priority=CliParser.PRIO_LOW ),
            'DEFAULT_VRF': defaultVrfMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'LEAK_TO_VRF' ] = ( args.pop( 'CONFIG_VRFNAME', None ) or
                                args.pop( 'DEFAULT_VRF', None ) )

class EgressVrfCliExpr( CliCommand.CliExpression ):
   expression = 'egress-vrf ( NON_DEFAULT | DEFAULT )'
   data = { 'egress-vrf': IraCommonCli.egressVrfMatcher,
            'NON_DEFAULT': CliMatcher.DynamicNameMatcher( getVrfNames,
               'VRF name', priority=CliParser.PRIO_LOW ),
            'DEFAULT': defaultVrfMatcher,
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'EGRESS_VRF' ] = ( args.pop( 'NON_DEFAULT', None ) or
                               args.pop( 'DEFAULT', None ) )

class Ipv6RouteMplsLabelCmd( CliCommand.CliCommandClass ):
   '''This class covers mpls route'''
   syntax = 'ipv6 route [ VRF ] PREFIX NEXTHOP_LABEL [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_LABEL': NhMplsLabelExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=False,
                              includeNexthopName=True ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteMplsLabelCmd"

configMode.addCommandClass( Ipv6RouteMplsLabelCmd )

class Ipv6RouteNexthopGroupCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-group route'''
   syntax = 'ipv6 route [ VRF ] PREFIX NEXTHOP_GROUP [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_GROUP': NexthopGroupExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteNexthopGroupCmd"

configMode.addCommandClass( Ipv6RouteNexthopGroupCmd )

class Ipv6RouteNexthopOrComboCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes, combo routes(nexthop+intf)'''
   syntax = '''ipv6 route [ VRF ] PREFIX NEXTHOP_ADDR_OR_INTF [ ROUTE_OPTIONS ]
               [ LEAK_VRF ]'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled,
                              trackKwargs={
                                 'trackingProtocols': trackingProtocols,
                                 'trackingProtocolsHidden': trackingProtocolsHidden,
                                 'includeIp4': False,
                                 'includeIp6': True,
                                 'ip6Guard': None,
                                 'ip6IsHidden': False,
                                 } ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteNexthopOrComboCmd"

configMode.addCommandClass( Ipv6RouteNexthopOrComboCmd )

class Ipv6RouteNexthopResRibProfileCmd( CliCommand.CliCommandClass ):
   '''This class covers nexthop-only routes with resolution rib profile specified
   for the nexthop'''
   syntax = '''ipv6 route [ VRF ] PREFIX NEXTHOP_ADDR [ ROUTE_OPTIONS ]
               resolution ribs SYS_CON_ONLY_RIBS'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'NEXTHOP_ADDR': nexthopAddrMatcher,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled,
                              trackKwargs={
                                 'trackingProtocols': trackingProtocols,
                                 'trackingProtocolsHidden': trackingProtocolsHidden,
                                 'includeIp4': True,
                                 'includeIp6': True,
                                 'ip6Guard': None,
                                 'ip6IsHidden': True,
                                 } ),
            'resolution': IpRibLibCliTokens.matcherResolution,
            'ribs': IpRibLibCliTokens.matcherRibs,
            'SYS_CON_ONLY_RIBS': ResolutionRibsExprFactory()
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteNexthopResRibProfileCmd"

if toggleStaticResRibProfilePerViaEnabled():
   configMode.addCommandClass( Ipv6RouteNexthopResRibProfileCmd )

class Ipv6RouteInterfaceCmd( CliCommand.CliCommandClass ):
   '''This class covers interface routes'''
   syntax = 'ipv6 route [ VRF ] PREFIX INTF [ ROUTE_OPTIONS ] [ LEAK_VRF ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'INTF': IntfCli.Intf.matcher,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeCommunity=True,
                              includeNexthopName=True,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteInterfaceCmd"

configMode.addCommandClass( Ipv6RouteInterfaceCmd )

class Ipv6RouteEgressVrfCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR [ ROUTE_OPTIONS ]
               [ LEAK_VRF ]'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteEgressVrfCmd"

class Ipv6RouteEgressVrfNexthopOrComboCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX EGRESS_VRF NEXTHOP_ADDR_OR_INTF
               [ ROUTE_OPTIONS ] [ LEAK_VRF ]'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopOrIntfNexthopExpr,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   handler = "IraIp6CliHandler.handlerIpv6RouteEgressVrfNexthopOrComboCmd"

configMode.addCommandClass( Ipv6RouteEgressVrfNexthopOrComboCmd )

class Ipv6RouteIntfEgressVrfCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv6 route [ VRF ] PREFIX EGRESS_VRF INTF [ ROUTE_OPTIONS ]
               [ LEAK_VRF ]'''
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'INTF': IntfCli.Intf.matcher,
            'ROUTE_OPTIONS': RouteOptionsExprFactory(
                              includePreference=True,
                              includeTag=True,
                              includeNexthopName=True,
                              includeCommunity=False,
                              includeMetric=True,
                              includeWeight=weightEnabled ),
            'LEAK_VRF': LeakVrfCliExpr
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_INTF' not in args:
         args[ 'NEXTHOP_INTF' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
      args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

   handler = "IraIp6CliHandler.handlerIpv6RouteIntfEgressVrfCmd"

if toggleIntfEgressVrfRoutesEnabled():
   configMode.addCommandClass( Ipv6RouteIntfEgressVrfCmd )

class NoIpv6RouteCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 route [ VRF ] PREFIX ' \
                       '[ ( EGRESS_VRF NEXTHOP_ADDR ) | NEXTHOP_ADDR_OR_INTF ] ' \
                       '[ PREFERENCE ] ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'route': routeMatcherForIpv6Config,
            'VRF': vrfExprFactoryForConfig,
            'PREFIX': IraIp6RouteCliLib.ipv6PrefixMatcher,
            'EGRESS_VRF': EgressVrfCliExpr,
            'NEXTHOP_ADDR': NexthopAddrExpr,
            'NEXTHOP_ADDR_OR_INTF': NexthopAddrOrIntfExpr,
            'PREFERENCE': preferenceRangeMatcher
          }

   noOrDefaultHandler = \
      "IraIp6CliHandler.noOrDefaultHandlerIpv6RouteIntfEgressVrfCmd"

configMode.addCommandClass( NoIpv6RouteCmd )

#-------------------------------------------------------------------------------
# Ipv6 info for "show ip" command. This is called from IraIpCli.showIp
#-------------------------------------------------------------------------------

def generateIp6EcmpModel( mode, vrfName=DEFAULT_VRF ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName
   r = routing6.config( vrfName )
   v6RoutingEnabled = r.routing

   ip6Model = V6EcmpModel()
   if not routing6.hwStatus().ecmpSupported:
      hwConfig = routing6.hwConfig( vrfName )
      ip6Model.v6EcmpRouteSupport = False
      if hwConfig.ecmpNexthopIndex == hwConfig.defaultEcmpNexthopIndex:
         nexthopIndex = 0
         numPrefixBits = 0
      else:
         nexthopIndex = hwConfig.ecmpNexthopIndex
         numPrefixBits = hwConfig.numPrefixBitsForEcmpNexthopIndex

      ip6Model.nextHopIndex = nexthopIndex
      ip6Model.numPrefixBits = numPrefixBits
   else:
      ip6Model.v6EcmpRouteSupport = True

   return ( v6RoutingEnabled, ip6Model )

#-------------------------------------------------------------------------------
# The "show ipv6 interface [ intf | ipv6addr ]" command.
#-------------------------------------------------------------------------------
def ip6IntfShow( mode, intf, raiseLookupError=True ):
   if not IraCommonCli.lookupCommon( intf, useException=raiseLookupError ):
      return None
   ip6Intf = IraIp6IntfCli.Ip6Intf( intf, mode.sysdbRoot, createIfMissing=False )
   result = ip6Intf.show()
   result.name = intf.name
   result.mtu = intf.status().mtu
   result.mtuConfigError = IraCommonCli.isInvalidIpv6Mtu( result.mtu )
   ( result.lineProtocolStatus, result.interfaceStatus ) = intf.getStatus()
   return result

def ip6ConfiguredIntf( ifName ):
   if ifName in ip6.status.intf:
      return True
   ip6IntfConfig = ip6.config.intf.get( ifName )
   return ip6IntfConfig and ip6IntfConfig.ipv6Configured()

def filterIntfByVrf( mode, intfs, vrfName ):
   if vrfName is None or not vrfExists( vrfName ):
      raise UnknownEntityError( noIpv6RouteTableForVrfMsg % vrfName )
   return [ intf for intf in intfs
      if IraIp6IntfCli.Ip6Intf( intf, mode,
                                createIfMissing=False ).config().vrf == vrfName ]

def routedIntfsWithIp6Configured( mode, intfs, vrfName=None ):
   """
   returns a list of lists "l3IntfInfoList" with two elements
   listoflist[0] includes L3 interfaces
   listoflist[1] includes L3 interfaces with ipv6 configured
   """
   intfs = IntfCli.Intf.getAll( mode, intfs )
   if intfs is None:
      return None
   intfsL3 = [ i for i in intfs if i.routingCurrentlySupported() ]
   intfsIp6Configured = [ i for i in intfsL3 if ip6ConfiguredIntf( i.name ) ]
   if vrfName:
      intfsL3 = filterIntfByVrf( mode, intfsL3, vrfName )
      intfsIp6Configured = filterIntfByVrf( mode, intfsIp6Configured, vrfName )
   return [ intfsL3, intfsIp6Configured ]

#-------------------------------------------------------------------------------
# The "show ipv6 interface" command.
#-------------------------------------------------------------------------------
def showIp6InterfaceDetail( mode, intfs, vrfName=None ):
   intfList = None
   l3IntfInfoList = routedIntfsWithIp6Configured( mode, intfs, vrfName )
   result = Ip6Statuses()
   if l3IntfInfoList:
      intfList = l3IntfInfoList[ 0 ]
      for x in l3IntfInfoList[ 1 ]:
         ip6Status = ip6IntfShow( mode, x, raiseLookupError=bool( intfs ) )
         if ip6Status:
            result.interfaces[ x.name ] = ip6Status

   # Print an error message if no interface got displayed.
   if not result.interfaces:
      if intfList:
         # There are ipv6 capable interfaces, but none has ipv6 configured.
         if intfs:
            intfNames = list( intfs )
            if len( intfNames ) == 1:
               msgStr = "IPv6 not configured on %s" % intfNames[ 0 ]
               if vrfName:
                  msgStr = " %s or %s doesn't exist in vrf %s" % \
                           ( msgStr, intfNames[ 0 ], vrfName )
            else:
               msgStr = "No IPv6 configured interfaces in range"
               if vrfName:
                  msgStr = " %s or the range doesn\'t exist in vrf %s" % \
                            ( msgStr, vrfName )
         else:
            msgStr = "No IPv6 configured interfaces"
            if vrfName:
               msgStr = " %s in vrf %s " % ( msgStr, vrfName )

      else:
         # no ipv6 capable interfaces
         if intfs:
            intfNames = list( intfs )
            if len( intfNames ) == 1:
               msgStr = "%s does not support IPv6" % intfNames[ 0 ]
               if vrfName:
                  msgStr = " %s or %s doesn\'t exist in vrf %s" % \
                           ( msgStr, intfNames[ 0 ], vrfName )
            else:
               msgStr = "No IPv6 capable interfaces in range"
               if vrfName:
                  msgStr = " %s or the range doesn\'t exist in vrf %s" % \
                           ( msgStr, vrfName )
         else:
            msgStr = "No IPv6 capable interfaces"
            if vrfName:
               msgStr = " %s in vrf %s " % ( msgStr, vrfName )
      mode.addError( msgStr )
   return result

#-------------------------------------------------------------------------------
# The "show ipv6 interface brief" command.
#-------------------------------------------------------------------------------
def showIp6InterfaceBrief( mode, args ):
   intfs = args.get( 'INTFS' )
   vrfName = args.get( 'VRF' )
   briefModel = Ip6StatusesBrief()
   detailModel = showIp6InterfaceDetail( mode, intfs, vrfName=vrfName )

   for intfDetail in detailModel.interfaces.values():
      intfBrief = Ip6StatusBrief(
         name=intfDetail.name,
         mtu=intfDetail.mtu,
         interfaceStatus=intfDetail.interfaceStatus,
         lineProtocolStatus=intfDetail.lineProtocolStatus,
         linkLocal=intfDetail.linkLocal,
         addresses=intfDetail.addresses,
         addrSource=intfDetail.addrSource,
      )

      briefModel.interfaces[ intfBrief.name ] = intfBrief

   return briefModel

#-------------------------------------------------------------------------------
# Wrapper function for "show ipv6 interface [brief]"
#-------------------------------------------------------------------------------
def showIpv6IntfByAddr( mode, ipv6Addr=None, vrfName=None ):
   result = Ip6Statuses()
   if not ipv6Addr:
      return result
   for intfStatus in ip6.status.intf.values():
      # pylint: disable-next=consider-using-in,singleton-comparison
      if vrfName != None and intfStatus.vrf != vrfName:
         continue
      for addrInfo in intfStatus.addr:
         if addrInfo.address == ipv6Addr:
            intfName = intfStatus.intfId
            intfType, _ = Intf.IntfRange.intfTypeFromName( intfName )
            intf = intfType.getCliIntf( mode, intfName )
            ip6Status = ip6IntfShow( mode, intf, raiseLookupError=False )
            if ip6Status:
               result.interfaces[ intf.name ] = ip6Status
   return result

def showIpv6Intf( mode, args ):
   vrfName = args.get( 'VRF' )
   addr = args.get( 'ADDR' )
   if addr is not None:
      return showIpv6IntfByAddr( mode, ipv6Addr=addr, vrfName=vrfName )
   intfs = args.get( 'INTFS' )
   return showIp6InterfaceDetail( mode, intfs, vrfName=vrfName )

interfaceAfterShowMatcher = CliMatcher.KeywordMatcher(
   'interface',
   helpdesc='Details related to IPv6 interfaces' )

briefKwMatcher = CliMatcher.KeywordMatcher(
   'brief',
   helpdesc='Condensed output' )

vrfKwNode = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF instance' ),
   guard=ipV6Vrf )

vrfNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                                'VRF name' )

class ShowIpv6IntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """ show ipv6 interface
                [ INTFS | ADDR ]
                [ vrf VRF ]
            """
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPV6 Address' ),
            'vrf': vrfKwNode,
            'VRF': vrfNameMatcher
          }
   cliModel = Ip6Statuses

   handler = showIpv6Intf

BasicCli.addShowCommandClass( ShowIpv6IntfCmd )

class ShowIpv6IntfBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 interface
               [ INTFS ]
               [ vrf VRF ]
               brief
            """
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'interface': interfaceAfterShowMatcher,
            'INTFS': IntfCli.Intf.rangeMatcher,
            'vrf': vrfKwNode,
            'VRF': vrfNameMatcher,
            'brief': briefKwMatcher
          }
   cliModel = Ip6StatusesBrief

   handler = showIp6InterfaceBrief

BasicCli.addShowCommandClass( ShowIpv6IntfBriefCmd )

def warnIfRoutingDisabled( mode, vrfName=DEFAULT_VRF ):
   # OSPF and other protocols allow creating instances even before
   # corresponding VRF is defined. In such cases, vrf/config/<vrfname> may not
   # exist yet.
   if not ( vrfExists( vrfName ) and routing6.config( vrfName ).routing ):
      mode.addWarning( "IPv6 routing not enabled in VRF %s" % vrfName )

def warnIfRoutingDisabledGeneral( mode, vrfName=DEFAULT_VRF ):
   if mode.session_.outputFormat_ != 'json':
      if vrfName and not vrfExists( vrfName ):
         mode.addWarning( "IPv6 routing not enabled" )
      elif not routing6.config( vrfName ).routing:
         mode.addWarning( "IPv6 routing not enabled" )

#-------------------------------------------------------------------------------
# The "show ipv6 route host" command.
#-------------------------------------------------------------------------------
def showRouteHost( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName and not vrfExists( vrfName ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
         return None
   model = IraRouteCommon.showRouteHostCommon( mode, vrfName, quantify,
                        allVrfStatusLocal, routing6, allVrfConfig )
   return model

#-------------------------------------------------------------------------------
# The "show ipv6 route" command.
#-------------------------------------------------------------------------------
routeAfterShowIpv6Matcher = CliMatcher.KeywordMatcher( 'route',
                                                 helpdesc='IPv6 routing table' )
ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( 'Match this IPv6 prefix' )
ipv6AddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPv6 address' )
showIpv6StaticMatcher = CliMatcher.KeywordMatcher(
   'static',
   helpdesc='Show only static routes' )
detailAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'detail',
                                                             helpdesc='All routes')

sharedCountObj = object()

def _protocolMatcher( protocolName, helpdesc, alias='PROTOCOL' ):
   return CliCommand.Node( CliMatcher.KeywordMatcher(
                              protocolName, helpdesc=helpdesc ),
                           sharedMatchObj=sharedCountObj, maxMatches=1,
                           alias=alias )

class ProtocolCliExpression( CliCommand.CliExpression ):
   expression = "( bgp | connected | kernel | martian | " \
                   "dynamic-policy | ospf | static | aggregate"\
                   " | isis | mpls | dhcp | gribi | cbf )"
   data = { 'bgp': _protocolMatcher( 'bgp', 'Show only BGP routes' ),
            'connected': _protocolMatcher( 'connected',
                                           'Show only connected routes' ),
            'kernel': _protocolMatcher( 'kernel', 'Show only kernel routes' ),
            'martian': _protocolMatcher( 'martian', 'Show only kernel routes' ),
            'dynamic-policy': _protocolMatcher( 'dynamic-policy',
                                                'Show only Dynamic Policy routes' ),
            'ospf': _protocolMatcher( 'ospf', 'Show only OSPF routes' ),
            'isis': _protocolMatcher( 'isis', 'Show only IS-IS routes' ),
            'static': _protocolMatcher( 'static', 'Show only static routes' ),
            'aggregate': _protocolMatcher( 'aggregate',
                                           'Show only aggregate routes' ),
            'mpls': _protocolMatcher( 'mpls', 'Show only static MPLS routes' ),
            'dhcp': _protocolMatcher( 'dhcp', 'Show only DHCP routes' ),
            'gribi': _protocolMatcher( 'gribi', 'Show only gRIBI routes' ),
            'cbf': _protocolMatcher( 'cbf', 'Show only CBF Leaked routes' ) }

quantifyKwMatcher = CliMatcher.KeywordMatcher(
   'quantify',
   helpdesc='Print the time taken' )

class ShowIpv6RouteOptionalItemsCliExpr( CliCommand.CliExpression ):
   expression = '[ { detail | PROTOCOL | quantify } ]'
   data = { 'detail': CliCommand.Node(
               detailAfterShowIpv6RouteMatcher, maxMatches=1 ),
            'PROTOCOL': ProtocolCliExpression,
            'quantify': CliCommand.Node(
               quantifyKwMatcher, maxMatches=1, hidden=True )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      options = set()
      for token in ( 'detail', 'PROTOCOL', 'quantify' ):
         value = args.pop( token, None )
         if value is not None:
            options.add( ( token.lower(), value ) )
      args[ 'OPTIONS' ] = options

def showRoute( mode, prefix=None, optionsSet=None, vrfName=None,
               routeFilter=None, flattened=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfName == ALL_VRF_NAME and not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return None

   if flattened:
      mode.addWarning( "This command will be deprecated and superseded"
                       " by show ipv6 route" )

   fd = sys.stdout.fileno()
   outputFormat = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   # Use old model only for revision 1 JSON output
   if mode.session_.outputFormat_ == 'json' and revision == 1:
      if vrfName == ALL_VRF_NAME:
         mode.addError( "'vrf all' option not supported in revision 1" )
         return None
      p = printer.initPrinter( fd, outputFormat, True )
      printer.startRender( p )
      showRoutePerVrf( p, mode, prefix, optionsSet, vrfName,
                       routeFilter=routeFilter, flattened=flattened )
      printer.endRender( p )
      printer.deinitPrinter( p )
      return Ip6Routes
   # For everything else, use revision 2 model
   else:
      p = printer.initPrinter( fd, outputFormat, True )
      printer.startRender( p )
      printer.startDict( p, "vrfs" )
      if vrfName == ALL_VRF_NAME:
         vrfs = sorted( v for v in allVrfStatusLocal.vrf )
         vrfs.insert( 0, 'default' )
      else:
         vrfs = [ vrfName ]
      for vrf in vrfs:
         # make sure we don't attempt to write if SIGINT happened outside the
         # immediateMode section (since RouteSorter handles those internally)
         #
         # Additionally, "VRF: ..." header is printed inside RouterSorter C++ code
         # for safety when pager exits, see BUG284545I
         TacSigint.check()
         printer.startDict( p, vrf )
         keepGoing = showRoutePerVrf( p, mode, prefix=prefix, optionsSet=optionsSet,
                                      vrfName=vrf, routeFilter=routeFilter,
                                      flattened=flattened )
         printer.endDict( p, vrf )
         if not keepGoing:
            break
      printer.endDict( p, "vrfs" )
      printer.endRender( p )
      printer.deinitPrinter( p )
      return VrfIp6Routes

def showRoutePerVrf( p, mode, prefix=None, optionsSet=None, vrfName=DEFAULT_VRF,
                     routeFilter=None, flattened=False ):
   optionsDict = dict( optionsSet or [] )
   detail = 'detail' in optionsDict
   protocol = optionsDict.get( 'protocol' )
   longerPrefixes = 'longerPrefixes' in optionsDict
   quantify = 'quantify' in optionsDict.values()
   intf = None

   originalMask = True
   if prefix != None: # pylint: disable=singleton-comparison
      # pass down to c++ file so it knows whether the user specified a mask,
      # or we appended a /128 for them.  Used to print "Routing entry for %s".
      # handle the printing in c++ with cliprint
      # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
      if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
         originalMask = False
         prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   sys.stdout.flush()
   # create the outputFormat here, since we will not have access to mode later
   # down the stack, in routing.showRoute
   fmt = mode.session_.outputFormat()
   requestedRevision = mode.session_.requestedModelRevision()
   # In revision 3 of the VrfIp6Routes CAPI model (IraIp6Model.py) we moved
   # the Nexthop-Group submodel from the per-route level to the per-via level.
   # If revision 2 or below is requested, we "degrade" the JSON output by picking
   # the first Nexthop-Group that a route points to and putting it in the per-route
   # level Nexthop-Group submodel.
   nhgModelRevision = 3

   # revision 3 prints the nested tunnel representation for routes resolving over
   # tunnels. If a lower revision is requested, it will flatten the tunnel entries,
   # and print the final nexthop, interface and combined label stack
   flatTunnelRevision = 3

   # revision 4 adds iBGP and eBGP to RouteType6.
   # If a lower revision is requested, it will replace these types with
   # "BGP"
   ip6RouteTypeRevision = 4

   # revision 5 makes srTePolicy in cbfVias to be optional when RSVP CBF is used
   rsvpCbfViaRevision = 5

   json = not mode.session_.shouldPrint()
   degradeNhgModel = json and requestedRevision < nhgModelRevision
   flattenTunnelOverTunnel = json and requestedRevision < flatTunnelRevision
   degradeIp6RouteTypeModel = json and requestedRevision < ip6RouteTypeRevision
   degradeCbfViasModel = json and requestedRevision < rsvpCbfViaRevision

   keepGoing = routing6.showRoute(
      prefix, intf, protocol, detail, vrfName=vrfName,
      longerPrefixes=longerPrefixes, quantify=quantify, fmt=fmt,
      originalMask=originalMask, degradeNhgModel=degradeNhgModel,
      degradeIp6RouteTypeModel=degradeIp6RouteTypeModel,
      flattenTunnelOverTunnel=flattenTunnelOverTunnel,
      degradeCbfViasModel=degradeCbfViasModel, routeFilter=routeFilter,
      flattened=flattened )
   warnIfRoutingDisabledGeneral( mode, vrfName )
   sys.stdout.flush()
   return keepGoing

showRouteVrfNameExprFactory = VrfExprFactory( helpdesc='Show routes in a VRF',
                                              inclDefaultVrf=True )

class ShowIpv6RouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 route [ VRF ] [ PREFIX | ADDR | flattened ]
               [ OPTIONS ] [ NEXTHOP ]"""

   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'flattened': 'show flattened adjacencies',
            'OPTIONS': ShowIpv6RouteOptionalItemsCliExpr,
            'NEXTHOP': NexthopCliExpression
          }

   cliModel = VrfIp6Routes

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   @staticmethod
   def handler( mode, args ):
      return showRoute( mode, prefix=args.get( 'PREFIX' ),
                        optionsSet=args[ 'OPTIONS' ],
                        vrfName=args.get( 'VRF' ),
                        routeFilter=args.get( 'nexthop-matchers' ),
                        flattened=bool( args.get( 'flattened' ) ) )

BasicCli.addShowCommandClass( ShowIpv6RouteCmd )

hostAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'host',
                                     helpdesc='Show only Host routes' )

fecAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'fec',
                                       helpdesc='show route adjacencies' )

summaryAfterShowIpv6FecMatcher = CliMatcher.KeywordMatcher( 'summary',
                                   helpdesc='show adjacency size and distribution' )

class ShowIpv6RouteHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] host [ quantify ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'host': hostAfterShowIpv6RouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = VrfIp6RoutesHost

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showRouteHost( mode, args.get( 'VRF' ), quantify=quantify )

BasicCli.addShowCommandClass( ShowIpv6RouteHostCmd )

class ShowIpv6RouteFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] [ PREFIX | ADDR ] fec'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'fec': fecAfterShowIpv6RouteMatcher
          }

   cliModel = IraRouteCommon.VrfFecRoutesModel
   handler = "IraIp6CliHandler.handlerShowIpv6RouteFecCmd"

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

BasicCli.addShowCommandClass( ShowIpv6RouteFecCmd )

class ShowIpv6FecSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] fec summary'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'fec': fecAfterShowIpv6RouteMatcher,
            'summary': summaryAfterShowIpv6FecMatcher
            }
   cliModel = IraRouteCommon.FecSummaryModel
   handler = "IraIp6CliHandler.handlerShowIpv6FecSummaryCmd"

BasicCli.addShowCommandClass( ShowIpv6FecSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route next-hop nexthop-group" command
#-------------------------------------------------------------------------------

class ShowIpv6RouteNextHopGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] next-hop nexthop-group [ NHG_NAME ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'next-hop': 'Filter output by next hop',
            'nexthop-group': 'Show only via next hop group',
            'NHG_NAME': nexthopGroupNameMatcher
   }
   cliModel = VrfIp6Routes
   handler = "IraIp6CliHandler.handlerShowIpv6RouteNextHopGroupCmd"

BasicCli.addShowCommandClass( ShowIpv6RouteNextHopGroupCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route static not-installed bfd" command
#-------------------------------------------------------------------------------

class ShowIpv6NotInstalledBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] static not-installed bfd'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'static': showIpv6StaticMatcher,
            'not-installed': IraCommonCli.notInstalledMatcher,
            'bfd': IraCommonCli.bfdMatcher
          }
   cliModel = VrfNotInstalledBfd
   handler = "IraIp6CliHandler.handlerShowIpv6NotInstalledBfdCmd"

BasicCli.addShowCommandClass( ShowIpv6NotInstalledBfdCmd )

def vrfSummaryCommon( mode, ip6RouteSummaryModel, vrfs ):
   routeCountByTypeDict = { 'connected': 0,
                            'static': 0,
                            'staticPersistent': 0,
                            'staticNonPersistent': 0,
                            'vcs': 0,
                            'staticNexthopGroup': 0,
                            'internal': 0,
                            'attached': 0,
                            'dhcp': 0,
                            'dynamicPolicy': 0,
                            'gribi': 0,
                            'aggregate': 0,
                            'ospfTotal': 0,
                            'ospfIntraArea': 0,
                            'ospfInterArea': 0,
                            'ospfExternal1': 0,
                            'ospfExternal2': 0,
                            'nssaExternal1': 0,
                            'nssaExternal2': 0,
                            'bgpTotal': 0,
                            'bgpInternal': 0,
                            'bgpExternal': 0,
                            'bgpLocal': 0,
                            'isisTotal': 0,
                            'isisLevel1': 0,
                            'isisLevel2': 0,
                            # Below keys will be populated and used when
                            # toggleOspf3RouteTypesDisplay is enabled.
                            'ospfv3Total': 0,
                            'ospfv3IntraArea': 0,
                            'ospfv3InterArea': 0,
                            'ospfv3External1': 0,
                            'ospfv3External2': 0,
                            'ospfv3NssaExternal1': 0,
                            'ospfv3NssaExternal2': 0}

   useAllVrfRouteSummary = (
      getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent and
      allVrfRouteSummary and allVrfRouteSummary.vrfRouteSummary )

   for vrfName in vrfs:
      if useAllVrfRouteSummary:
         vrfRouteSummaryMultiAgent( vrfName, ip6RouteSummaryModel,
                                    routeCountByTypeDict )
      else:
         vrfRouteSummarySingleAgent( vrfName, ip6RouteSummaryModel,
                                     routeCountByTypeDict )
      # This is done to reflect the special case of 'fe80::/10' route which
      # is listed as connnected route in 'show ipv6 route detail' but not
      # as receive route in 'show ipv6 route host' even though its
      # routeType is receive
      if Arnet.Ip6Prefix( 'fe80::/10' ) in routing6.rStatus( vrfName=vrfName ).route:
         routeCountByTypeDict[ 'connected' ] += 1
         routeCountByTypeDict[ 'internal' ] -= 1

   if useAllVrfRouteSummary:
      # CountByType totals
      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpInternal' ]
      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpExternal' ]
      routeCountByTypeDict[ 'bgpTotal' ] += routeCountByTypeDict[ 'bgpLocal' ]

      routeCountByTypeDict[ 'isisTotal' ] += routeCountByTypeDict[ 'isisLevel1' ]
      routeCountByTypeDict[ 'isisTotal' ] += routeCountByTypeDict[ 'isisLevel2' ]
      
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3InterArea' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3IntraArea' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3External1' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3External2' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3NssaExternal1' ]
      routeCountByTypeDict[ 'ospfv3Total' ] += \
         routeCountByTypeDict[ 'ospfv3NssaExternal2' ]

   ip6RouteSummaryModel.connected = routeCountByTypeDict[ 'connected' ]
   ip6RouteSummaryModel.static = routeCountByTypeDict[ 'static' ]
   ip6RouteSummaryModel.staticPersistent = routeCountByTypeDict[ 'staticPersistent' ]
   ip6RouteSummaryModel.staticNonPersistent = \
         routeCountByTypeDict[ 'staticNonPersistent' ]
   ip6RouteSummaryModel.vcs = routeCountByTypeDict[ 'vcs' ]
   ip6RouteSummaryModel.staticNexthopGroup = \
       routeCountByTypeDict[ 'staticNexthopGroup' ]
   ip6RouteSummaryModel.internal = routeCountByTypeDict[ 'internal' ]
   ip6RouteSummaryModel.attached = routeCountByTypeDict[ 'attached' ]
   ip6RouteSummaryModel.aggregate = routeCountByTypeDict[ 'aggregate' ]
   ip6RouteSummaryModel.dhcp = routeCountByTypeDict[ 'dhcp' ]
   ip6RouteSummaryModel.dynamicPolicy = routeCountByTypeDict[ 'dynamicPolicy' ]
   ip6RouteSummaryModel.gribi = routeCountByTypeDict[ 'gribi' ]

   ospfTotalsModel = ospfTotals6()
   ospfTotalsModel.ospfTotal = routeCountByTypeDict[ 'ospfTotal' ]
   ospfTotalsModel.ospfIntraArea = routeCountByTypeDict[ 'ospfIntraArea' ]
   ospfTotalsModel.ospfInterArea = routeCountByTypeDict[ 'ospfInterArea' ]
   ospfTotalsModel.ospfExternal1 = routeCountByTypeDict[ 'ospfExternal1' ]
   ospfTotalsModel.ospfExternal2 = routeCountByTypeDict[ 'ospfExternal2' ]
   ospfTotalsModel.nssaExternal1 = routeCountByTypeDict[ 'nssaExternal1' ]
   ospfTotalsModel.nssaExternal2 = routeCountByTypeDict[ 'nssaExternal2' ]
   if toggleOspf3RouteTypesDisplayEnabled():
      ospfTotalsModel.ospfTotal = routeCountByTypeDict[ 'ospfv3Total' ]
      ospfTotalsModel.ospfIntraArea = routeCountByTypeDict[ 'ospfv3IntraArea' ]
      ospfTotalsModel.ospfInterArea = routeCountByTypeDict[ 'ospfv3InterArea' ]
      ospfTotalsModel.ospfExternal1 = routeCountByTypeDict[ 'ospfv3External1' ]
      ospfTotalsModel.ospfExternal2 = routeCountByTypeDict[ 'ospfv3External2' ]
      ospfTotalsModel.nssaExternal1 = \
         routeCountByTypeDict[ 'ospfv3NssaExternal1' ]
      ospfTotalsModel.nssaExternal2 = \
         routeCountByTypeDict[ 'ospfv3NssaExternal2' ]
   ip6RouteSummaryModel.ospfCounts6 = ospfTotalsModel

   bgpTotalsModel = bgpTotals6()
   bgpTotalsModel.bgpTotal = routeCountByTypeDict[ 'bgpTotal' ]
   bgpTotalsModel.bgpExternal = routeCountByTypeDict[ 'bgpExternal' ]
   bgpTotalsModel.bgpInternal = routeCountByTypeDict[ 'bgpInternal' ]
   bgpTotalsModel.bgpLocal = routeCountByTypeDict[ 'bgpLocal' ]
   ip6RouteSummaryModel.bgpCounts6 = bgpTotalsModel

   isisTotalsModel = isisTotals6()
   isisTotalsModel.isisTotal = routeCountByTypeDict[ 'isisTotal' ]
   isisTotalsModel.isisLevel1 = routeCountByTypeDict[ 'isisLevel1' ]
   isisTotalsModel.isisLevel2 = routeCountByTypeDict[ 'isisLevel2' ]
   ip6RouteSummaryModel.isisCounts6 = isisTotalsModel

   return ip6RouteSummaryModel

def vrfRouteSummarySingleAgent( vrfName, ip6RouteSummaryModel,
                                routeCountByTypeDict ):
   routeConfig = routing6.routeConfig( vrfName )
   assert routeConfig
   routeConfigDynamic = routing6.routeConfig( vrfName, dynamic=True )
   assert routeConfigDynamic
   route6Summary = Tac.newInstance( "Ira::Route6Summary",
                                routing6.config( vrfName ),
                                routeConfig,
                                routeConfigDynamic,
                                routing6.rStatus( vrfName ),
                                routing6.attachRouteStatus( vrfName ),
                                routing6.vrfIdMap(),
                                routing6.ip.ribBypassConfig,
                                vrfName )
   route6Summary.loadInternalStructures()
   route6Summary.loadAttachedRoutes()
   route6Summary.loadRouteCacheRoutes()

   maskLenCounter = defaultdict( int, ip6RouteSummaryModel.maskLen )
   for key, value in route6Summary.routeCountByMask.items():
      maskLenCounter[ key ] += value
   ip6RouteSummaryModel.maskLen = dict( maskLenCounter )

   fibSummaryTable = route6Summary.routeCountByType
   for source in routeCountByTypeDict:
      if source in fibSummaryTable:
         routeCountByTypeDict[ source ] += fibSummaryTable[ source ]

   ip6RouteSummaryModel.totalRoutes += route6Summary.totalNumRoutes

def vrfRouteSummaryMultiAgent( vrfName, ip6RouteSummaryModel,
                               routeCountByTypeDict ):
   # TODO BUG804766:
   # Remove this when routeCache is part of the aggregated allVrfRouteSummary
   routeConfig = routing6.routeConfig( vrfName )
   assert routeConfig
   routeConfigDynamic = routing6.routeConfig( vrfName, dynamic=True )
   assert routeConfigDynamic
   route6Summary = Tac.newInstance( "Ira::Route6Summary",
                                    routing6.config( vrfName ),
                                    routeConfig,
                                    routeConfigDynamic,
                                    routing6.rStatus( vrfName ),
                                    routing6.attachRouteStatus( vrfName ),
                                    routing6.vrfIdMap(),
                                    routing6.ip.ribBypassConfig,
                                    vrfName )
   route6Summary.loadRouteCacheRoutes()
   staticNonPersistent = route6Summary.loadStaticNonPersistentCounter()
   routeCountByTypeDict[ 'staticNonPersistent' ] += staticNonPersistent

   maskLenCounter = defaultdict( int, ip6RouteSummaryModel.maskLen )
   for key, value in route6Summary.routeCountByMask.items():
      maskLenCounter[ key ] += value
   
   # routeCache has only static routes
   oldFibSummaryByType = route6Summary.routeCountByType
   totalTypesCount = oldFibSummaryByType.get( 'static', 0 )
   routeCountByTypeDict[ 'static' ] += totalTypesCount
   # routeCache route is always staticPersistent
   routeCountByTypeDict[ 'staticPersistent' ] += totalTypesCount

   # Shark based approach starts from here
   vrfId = routing6.vrfId( vrfName )
   assert allVrfRouteSummary
   vrfRouteSummary = allVrfRouteSummary.vrfRouteSummary.get( vrfId )
   assert vrfRouteSummary

   maskLenSummary = vrfRouteSummary.v6MaskLen
   for key, maskLenInfo in maskLenSummary.items():
      maskLenCounter[ key ] += maskLenInfo.count

   fibSummaryTable = vrfRouteSummary.v6RouteCount
   enumTypes = Tac.Type( "Routing6::Fib::RouteType" ).attributes
   protocolCodeWordHelper = Tac.Type( 'Ira::ProtocolCodeWordHelper' )
   for enumName in enumTypes:
      fibType = Tac.enumValue( 'Routing6::Fib::RouteType', enumName )
      routeType = Tac.newInstance( 'Routing::RouteSummary::V6RouteType', fibType )
      if routeType in fibSummaryTable:
         typeCount = fibSummaryTable[ routeType ].count
         protocolTypeWord = (
            protocolCodeWordHelper.routeType6ToProtocolTypeWord( fibType ) )
         
         if enumName == 'staticConfig':
            routeCountByTypeDict[ 'staticPersistent' ] += (
               typeCount - staticNonPersistent )
            
         if protocolTypeWord in routeCountByTypeDict:
            routeCountByTypeDict[ protocolTypeWord ] += typeCount
            totalTypesCount += typeCount

   ip6RouteSummaryModel.maskLen = dict( maskLenCounter )
   ip6RouteSummaryModel.totalRoutes += totalTypesCount

#-------------------------------------------------------------------------------
# The "show ipv6 route [vrf <vrfname>] summary" command.
#-------------------------------------------------------------------------------
def showRouteSummary( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfsToShow = []

   if not ( vrfName is None or vrfExists( vrfName ) ):
      if vrfName != ALL_VRF_NAME:
         mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
         return None

   if vrfName == ALL_VRF_NAME:
      vrfsToShow = [ "default" ]
      vrfsToShow.extend( sorted( allVrfStatusLocal.vrf ) )
   else:
      vrfsToShow = [ vrfName ]

   svcRoutingProtoModelStatus = ServiceRoutingProtoModelStatus(
      operatingProtoModel=l3Config.protocolAgentModel,
      configuredProtoModel=l3Config.configuredProtocolAgentModel )
   ip6Summary = Ip6RouteSummary()

   ip6Summary.protoModelStatus = svcRoutingProtoModelStatus
   for vrf in vrfsToShow:
      result = showRouteSummaryForVrf( mode, vrfName=vrf,
                                       quantify=quantify )
      if result:
         ip6Summary.vrfs[ vrf ] = result

   return ip6Summary

def showRouteSummaryForVrf( mode, vrfName=None, quantify=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return None

   beforeSum = Tac.now()
   ip6RouteSummaryModel = Ip6RouteSummaryForVrf()
   # pylint: disable=protected-access
   vrfs = [ vrfName ]

   ip6RouteSummaryModel.totalRoutes = 0
   ip6RouteSummaryModel = vrfSummaryCommon( mode, ip6RouteSummaryModel, vrfs )
   if quantify:
      afterSum = Tac.now()
      ip6RouteSummaryModel._quantify = afterSum - beforeSum
   return ip6RouteSummaryModel

summaryAfterShowIpv6RouteMatcher = CliMatcher.KeywordMatcher( 'summary',
                                     helpdesc='IPv6 routing table summary' )

class ShowIpv6RouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route [ VRF ] summary [ quantify ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'summary': summaryAfterShowIpv6RouteMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = Ip6RouteSummary

   @staticmethod
   def handler( mode, args ):
      quantify = 'quantify' in args
      return showRouteSummary( mode, vrfName=args.get( 'VRF' ),
                               quantify=quantify )

BasicCli.addShowCommandClass( ShowIpv6RouteSummaryCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route (prefix) longer-prefixes" command.
#-------------------------------------------------------------------------------

class ShowIpv6RouteLongerPrefixesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ipv6 route [ VRF ] ( PREFIX | ADDR )
               longer-prefixes [ OPTIONS ]
            """
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'VRF': showRouteVrfNameExprFactory,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': CliCommand.Node( ipv6AddrMatcher, alias='PREFIX' ),
            'longer-prefixes': 'Match longer-prefixes',
            'OPTIONS': ShowIpv6RouteOptionalItemsCliExpr
          }
   cliModel = VrfIp6Routes
   handler = "IraIp6CliHandler.handlerShowIpv6RouteLongerPrefixesCmd"

BasicCli.addShowCommandClass( ShowIpv6RouteLongerPrefixesCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route vrf all summary brief " command.
#-------------------------------------------------------------------------------

allKwMatcher = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
                                          helpdesc='Show summary of all vrfs' )

vrfForShowMatcher = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Specify the vrf' )

class ShowIpv6RouteSummaryBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route vrf all summary brief [ quantify ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'vrf': vrfForShowMatcher,
            'all': allKwMatcher,
            'summary': summaryAfterShowIpv6RouteMatcher,
            'brief': briefKwMatcher,
            'quantify': quantifyKwMatcher
          }
   cliModel = Ip6RouteSummaryForVrf
   handler = "IraIp6CliHandler.handlerShowIpv6RouteSummaryBriefCmd"

BasicCli.addShowCommandClass( ShowIpv6RouteSummaryBriefCmd )

#-------------------------------------------------------------------------------
#[no|default] ipv6 hardware fib aggregate-address <pfx> summary-only software-forward
#-------------------------------------------------------------------------------

summaryOnlyKwMatcher = CliMatcher.KeywordMatcher( 'summary-only',
      helpdesc='Filters all more-specific routes from hardware routing table' )
softwareForwardMatcherForConfig = CliMatcher.KeywordMatcher( 'software-forward',
      helpdesc='Software Forward the packets' )
aggPrefixForConfigMatcher = CliMatcher.KeywordMatcher( 'aggregate-address',
   helpdesc='Configure aggregate address' )

class Ipv6FibAggregateAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib aggregate-address PREFIX ' \
            'summary-only software-forward'
   noOrDefaultSyntax = 'ipv6 hardware fib aggregate-address PREFIX ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'aggregate-address': aggPrefixForConfigMatcher,
            'PREFIX': ipv6PrefixMatcher,
            'summary-only': summaryOnlyKwMatcher,
            'software-forward': softwareForwardMatcherForConfig
          }

   handler = "IraIp6CliHandler.handlerIpv6FibAggregateAddrCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerIpv6FibAggregateAddrCmd"

configMode.addCommandClass( Ipv6FibAggregateAddrCmd )

#-------------------------------------------------------------------------------
#show ipv6 hardware fib aggregate-address [<pfx>] [software-forward]
#-------------------------------------------------------------------------------

softwareForwardForShowMatcher = CliMatcher.KeywordMatcher( 'software-forward',
      helpdesc='Software Forwarded packets' )
aggPrefixForShowMatcher = CliMatcher.KeywordMatcher( 'aggregate-address',
   helpdesc='Display aggregate address' )

class ShowIpv6FibAggregateAddrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib aggregate-address ' \
            '[ PREFIX | ADDR ] [ software-forward ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'hardware': CliToken.Hardware.hardwareForShowMatcher,
            'fib': IraCommonCli.fibMatcher,
            'aggregate-address': aggPrefixForShowMatcher,
            'PREFIX': ipv6PrefixMatcher,
            'ADDR': ipv6AddrMatcher,
            'software-forward': softwareForwardForShowMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   handler = "IraIp6CliHandler.handlerShowIpv6FibAggregateAddrCmd"

BasicCli.addShowCommandClass( ShowIpv6FibAggregateAddrCmd )

#-------------------------------------------------------------------------------
# [no|default] ipv6 hardware fib optimize prefix-length <length>
#-------------------------------------------------------------------------------
def routesV6OptimizeGuard( mode, token ):
   if routing6HardwareStatus.prefixLenSupportedInExactMatch:
      return None
   return CliParser.guardNotThisPlatform

def vrfRoutesV6OptimizeGuard( mode, token ):
   if routing6HardwareStatus.vrfPrefixLensInExactMatchSupported:
      return None
   return CliParser.guardNotThisPlatform

optimizeMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'optimize',
                                      helpdesc='Optimize IPv6 routes' ),
                                   guard=routesV6OptimizeGuard )

def prefixV6LengthGuard( mode, token ):
   if int( token ) not in routing6HardwareStatus.prefixLenSupportedInExactMatch:
      return CliParser.guardNotThisPlatform
   return None

prefixLengthData = {}
for prefixLen in range( 0, 129 ):
   prefixLengthData[ str( prefixLen ) ] = 'Prefix length %d' % prefixLen

class OptimizeV6RoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib optimize prefix-length LENGTH'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'optimize': optimizeMatcher,
            'prefix-length': IraCommonCli.prefixLengthMatcher,
            'LENGTH': CliCommand.SetEnumMatcher( prefixLengthData,
               guard=prefixV6LengthGuard )
            }

   handler = "IraIp6CliHandler.handlerOptimizeV6RoutesCmd"

configMode.addCommandClass( OptimizeV6RoutesCmd )

# The noOrDefault command has been moved down

#-------------------------------------------------------------------------------
# "[no|default] ipv6 hardware fib optimize vrf (<vrf> [ <vrf2> ...])
#               prefix-length <length>"
#-------------------------------------------------------------------------------
class VrfOptimizeV6RoutesCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 hardware fib optimize " \
            "vrf { VRFS } " \
            "prefix-length LENGTH"

   noOrDefaultSyntax = "ipv6 hardware fib optimize vrf { VRFS } ..."

   data = mergedDict(
         OptimizeV6RoutesCmd.data,
         {
            "vrf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     "vrf", helpdesc="Optimize IPv6 routes for specific VRF(s)" ),
                  guard=vrfRoutesV6OptimizeGuard ),
            "VRFS": VrfCli.VrfNameExprFactory( inclDefaultVrf=True ),
         } )

   handler = "IraIp6CliHandler.handlerVrfOptimizeV6RoutesCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerVrfOptimizeV6RoutesCmd"

configMode.addCommandClass( VrfOptimizeV6RoutesCmd )

#-------------------------------------------------------------------------------
# [no|default] ipv6 hardware fib optimize prefixes profile internet
#-------------------------------------------------------------------------------
def supportedProfileTypeIpv6( mode ):
   return supportedProfileType( mode, routing6HardwareStatus )

class Ipv6OptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 hardware fib optimize prefixes profile PROFILE'

   noOrDefaultSyntax = 'ipv6 hardware fib optimize ...'

   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'optimize': optimizeMatcher,
            'prefixes': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'prefixes',
                        helpdesc='Optimize prefixes' ),
                  guard=routesV6OptimizeGuard ),
            'profile': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'profile',
                        helpdesc='Optimize based on profiles' ),
               guard=routesV6OptimizeGuard ),
            'PROFILE': CliMatcher.DynamicKeywordMatcher( supportedProfileTypeIpv6 )
          }

   handler = "IraIp6CliHandler.handlerIpv6OptimizePrefixesCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerIpv6OptimizePrefixesCmd"

configMode.addCommandClass( Ipv6OptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 hardware fib optimize vrf (<vrf> [ <vrf2> ...])
#               prefixes profile internet"
# command
#-------------------------------------------------------------------------------
class VrfIpv6OptimizePrefixesCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 hardware fib optimize " \
            "vrf { VRFS } " \
            "prefixes profile PROFILE"

   # The no command for the below commands are the same:
   # - "ipv6 hardware fib optimize vrf { VRFS } prefixes" AND
   # - "ipv6 hardware fib optimize vrf { VRFS } prefix-length"
   #
   # The no command is "no ipv6 hardware fib optimize vrf { VRFS }". The no
   # handler is defined in VrfOptimizeV6RoutesCmd class.

   data = mergedDict(
         Ipv6OptimizePrefixesCmd.data,
         {
            "vrf": CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     "vrf", helpdesc="Optimize IPv6 routes for specific VRF(s)" ),
                  guard=vrfRoutesV6OptimizeGuard ),
            "VRFS": VrfCli.VrfNameExprFactory( inclDefaultVrf=True ),
         } )

   handler = "IraIp6CliHandler.handlerVrfIpv6OptimizePrefixesCmd"

configMode.addCommandClass( VrfIpv6OptimizePrefixesCmd )

#-------------------------------------------------------------------------------
# Determine if assigning the given address/mask to some interface in Ira
# should be allowed
# return a pair of a boolean with a string, where the string is an error or
# warning message
#-------------------------------------------------------------------------------
def canSetIntfIp( intfName, ipAddrWithMask, mode=None ):
   # skip IPv6 address sanity check only during config replace
   if mode and mode.session_.skipConfigCheck():
      return [True, ""]
   # This gets the right vrf route config based on the intf
   config = routing6.routeConfig( vrfName=None, intfName=intfName )
   result = IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                               'ipv6', True )
   if result[ 1 ]:
      return result
   config = routing4.routeConfig( vrfName=None, intfName=intfName )
   if config is None:
      return result
   return IraRouteCommon.canSetIntfIpHelper( config, intfName, ipAddrWithMask,
                                             'ipv6', True )

#-------------------------------------------------------------------------------
# The Ipv6 Resilient Ecmp command
# The syntax of the command is:
#  ipv6 hardware fib ecmp <prefix> capacity <C> redundancy <R>
#  no/default ipv6 hardware fib ecmp <prefix>
#  no/default ipv6 hardware fib ecmp
#-------------------------------------------------------------------------------

def maxResilientEcmpIp6CapacityRangeFn( mode, context ):
   if ( routing6HardwareStatus is None or
        routing6HardwareStatus.maxLogicalProtocolEcmp == 0 ):
      return( 1, sys.maxsize )
   else :
      return( 2, routing6HardwareStatus.maxLogicalProtocolEcmp )

def maxResilientEcmpIp6RedundancyRangeFn( mode, context ):
   if routing6HardwareStatus is None or routing6HardwareStatus.maxResilientEcmp == 0:
      return( 1, sys.maxsize )
   else :
      return( 1, routing6HardwareStatus.maxResilientEcmp // 2 )

def resilientEcmpIp6Guard( mode, token ):
   if routing6HardwareStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

def orderedRecmpGuard( mode, token ):
   if routingHardwareStatus.orderedEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform


orderedRecmpHelpDesc = "Enforce order of next hops in FEC" \
                       " for this resilient configuration"
markedRecmpHelpDesc = "Applies to routes marked for ecmp resilience using policy"

class Ipv6ResilientEcmpCmd( CliCommand.CliCommandClass ):
   syntax = """ipv6 hardware fib ecmp resilience
            ( ( PREFIX capacity CAPACITY redundancy REDUNDANCY [ ordered ] )
            | ( marked capacity CAPACITY redundancy REDUNDANCY ) )"""
   noOrDefaultSyntax = 'ipv6 hardware fib ecmp resilience [ PREFIX | marked ]  ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'hardware': CliToken.Hardware.hardwareForConfigMatcher,
            'fib': IraCommonCli.fibMatcher,
            'ecmp': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ecmp',
                                       helpdesc='Configure ECMP routes'),
                                     guard=resilientEcmpIp6Guard ),
            'resilience': IraCommonCli.resilienceMatcher,
            'PREFIX': ipv6PrefixMatcher,
            "marked": CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                    "marked", helpdesc=markedRecmpHelpDesc ) ),
            'capacity': IraCommonCli.resilientEcmpCapacityMatcher,
            'CAPACITY': CliMatcher.DynamicIntegerMatcher(
                              rangeFn=maxResilientEcmpIp6CapacityRangeFn,
                              helpdesc='capacity value' ),
            'redundancy': IraCommonCli.resilientEcmpRedundancyMatcher,
            'REDUNDANCY': CliMatcher.DynamicIntegerMatcher(
                              rangeFn=maxResilientEcmpIp6RedundancyRangeFn,
                              helpdesc='redundancy value' ),
            "ordered": CliCommand.guardedKeyword(
                                    "ordered",
                                    helpdesc=orderedRecmpHelpDesc,
                                    guard=orderedRecmpGuard,
                                    hidden=False )
          }

   handler = "IraIp6CliHandler.handlerIpv6ResilientEcmpCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerIpv6ResilientEcmpCmd"

configMode.addCommandClass( Ipv6ResilientEcmpCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 route rpf unicast [ prefix ]" command.
#-------------------------------------------------------------------------------

class ShowIpv6UrpfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 route rpf unicast [ VRF ] [ PREFIX | ADDR ]'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'route': routeAfterShowIpv6Matcher,
            'rpf': 'Reverse Path Forwarding information',
            'unicast': 'Unicast Reverse Path Forwarding information',
            'VRF': VrfExprFactory( helpdesc='Show Unicast Reverse Path '
                                            'Forwarding information in a VRF',
                                   guard=ipV6Vrf,
                                   inclDefaultVrf=True ),
            'PREFIX': Ip6AddrMatcher.Ip6PrefixMatcher( 'Destination prefix' ),
            'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Destination address' )
          }

   cliModel = Ip6UrpfInterfaces

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PREFIX' ] = args.pop( 'ADDR' )

   handler = "IraIp6CliHandler.handlerShowIpv6UrpfCmd"

BasicCli.addShowCommandClass( ShowIpv6UrpfCmd )

myEntManager = None

#-------------------------------------------------------------------------------
# The "ipv6 fib compression redundant-specifics filter" and
# "[no|default] ip fib compression redundant-specifics filter"
# commands.
#-------------------------------------------------------------------------------
def fibSuppressionSupportedGuard( mode, token ):
   rhs = routing6HardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing6' ][ 'hardware' ][ 'status' ]
   if rhs.fibSuppressionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

compressionMatcher = CliCommand.Node( CliMatcher.KeywordMatcher( 'compression',
                                          helpdesc='FIB compression' ),
                                      guard=fibSuppressionSupportedGuard )

class Ipv6FibFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 fib compression redundant-specifics filter'
   noOrDefaultSyntax = 'ipv6 fib compression redundant-specifics filter'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
            'fib': IraCommonCli.fibMatcher,
            'compression': compressionMatcher,
            'redundant-specifics': IraCommonCli.redundantSpecificMatcher,
            'filter': IraCommonCli.filterMatcher }

   handler = "IraIp6CliHandler.handlerIpv6FibFilterCmd"
   noOrDefaultHandler = "IraIp6CliHandler.noOrDefaultHandlerIpv6FibFilterCmd"

configMode.addCommandClass( Ipv6FibFilterCmd )

#-------------------------------------------------------------------------------
# The "show ipv6 user route" command
#-------------------------------------------------------------------------------
class ShowIpv6UserRouteCmd( ShowCommand.ShowCliCommandClass ):
   '''Display routes from the RIB that have the 'dynamic' attribute set to True
      These routes have been programmed via the EOS SDK with the 'persistent' flag
      set to False.'''
   syntax = 'show ipv6 user route [ vrf VRF ] [ tag TAGNUM ] [ CONFIG_TAG_EXPR ]'
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
         'user': 'Display non-persistent routes',
         'route': 'Static routes',
         'vrf': vrfForShowMatcher,
         'VRF': vrfNameMatcher,
         'tag': tagMatcherForConfig,
         'TAGNUM': tagNumberMatcher,
         'CONFIG_TAG_EXPR': configTagExpr
         }
   cliModel = VrfIpRibRoutes
   handler = "IraIp6CliHandler.handlerShowIpv6UserRouteCmd"

BasicCli.addShowCommandClass( ShowIpv6UserRouteCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHardwareStatusCommon
   global routingHardwareStatus, routing6HardwareStatus
   global allVrfConfig
   global allVrfStatusLocal
   global l3Config
   global myEntManager
   global allIntfStatusDir
   global allVrfRouteSummary

   myEntManager = entityManager

   routing4.plugin( entityManager )
   routing6.plugin( entityManager )
   allVrfConfig = ConfigMount.mount( entityManager, "ip/vrf/config",
                                     "Ip::AllVrfConfig", "w" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                       Cell.path( "ip/vrf/status/local" ),
                                       "Ip::AllVrfStatusLocal", "r" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager, "routing/hardware/statuscommon",
      "Routing::Hardware::StatusCommon", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
         "routing6/hardware/status", "Routing6::Hardware::Status", "r" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", "r" )
   if toggleMixedRouteSummaryOutputEnabled():
      allVrfRouteSummary = SharkLazyMount.mount(
                              entityManager,
                              "routing/routeSummary/aggregated",
                              "Routing::RouteSummary::AllVrfRouteSummary",
                              SharkLazyMount.mountInfo( "shadow" ),
                              autoUnmount=True,
                              unmountTimeout=600 )
      
   IraIp6IntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )

   registerShowTechSupportCmd(
      '2012-08-22 09:22:00',
      cmds=[ 'show ipv6 route vrf all detail',
             'show ipv6 interface',
             'show ipv6 route vrf all host' ] )
   registerShowTechSupportCmd(
      '2012-08-22 09:22:00',
      cmds=[ 'show kernel ipv6 route vrf all',
             'show kernel ipv6 interface addr vrf all',
             'show kernel ipv6 acl vrf all' ],
      cmdsGuard=lambda: routingHardwareStatusCommon.vrfCapability.\
      ipv6EnabledDefault )

   # Timestamps are made up to maintain historical order within show tech-support
   registerShowTechSupportCmd(
      '2017-04-07 09:22:00',
      cmds=[ 'show ipv6 route vrf all summary' ],
      summaryCmds=[ 'show ipv6 interface brief',
                    'show ipv6 route vrf all summary' ] )
