# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=cell-var-from-loop

import Arnet
import CliMatcher
import Ethernet
from CliToken.Community import(
   getCommIntValue,
)
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IraCommonCli
from CliPlugin import IraRouteCommon
from CliPlugin.IraNexthopGroupCli import nexthopGroupSupportedGuard
from IpLibConsts import DEFAULT_VRF
from RoutingConsts import (
      defaultStaticRouteName,
      defaultStaticRoutePreference,
      defaultStaticRouteTag,
      defaultStaticRouteMetric,
      defaultStaticRouteWeight
)
import Tac

routeMatcherForIpv6Config = CliMatcher.KeywordMatcher( 'route',
      helpdesc='Manage static IPv6 routes' )
ipv6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( "Destination prefix" )

ip6 = IraRouteCommon.Ip6()
routing6 = IraRouteCommon.routing( ip6 )

noIpv6RouteTableForVrfMsg = "IPv6 Routing table for VRF %s does not exist."

# dynamic and non-dynamic static routes have separate routeConfigs
def staticIpv6RoutingTable( vrfName=DEFAULT_VRF, dynamic=False ):
   return routing6.routeConfig( vrfName, dynamic=dynamic )

def isValidIpv6PrefixWithError( mode, prefix ):
   """Returns True if a prefix is valid.  If invalid, it addas a CLI error
   and returns False."""

   if not prefix.validAsPrefix:
      mode.addError( "Host part of destination prefix must be zero" )
      return False

   return True

def manageIpv6Route( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                     leakToVrf=None, egressVrf=None, configTag=False,
                     configTagStr=None, viaResRibProfile=None ):
   if not isValidIpv6PrefixWithError( mode, prefix ):
      return

   # TODO We're unnecessarily matching an Ip6AddrWithMask with the CLI
   # matcher, then manually converting it to an Ip6Prefix here. This
   # is wasteful, and is presumably done so that we can allow invalid
   # prefixes to be transparently converted (masking off the host
   # bits). When BUG35676 is fixed, this can be refactored
   prefix = Arnet.Ip6Prefix( prefix.stringValue )

   if prefix.address.isMulticast:
      mode.addError( "Destination prefix must be unicast" )
      return

   if not mode.session_.startupConfig() and 'nexthopGroupName' in nexthop and \
         nexthopGroupSupportedGuard( mode, None ) is not None:
      mode.addError( "Nexthop-Group not supported on this hardware platform" )
      return

   if routeOptions is None:
      routeOptions = {}
   preference = routeOptions.get( 'preference', defaultStaticRoutePreference )
   tag = routeOptions.get( 'tag', defaultStaticRouteTag )
   community = routeOptions.get( 'community' )
   nexthopName = routeOptions.get( 'nextHopName', defaultStaticRouteName )
   trackingProto = routeOptions.get( 'trackingProto' )
   localAddress = routeOptions.get( 'localAddress', {} )
   localAddr = localAddress.get( 'localAddr6' )
   metric = routeOptions.get( 'metric', defaultStaticRouteMetric )
   weight = routeOptions.get( 'weight', defaultStaticRouteWeight )

   routeAttributes = Tac.Value( 'Routing::RouteAttributes' )
   if community is not None:
      if len( community ) > IraCommonCli.COMMUNITY_LIMIT:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "You can only add up to {} communities".format(
            IraCommonCli.COMMUNITY_LIMIT ) )
         return
      communityValues = [ getCommIntValue( value ) for value in community ]
      for commValue in sorted( set( communityValues ) ):
         routeAttributes.community.push( commValue )

   label = None
   vtepAddr = None
   vni = None
   routerMac = None
   vxlanSrcIntf = None
   dynamic = False

   # nexthop is actually either a rule with mpls, evpn, nexthop or an intf,
   # and it is a dict, produced by a Cli OrRule.
   routeType = 'forward'
   if 'mpls' in nexthop:
      hop = nexthop[ 'mpls' ][ 'nexthop' ]
      label = nexthop[ 'mpls' ][ 'label' ]

      validNextHop = IraCommonCli.validNextHopGen
      if isinstance( hop, Tac.Type( "Arnet::Ip6Addr" ) ):
         validNextHop = IraCommonCli.validNextHop6
      if not validNextHop( mode, hop, vrfName, egressVrfName=egressVrf ):
         return
      intf = None
   elif 'evpn' in nexthop:
      hop = Arnet.Ip6Addr( '::' )
      vtepAddr = nexthop[ 'evpn' ][ 'vtepAddr' ]
      vni = nexthop[ 'evpn' ][ 'vni' ]
      routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
      vxlanSrcIntf = nexthop[ 'evpn' ][ 'vxlanSrcIntf' ]
      dynamic = nexthop[ 'evpn' ][ 'dynamic' ]
      ribBypass = nexthop[ 'evpn' ][ 'ribBypass' ]
      vtepSipValidation = nexthop[ 'evpn' ][ 'source-vtep-validation' ]
      intf = None
   elif 'nexthop' in nexthop:
      hop = nexthop[ 'nexthop' ]

      validNextHop = IraCommonCli.validNextHopGen
      if isinstance( hop, Tac.Type( "Arnet::Ip6Addr" ) ):
         validNextHop = IraCommonCli.validNextHop6
      if not validNextHop( mode, hop, vrfName, egressVrfName=egressVrf ):
         return
      intf = None
   elif 'Null0' in nexthop or 'Null 0' in nexthop:
      routeType = 'drop'
   elif 'nexthopGroupName' in nexthop:
      routeType = 'nexthopGroup'
      nexthopGroupName = nexthop[ 'nexthopGroupName' ]
   else:
      nhi = nexthop[ 'intf' ][ 'intf' ]
      hop = ip6.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
      if hop:
         validNextHop = IraCommonCli.validNextHopGen
         if isinstance( hop, Tac.Type( "Arnet::Ip6Addr" ) ):
            validNextHop = IraCommonCli.validNextHop6
         if not validNextHop( mode, hop, vrfName, nhi, egressVrfName=egressVrf ):
            return
      else:
         hop = Arnet.Ip6Addr( '::' )

      if not nhi.config():
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Interface %s does not exist" % nhi.name )
         return

      if not nhi.routingSupported():
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Interface %s is not routable" % nhi.name )
         return

      intf = ip6.config.intf.get( nhi.name )
      if not intf:
         intf = ip6.config.newIntf( nhi.name )
         intf.l3Config = ip6.l3ConfigDir.newIntfConfig( nhi.name )

   key = Tac.Value( "Routing::RouteKey",
                    prefix=Arnet.IpGenPrefix( prefix.stringValue ),
                    preference=preference )

   rt = staticIpv6RoutingTable( vrfName, dynamic=dynamic )
   r = rt.route.get( key )
   if r:
      if ( ( r.routeType != 'nexthopGroup' and routeType == 'nexthopGroup' ) or
           ( routeType != 'nexthopGroup' and r.routeType == 'nexthopGroup' ) ):
         # pylint: disable-next=consider-using-f-string
         mode.addError( "Route %s already exists, cannot ECMP with nexthop group "
                        "and non-nexthop group routes " % prefix )
         return
      if( ( routeType == 'drop' or r.routeType == 'drop' ) and
          routeType != r.routeType ):
         mode.addError( "Cannot ECMP to Null0 interface." )
         return
   else:
      r = rt.route.newMember( key, routeType )
   r.tag = tag

   # `routeAttributes` are stored in a dynOptional, prevent the allocation of the
   # `routeAttributes` if the object is in its default state (no communities added).
   if routeAttributes != Tac.Value( 'Routing::RouteAttributes' ):
      r.routeAttributes = routeAttributes
   else:
      r.routeAttributes = None

   if leakToVrf:
      r.leakToVrf = leakToVrf
   else:
      r.leakToVrf = ''

   if routeType == 'forward':
      hop = Arnet.IpGenAddr( str( hop ) )
      if intf:
         via = Tac.Value( "Routing::Via", hop=hop, intfId=intf.intfId )
      else:
         via = Tac.Value( "Routing::Via", hop=hop, intfId='' )
      # If egress VRF is the same as the VRF route is pointing to, there is no
      # meaning setting egress VRF in via
      if egressVrf is not None and egressVrf != vrfName:
         via.egressVrf = egressVrf
      if label != None: # pylint: disable=singleton-comparison
         via.mplsLabel = Tac.Value( "Arnet::MplsLabel", int( label[ 0 ] ) )

      if vtepAddr is not None:
         if vni is None or routerMac is None:
            mode.addError( "Static EVPN routes must specify VNI and "
                           "router-mac-address " )
            return

         vtepAddr = Arnet.IpGenAddr( vtepAddr )
         vni = int( vni )
         routerMac = Ethernet.convertMacAddrToCanonical( routerMac )
         via.vtepAddr = vtepAddr
         via.vni = vni
         via.routerMac = routerMac
         via.vxlanIntf = vxlanSrcIntf

         r.ribBypass = ribBypass

      if configTag:
         if configTagStr:
            IraCommonCli.manageStaticEvpnRouteConfigTag( mode, prefix.stringValue,
               preference, vrfName, dynamic, rt, tagStr=configTagStr, af='ipv6' )
         else:
            # If a route exists with a configTag association, delete it
            IraCommonCli.manageStaticEvpnRouteConfigTag( mode, prefix.stringValue,
               preference, vrfName, dynamic, rt, af='ipv6' )

      maxEcmp = None
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
      if rhs is not None and rhs.maxLogicalProtocolEcmp > 0:
         maxEcmp = rhs.maxLogicalProtocolEcmp

      if maxEcmp and len( r.via ) == maxEcmp and via not in r.via:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "A maximum of %d routes are allowed "
                        "with the same prefix and preference" % maxEcmp )
         return

      if vtepAddr is not None:
         via.vtepSipValidation = not vtepSipValidation
         del r.via[ via ]
         via.vtepSipValidation = vtepSipValidation

      if trackingProto == 'bfd' and localAddr is not None:
         via.localAddrOpt = Arnet.IpGenAddr( str( localAddr ) )

      # BFD tracking is a property of the nexthop.
      # BFD tracked vias are kept in the bfdTracked collection while
      # local-address is in the via.
      # For a given nexthop, if there is a bfdTracked via it should be
      # the only via for the nexthop.
      if trackingProto == 'bfd':
         for v in r.via:
            if v.hop == via.hop and v != via:
               del r.via[ v ]
               del r.viaNextHopName[ v ]
               del r.bfdTracked[ v ]
      else:
         for v in r.bfdTracked:
            if v.hop == via.hop:
               del r.via[ v ]
               del r.viaNextHopName[ v ]
               del r.bfdTracked[ v ]

      routeViaAttr = Tac.Value( "Routing::RouteViaAttr", metric, weight )
      r.via[ via ] = routeViaAttr
      if viaResRibProfile:
         r.viaResRibProfileConfig[ via ] = viaResRibProfile
      else:
         del r.viaResRibProfileConfig[ via ]
      r.viaNextHopName[ via ] = nexthopName
      if trackingProto == 'bfd':
         r.bfdTracked[ via ] = True
      else:
         del r.bfdTracked[ via ]
   elif routeType == 'drop':
      r.dropNextHopName = nexthopName
   else:
      r.nexthopGroupName = nexthopGroupName
      r.nexthopGroupMetric = metric
      r.nexthopGroupWeight = weight

def noIpv6Route( mode, prefix, nexthop, preference, vrfName, egressVrf=None,
      configTag=False ):
   if not isValidIpv6PrefixWithError( mode, prefix ):
      return

   # If egress VRF is the same as the VRF route is pointing to, there is no
   # meaning setting egress VRF in via
   if egressVrf is None or egressVrf == vrfName:
      egressVrf = ""

   prefix = Arnet.IpGenPrefix( prefix.stringValue )

   dynamic = nexthop[ 'evpn' ][ 'dynamic' ] \
         if 'evpn' in nexthop and 'dynamic' in nexthop[ 'evpn' ] else False
   rt = staticIpv6RoutingTable( vrfName, dynamic=dynamic )

   # optimization
   if preference:
      key = Tac.Value( "Routing::RouteKey", prefix=prefix, preference=preference )
      if key in rt.route:
         keys = [ key ]
      else:
         keys = []
   else:
      keys = list( rt.route )

   # Remove the command tag association if it exists. We do this before removing
   # the route from routeConfig since the tagId present in the route is required
   # to figure out the command tag association.
   if configTag:
      IraCommonCli.manageStaticEvpnRouteConfigTag( mode, prefix.stringValue,
            preference, vrfName, dynamic, rt, af='ipv6' )

   for key in keys:
      if key.prefix != prefix:
         continue

      if ( preference and preference != key.preference ):
         continue

      if not nexthop:
         # no nexthop or intf specified. Delete every route with this prefix.
         del rt.route[ key ]
         continue

      if 'Null0' in nexthop or 'Null 0' in nexthop:
         if rt.route[ key ].routeType == 'drop':
            del rt.route[ key ]
         continue

      def delRoutes( matching ):
         r = rt.route.get( key )
         rvias = list( r.via )
         for v in rvias:
            if matching( v ):
               if len( rvias ) == 1:
                  del rt.route[ key ]
               else:
                  del r.via[ v ]
                  del r.viaNextHopName[ v ]
                  del r.bfdTracked[ v ]

      nullLabel = Tac.Type( 'Arnet::MplsLabel' ).null
      if 'intf' in nexthop:
         intf = nexthop[ 'intf' ][ 'intf' ]
         hop = ip6.intfAddrRoutesSupported and nexthop[ 'intf' ][ 'intfnexthop' ]
         if not hop:
            hop = Arnet.IpGenAddr( '::' )
         else:
            hop = Arnet.IpGenAddr( str( hop ) )
         delRoutes( lambda via: via.intfId == intf.name and via.hop == hop
                    and via.mplsLabel == nullLabel )
      elif 'nexthopGroupName' in nexthop:
         r = rt.route.get( key )
         if ( r.routeType == 'nexthopGroup' and
              r.nexthopGroupName == nexthop[ 'nexthopGroupName' ] ):
            del rt.route[ key ]
      elif 'mpls' in nexthop:
         hop = Arnet.IpGenAddr( str( nexthop[ 'mpls' ][ 'nexthop' ] ) )
         label = nexthop[ 'mpls' ][ 'label' ]
         delRoutes( lambda via: via.hop == hop and via.mplsLabel == label
                    and via.intfId == '' )
      elif 'evpn' in nexthop:
         vtepAddr = nexthop[ 'evpn' ].get( 'vtepAddr', None )
         if vtepAddr is None:
            if configTag:
               # command-tags are configured on a per-routeKey basis and
               # not per via/nexthop.
               # ip route <prefix> vtep 1.1.1.1 ... command-tag TAG
               # ip route <prefix> vtep 2.2.2.2 ... command-tag TAG2
               # no command-tag TAG2 all-configuration
               # The last command should delete both the above routes.
               del rt.route[ key ]
            else:
               mode.addError( "VtepAddr not provided for evpn next hop" )
               return
         else:
            vtepAddr = Tac.Value( 'Arnet::IpGenAddr', vtepAddr )
            vni = nexthop[ 'evpn' ][ 'vni' ]
            vni = int( vni )
            routerMac = nexthop[ 'evpn' ][ 'routerMac' ]
            routerMac = Ethernet.convertMacAddrToCanonical( routerMac )
            localIntf = nexthop[ 'evpn' ][ 'local-intf' ]
            delRoutes( lambda via: via.intfId == '' and
                       via.vni == vni and via.vtepAddr == vtepAddr and
                       via.routerMac == routerMac and
                       via.vxlanIntf == localIntf )
      else:
         hop = Arnet.IpGenAddr( str( nexthop[ 'nexthop' ] ) )
         delRoutes( lambda via: via.hop == hop and via.intfId == ''
                    and via.mplsLabel == nullLabel and via.egressVrf == egressVrf )
